/*
 * INSDROP.C
 *
 * Object creation functions for Insert Object and FCreateFromDropFiles.
 *
 * FCreateFromDropFiles creates embedded Packager objects from files
 * dropped from File Manager.
 *
 * Copyright(c) Microsoft Corp. 1992 All Rights Reserved
 */

#include <windows.h>
#include <ole.h>
#include <shellapi.h>
#include "oclient.h"
#include "blackbox.h"
#include "patron.h"


/*
 * FEditInsertObject
 *
 * Purpose:
 *  Invokes the Insert Object dialog handler that shows the dialog
 *  and creates an object if the user selects a valid class name.
 *  We need to create an object name before calling the dialog, and
 *  on return we create a BlackBox window for that object.
 *
 * Parameters:
 *  hWnd            HWND to use as the parent of the dialog.
 *  hInst           HANDLE to the application instance.
 *  pDoc            LPDOCUMENT that contains OLE information.
 *
 * Return Value:
 *  BOOL            TRUE if an object was created, FALSE otherwise.
 */

BOOL FAR PASCAL FEditInsertObject(HWND hWnd, HANDLE hInst, LPDOCUMENT pDoc)
    {
    LPOBJECT        pObj;
    FILEOBJECT      fo;
    HWND            hWndT;

    if (NULL==pDoc || NULL==hInst)
        return FALSE;

    FileObjectDefaults(&fo);

    pObj=PObjectInsertDialog(hWnd, hInst, pDoc, fo.szName);

    if (NULL==pObj)
        return FALSE;

    hWndT=HBlackBoxCreate(hWnd, &fo, TRUE, pObj);

    if (NULL==hWndT)
        return FALSE;

    return TRUE;
    }






/*
 * FCreateFromDropFiles
 *
 * Purpose:
 *  Creates embedded OLE Packager objects from dropped files in
 *  response to a WM_DROPFILES message.
 *
 * Parameters:
 *  hWnd            HWND on which the files were dropped.
 *  hDrop           HANDLE identifying the dropped files.
 *  pDoc            LPDOCUMENT owner of all OLE objects in this application.
 *
 * Return Value:
 *  BOOL            TRUE if objects were ALL created successfully, FALSE
 *                  otherwise.  Returning FALSE does not mean that some
 *                  objects were not created.
 *
 */

BOOL FAR PASCAL FCreateFromDropFiles(HWND hWnd, HANDLE hDrop, LPDOCUMENT pDoc)
    {
    LPOBJECT        pObj;
    FILEOBJECT      fo;
    HWND            hWndT;
    char            szPath[CCHPATHMAX];
    WORD            cFiles;
    WORD            i;
    BOOL            fRet=TRUE;
    BOOL            fTemp;
    OLESTATUS       os;

    //Get the number of files dropped.
    cFiles=DragQueryFile(hDrop, (WORD)-1, szPath, CCHPATHMAX);

    //For each file, create an object window and call OleCreateFromFile.

    for (i=0; i< cFiles; i++)
        {
        //Get the filename for the object.
        DragQueryFile(hDrop, i, szPath, CCHPATHMAX);

        //Get the next ID to use for a window.
        FileObjectDefaults(&fo);
        pObj=PObjectAllocate(&fTemp, pDoc);

        if (!fTemp)
            {
            PObjectFree(pDoc, pObj);
            fRet=FALSE;
            continue;
            }

        /*
         * Create a Packager object from a file, using "Package" as
         * the class and the filename given from DragQueryFile.
         */
        os=OleCreateFromFile(PSZOLE(IDS_STDFILEEDITING), (LPOLECLIENT)pObj,
                             PSZOLE(IDS_PACKAGE), szPath, pDoc->lh, fo.szName,
                             &pObj->pObj, olerender_draw, 0);

        if (OLE_OK!=OsError(os, pDoc, pObj, TRUE))
            {
            PObjectFree(pDoc, pObj);
            fRet=FALSE;
            continue;
            }

        /*
         * If we really wanted to be picky, we could call DragQueryPoint to
         * determine where the object was dropped, but it's of little
         * consequence in this application, so we'll simply use default
         * placement.
         */

        /*
         * Creating a window make the file dirty, so we don't have to.
         * This also allocates an OBJECT and initializes everything it
         * cares to know about.
         */
        hWndT=HBlackBoxCreate(hWnd, &fo, TRUE, pObj);

        if (NULL==hWndT)
            fRet=FALSE;
        }

    DragFinish(hDrop);
    return fRet;
    }
