/*
 * OLELIB.C
 *
 * Miscellaneous library functions necessary for OLE.  Does not
 * make any OLE calls.
 *
 *  DwReadHuge          Reads potentially > 64K data from a file.
 *  DwWriteHuge         Writes potentially > 64K data to a file
 *  RectConvertMappings Converts a RECT between two logical mapping modes.
 *
 * Copyright(c) Microsoft Corp. 1992 All Rights Reserved
 *
 */

#include <windows.h>
#include <ole.h>
#include <commdlg.h>
#include "oclient.h"


#define CBREADBLOCK 32768


/*
 * FFileDialog
 *
 * Purpose:
 *  Invokes the COMMDLG.DLL GetOpenFileName dialog and retrieves
 *  a filename for saving or opening.
 *
 * Parameters:
 *  hWnd            HWND of the owning application.
 *  hInst           HANDLE of the application instance.
 *  pszExt          LPSTR of the default extension
 *  pszFilter       LPSTR of the filter desciption.
 *  pszFile         LPSTR buffer to receive the entered filename.
 *                  Must be at least CCHPATHMAX long.
 *  fOpen           BOOL indicating if we want file open or save.
 *
 * Return Value:
 *  BOOL            TRUE if the function retrieved a filename,
 *                  FALSE if the user pressed CANCEL.
 */

BOOL FAR PASCAL FFileDialog(HWND hWnd, HANDLE hInst, LPSTR pszExt,
                            LPSTR pszFilter, LPSTR pszFile, LPSTR pszCaption,
                            BOOL fOpen)
    {
    OPENFILENAME    ofn;
    char            szTitle[CCHFILENAMEMAX];
    char            szFilter[80];
    WORD            cch1;
    WORD            cch2;


    ofn.lStructSize      =sizeof(OPENFILENAME);
    ofn.hwndOwner        =hWnd;
    ofn.hInstance        =hInst;

    ofn.lpstrFilter      =szFilter;
    ofn.lpstrCustomFilter=NULL;
    ofn.nMaxCustFilter   =0L;
    ofn.nFilterIndex     =1L;                //We only have 1 extension.

    ofn.lpstrFile        =pszFile;
    ofn.nMaxFile         =CCHPATHMAX;
    ofn.lpstrFileTitle   =(LPSTR)szTitle;
    ofn.nMaxFileTitle    =CCHFILENAMEMAX;

    ofn.lpstrInitialDir  =NULL;
    ofn.lpstrTitle       =pszCaption;

    ofn.Flags            =OFN_HIDEREADONLY;
    ofn.nFileOffset      =0;
    ofn.nFileExtension   =0;
    ofn.lpstrDefExt      =pszExt;
    ofn.lCustData        =NULL;
    ofn.lpfnHook         =NULL;
    ofn.lpTemplateName   =NULL;


    //Modify the flags as appropriate.
    if (fOpen)
        ofn.Flags        |= OFN_FILEMUSTEXIST;
    else
        ofn.Flags        |= OFN_PATHMUSTEXIST | OFN_OVERWRITEPROMPT;


    //Build a filter like "pszFilter\0*.pszExt\0\0"
    lstrcpy(szFilter, pszFilter);
    cch1=1+lstrlen(szFilter);

    cch2=wsprintf(pszFile, "*.%s", pszExt);  //Initial edit control contents.
    lstrcpy(szFilter+cch1, pszFile);         //Append to filter.

    //Add the second null-terminator.
    *(szFilter+cch1+cch2+1)=0;

    return GetOpenFileName(&ofn);
    }






/*
 * PszFileFromPath
 *
 * Purpose:
 *  Returns a pointer within an existing pathname string to the
 *  last file of that string.  Used to extract the filename from
 *  a path for use in window titles and message boxes.
 *
 *  Note:  This function does character comparisons to '\' and so
 *  may require more work to localize.
 *
 * Parameters:
 *  pszPath         LPSTR pointer to full pathname.
 *
 * Return Value:
 *  LPSTR           Pointer to a filename within pszPath or NULL if
 *                  there is an error.
 */

LPSTR FAR PASCAL PszFileFromPath(LPSTR pszPath)
    {
    LPSTR   psz=NULL;

    if (NULL!=pszPath)
        {
        psz=pszPath + lstrlen(pszPath)-1;

        while ((psz > pszPath) && (psz[-1]!='\\'))
            psz--;
        }

    return psz;
    }



/*
 * PszExtensionFromFile
 *
 * Purpose:
 *  Returns a pointer within an existing filename string to the
 *  extension of that file.  Used to extract the extension from a file
 *  for use in File dialogs and so forth.  The file is scanned backwards
 *  looking for a '.' or '\'.  If no '.' is found before a '\' then
 *  this function returns a pointer to the null terminator.
 *
 *  Note:  This function does character comparisons to '.' and '\' and so
 *  may require more work to localize.
 *
 * Parameters:
 *  pszFile         LPSTR pointer to a filename.
 *
 * Return Value:
 *  LPSTR           Pointer to an extension (starting with the .) within
 *                  pszFile or NULL if there is an error.
 */

LPSTR FAR PASCAL PszExtensionFromFile(LPSTR pszFile)
    {
    LPSTR   psz;
    LPSTR   pszT=NULL;

    if (NULL!=pszFile)
        {
        //Point to null terminator.
        pszT=pszFile + lstrlen(pszFile);
        psz =pszT-1;

        while ((psz > pszFile) && (psz[-1]!='\\') && (psz[-1]!='.'))
            psz--;

        if ('.'==psz[-1])
            return --psz;
        }

    return pszT;
    }





/*
 * PszWhiteSpaceScan
 *
 * Purpose:
 *  Skips characters in a string until a whitespace or non-whitespace
 *  character is seen.  Whitespace is defined as \n, \r, \t, or ' '.
 *
 * NOTE:  This function is not extremely well suited to localization.
 *        It assumes that an existing application seeking to become
 *        and OLE client probably already has such a string function
 *        available.
 *
 * Parameters:
 *  psz             LPSTR to string to manipulate
 *  fSkip           BOOL  TRUE if we want to skip whitespace.
 *                  FALSE if we want to skip anything but whitespace.
 *
 * Return Value:
 *  LPSTR           Pointer to first character in the string that either
 *                  non-whitespace (fSkip=TRUE) or whitespace (fSkip=FALSE),
 *                  which may be the null terminator.
 */

LPSTR FAR PASCAL PszWhiteSpaceScan(LPSTR psz, BOOL fSkip)
    {
    char        ch;
    BOOL        fWhite;

    while (ch=*psz)
        {
        fWhite=('\n'==ch || '\r'==ch || '\t'==ch || ' '==ch);

        //Too bad C doesn't have a logical XOR (^^) operator.
        if ((fSkip && !fWhite) || (!fSkip && fWhite))
            break;

        psz++;
        }

    return psz;
    }









/*
 * DwReadHuge
 *
 * Purpose:
 *  Provides an _lread function that reads data from a > 64K file
 *  32K at a time until the entire file has been read.  Of course, this
 *  uses huge pointers.
 *
 * Parameters:
 *  hFile       WORD file handle.
 *  pv          LPSTR pointer to data buffer.
 *  dwRead      DWORD bytes to read.
 *
 * Returns:
 *  DWORD       Number of bytes read of 0 if function failed.
 */


DWORD FAR PASCAL DwReadHuge(WORD hFile, LPVOID pv, DWORD dwRead)
    {
    DWORD     dwT;
    BYTE huge *hp;

    dwT=dwRead;
    hp=(BYTE huge *)pv;

    while (dwT > (DWORD)CBREADBLOCK)
        {
        if (_lread(hFile, (LPSTR)hp, (WORD)CBREADBLOCK) != CBREADBLOCK)
            return 0L;

        dwT-= CBREADBLOCK;
        hp  = hp+(DWORD)CBREADBLOCK;
        }

    if (_lread(hFile, (LPSTR)hp, (WORD)dwT) != (WORD)dwT)
        return 0;

    return dwRead;
    }





/*
 * DwWriteHuge
 *
 * Purpose:
 *  Provides an _lwrite function that writes data from a > 64K file
 *  32K at a time until the entire file has been written.  Of course, this
 *  uses huge pointers.
 *
 * Parameters:
 *  hFile       WORD file handle.
 *  pv          LPSTR pointer to data buffer.
 *  dwRead      DWORD bytes to write.
 *
 * Returns:
 *  DWORD       Number of bytes written of 0 if function failed.
 */


DWORD FAR PASCAL DwWriteHuge(WORD hFile, LPVOID pv, DWORD dwWrite)
    {
    DWORD     dwT;
    BYTE huge *hp;

    dwT=dwWrite;
    hp=(BYTE huge *)pv;

    while (dwT > (DWORD)CBREADBLOCK)
        {
        if (_lwrite(hFile, (LPSTR)hp, (WORD)CBREADBLOCK) != CBREADBLOCK)
            return 0L;

        dwT-= CBREADBLOCK;
        hp  = hp+(DWORD)CBREADBLOCK;
        }

    if (_lwrite(hFile, (LPSTR)hp, (WORD)dwT) != (WORD)dwT)
        return 0;

    return dwWrite;
    }





/*
 * RectConvertMappings
 *
 * Purpose:
 *  Converts the contents of a rectangle from one logical mapping
 *  into another.  This function is useful since you have to convert
 *  logical->device then device->logical to do this transformation.
 *
 * Parameters:
 *  pRect           LPRECT containing the rectangle to convert.
 *  mm              WORD mapping mode or source rectangle.
 *
 * Return Value:
 *  None.
 */

void FAR PASCAL RectConvertMappings(LPRECT pRect, WORD mmSrc, WORD mmDst)
    {
    HDC      hDC;
    POINT    rgpt[2];
    WORD     mmTemp;

    if (NULL==pRect)
        return;

    hDC=GetDC(NULL);

    rgpt[0].x=pRect->left;
    rgpt[0].y=pRect->top;
    rgpt[1].x=pRect->right;
    rgpt[1].y=pRect->bottom;

    //Convert the source units into device units.
    mmTemp=SetMapMode(hDC, mmSrc);
    LPtoDP(hDC, rgpt, 2);

    //Convert the device units into the destination units.
    SetMapMode(hDC, mmDst);
    DPtoLP(hDC, rgpt, 2);

    pRect->left   =rgpt[0].x;
    pRect->top    =rgpt[0].y;
    pRect->right  =rgpt[1].x;
    pRect->bottom =rgpt[1].y;

    //Cleanup and return.
    SetMapMode(hDC, mmTemp);
    ReleaseDC(NULL, hDC);
    return;
    }
