/*
 * OLEVTBL.C
 *
 * Constructors and Destructors for OLECLIENTVTBL and OLESTREAMVTBL
 * structures.  The constructors allocate the structure and initialize
 * the VTBLs within.  The destructors free the VTBL and the structure.
 *
 * Copyright(c) Microsoft Corp. 1992 All Rights Reserved
 */


#include <windows.h>
#include <ole.h>
#include "oclient.h"



/*
 * PVtblClientAllocate
 *
 * Purpose:
 *  Allocates and initializes an OLECLIENTVTBL structure.
 *
 * Parameters:
 *  pfSuccess       LPBOOL indicating if the initialization succeeded.  If
 *                  this function returns non-NULL, but *pfSuccess==FALSE,
 *                  the caller must call the destructor function.
 *  hInst           HANDLE of the application instance.
 *  pfn             FARPROC to the single client method to initialize.
 *                  We call MakeProcInstance for this function.
 *
 * Return Value:
 *  LPOLECLIENTVTBL Pointer to the allocated VTBL if successful, NULL
 *                  if the allocation failed or a parameter is invalid.
 */

LPOLECLIENTVTBL FAR PASCAL PVtblClientAllocate(LPBOOL pfSuccess, HANDLE hInst,
                                               FARPROC pfn)
    {
    LPOLECLIENTVTBL     pvt;
    HANDLE              hMem;

    if (NULL==pfSuccess)
        return NULL;

    *pfSuccess=FALSE;

    if (NULL==hInst || NULL==pfn)
        return NULL;

    hMem=LocalAlloc(LPTR, sizeof(OLECLIENTVTBL));

    if (NULL==hMem)
        return NULL;

    pvt=(LPOLECLIENTVTBL)(PSTR)hMem;

    pvt->CallBack=(LPCLIENTCALLBACK)MakeProcInstance(pfn, hInst);

    //Indicate success of MakeProcInstance.
    *pfSuccess=(NULL!=pvt->CallBack);
    return pvt;
    }




/*
 * PVtblClientFree
 *
 * Purpose:
 *  Frees all procedure instances in the LPOLECLIENTVTBL and frees
 *  the structure as well.
 *
 * Parameters:
 *  pvt             LPOLECLIENTVTBL to the structure to free.
 *
 * Return Value:
 *  LPOLECLIENTVTBL NULL if the function succeeds, pvt otherwise
 */

LPOLECLIENTVTBL FAR PASCAL PVtblClientFree(LPOLECLIENTVTBL pvt)
    {
    if (NULL==pvt)
        return pvt;

    if (NULL!=pvt->CallBack)
        FreeProcInstance((FARPROC)pvt->CallBack);

    /*
     * Since we know in the constructor that this came from local memory,
     * we can assume here that dumping the selector will not hurt us
     * at all, which is a valid assumption (good place for an assert)
     */
    if (NULL!=LocalFree((HANDLE)(DWORD)pvt))
        return pvt;

    return NULL;
    }



/*
 * PVtblStreamAllocate
 *
 * Purpose:
 *  Allocates and initializes an OLESTREAMVTBL structure.  It depends
 *  on publics functions named "StreamGet" and "StreamPut" to exist.
 *
 * Parameters:
 *  pfSuccess       LPBOOL indicating if the initialization succeeded.  If
 *                  this function returns non-NULL, but *pfSuccess==FALSE,
 *                  the caller must call the destructor function.
 *  hInst           HANDLE of the application instance.
 *  pfnGet          FARPROC to the stream's Get method.
 *  pfnPut          FARPROC to the stream's Put method.
 *
 * Return Value:
 *  LPOLESTREAMVTBL Pointer to the allocated VTBL if successful, NULL
 *                  if the allocation failed or a parameter is invalid.
 */

LPOLESTREAMVTBL FAR PASCAL PVtblStreamAllocate(LPBOOL pfSuccess, HANDLE hInst,
                                               FARPROC pfnGet, FARPROC pfnPut)
    {
    LPOLESTREAMVTBL     pvt;
    HANDLE              hMem;

    if (NULL==pfSuccess)
        return NULL;

    *pfSuccess=FALSE;

    if (NULL==hInst || NULL==pfnGet || NULL==pfnPut)
        return NULL;

    hMem=LocalAlloc(LPTR, sizeof(OLESTREAMVTBL));

    if (NULL==hMem)
        return NULL;

    pvt=(LPOLESTREAMVTBL)(PSTR)hMem;

    pvt->Get=(LPSTREAMMETHOD)MakeProcInstance(pfnGet, hInst);
    pvt->Put=(LPSTREAMMETHOD)MakeProcInstance(pfnPut, hInst);

    *pfSuccess=(NULL!=pvt->Get) & (NULL!=pvt->Put);
    return pvt;
    }




/*
 * PVtblStreamFree
 *
 * Purpose:
 *  Frees all procedure instances in the OLESTREAMVTBL and frees the
 *  structure.
 *
 * Parameters:
 *  pvt             LPOLESTREAMVTBL to the structure to free.
 *
 * Return Value:
 *  LPOLESTREAMVTBL NULL if the function succeeds, pvt otherwise
 *
 */

LPOLESTREAMVTBL FAR PASCAL PVtblStreamFree(LPOLESTREAMVTBL pvt)
    {
    if (NULL==pvt)
        return NULL;

    if (NULL!=pvt->Get)
        FreeProcInstance((FARPROC)pvt->Get);

    if (NULL!=pvt->Put)
        FreeProcInstance((FARPROC)pvt->Put);

    /*
     * Since we know in the constructor that this came from local memory,
     * we can assume here that dumping the selector will not hurt us
     * at all, which is a valid assumption (good place for an assert)
     */
    if (NULL!=LocalFree((HANDLE)(DWORD)pvt))
        return pvt;

    return NULL;
    }
