{************************************************}
{                                                }
{   Turbo Pascal for Windows                     }
{   Paradox Engine demo access unit              }
{   Copyright (c) 1991 by Borland International  }
{                                                }
{************************************************}

unit PXAccess;

interface

{$N+}

uses WObjects, PXEngine;

type
  PFieldArray = ^TFieldArray;
  TFieldArray = array[1..256] of PChar;

type
  PPXTable = ^TPXTable;
  TPXTable = object(TObject)
    Status: Integer;
    constructor Init(TableName: PChar);
    destructor Done; virtual;
    procedure ClearError;
    function FieldName(Field: Integer): PChar;
    function FieldType(Field: Integer): PChar;
    function FieldWidth(Field: Integer): Integer;
    function Find(Value: PChar): RecordNumber;
    function FindNext: RecordNumber;
    function GetField(Rec, Fld: Integer): PChar;
    function NumRecords: LongInt;
    function NumFields: Integer;
    procedure PXError(Error: Integer); virtual;
    function Update: Boolean;
  private
    CurRecord: Integer;
    TblHandle: TableHandle;
    RecHandle: RecordHandle;
    NumFlds: Integer;
    NumRecs: LongInt;
    FieldNames: PFieldArray;
    FieldTypes: PFieldArray;
    Cache: Pointer;
    TableTime: LongInt;
    function CheckError(Code: Integer): Boolean;
    procedure CacheTable;
    procedure FlushTable;
    function TblName: PChar;
  end;

implementation

uses WinTypes, WinProcs, Strings, WinDos;

type
  PCache = ^TCache;
  TCache = object(TCollection)
    constructor Init(CacheSize: Integer);
    procedure Add(Index: LongInt; P: PChar);
    function Get(Index: LongInt): PChar;
    procedure FreeItem(P: Pointer); virtual;
  end;

type
  PCacheElement = ^TCacheElement;
  TCacheElement = record
    Index: LongInt;
    Item: PChar;
  end;

constructor TCache.Init(CacheSize: Integer);
begin
  TCollection.Init(CacheSize, 0);
end;

procedure TCache.Add(Index: LongInt; P: PChar);
var
  CE: PCacheElement;
begin
  New(CE);
  CE^.Index := Index;
  CE^.Item := P;
  if Count = Limit then AtFree(Count - 1);
  AtInsert(0, CE);
end;

function TCache.Get(Index: LongInt): PChar;
var
  P: PCacheElement;

  function ItemWithIndex(P: PCacheElement): Boolean; far;
  begin
    ItemWithIndex := P^.Index = Index;
  end;

begin
  Get := nil;
  P := FirstThat(@ItemWithIndex);
  if P <> nil then Get := P^.Item;
end;

procedure TCache.FreeItem(P: Pointer);
begin
  StrDispose(PCacheElement(P)^.Item);
  Dispose(P);
end;

{ TPXTable }

function FileTime(F: PChar): LongInt;
var
  S: TSearchRec;
begin
  FindFirst(F, 0, S);
  FileTime := S.Time;
end;

constructor TPXTable.Init(TableName: PChar);
begin
  FieldTypes := nil;
  FieldNames := nil;
  Status := 0;
  CurRecord := -1;
  if CheckError(PXTblOpen(TableName, TblHandle, 0, False)) and
     CheckError(PXRecBufOpen(TblHandle, RecHandle)) then
    CacheTable;
end;

destructor TPXTable.Done;
begin
  TObject.Done;
  PXRecBufClose(RecHandle);
  PXTblClose(TblHandle);
  FlushTable;
end;

procedure TPXTable.CacheTable;
var
  Temp: array[0..25] of Char;
  I: Integer;
begin
  PXNetTblRefresh(TblHandle);
  if CheckError(PXRecNFlds(TblHandle, NumFlds)) and
     CheckError(PXTblNRecs(TblHandle, NumRecs)) then
  begin
    GetMem(FieldTypes, NumFields * SizeOf(PChar));
    GetMem(FieldNames, NumFields * SizeOf(PChar));
    for I := 1 to NumFields do
    begin
      CheckError(PXFldName(TblHandle, I, SizeOf(Temp), Temp));
      FieldNames^[I] := StrNew(Temp);
      CheckError(PXFldType(TblHandle, I, SizeOf(Temp), Temp));
      FieldTypes^[I] := StrNew(Temp);
    end;
    Cache := New(PCache, Init(300));
    TableTime := FileTime(TblName);
  end;
end;

function TPXTable.CheckError(Code: Integer): Boolean;
begin
  if Status = 0 then
  begin
    case Code of
      { ignore the following errors }
{      PXErr_RecDeleted,}
      PXSuccess:
	begin
	end;
    else
      PXError(Code);
      Status := Code;
    end;
  end;
  CheckError := Status = 0;
end;

procedure TPXTable.ClearError;
begin
  Status := 0;
end;

function TPXTable.FieldName(Field: Integer): PChar;
begin
  if FieldNames <> nil then FieldName := FieldNames^[Field]
  else FieldName := nil;
end;

function TPXTable.FieldType(Field: Integer): PChar;
begin
  if FieldTypes <> nil then FieldType := FieldTypes^[Field]
  else FieldType := nil;
end;

function TPXTable.FieldWidth(Field: Integer): Integer;
var
  Width, Code: Integer;
begin
  FieldWidth := 0;
  if FieldTypes <> nil then
    case FieldTypes^[Field][0] of
      'N',
      '$': FieldWidth := 14;
      'A':
	begin
	  Val(PChar(@FieldTypes^[Field][1]), Width, Code);
	  FieldWidth := Width
	end;
      'D': FieldWidth := 12;
      'S': FieldWidth := 8;
    end;
end;

function TPXTable.Find(Value: PChar): RecordNumber;
var
  RecNum: RecordNumber;
begin
  Find := 0;
  CurRecord := -1;
  if Status = 0 then
    if CheckError(PXPutAlpha(RecHandle, 1, Value)) then
      if PXSrchFld(TblHandle, RecHandle, 1, SearchFirst) = 0 then
	if CheckError(PXRecNum(TblHandle, RecNum)) then
	  Find := RecNum;
end;

function TPXTable.FindNext: RecordNumber;
var
  RecNum: RecordNumber;
begin
  FindNext := 0;
  CurRecord := -1;
  if Status = 0 then
    if PXSrchFld(TblHandle, RecHandle, 1, SearchNext) = 0 then
      if CheckError(PXRecNum(TblHandle, RecNum)) then
	FindNext := RecNum;
end;

procedure TPXTable.FlushTable;
var
  I: Integer;
begin
  if (FieldTypes <> nil) and (FieldNames <> nil) then
  begin
    for I := 1 to NumFields do
    begin
      StrDispose(FieldNames^[I]);
      StrDispose(FieldTypes^[I]);
    end;
    FreeMem(FieldTypes, NumFields * SizeOf(PChar));
    FreeMem(FieldNames, NumFields * SizeOf(PChar));
    FieldTypes := nil;
    FieldNames := nil;
    Dispose(PCache(Cache), Done);
    Cache := nil;
    CurRecord := -1;
  end;
end;

function TPXTable.GetField(Rec, Fld: Integer): PChar;
const
  TheData: array[0..255] of Char = '';
var
  Tmp: array[0..255] of Char;
  N: Double;
  I: Integer;
  L: LongInt;
  ArgList: array[0..2] of Integer;
  Index: LongInt;
  P: PChar;
begin
  TheData[0] := #0;
  GetField := TheData;
  if Status <> 0 then Exit;
  if Cache = nil then Exit;
  if (Rec < 1) or (Rec > NumRecords) then Exit;
  if (Fld < 1) or (Fld > NumFields) then Exit;
  Index := Rec * NumFields + Fld;
  P := PCache(Cache)^.Get(Index);
  if P = nil then
  begin
    if Rec <> CurRecord then
    begin
      CheckError(PXRecGoto(TblHandle, Rec));
      CheckError(PXRecBufEmpty(RecHandle));
      CheckError(PXRecGet(TblHandle, RecHandle));
      CurRecord := Rec;
    end;
    FillChar(TheData, SizeOf(TheData), ' ');
    Tmp[0] := #0;
    case FieldTypes^[Fld][0] of
      'A':
	CheckError(PXGetAlpha(RecHandle, Fld, SizeOf(Tmp), Tmp));
      'N':
	begin
	  CheckError(PXGetDoub(RecHandle, Fld, N));
	  if not IsBlankDouble(N) then
	    Str(N:12:4, Tmp);
	end;
      '$':
	begin
	  CheckError(PXGetDoub(RecHandle, Fld, N));
	  if not IsBlankDouble(N) then
	    Str(N:12:2, Tmp);
	end;
      'S':
	begin
	  CheckError(PXGetShort(RecHandle, Fld, I));
	  if not IsBlankShort(i) then
	    Str(I:6, Tmp)
	end;
      'D':
	begin
	  CheckError(PXGetDate(RecHandle, Fld, L));
	  if Not IsBlankDate(L) then
	  begin
	    CheckError(PXDateDecode(L, ArgList[0], ArgList[1], ArgList[2]));
	    wvSprintf(Tmp, '%2d/%2d/%4d', ArgList);
	  end;
	end;
    end;
    StrMove(TheData, Tmp, StrLen(Tmp));
    TheData[FieldWidth(Fld)] := #0;
    P := StrNew(TheData);
    PCache(Cache)^.Add(Index, P);
    GetField := P;
  end
  else
    GetField := P;
end;

function TPXTable.NumRecords: LongInt;
begin
  NumRecords := NumRecs;
end;

function TPXTable.NumFields: Integer;
begin
  NumFields := NumFlds;
end;

procedure TPXTable.PXError(Error: Integer);
begin
  MessageBox(GetFocus, PXErrMsg(Error), 'PXAccess', mb_OK)
end;

function TPXTable.TblName: PChar;
const
  TableName: array[0..81] of Char = '';
begin
  if CheckError(PXTblName(TblHandle, SizeOf(TableName), TableName)) then
    StrCat(TableName, '.DB');
  TblName := @TableName;
end;

function TPXTable.Update: Boolean;
begin
  if FileTime(TblName) = TableTime then Update:= False
  else
  begin
    FlushTable;
    CacheTable;
    Update := True;
  end;
end;

end.
