//========================================================================
//
//  WINNAV.C -- A File manager for Microsoft Windows
//
//  (c) Douglas Boling, 1991
//
//
//  Set tab stops to every 3 spaces for better readability
//
//========================================================================

#define  BUTTONWIDTH     80
#define  BUTTONHEIGHT    50

#define  BUTTONMAX      10
#define  LBOXMAX        10

#include <dos.h>
#include <windows.h>
#include <string.h>
#include <stdlib.h>
#include <stdio.h>
#include <direct.h>
#include <errno.h>

#define  STRTOLP(l)	    ((LONG)(LPSTR)(l))

#include "winnav.h"

//------------------------------------------------------------------------
//Control Window Procedure pointers used in subclassing
//------------------------------------------------------------------------
FARPROC	lpfnOldLBoxProc[LBOXMAX+1];
FARPROC	lpfnOldEditBoxProc;

FARPROC	lpfnListBoxSubclassProc;
FARPROC	lpfnEditBoxSubclassProc;

//------------------------------------------------------------------------
//Global Data used to manipulate child Windows
//------------------------------------------------------------------------
int   sLBoxCount = LBOXMAX;
int	sLBoxVStart = 0;

typedef struct {
	DWORD	lSize;
	WORD	wDate;
	WORD	wTime;
	WORD	wAttr;
	char	szName[13];
} FILEINFO;

//typedef FILEINFO *PFILEINFO;
typedef struct find_t FIND_T;
typedef struct DOSERROR DOSERR;
typedef struct diskfree_t DISKFREE_T;


struct LISTBOXSTRUCT {
	char	szPath [128];
	char	szSel [16];
} lsLBoxes [LBOXMAX];

int   sLBoxVisible = 0;
HWND	hwndLBoxArray [LBOXMAX];


struct BUTTONSTRUCT {
	HWND	hwndButton;
	int	sFunction;
} bsButtons [BUTTONMAX] = {{0, 0},
                           {0, 1},
                           {0, 2},
                           {0, 3},
                           {0, 4},
                           {0, 5},
                           {0, 6},
                           {0, 7},
                           {0, 8},
                           {0, 9}};

struct FUNCTIONSTRUCT {
	int	sFunction;
	char	szText [8];
} fsFunctions [10] = {{IDM_OPEN,		"Open"},
                      {IDM_MOVE,		"Move"},
                      {IDM_COPY,		"Copy"},
                      {IDM_RENAME,	"Rename"},
                      {IDM_MKDIR,	"MkDir"},
                      {IDM_DELETE,	"Delete"},
                      {IDM_ATTRIB,	"Attrib"},
                      {IDM_INFO,		"Info"},
                      {IDM_CONFIG,	"Config"},
                      {IDM_EXIT,		"Exit"}};

int	sButtonCount;
BOOL	bIncDir;

HWND  hwndEditBox;
HWND  hwndScrollBar;
HWND  hwndDrvBox;

HWND	hwndFocusWin;

char	szMasterPath [128];
char	szDestPath [128];
char	szFunction [8];

BOOL	bDir;
BOOL	bExecutable;

//------------------------------------------------------------------------
//Function prototypes
//------------------------------------------------------------------------
int  PASCAL		 WinMain (HANDLE, HANDLE, LPSTR, int);
long FAR PASCAL WndProc (HWND, WORD, WORD, LONG);
long FAR PASCAL ListBoxSubclassProc (HWND, WORD, WORD, LONG);
long FAR PASCAL EditBoxSubclassProc (HWND, WORD, WORD, LONG);
long FAR PASCAL ListBoxSubclassProc (HWND, WORD,WORD, LONG);
long FAR PASCAL EditBoxSubclassProc (HWND, WORD,WORD, LONG);
int  FAR PASCAL ConfigDlgProc (HWND,WORD, WORD, LONG);
int  FAR PASCAL CopyRenDlgProc (HWND,WORD, WORD, LONG);
int  FAR PASCAL MkDirDlgProc (HWND,WORD, WORD, LONG);
int  FAR PASCAL SetAttrDlgProc (HWND,WORD, WORD, LONG);
int  FAR PASCAL FileInfoDlgProc (HWND,WORD, WORD, LONG);
int  FAR PASCAL AboutDlgProc (HWND,WORD, WORD, LONG);
long FAR PASCAL WndProc (HWND, WORD, WORD, LONG);

int	NumLBoxes (int);
int  	LBoxPos (int);
int  	LBoxHeight (int);
int  	EBoxWidth (int);
int  	GetButtonHeight (int);
void  SetButtonPos (int, int);
LONG  SendLBoxMsg (int,WORD, WORD, LONG);
void  GetLBoxText (int, char *);
void  ScrollLBoxes (int);
int	TranslateID (WORD);
void	FillListBox (HWND, char *, int);
int  	ParsePath (HWND, int, char *, int);
int	GetDirDepth (void);
void	RefreshListBoxes (HWND);
int	FillListBoxes (HWND);
int	IsProgram (char  *,char  *);
int	EnableButtons (HWND, BOOL, BOOL);
long	GetDiskFree (int);
int	GetFileInfo (char  *, FILEINFO *);
void	ConvertDate (WORD, char *);
void	ConvertTime (WORD, char *);
void  GetFileType (char  *, WORD, char  *, char  *);
int	GetErrorCode (void);
void	PrintError (HWND,int, char  *);
int	MakeDir (char  *);
int	CopyFile (HWND,char  *,char  *);
int	CopyDir (HWND,char  *, char  *, int);
int	DeleteFile (HWND, char  *);
int	DeleteDir (HWND, char  *, int);

//------------------------------------------------------------------------
// Child window size data
//------------------------------------------------------------------------
int	sDrvBoxWidth, sLBoxWidth, sEditBoxHeight, sBMinHeight;

//------------------------------------------------------------------------
// Configuration data
//------------------------------------------------------------------------
char	szDefaultEditor [80];
int	sSearchFlags;
BOOL	bDirFirst;
BOOL	bConfirmFileDel;
BOOL	bConfirmDirDel;

char		szProfileName[] = "WINNAV.INI";
char		szAppName[] = "WinNav";
HANDLE	hInstance;

//=======================================================================
//
// Program Entry Point (WinMain)
//
//=======================================================================
int PASCAL WinMain (HANDLE hInstance, HANDLE hPrevInstance,
	            LPSTR lpszCmdParam, int nCmdShow) {

   HWND		hwnd;
   MSG			msg;
   WNDCLASS	wndclass;

	HDC			hdc;
	TEXTMETRIC	tm;

	int			i, x, y, cx, cy;
	char		szTemp [32];

	if (!hPrevInstance) {
      wndclass.style          = CS_HREDRAW | CS_VREDRAW;
      wndclass.lpfnWndProc    = WndProc;
      wndclass.cbClsExtra     = 0;
      wndclass.cbWndExtra     = 0;
      wndclass.hInstance      = hInstance;
      wndclass.hIcon          = LoadIcon (hInstance, szAppName);
      wndclass.hCursor        = LoadCursor (NULL, IDC_ARROW);
      wndclass.hbrBackground  = COLOR_WINDOW + 1;
      wndclass.lpszMenuName   = szAppName;
      wndclass.lpszClassName  = szAppName;

      RegisterClass (&wndclass);
   }
	//
	//Determine the size of the system font to set the sizes of the
	//window controls.
	//
	hdc = GetDC (GetDesktopWindow());
	SelectObject (hdc, GetStockObject (SYSTEM_FONT));
	GetTextMetrics (hdc, &tm);
	sDrvBoxWidth = tm.tmAveCharWidth * 4;
	sLBoxWidth = tm.tmAveCharWidth * 20;
	sEditBoxHeight = 3 * (tm.tmHeight + tm.tmExternalLeading) / 2;
	sBMinHeight = 3 * (tm.tmHeight + tm.tmExternalLeading) / 2;
	ReleaseDC (GetDesktopWindow(), hdc);

	//
	// Get data on the last state of the program.
	//
	x = GetPrivateProfileInt (szAppName, "x", 40, szProfileName);
	y = GetPrivateProfileInt (szAppName, "y", 40, szProfileName);

	cx = (3 * (sLBoxWidth+5)) + 15 + BUTTONWIDTH + sDrvBoxWidth
	     + (GetSystemMetrics (SM_CXFRAME) * 2);
	cx = GetPrivateProfileInt (szAppName, "cx", cx, szProfileName);
	cy = GetPrivateProfileInt (szAppName, "cy", 400, szProfileName);

	sSearchFlags = GetPrivateProfileInt (szAppName, "IncFlags", 6, 
	                                     szProfileName);
	bDirFirst = GetPrivateProfileInt (szAppName, "DirFirst", TRUE,
	                                  szProfileName);
	bConfirmFileDel = GetPrivateProfileInt (szAppName, "ConfFileDel", TRUE, 
	                                        szProfileName);
	bConfirmDirDel = GetPrivateProfileInt (szAppName, "ConfDirDel", TRUE, 
	                                       szProfileName);
	GetPrivateProfileString (szAppName, "DefaultEditor" , "notepad.exe",
	                         szDefaultEditor, sizeof (szDefaultEditor),
	                         szProfileName);

	sButtonCount = GetPrivateProfileInt (szAppName, "NumberButtons", 6, 
	                                     szProfileName);
	for (i = 0; i < BUTTONMAX; i++) {
		strcpy (szTemp, "Button");
		itoa (i, szTemp+6, 10);
		bsButtons[i].sFunction = GetPrivateProfileInt (szAppName, szTemp, i,
		                                               szProfileName);
	}
   hwnd = CreateWindow (szAppName,            // window class name
	                "WinNav",                  // window caption
	                WS_OVERLAPPEDWINDOW,       // window style
	                x,                         // initial x position
	                y,                         // initial y position
	                cx,                        // initial x size
	                cy,                        // initial y size
	                NULL,                      // parent window handle
	                NULL,                      // window menu handle
	                hInstance,                 // program instance handle
	                NULL);                     // creation parameters

	if (!(nCmdShow & SW_SHOWMINIMIZED) &&
	    GetPrivateProfileInt (szAppName, "Zoom", 0, szProfileName))
		nCmdShow |= SW_SHOWMAXIMIZED;

   ShowWindow (hwnd, nCmdShow);

   UpdateWindow (hwnd);

   while (GetMessage (&msg, NULL, 0, 0)) {
	   TranslateMessage (&msg);
	   DispatchMessage (&msg);
   }
   return msg.wParam;
}

//========================================================================
//
// Window formatting routines
//
//========================================================================
//------------------------------------------------------------------------
// Routine to compute the proper number of list boxes
//------------------------------------------------------------------------
int NumLBoxes (int sWinWidth) {

	int	sCount;

	if (sButtonCount)
		sCount = (sWinWidth - BUTTONWIDTH - sDrvBoxWidth - 10) / 
		         (sLBoxWidth+3);
	else
		sCount = (sWinWidth - sDrvBoxWidth - 10) / (sLBoxWidth+3);
	return max (1, sCount);
}

//------------------------------------------------------------------------
// Routine to compute the position of a list box
//------------------------------------------------------------------------
int LBoxPos (int sBoxNum) {

   return max (1, (10 + (sBoxNum * (sLBoxWidth + 3)) + sDrvBoxWidth));
}

//------------------------------------------------------------------------
// Routine to compute the height of the list boxes
//------------------------------------------------------------------------
int LBoxHeight (int sWinHeight) {

   return max (0, sWinHeight - (2*sEditBoxHeight) - 20);
}

//------------------------------------------------------------------------
// Routine to compute the proper width of the edit box
//------------------------------------------------------------------------
int EBoxWidth (int sLBoxCnt) {

   return max (sLBoxWidth, (sLBoxCnt * (sLBoxWidth + 3))) + sDrvBoxWidth + 2;
}

//------------------------------------------------------------------------
// Routine to compute the proper width of the edit box
//------------------------------------------------------------------------
int GetButtonHeight (int sWinHeight) {

	int	sBHeight;

	sBHeight = 0;
	if (sButtonCount) {
		sBHeight = min (((sWinHeight-5)/sButtonCount)-5, BUTTONWIDTH);
		sBHeight = max (sBMinHeight, sBHeight);
	}
	return sBHeight;
}	

//------------------------------------------------------------------------
// Set Button Position
//------------------------------------------------------------------------
void SetButtonPos (int sHeight, int sWidth) {
	int	i, sBHeight;
	
	sBHeight = GetButtonHeight (sHeight);
	if (sBHeight) {

		for (i = 0; i < sButtonCount; i++)
			SetWindowPos (bsButtons [i].hwndButton, NULL,
	 	      	              sWidth - BUTTONWIDTH - 5, 
			              5 + (i * (sBHeight+5)), 
			              BUTTONWIDTH, sBHeight, 0);
	}
}

//========================================================================
//
// Virtual Listbox Support routines
//
//========================================================================
//------------------------------------------------------------------------
// Virtualize listboxes by using this function to send messages to LBoxes
//------------------------------------------------------------------------
LONG SendLBoxMsg (int sLBoxNum, WORD msg, WORD wParam, LONG lParam) {
	LONG	lRetCode;
	
	switch (msg) {
		case LB_DIR:
			strcpy (lsLBoxes[sLBoxNum].szPath, 
			        (char *)LOWORD (lParam));
			lRetCode = 0;
			break;

		case LB_SELECTSTRING:
			strcpy (lsLBoxes[sLBoxNum].szSel, 
			        (char *)LOWORD (lParam));
			lRetCode = 0;
			break;

		case LB_SETCURSEL:
			if (wParam == -1)
				strcpy (lsLBoxes[sLBoxNum].szSel, "");
			lRetCode = 0;
			break;

		case LB_RESETCONTENT:
			strcpy (lsLBoxes[sLBoxNum].szPath, "");
			strcpy (lsLBoxes[sLBoxNum].szSel, "");
			lRetCode = 0;
			break;
	}
	if (sLBoxNum >= sLBoxVStart && sLBoxNum < sLBoxVStart + sLBoxVisible)
		return SendMessage (hwndLBoxArray[sLBoxNum - sLBoxVStart], 
		                    msg, wParam, lParam);
   return lRetCode;
}                            

//------------------------------------------------------------------------
// Returns the selected text of a virtual list box.
//------------------------------------------------------------------------
void GetLBoxText (int sLBoxNum, char *spOutText) {

	int	i;

	if (sLBoxNum >= sLBoxVStart && sLBoxNum < sLBoxVStart + sLBoxVisible) {

		i = sLBoxNum - sLBoxVStart;

  		SendMessage (hwndLBoxArray [i], LB_GETTEXT,
            (WORD) SendMessage (hwndLBoxArray [i], LB_GETCURSEL, 0, 0L), 
            STRTOLP (lsLBoxes[sLBoxNum].szSel));
	} 
	strcpy (spOutText, lsLBoxes[sLBoxNum].szSel);
	return;
}

//------------------------------------------------------------------------
// Scroll virtual list boxes
//------------------------------------------------------------------------
void ScrollLBoxes (int sScrollCnt) {

	int	i;

	i = min (GetDirDepth (), LBOXMAX);
	sLBoxVStart = max (0, sLBoxVStart + sScrollCnt);
	sLBoxVStart = min (sLBoxVStart, max (0, i - sLBoxVisible));
	return;
}

//------------------------------------------------------------------------
// Translate a ListBox notification ID into a virtual LB ID
//------------------------------------------------------------------------
int TranslateID (WORD wParam) {
	return wParam + sLBoxVStart;
}

//========================================================================
//
// Listbox Data routines
//
//========================================================================
//------------------------------------------------------------------------
// Routine to fill a list box with a list of files
//------------------------------------------------------------------------
void FillListBox (HWND hwnd, char *szPath, int sBoxNum) {
	char	szTemp [128];
	int	sSort1, sSort2;
	LONG	lRetCode;

	sSort1 = 0xA010;								//Directory search
	sSort2 = 0x2001 | sSearchFlags;			//File search

	if (!bDirFirst) {
		sSort2 = sSort1;
		sSort1 = 0x2001;
	}
	SendLBoxMsg (sBoxNum, WM_SETREDRAW, FALSE, 0L);

	strcpy (szTemp, szPath);
	strcat (szTemp, "*.*");
	SendLBoxMsg (sBoxNum, LB_DIR, sSort1, STRTOLP (szTemp));
	SendLBoxMsg (sBoxNum, WM_SETREDRAW, TRUE, 0L);	
	strcpy (szTemp, szPath);
	strcat (szTemp, "*.*");
	SendLBoxMsg (sBoxNum, LB_DIR, sSort2, STRTOLP (szTemp));

	lRetCode = SendLBoxMsg (sBoxNum, LB_FINDSTRING, -1, STRTOLP ("[..]")); 
	if (lRetCode != LB_ERR)
		SendLBoxMsg (sBoxNum, LB_DELETESTRING, (WORD) lRetCode, 0L); 

   return;
}

//------------------------------------------------------------------------
// Routine to fill the list boxes using the default path
//------------------------------------------------------------------------
int ParsePath (HWND hwnd, int sBoxNum, char *szPath, int sPtr) {

   char 	chr, szDir [15];
	int	i;

	chr = szPath [sPtr+1];
	szPath [sPtr+1] = 0;
	FillListBox (hwnd, szPath, sBoxNum);
	szPath [sPtr+1] = chr;
	strcpy (szDir, "[");

	for (i = 1; i < 12 && szPath[sPtr+i] && szPath[sPtr+i] != '\\'; i++) 
	   szDir [i] = szPath [sPtr+i];

	szDir [i] = 0;
	strcat (szDir, "]");

	SendLBoxMsg (sBoxNum, LB_SELECTSTRING, -1, STRTOLP (szDir));

	if (szPath [sPtr+i] && sBoxNum < sLBoxCount)
		ParsePath (hwnd, sBoxNum+1, szPath, sPtr+i);
	else {
		strcat (szPath , "\\");
		FillListBox (hwnd, szPath, sBoxNum+1);
	}
   return 1;
}

//------------------------------------------------------------------------
// Routine to fill all list boxes
//------------------------------------------------------------------------
int GetDirDepth () {
	char	szPath [128];
	int	i, j, sPathLen;

   getcwd (szPath, sizeof (szPath));
	//	
	// Count the depth of the current directory by counting '\' chars.
	//
	sPathLen = strlen (szPath);
	j = 1;
	if (sPathLen != 3)
		for (i = 0; i < sPathLen; i++)
			if (szPath[i] == '\\')
				j++;
   return j;
}

//------------------------------------------------------------------------
// Routine to fill all list boxes
//------------------------------------------------------------------------
void RefreshListBoxes (HWND hwnd) {
	char	szPath [128], szTemp[10];
	int	i;

	for (i = 0; i < sLBoxCount; i++)
		SendLBoxMsg (i, LB_RESETCONTENT, 0, 0L);

	getcwd (szPath, sizeof (szPath));
	strcpy (szMasterPath, szPath);

	SetWindowText (hwndEditBox, szPath);
	bDir = TRUE;
	bExecutable = FALSE;
	EnableButtons (hwnd, bExecutable, bDir);

	strcpy (szTemp, "[---]");
	szTemp [2] = szPath [0] + (char) 0x20;
	SendMessage (hwndDrvBox, LB_SELECTSTRING, -1, STRTOLP (szTemp));

	SetScrollPos (hwndScrollBar, SB_CTL, sLBoxVStart, TRUE);

	ParsePath (hwnd, 0, szPath, 2);

	return;
}

//------------------------------------------------------------------------
// Routine to fill all list boxes. 
// Returns the ID of the last list box filled.
//------------------------------------------------------------------------
int FillListBoxes (HWND hwnd) {
	int	sDDepth;

	sDDepth = GetDirDepth ();
	//
	// If directory deeper than number of listboxes, shift starting dir
	//	
	if (sDDepth <= sLBoxVisible) {
		if (sLBoxVStart > 0) 
			ScrollLBoxes (-sLBoxVStart);
		SetScrollRange (hwndScrollBar, SB_CTL, 0, 0, FALSE);
	} else {
		ScrollLBoxes (sDDepth - sLBoxVStart - sLBoxVisible);
		SetScrollRange (hwndScrollBar, SB_CTL, 0, sLBoxVStart, FALSE);
	}
	RefreshListBoxes (hwnd);
   return min (sDDepth, sLBoxVisible) - 1;
}

//========================================================================
//
// Functional support routines
//
//========================================================================

//------------------------------------------------------------------------
// Routine to determine a file is executable.
//------------------------------------------------------------------------
BOOL IsProgram (char *szPrograms, char *szInPath) {

  	char	szDrive [4], szDir [64], szName [12], szExt[8];

	_splitpath (szInPath, szDrive, szDir, szName, szExt);

	if (strlen (szExt) > 1) {
		strcat (szExt, " ");
		if (strstr (szPrograms, szExt+1))
			return TRUE;
	}
  	return FALSE;
}

//------------------------------------------------------------------------
// Enable/Disable Menu items and Buttons.
//------------------------------------------------------------------------
BOOL EnableButtons (HWND hwnd, BOOL bProgram, BOOL bDir) {

	int	i;

	for (i = 0; i < sButtonCount; i++) {

		if (bsButtons[i].sFunction == 0) {
			if (bDir) {
				EnableWindow (bsButtons[i].hwndButton, FALSE);
				EnableMenuItem (GetMenu (hwnd), IDM_OPEN, 
				                MF_GRAYED);
			} else {
				EnableWindow (bsButtons[i].hwndButton, TRUE);
				EnableMenuItem (GetMenu (hwnd), IDM_OPEN, 
				                MF_ENABLED);
			}
		} else 
			EnableWindow (bsButtons[i].hwndButton, TRUE);
	}
	return bProgram;	
}

//------------------------------------------------------------------------
// Get Disk Bytes Free
//------------------------------------------------------------------------
LONG GetDiskFree (int sDiskNum) {

	DISKFREE_T	dfData;
	int			rc;
	LONG			lBytesFree;

	rc = _dos_getdiskfree (sDiskNum, &dfData);
	if (rc)
		return -1;
	else {
		lBytesFree = (LONG) dfData.avail_clusters;		
		lBytesFree *= (LONG) dfData.sectors_per_cluster;
		lBytesFree *= (LONG) dfData.bytes_per_sector;
		return lBytesFree;
	}
}

//------------------------------------------------------------------------
// Get File Info
//------------------------------------------------------------------------
int GetFileInfo (char *szFileName, FILEINFO *pfiData) {

	FIND_T	fs;
	int	rc, sFlags = _A_ARCH | _A_NORMAL | _A_HIDDEN | _A_RDONLY | 
		             _A_SYSTEM | _A_SUBDIR;

	//
	// If Root dir, look for Volume ID. If not found, fake.
	//
	if (strlen (szFileName) == 3) {
		strcat (szFileName, "*.*");
		rc = _dos_findfirst (szFileName, _A_VOLID, &fs);
		szFileName[4] = '\0';		
		if (rc == 0x12) {
			strcpy (fs.name, "Drive A");
			fs.name[6] = szFileName[0];
			fs.size = GetDiskFree (szFileName[0] - '@');
			fs.wr_date = 1;
			fs.wr_time = 0;
			fs.attrib = _A_VOLID;
			rc = 0;
		}
	} else
		rc = _dos_findfirst (szFileName, sFlags, &fs);

	pfiData->lSize = fs.size;
	pfiData->wDate = fs.wr_date;
	pfiData->wTime = fs.wr_time;
	pfiData->wAttr = fs.attrib;
	strcpy (pfiData->szName, fs.name);
	
	return rc; 
}

//------------------------------------------------------------------------
// Convert date
//------------------------------------------------------------------------
void ConvertDate (WORD wDate, char *szBuffer) {

	int	i;
	char	szTemp [20];
	static	char szMonth[12][10] = {"January",
	 			                        "February",
	 			                        "March",
	 			                        "April",
	 			                        "May",
	 			                        "June",
	 			                        "July",
	 			                        "August",
	 			                        "September",
	 			                        "October",
	 			                        "November",
	 			                        "December"};

	i = wDate;                        //Month
	i >>= 5;
	i &= 0x000F;
	i = max (1, min (i, 12));
	strcpy (szBuffer, szMonth[i-1]);
	strcat (szBuffer, " ");

	i = wDate;                        //Day
	i &= 0x001F;
	itoa (i, szTemp, 10);
	strcat (szBuffer, szTemp);
	strcat (szBuffer, ",");

	wDate >>= 9;                      //Year
	wDate &= 0x007F;
	wDate += 1980;
	itoa (wDate, szTemp, 10);
	strcat (szBuffer, szTemp);

	return;
}

//------------------------------------------------------------------------
// Convert time
//------------------------------------------------------------------------
void ConvertTime (WORD wTime, char *szBuffer) {

	int	i;
	char	szTemp [20], szNoon[5];


	i = wTime;                           // Hours
	i >>= 11;
	i &= 0x001F;
	if (i > 12) {
		i -= 12;
		strcpy (szNoon, "  PM");
	} else
		strcpy (szNoon, "  AM");
	if (i == 0) i = 12;
	itoa (i, szTemp, 10);
	strcpy (szBuffer, szTemp);
	strcat (szBuffer, " : ");

	i = wTime;                          //Minutes
	i &= 0x07E0;
	i >>= 5;
	itoa (i, szTemp, 10);
	if (strlen (szTemp) == 1)
		strcat (szBuffer, "0");
	strcat (szBuffer, szTemp);

	strcat (szBuffer, szNoon);

	return;
}

//------------------------------------------------------------------------
// Get File Type
//------------------------------------------------------------------------
void GetFileType (char *szFileName, WORD wFlags, char *szFileType, 
                  char *szFileInfo) {

	char	szDrive[3], szDir[64], szName[9], szExt[5];

	_splitpath (szFileName, szDrive, szDir, szName, szExt);

	if (wFlags & _A_VOLID) {
		strcpy (szFileType, "Drive");
		return;
	}
	if (wFlags & _A_SUBDIR) {
		strcpy (szFileType, "Directory");
		return;
	}
	strupr (szExt);
	//
	// Check for standard COM file
	//
	if (strcmp (szExt, ".COM") == 0) {
		strcpy (szFileType, "DOS Program");
		strcpy (szFileInfo, "");
	}
	//
	// Check for EXE file
	//
	else if (strcmp (szExt, ".EXE")==0) {
		strcpy (szFileType, "DOS Program");
		strcpy (szFileInfo, "");			
	}
	//
	// Check for Bitmap file
	//
	else if (strcmp (szExt, ".BMP") == 0) {
		strcpy (szFileType, "Bitmap");
		strcpy (szFileInfo, "");
	}
	else
		strcpy (szFileType, "File");

	return;
}

//------------------------------------------------------------------------
// Get DOS extended error info
//------------------------------------------------------------------------
int GetErrorCode () {
	int		rc;
	DOSERR	doserr;
	 
	return rc = dosexterr (&doserr);
}

//------------------------------------------------------------------------
// Print DOS error code
//------------------------------------------------------------------------
void PrintError (HWND hwnd, int ErrorCode, char *szFileName) {
	char		szTemp [64], szTemp1[128];

	//
	// If fail on critical error, don't put up another box.	
	if (ErrorCode == 53)
		return;
 
	if (LoadString (hInstance, ErrorCode+200, szTemp, sizeof (szTemp)))
		sprintf (szTemp1, szTemp, szFileName);
	else
		sprintf (szTemp1, "DOS Error %x", ErrorCode);
	MessageBox (hwnd, szTemp1, szAppName, MB_OK);
}

//========================================================================
//
// Function procedures
//
//========================================================================
//------------------------------------------------------------------------
// Create a directory
//------------------------------------------------------------------------
BOOL	MakeDir (char *szNewDir) {

   if (mkdir (szNewDir))
		return FALSE;
	else
		return TRUE;
}

//------------------------------------------------------------------------
// Copy a file
//------------------------------------------------------------------------
BOOL	CopyFile (HWND hwnd, char *szDestFile, char *szSrcFile) {

	char		szTempDest[128];
	int		rc, hSrc, hDest, i;
	WORD		wByteCount, wBuffSize;		
	DWORD		lBytesRead, lBytesFree;
	HANDLE	hMem;
	OFSTRUCT	ofFile;
	LPSTR		lpBuffer;
	FILEINFO	fiData;
	
	//
	// See if destination disk valid
	//
	if (rc = GetFileInfo (szSrcFile, &fiData)) {
		PrintError (hwnd, rc, szSrcFile);
		return FALSE;
	} 
	strcpy (szTempDest, szDestFile);
	//
	// See if room on destination disk 
	//
	lBytesFree = GetDiskFree (szTempDest[0] - '@');
	if (lBytesFree == -1) {
		rc = GetErrorCode ();
		return FALSE;
	}
	if (fiData.lSize > lBytesFree) {
		PrintError (hwnd, 0, "");
		return 0;
	}	
	//
	// Open Source file
	//	
	hSrc = OpenFile (szSrcFile, &ofFile, OF_READ);
	if (hSrc == -1) {
		PrintError (hwnd, ofFile.nErrCode, szSrcFile);
		return FALSE;
	}
	//
	// Open Destination file, first, append src filename.
	//	
	i = strlen (szTempDest);
	if (szTempDest[i-1] != '\\')
		strcat (szTempDest, strrchr (szSrcFile, '\\'));
	else
	 	strcat (szTempDest, strrchr (szSrcFile, '\\')+1);

	hDest = OpenFile (szTempDest, &ofFile, OF_CREATE);
	if (hDest == -1) {
		//
		// If path not found error, delete src filename
		//
		if (ofFile.nErrCode == 3) {
			szTempDest[i] ='\0';
			hDest = OpenFile (szTempDest, &ofFile, OF_CREATE);
		}
		if (hDest == -1) {
			PrintError (hwnd, ofFile.nErrCode, szTempDest);
			_lclose (hSrc);
			return FALSE;
		} 
	}
	//
	// Allocate memory for a copy buffer
	//	
	if (fiData.lSize > 0xC000)
		wBuffSize = 0xC000;
	else	
		wBuffSize = (WORD) fiData.lSize;
	hMem = GlobalAlloc (GMEM_DISCARDABLE | GMEM_MOVEABLE, (DWORD) wBuffSize);
	if (!hMem) {
		PrintError (hwnd, 8, "");
		_lclose (hSrc);
		_lclose (hDest);
		return FALSE;
	}
	//
	// Copy loop
	//	
	rc = 0;
	lBytesRead = 0;
	while (lBytesRead < fiData.lSize && rc == 0) {
		//
		// Lock memory block. If discarded, reallocate.
		//
		lpBuffer = GlobalLock (hMem);
		if (!lpBuffer) {
			hMem = GlobalReAlloc (hMem, (DWORD) wBuffSize, 
			                      GMEM_DISCARDABLE | GMEM_MOVEABLE);
			if (!hMem) {
				_lclose (hSrc);
				_lclose (hDest);
				PrintError (hwnd, 8, "");
				return FALSE;
			}
			lpBuffer = GlobalLock (hMem);
			if (!lpBuffer) {
				_lclose (hSrc);
				_lclose (hDest);
				PrintError (hwnd, 8, "");
				return FALSE;
			}
		}
		wByteCount = _lread (hSrc, lpBuffer, wBuffSize);
		if (wByteCount == 0xFFFF)
			rc = GetErrorCode ();
		else {
			lBytesRead += wByteCount;
			wByteCount = _lwrite (hDest, lpBuffer, wByteCount);
			if (wByteCount == 0xFFFF)
				rc = GetErrorCode ();
		}
		GlobalUnlock (hMem);
	}
	//
	// Clean up
	//	
	GlobalFree (hMem);
	_lclose (hSrc);
	_lclose (hDest);

	if (rc) {
		PrintError (hwnd, rc, "");
		return FALSE;
	} else
		return TRUE;
}

//------------------------------------------------------------------------
// Copy directory
//------------------------------------------------------------------------
BOOL	CopyDir (HWND hwnd, char *szDestDir, char *szSrcDir, BOOL bRecurse) {

	FIND_T	fs;
	int		rc, sFlags = _A_ARCH | _A_SUBDIR;

	rc = _dos_findfirst (szDestDir, sFlags | _A_HIDDEN | _A_SYSTEM, &fs);
	if (rc)
		if (!MakeDir (szDestDir)) {
			PrintError (hwnd, GetErrorCode(), szDestDir);
			return FALSE;
		}

	strcat (szSrcDir, "\\*.*");
	rc = _dos_findfirst (szSrcDir, sFlags | sSearchFlags, &fs);
	*strrchr (szSrcDir, '\\') = '\0';

	while (rc == 0) {
		strcat (szSrcDir, "\\"); 
		strcat (szSrcDir, fs.name);

		if (fs.attrib & _A_SUBDIR) {
			if (bRecurse && strcmp (fs.name, ".") != 0 
			             && strcmp (fs.name, "..") != 0) {
				strcat (szDestDir, "\\");
				strcat (szDestDir, fs.name);
				if (!CopyDir (hwnd, szDestDir, szSrcDir, 
				              bRecurse))
					rc = 1;
				*strrchr (szDestDir, '\\') = '\0';
			}
		} else
	 		if (!CopyFile(hwnd, szDestDir, szSrcDir))
				rc = 1;
		*strrchr (szSrcDir, '\\') = '\0';

		if (!rc)
			rc = _dos_findnext (&fs);
	}
	if (rc != 0x12) {
		PrintError (hwnd, GetErrorCode(), szSrcDir);
		return FALSE;
	}
	return TRUE;
}
//------------------------------------------------------------------------
// Delete a file
//------------------------------------------------------------------------
BOOL	DeleteFile (HWND hwnd, char *szFile) {

	OFSTRUCT	ofFileData;

	OpenFile (szFile, &ofFileData, OF_EXIST | OF_DELETE);
	
	if (ofFileData.nErrCode && ofFileData.nErrCode != 2) {
		PrintError (hwnd, ofFileData.nErrCode, szFile);
		return FALSE;
	} 
	return TRUE;
}

//------------------------------------------------------------------------
// Delete directories
//------------------------------------------------------------------------
BOOL	DeleteDir (HWND hwnd, char *szDir, BOOL bRecurse) {

	FIND_T	fs;
	int		rc, sFlags = _A_ARCH | _A_SUBDIR;

	strcat (szDir, "\\*.*");
	rc = _dos_findfirst (szDir, sFlags | sSearchFlags, &fs);
	*strrchr (szDir, '\\') = '\0';

	while (rc == 0) {
		strcat (szDir, "\\"); 
		strcat (szDir, fs.name);

		if (fs.attrib & _A_SUBDIR) {
			if (bRecurse && strcmp (fs.name, ".") != 0 
			             && strcmp (fs.name, "..") != 0) 
				if (!DeleteDir (hwnd, szDir, bRecurse))
					rc = 1;
		} else
	 		if (!DeleteFile(hwnd, szDir))
				rc = 1;
		*strrchr (szDir, '\\') = '\0';

		if (!rc)
			rc = _dos_findnext (&fs);
	}
	if (rc != 0x12) {
		PrintError (hwnd, GetErrorCode(), szDir);
		return FALSE;
	}
	if (strlen (szDir) != 3) {		
		if (rmdir (szDir)) {
			rc = GetErrorCode();
			if (rc == 16 || rc == 5) {
				char	szTemp[128];
				strcpy (szTemp, szDir);
				*strrchr (szTemp, '\\') = '\0';
				if (strlen (szTemp) == 2) strcat (szTemp, "\\");
				chdir (szTemp);
				rc = 0;
				if (rmdir (szDir)) rc = GetErrorCode();
			}
			if (rc) {
				PrintError (hwnd, GetErrorCode(), szDir);
				return FALSE;
			}
		}
	}
	return TRUE;
}

//========================================================================
//
// Control Subclass Procedures
//
//========================================================================
//------------------------------------------------------------------------
//
// Listbox subclass procedure
//
//------------------------------------------------------------------------
long FAR PASCAL ListBoxSubclassProc (HWND hwnd, WORD message, WORD wParam, 
                                     LONG lParam) {
int	sLBoxID;
sLBoxID = GetWindowWord (hwnd, GWW_ID);
switch (message) {

	case WM_SETFOCUS:
		hwndFocusWin = hwnd;
		break;

	case WM_KEYDOWN:
		switch (wParam) {

			case VK_RETURN:
				SendMessage (GetParent (hwnd), WM_COMMAND, 
				             sLBoxID,
				             MAKELONG (hwnd, LBN_DBLCLK));
				break;

			case VK_TAB:

				if (sLBoxID == IDD_DRVBOX) {
					if (GetKeyState (VK_SHIFT) < 0)
						SetFocus (hwndEditBox);
					else
						SetFocus (hwndLBoxArray[0]);
				} else 					       
					if (GetKeyState (VK_SHIFT) < 0) {
						if (sLBoxID == IDD_1STBOX)
							SetFocus (hwndDrvBox);
						else 
							SetFocus(hwndLBoxArray[sLBoxID-IDD_1STBOX-1]);
					} else {
						if (sLBoxID == IDD_1STBOX + sLBoxVisible - 1)
							SetFocus (hwndEditBox);
						else 
							SetFocus(hwndLBoxArray[sLBoxID-IDD_1STBOX+1]);
					}
				break;
		}
		break;
}
return CallWindowProc (lpfnOldLBoxProc[sLBoxID-IDD_DRVBOX], hwnd, 
                       message, wParam, lParam);
}

//------------------------------------------------------------------------
//
// Editbox subclass procedure
//
//------------------------------------------------------------------------
long FAR PASCAL EditBoxSubclassProc (HWND hwnd, WORD message, WORD wParam, 
                                     LONG lParam) {

	switch (message) {

		case WM_SETFOCUS:
			hwndFocusWin = hwnd;
			break;

		case WM_KEYDOWN:
			switch (wParam) {

				case VK_RETURN:
					SendMessage (GetParent (hwnd), 
					             WUM_SETPATH, 0, 0);
					return 0;

				case VK_TAB:

					if (GetKeyState (VK_SHIFT) < 0)
						SetFocus (hwndLBoxArray[sLBoxVisible-1]);
					else
						SetFocus (hwndDrvBox);
					return 0;
			}
			break;
		case WM_CHAR:
			switch (wParam) {

				case '\t':
					return 0;

				case '\r':
					return 0;
			}
			break;
	}
	return CallWindowProc (lpfnOldEditBoxProc, hwnd, message, 
	                       wParam, lParam);
}


//========================================================================
//
// Window Proceedures for Dialog Boxes
//
//========================================================================
//------------------------------------------------------------------------
//
//Configure Dialog procedure
//
//------------------------------------------------------------------------
BOOL FAR PASCAL ConfigDlgProc (HWND hwnd, WORD message, WORD wParam, 
                               LONG lParam) {

static	int	sLocalButtons [BUTTONMAX], sLocalTotal;
int		i;
char 		szTemp [128], szTemp1[8];
BOOL		bGood;

switch (message) {
	case WM_INITDIALOG:

		for (i = 0; i < BUTTONMAX; i++) 
			sLocalButtons[i] = bsButtons[i].sFunction;
		sLocalTotal = sButtonCount;
			
		SendDlgItemMessage (hwnd, IDD_EDITORNAME, EM_LIMITTEXT, 
		                    sizeof (szDefaultEditor), 0L);	
		SetDlgItemText (hwnd, IDD_EDITORNAME, szDefaultEditor);

		if (sSearchFlags && 0x0002)
			CheckDlgButton (hwnd, IDD_INCHIDDEN, TRUE);
		else
			CheckDlgButton (hwnd, IDD_INCHIDDEN, FALSE);
		if (sSearchFlags && 0x0004)
			CheckDlgButton (hwnd, IDD_INCSYSTEM, TRUE);
		else
			CheckDlgButton (hwnd, IDD_INCSYSTEM, FALSE);

		if (bDirFirst)
			CheckRadioButton (hwnd, IDD_SRTDIR1ST, 
			                IDD_SRTDIRLST, IDD_SRTDIR1ST);
		else
			CheckRadioButton (hwnd, IDD_SRTDIR1ST, 
			                IDD_SRTDIRLST, IDD_SRTDIRLST);

		CheckDlgButton (hwnd, IDD_CONFFDEL, bConfirmFileDel);
		CheckDlgButton (hwnd, IDD_CONFDDEL, bConfirmDirDel);

		SendDlgItemMessage (hwnd, IDD_BTNTOTAL, 
		                    EM_LIMITTEXT, 2, 0L);
		SetDlgItemInt (hwnd, IDD_BTNTOTAL, sButtonCount, 
		               FALSE);
		SendDlgItemMessage (hwnd, IDD_BTNNUM, 
		                    EM_LIMITTEXT, 2, 0L);
		SetDlgItemInt (hwnd, IDD_BTNNUM, 1, FALSE);
		SendDlgItemMessage (hwnd, IDD_BTNNUM, 
		                    CB_RESETCONTENT, 0, 0L);
		for (i = 0; i < 10; i++) {
			itoa (i+1, szTemp1, 10);
			strcpy (szTemp, "Button ");
			strcat (szTemp, szTemp1);
			SendDlgItemMessage (hwnd, IDD_BTNNUM, 
			                    CB_ADDSTRING, 0,
			                    STRTOLP (szTemp));
		}
		SendDlgItemMessage (hwnd, IDD_BTNNUM, 
		                    CB_SETCURSEL, 0, 0);
		SendDlgItemMessage (hwnd, IDD_BTNFUNC, 
		                    CB_RESETCONTENT, 0, 0L);
		for (i = 0; i < 10; i++) 
			SendDlgItemMessage (hwnd, IDD_BTNFUNC, 
			              CB_ADDSTRING, 0, 
			              STRTOLP (fsFunctions[i].szText));
		SendDlgItemMessage (hwnd, IDD_BTNFUNC, CB_SETCURSEL, 
		                    sLocalButtons[0], 0);
		return TRUE;

	case WM_COMMAND:
		switch (wParam) {
			case IDD_BTNTOTAL:
				if (HIWORD (lParam) == EN_KILLFOCUS) {
					i=GetDlgItemInt (hwnd, 
					        IDD_BTNTOTAL, &bGood, FALSE);
					if (!bGood) 
						SetDlgItemInt (hwnd, 
						    IDD_BTNTOTAL, 
						    sLocalTotal, FALSE);
					else {
						if (i<0 || i>BUTTONMAX) {
							i = max (1, min (i, BUTTONMAX));
							SetDlgItemInt (hwnd, IDD_BTNTOTAL, 
							               sLocalTotal, FALSE);
						}
						sLocalTotal = i;
					}
				}
				break;

			case IDD_BTNNUM:
				if (HIWORD (lParam) == CBN_SELCHANGE) {

					i = (int) SendDlgItemMessage(hwnd, IDD_BTNNUM, 
					                             CB_GETCURSEL, 0, 0);
					SendDlgItemMessage (hwnd, IDD_BTNFUNC, CB_SETCURSEL, 
					                    sLocalButtons[i], 0);
				}
				break;

			case IDD_BTNFUNC:
				if (HIWORD (lParam) == CBN_SELCHANGE) {
					i = (int) SendDlgItemMessage (hwnd, IDD_BTNNUM, 
					                              CB_GETCURSEL, 0, 0L);
					sLocalButtons[i] = (int) SendDlgItemMessage (hwnd, 
					                          IDD_BTNFUNC, CB_GETCURSEL, 
					                          0, 0L);
				}
				break;

			case IDD_SRTDIR1ST:
			case IDD_SRTDIRLST:
				CheckRadioButton (hwnd, IDD_SRTDIR1ST, IDD_SRTDIRLST, 
				                  wParam);
				break;

			case IDOK:
				//
				// Set include flags from check box settings
				//
				sSearchFlags &= 0xFFF9;
				if (IsDlgButtonChecked (hwnd, IDD_INCHIDDEN) == 1)
					sSearchFlags |= 0x0002;
				if (IsDlgButtonChecked (hwnd, IDD_INCSYSTEM) == 1)
					sSearchFlags |= 0x0004;
				//
				// Set Confirmation flags from check box settings
				//
				if (IsDlgButtonChecked (hwnd, IDD_CONFFDEL) == 1)
					bConfirmFileDel = TRUE;
				else
					bConfirmFileDel = FALSE;
				if (IsDlgButtonChecked (hwnd, IDD_CONFDDEL) == 1)
					bConfirmDirDel = TRUE;
				else
					bConfirmDirDel = FALSE;
				//
				// Set Directory sort flag
				//
				if (IsDlgButtonChecked (hwnd, IDD_SRTDIR1ST) == 1)
					bDirFirst = TRUE;
				else
					bDirFirst = FALSE;
				//
				// Assign new button count and functions. 
				//
				sButtonCount = GetDlgItemInt (hwnd, IDD_BTNTOTAL, 
				                              &bGood, FALSE);

				for (i = 0; i < BUTTONMAX; i++) 
					if (bsButtons[i].sFunction != sLocalButtons[i]) {
						bsButtons[i].sFunction = sLocalButtons[i];
						if (bsButtons[i].hwndButton != 0)
							SetWindowText (bsButtons[i].hwndButton,
						     fsFunctions [bsButtons [i].sFunction].szText);
					}
				//
				// Assign new default editor name.
				//
				GetDlgItemText (hwnd, IDD_EDITORNAME, szDefaultEditor,
				                sizeof (szDefaultEditor));
				EndDialog (hwnd, 1);
				return TRUE;

			case IDCANCEL:	
				EndDialog (hwnd, 0);
				return TRUE;
		}
		break;
	case WM_CLOSE:
		EndDialog (hwnd, 0);
		return TRUE;
}
return FALSE;	
}

//------------------------------------------------------------------------
//
//Copy/Rename Dialog procedure
//
//------------------------------------------------------------------------
BOOL FAR PASCAL CopyRenDlgProc (HWND hwnd, WORD message, WORD wParam, 
                               LONG lParam) {
char	szTemp[80];
char	szSelect[16];
char	szSavePath[80];
int	i;

switch (message) {

	case WM_INITDIALOG:

		SendDlgItemMessage (hwnd, IDD_DESTNAME, EM_LIMITTEXT, 
		                    sizeof (szTemp), 0L);	

		getcwd (szSavePath, sizeof (szSavePath));
		strcpy (szTemp, szMasterPath);
		if (!bDir) *strrchr (szTemp, '\\') = '\0';
		strcpy (szDestPath, szTemp);
		DlgDirList (hwnd, szDestPath, IDD_FDRVBOX, 0, 0xC000);
		strcpy (szDestPath, szTemp);
		DlgDirList (hwnd, szDestPath, IDD_FDIRBOX, 0, 0x8010);
		strcpy (szDestPath, szTemp);
		DlgDirList (hwnd, szDestPath, IDD_FFILEBOX, 0, 0x0007);
		SetDlgItemText (hwnd, IDD_DESTNAME, szTemp);
		strcpy (szTemp, szFunction);
		if (bDir)
			strcat (szTemp, " Directory:");
		else
			strcat (szTemp, " File:");
		SetDlgItemText (hwnd, IDD_FUNCNAME, szTemp);			
		SetDlgItemText (hwnd, IDD_SRCNAME, szMasterPath);
		if (strcmp (szFunction, "Rename") == 0) {
			CheckDlgButton (hwnd, IDD_SRCRECURSE, 2);
			ShowWindow (GetDlgItem (hwnd, IDD_SRCRECURSE), SW_HIDE);
		} else {
			ShowWindow (GetDlgItem (hwnd, IDD_SRCRECURSE), SW_SHOW);
			CheckDlgButton (hwnd, IDD_SRCRECURSE, 0);
		}
		return TRUE;

	case WM_COMMAND:

		if ((wParam == IDD_FDRVBOX || wParam == IDD_FDIRBOX ||
		     wParam == IDD_FFILEBOX) && 
			 (HIWORD (lParam) == LBN_DBLCLK ||
			  HIWORD (lParam) == LBN_SELCHANGE)) {

			DlgDirSelect (hwnd, szSelect, wParam);

			switch (wParam) {
				case IDD_FDRVBOX:
					if (HIWORD (lParam) == LBN_DBLCLK) {
						_chdrive (szSelect [0] - 'a' + 1);
					   getcwd (szTemp, sizeof (szTemp));
					} else {
						strcpy (szTemp, szSelect);
						strcat (szTemp, "\\");
					}
					break;

				case IDD_FDIRBOX:

				   getcwd (szTemp, sizeof (szTemp));
					if (strcmp (szSelect, "..\\") == 0) {
						*strrchr (szTemp, '\\') = '\0';
						if (strlen (szTemp) == 2) strcat (szTemp, "\\");
					} else {
						if (strlen (szTemp) != 3) strcat (szTemp, "\\");
						strcat (szTemp, szSelect);
						*strrchr (szTemp, '\\') = '\0';
					}	
					if (HIWORD (lParam) == LBN_DBLCLK) 
							chdir (szTemp);
					break;	

				case IDD_FFILEBOX:
				   getcwd (szTemp, sizeof (szTemp));
					if (strlen (szTemp) != 3) strcat (szTemp, "\\");
					strcat (szTemp, szSelect);
					if (HIWORD (lParam) == LBN_DBLCLK) 
						PostMessage (hwnd, WM_COMMAND, IDOK, 0L);
					break;
			}
			if (HIWORD (lParam) == LBN_DBLCLK) {
				strcpy (szDestPath, szTemp);
				DlgDirList (hwnd, szDestPath, IDD_FDRVBOX, 0, 0xC000);
				strcpy (szDestPath, szTemp);
				DlgDirList (hwnd, szDestPath, IDD_FDIRBOX, 0, 0x8010);
				strcpy (szDestPath, szTemp);
				DlgDirList (hwnd, szDestPath, IDD_FFILEBOX, 0, 0x0007);
			}
			SetDlgItemText (hwnd, IDD_DESTNAME, szTemp);
		}
		switch (wParam) {

			case IDD_DESTNAME:
				//
				// Disable automatic text selection when gaining focus
				//
				if (HIWORD (lParam) == EN_SETFOCUS) {
					GetDlgItemText (hwnd, IDD_DESTNAME, 
					              szTemp, sizeof (szTemp));
					i = strlen (szTemp);
					SendMessage ((HWND) LOWORD (lParam), 
					             EM_SETSEL, 0, 
					             MAKELONG (i,i));
				}
				break;
			case IDOK:
				//
				// On OK, get destination filename text.  Compare it with
				// the source filename.  If equal, reject dest name. If 
				// dest name ok, get Inc Directories button state.
				//
				GetDlgItemText (hwnd, IDD_DESTNAME, szTemp, 
				                sizeof (szTemp));
				_fullpath (szDestPath, szTemp, 
				           sizeof (szDestPath));
				if (strcmp (szMasterPath, szDestPath) == 0) {
					MessageBox (hwnd,
					 "Source and Destination names cannot be the same", 
					            szAppName, MB_OK);
					break;
				} 
				if (IsDlgButtonChecked(hwnd,IDD_SRCRECURSE)==1)
					bIncDir = TRUE;
				else
					bIncDir = FALSE;
				chdir (szSavePath);
				EndDialog (hwnd, 1);
				return TRUE;

			case IDCANCEL:	
				chdir (szSavePath);
				EndDialog (hwnd, 0);
				return TRUE;
		}
		break;

	case WM_CLOSE:
		EndDialog (hwnd, 0);
		return TRUE;
}
return FALSE;	
}

//------------------------------------------------------------------------
//
//Make Directory Dialog procedure
//
//------------------------------------------------------------------------
BOOL FAR PASCAL MkDirDlgProc (HWND hwnd, WORD message, WORD wParam, 
                               LONG lParam) {
	char	szTemp[80];

	switch (message) {

		case WM_INITDIALOG:

			SendDlgItemMessage (hwnd, IDD_DIRNAME, EM_LIMITTEXT, 
			                    sizeof (szTemp), 0L);	

			SetDlgItemText (hwnd, IDD_CURRDIR, szMasterPath);

			return TRUE;

		case WM_COMMAND:

			switch (wParam) {

				case IDOK:
					//
					// On OK, get new directory name.
					//
					GetDlgItemText (hwnd, IDD_DIRNAME, 
					                szTemp, 
					                sizeof (szTemp));
					_fullpath (szDestPath, szTemp, 
					           sizeof (szDestPath));
					EndDialog (hwnd, 1);
					return TRUE;

				case IDCANCEL:	
					EndDialog (hwnd, 0);
					return TRUE;
			}
			break;

		case WM_CLOSE:
			EndDialog (hwnd, 0);
			return TRUE;
	}
	return FALSE;	
}

//------------------------------------------------------------------------
//
//Set File Attribute dialog procedure
//
//------------------------------------------------------------------------
BOOL FAR PASCAL SetAttrDlgProc (HWND hwnd, WORD message, WORD wParam, 
                               LONG lParam) {
static	int	sFileFlags;
char		szTemp[128];

switch (message) {

	case WM_INITDIALOG:

		SetDlgItemText (hwnd, IDD_SRCNAME, szMasterPath);

		strcpy (szTemp, "Set file attributes for ");
		if (bDir)
			strcat (szTemp, "Directory:");
		else
			strcat (szTemp, "File:");
		SetDlgItemText (hwnd, IDD_FUNCNAME, szTemp);			

		if (_dos_getfileattr (szMasterPath, &sFileFlags))
			return FALSE;

		if (sFileFlags & _A_HIDDEN)
			CheckDlgButton (hwnd, IDD_FILEHIDDEN, TRUE);
		else
			CheckDlgButton (hwnd, IDD_FILEHIDDEN, FALSE);

		if (sFileFlags & _A_SYSTEM)
			CheckDlgButton (hwnd, IDD_FILESYSTEM, TRUE);
		else
			CheckDlgButton (hwnd, IDD_FILESYSTEM, FALSE);

		if (sFileFlags & _A_RDONLY)
			CheckDlgButton (hwnd, IDD_FILEREADONLY, TRUE);
		else
			CheckDlgButton (hwnd, IDD_FILEREADONLY, FALSE);

		if (sFileFlags & _A_ARCH)
			CheckDlgButton (hwnd, IDD_FILEARCHIVE, TRUE);
		else
			CheckDlgButton (hwnd, IDD_FILEARCHIVE, FALSE);

		return TRUE;

	case WM_COMMAND:

		switch (wParam) {

			case IDOK:

				//
				// Set include flags from check box settings
				//
				
				if (IsDlgButtonChecked (hwnd, IDD_FILEHIDDEN) == 1)
					sFileFlags |= _A_HIDDEN;
				else	
					sFileFlags &= ~_A_HIDDEN;

				if (IsDlgButtonChecked (hwnd, IDD_FILESYSTEM) == 1)
					sFileFlags |= _A_SYSTEM;
				else	
					sFileFlags &= ~_A_SYSTEM;

				if (IsDlgButtonChecked (hwnd, IDD_FILEREADONLY) == 1)
					sFileFlags |= _A_RDONLY;
				else	
					sFileFlags &= ~_A_RDONLY;

				if (IsDlgButtonChecked (hwnd, IDD_FILEARCHIVE) == 1)
					sFileFlags |= _A_ARCH;
				else	
					sFileFlags &= ~_A_ARCH;

				_dos_setfileattr (szMasterPath, sFileFlags);
				EndDialog (hwnd, 1);
				return TRUE;

			case IDCANCEL:	
				EndDialog (hwnd, 0);
				return TRUE;
		}
		break;
	case WM_CLOSE:
		EndDialog (hwnd, 0);
		return TRUE;
}
return FALSE;	
}

//------------------------------------------------------------------------
//
// File Infomation Dialog procedure
//
//------------------------------------------------------------------------
BOOL FAR PASCAL FileInfoDlgProc (HWND hwnd, WORD message, WORD wParam, 
                                 LONG lParam) {
	char		szTemp[128], szTemp1[32];
	FILEINFO	fiData;
	int		i, rc, sNumLen;
	int		sFlags = _A_ARCH | _A_NORMAL | _A_HIDDEN | 
			         _A_RDONLY | _A_SYSTEM | _A_SUBDIR;

	switch (message) {

		case WM_INITDIALOG:

			rc = GetFileInfo (szMasterPath, &fiData);
			if (rc) {
				PrintError (hwnd, rc, szMasterPath);
				return FALSE;
			}
			SetDlgItemText (hwnd, IDD_INFONAME, fiData.szName);

			GetFileType (szMasterPath, fiData.wAttr, szTemp1, 
			             szTemp);
			SetDlgItemText (hwnd, IDD_INFOTYPE, szTemp1);
			//
			// Insert commas in file size
			//
			ltoa (fiData.lSize, szTemp, 10);
			strcpy (szTemp1, "");
			sNumLen = strlen (szTemp);

			for (i = 0; i < sNumLen; i++) {
				if ((sNumLen - i) % 3 == 0 && i != 0)
					strcat (szTemp1, ",");
				strncat (szTemp1, szTemp+i,1);
			}
			strcat (szTemp1, " Bytes");
			SetDlgItemText (hwnd, IDD_INFOSIZE, szTemp1);

			ConvertDate (fiData.wDate, szTemp);
			SetDlgItemText (hwnd, IDD_INFODATE, szTemp);

			ConvertTime (fiData.wTime, szTemp);
			SetDlgItemText (hwnd, IDD_INFOTIME, szTemp);

			strcpy (szTemp, "");
			if (fiData.wAttr & _A_ARCH)
				strcat (szTemp, "Archive");
			if (fiData.wAttr & _A_HIDDEN) {
				if (strlen (szTemp) != 0)
					strcat (szTemp, ", ");
				strcat (szTemp, "Hidden");
			}
			if (fiData.wAttr & _A_SYSTEM) {
				if (strlen (szTemp) != 0)
					strcat (szTemp, ", ");
				strcat (szTemp, "System");
			}
			if (fiData.wAttr & _A_RDONLY) {
				if (strlen (szTemp) != 0)
					strcat (szTemp, ", ");
				strcat (szTemp, "Read Only");
			}
			SetDlgItemText (hwnd, IDD_INFOFLAGS, szTemp);

			return TRUE;

		case WM_COMMAND:

			switch (wParam) {
				case IDOK:
					EndDialog (hwnd, 1);
					return TRUE;
			}
			break;

		case WM_CLOSE:
			EndDialog (hwnd, 0);
			return TRUE;
	}	
	return FALSE;	
}
//------------------------------------------------------------------------
//
// About box dialog procedure
//
//------------------------------------------------------------------------
BOOL FAR PASCAL AboutDlgProc (HWND hwnd, WORD message, WORD wParam, 
                               LONG lParam) {

	switch (message) {

		case WM_COMMAND:

			switch (wParam) {

				case IDOK:

					EndDialog (hwnd, 1);
					return TRUE;
			}
			break;

		case WM_CLOSE:
			EndDialog (hwnd, 0);
			return TRUE;
	}
	return FALSE;	
}

//========================================================================
//
// Main Window Procedure.
//
//========================================================================
//------------------------------------------------------------------------
//
// Window Procedure
//
//------------------------------------------------------------------------
long FAR PASCAL WndProc (HWND hwnd, WORD message, WORD wParam, LONG lParam) {

	static		char  	szPrograms [80];
	static		int   	sWinHeight, sWinWidth, sOldBCnt;
	static		FARPROC	lpfnConfigDlgProc, lpfnCopyRenDlgProc;
	static		FARPROC	lpfnSetAttrDlgProc, lpfnAboutDlgProc;
	static		FARPROC	lpfnFileInfoDlgProc, lpfnMkDirDlgProc;

	char		szTemp [128], szSel [128];	
	char		szDrive [3], szDir [128], szName [9], szExt [5];
	int		i, sLBoxID, sNewCount, rc, sBHeight;
	RECT		rect;
	BOOL		bTemp;

	switch (message) {
	
		case WM_CREATE:
			//
			//Initialize Program ID string
			//
			GetProfileString ("windows", "Programs", "com exe", szPrograms,
			                  sizeof (szPrograms));
			strupr (szPrograms);
			strcat (szPrograms, " ");
			//
			//Create thunks for dialog box and subclass procedures 
			//
			hInstance = ((LPCREATESTRUCT) lParam)->hInstance;

			lpfnAboutDlgProc = MakeProcInstance (AboutDlgProc, hInstance);
			lpfnConfigDlgProc = MakeProcInstance (ConfigDlgProc, hInstance);
			lpfnCopyRenDlgProc = MakeProcInstance (CopyRenDlgProc, hInstance);
			lpfnMkDirDlgProc = MakeProcInstance (MkDirDlgProc, hInstance);
			lpfnSetAttrDlgProc = MakeProcInstance (SetAttrDlgProc, hInstance);
			lpfnFileInfoDlgProc = MakeProcInstance (FileInfoDlgProc, hInstance);

			lpfnListBoxSubclassProc = 
 			       MakeProcInstance ((FARPROC) ListBoxSubclassProc, hInstance);
			lpfnEditBoxSubclassProc = 
			       MakeProcInstance ((FARPROC) EditBoxSubclassProc, hInstance);
			//
			//Create Child Windows (ListBoxes, Buttons, and Edit Box)
			//
			GetClientRect (hwnd, &rect);
			sWinHeight = rect.bottom - rect.top;
			sWinWidth = rect.right - rect.left;
	
			hwndDrvBox = CreateWindow ("listbox",
	                      NULL,
	                      WS_CHILD | WS_VISIBLE | WS_VSCROLL |
	                      WS_BORDER | LBS_NOTIFY | LBS_NOINTEGRALHEIGHT,
	                      5,                            //x Pos
	                      5,                            //y Pos
	                      sDrvBoxWidth,                 //x Size
	                      LBoxHeight (sWinHeight),      //y Size
	                      hwnd,                         //Parent handle
	                      IDD_DRVBOX,                   //ID number
	                      hInstance,
	                      NULL);
			//
			// Subclass drive listbox
			//
			lpfnOldLBoxProc[0] = (FARPROC) GetWindowLong (hwndDrvBox, 
			                                              GWL_WNDPROC);
			SetWindowLong (hwndDrvBox, GWL_WNDPROC, 
			               (LONG) lpfnListBoxSubclassProc);
			//
			// Create directory listboxes
			//
			for (i = 0; i < NumLBoxes (sWinWidth) && i < LBOXMAX; i++) {
		   	hwndLBoxArray[i] = CreateWindow ("listbox",
	                      NULL, 
	                      WS_CHILD | WS_VISIBLE | WS_VSCROLL |
	                      WS_BORDER | LBS_NOTIFY | LBS_NOINTEGRALHEIGHT,
	                      LBoxPos (i),                  //x Pos
	                      5,                            //y Pos
	                      sLBoxWidth,                   //x Size
	                      LBoxHeight (sWinHeight),      //y Size
	                      hwnd,                         //Parent handle 
	                      i+IDD_1STBOX,                 //ID number
	                      hInstance,
	                      NULL);
		   	sLBoxVisible++;
				//
				// Subclass each directory listbox
				//
				lpfnOldLBoxProc[i+1]=(FARPROC) GetWindowLong (hwndLBoxArray[i], 
				                                              GWL_WNDPROC);
				SetWindowLong (hwndLBoxArray[i], GWL_WNDPROC, 
				               (LONG) lpfnListBoxSubclassProc);
			}
			//
			// Create Button windows
			//
			sBHeight = GetButtonHeight (rect.bottom);

			for (i = 0; i < sButtonCount; i++) 
			   bsButtons [i].hwndButton = CreateWindow ("button",
	                      fsFunctions [bsButtons [i].sFunction].szText,
	                      WS_CHILD | WS_VISIBLE | BS_PUSHBUTTON,
	                      rect.right - BUTTONWIDTH - 5, //x Pos
	                      5+(i * (sBHeight+5)),         //y Pos
	                      BUTTONWIDTH,                  //x Size
	                      sBHeight,		 			  	//y Size
	                      hwnd,                         //Parent handle
	                      i+IDD_1STBUTTON,              //ID number
	                      hInstance,
	                      NULL);

			hwndScrollBar = CreateWindow ("scrollbar",
	                   NULL,
	                   WS_CHILD | WS_VISIBLE | SBS_HORZ,
	                   5,                      				//Left of rect
	                   rect.bottom - (2*sEditBoxHeight)-10,//y Pos
	                   EBoxWidth (sLBoxVisible), 			//x Size
	                   sEditBoxHeight,						//y Size
	                   hwnd,                   				//Parent handle
	                   IDD_SCROLLBAR,          				//ID number
	                   hInstance,
	                   NULL);

			hwndEditBox = CreateWindow ("edit",
	                   NULL,
	                   WS_CHILD | WS_VISIBLE | WS_BORDER |
	                   ES_UPPERCASE | ES_LEFT,
	                   5,                      				//x Pos
	                   rect.bottom - sEditBoxHeight - 5,	//y Pos
	                   EBoxWidth (sLBoxVisible), 			//x Size
	                   sEditBoxHeight,          			//y Size
	                   hwnd,                   				//Parent handle
	                   IDD_EDITBOX,            				//ID number
	                   hInstance,
	                   NULL);
			//
			// Subclass edit box
			//
			lpfnOldEditBoxProc = (FARPROC) GetWindowLong (hwndEditBox, 
			                                              GWL_WNDPROC);
			SetWindowLong (hwndEditBox, GWL_WNDPROC, 
			               (LONG) lpfnEditBoxSubclassProc);

			//
			// Fill drive box, then select the active drive
			//
			SendMessage (hwndDrvBox, LB_DIR, 0xC000, STRTOLP ("*.*"));

			FillListBoxes (hwnd);

			hwndFocusWin = hwndDrvBox;

			bDir = TRUE;
			bExecutable = EnableButtons (hwnd, FALSE, bDir);
			return 0;


		case WM_SETFOCUS:
			SetFocus (hwndFocusWin);
			break;

		case WM_SIZE:
			//
		 	//Check for changes in Window Width
		 	//
			sNewCount = NumLBoxes (LOWORD (lParam));
			if (sLBoxVisible > sNewCount) {
				for (i = sNewCount; i <= sLBoxVisible; i++)
				DestroyWindow (hwndLBoxArray[i]);
			} else {
				if (sLBoxVisible < sNewCount && sLBoxVisible != LBOXMAX) 
					for (i=sLBoxVisible; i<sNewCount && i<LBOXMAX;i++) {
						hwndLBoxArray[i] = CreateWindow (
						       "listbox",
						       NULL,
						       WS_CHILD | WS_VISIBLE | 
						       WS_VSCROLL | WS_BORDER | 
						       LBS_NOTIFY | 
						       LBS_NOINTEGRALHEIGHT,
						       LBoxPos (i), 
						       5,           
						       sLBoxWidth,  
						       LBoxHeight (HIWORD (lParam)),
						       hwnd,                       
						       i+IDD_1STBOX,               
						       hInstance,
						       NULL);
						//
						// Subclass each new listbox
						//
						lpfnOldLBoxProc[i+1] = 
						    (FARPROC) GetWindowLong (
						           hwndLBoxArray[i], 
						           GWL_WNDPROC);
						SetWindowLong (hwndLBoxArray[i], 
						      GWL_WNDPROC, 
						      (LONG) lpfnListBoxSubclassProc);
			         	}
			}
			sWinWidth = LOWORD (lParam);
			//
			//Check for changes in Window Height
			//
			if (HIWORD (lParam) != (WORD) sWinHeight) {
				for (i = 0; i < sLBoxVisible; i++)
					SetWindowPos (hwndLBoxArray[i], NULL, 
					         0, 0, sLBoxWidth,
					         LBoxHeight (HIWORD (lParam)), 
					         SWP_NOMOVE);
				SendMessage (hwndDrvBox, LB_SETTOPINDEX, 0,0L);
				SetWindowPos (hwndDrvBox, NULL, 0, 0, 
				              sDrvBoxWidth,
				              LBoxHeight (HIWORD (lParam)), 
				              SWP_NOMOVE);
				sWinHeight = HIWORD (lParam);
			}
			if (sNewCount != sLBoxVisible) {
				sLBoxVisible = sNewCount;
				FillListBoxes (hwnd);
			}
	   		SetWindowPos (hwndScrollBar, NULL, 5, 
			              HIWORD (lParam) - (2*sEditBoxHeight) - 10,
			              EBoxWidth (sLBoxVisible), sEditBoxHeight, NULL);

			SetWindowPos (hwndEditBox, NULL, 5, 
			              HIWORD (lParam) - sEditBoxHeight - 5,
			              EBoxWidth (sLBoxVisible), sEditBoxHeight, NULL);
			SetButtonPos (sWinHeight, sWinWidth);
			return 0;


		case WM_COMMAND:
			//
			// First, check for any menu selections.
			//
			switch (wParam) {

				case IDM_OPEN:
					if (bExecutable)
						strcpy (szTemp, szMasterPath);
					else {
						_splitpath (szMasterPath, szDrive, szDir, szName, 
						            szExt);
						if (GetProfileString ("Extensions", szExt+1, "", 
						                      szSel, sizeof (szSel))) {

							strcpy (szTemp, szSel);
							szTemp [i = strcspn (szSel, "^")] = 0;
							strcat (szTemp, szDrive);
							strcat (szTemp, szDir);
							strcat (szTemp, szName);
							strcat (szTemp, szSel+i+1);	
						} else {
							strcpy (szTemp, szDefaultEditor);
							strcat (szTemp, " ");
							strcat (szTemp, szMasterPath);
						}
					}
					WinExec (szTemp, SW_SHOW);
					return 0;

				case IDM_COPY:
					strcpy (szFunction, "Copy");
					if (DialogBox (hInstance, "CopyRen", hwnd, 
					               lpfnCopyRenDlgProc)) {
						if (bDir) {
	  						if (CopyDir (hwnd, szDestPath, szMasterPath, 
							             bIncDir))
								FillListBoxes (hwnd);
						} else {
	  						if (CopyFile (hwnd, szDestPath, szMasterPath))
								FillListBoxes (hwnd);
						}
					}
					return 0;

				case IDM_MOVE:
					strcpy (szFunction, "Move");
					if (DialogBox (hInstance, "CopyRen", hwnd, 
					               lpfnCopyRenDlgProc)) {						
						if (bDir) {
							if (CopyDir (hwnd, szDestPath, szMasterPath, 
							             bIncDir)) 
								DeleteDir (hwnd, szMasterPath, bIncDir);
						} else {
							if (CopyFile (hwnd, szDestPath, szMasterPath)) 
								DeleteFile (hwnd, szMasterPath);
						}
						FillListBoxes (hwnd);
					}
					return 0;


				case IDM_RENAME:
					strcpy (szFunction, "Rename");
					if (DialogBox (hInstance, "CopyRen", hwnd, 
					               lpfnCopyRenDlgProc)) {
						getcwd (szTemp, sizeof (szTemp));
						if (bDir) chdir ("\\");
						rc = rename (szMasterPath, szDestPath);
						if (bDir) {
							strcpy (szMasterPath, szDestPath);
							chdir (szMasterPath);
						}						
						if (rc) 
							PrintError (hwnd, GetErrorCode (), szMasterPath);

						FillListBoxes (hwnd);
					}
					return 0;


				case IDM_MKDIR:
					if (DialogBox (hInstance, "MkDir", hwnd, 
					               lpfnMkDirDlgProc)) {

					   getcwd (szTemp, sizeof (szTemp));
						if (chdir (szDestPath)) {
							if (MakeDir (szDestPath)) 
								FillListBoxes (hwnd);
							else
								PrintError (hwnd, GetErrorCode(), szDestPath);
						} else {
							chdir (szTemp);
							strcpy (szTemp, "The directory: ");
							strcat (szTemp, szDestPath);
							strcat (szTemp, " already exists.");
						   MessageBox (hwnd, szTemp, szAppName, MB_OK);
						}
					}
					return 0;


				case IDM_ATTRIB:
					if (DialogBox (hInstance, "SetAttr", hwnd, 
					               lpfnSetAttrDlgProc)) 
						FillListBoxes (hwnd);
					return 0;


				case IDM_INFO:
					DialogBox (hInstance, "FileInfo", hwnd, 
					           lpfnFileInfoDlgProc);
					return 0;


				case IDM_DELETE:
					//
					// Before deleteing file or directory, 
					// display confirmation box if necessary.
					//
					strcpy (szTemp, "Do you wish to delete the ");
					if (bDir) {
						strcat (szTemp, "directory:\n");
						bTemp = bConfirmDirDel;
					} else {
						strcat (szTemp, "file:\n");
						bTemp = bConfirmFileDel;
					}
					strcat (szTemp, szMasterPath);
					if (bTemp)
						if (MessageBox (hwnd, szTemp, szAppName, 
						                MB_YESNO) != IDYES) 
							return 0;
					//
					// Delete file or directory.
					//
					if (bDir) {
						strcpy (szTemp, szMasterPath);
						if (strlen (szTemp) != 3)
							*strrchr (szTemp, '\\') = '\0';
						chdir (szTemp);
						if (DeleteDir (hwnd, szMasterPath, TRUE))
							FillListBoxes (hwnd);
					} else 
						if (DeleteFile (hwnd, szMasterPath))
							FillListBoxes (hwnd);
					return 0;


				case IDM_CONFIG:
					sOldBCnt = sButtonCount;
					if (DialogBox (hInstance, "Config", hwnd, 
					               lpfnConfigDlgProc)) {

						if (sOldBCnt > sButtonCount) {
							for (i = sOldBCnt-1; i >= sButtonCount; i--) {
					         DestroyWindow (bsButtons[i].hwndButton);
					         bsButtons[i].hwndButton = 0;
							}	
						}
						if (sOldBCnt < sButtonCount) {

							sBHeight = GetButtonHeight (sWinHeight);

							for (i = sOldBCnt; i < sButtonCount; i++) 
							   bsButtons [i].hwndButton = 
								 CreateWindow ("button",
	         	   	               fsFunctions [bsButtons [i].sFunction].szText,
	            	   	           WS_CHILD | WS_VISIBLE | BS_PUSHBUTTON,
	                     		   sWinWidth - BUTTONWIDTH - 5, //x Pos
			                       5+(i * (sBHeight+5)),        //y Pos
	   		                       BUTTONWIDTH,                 //x Size
	         		               sBHeight,					//y Size
	               		           hwnd,                        //Parent handle
	                     		   i+IDD_1STBUTTON,             //ID number
			                       hInstance,
	   				     	       NULL);
						}
						if (sOldBCnt != sButtonCount) {
							i = sWinWidth--;
							SendMessage (hwnd, WM_SIZE, 0, 
							             MAKELONG (i, sWinHeight));
							InvalidateRect (hwnd, 0, FALSE);
						}
						if (bDir)
							bExecutable = EnableButtons (hwnd, FALSE, bDir);
						else	
							bExecutable = EnableButtons (hwnd, 
 		                             IsProgram (szPrograms, szMasterPath), 
						             bDir);
						FillListBoxes (hwnd);

						itoa (sSearchFlags, szTemp, 10);
						WritePrivateProfileString (szAppName, "IncFlags", 
						                           szTemp, szProfileName);
						itoa (bDirFirst, szTemp, 10);
						WritePrivateProfileString (szAppName, "DirFirst", 
						                           szTemp, szProfileName);
						itoa (bConfirmFileDel, szTemp, 10);
						WritePrivateProfileString (szAppName, "ConfFileDel", 
						                           szTemp, szProfileName);
						itoa (bConfirmDirDel, szTemp, 10);
						WritePrivateProfileString (szAppName, "ConfDirDel", 
						                           szTemp, szProfileName);
						WritePrivateProfileString (szAppName, "DefaultEditor", 
						                           szDefaultEditor, 
						                           szProfileName);
						itoa (sButtonCount, szTemp, 10);
						WritePrivateProfileString (szAppName, "NumberButtons", 
						                           szTemp, szProfileName);

						for (i = 0; i < BUTTONMAX; i++) {
							strcpy (szTemp, "Button");
							itoa (i, szSel, 10);
							strcat (szTemp, szSel);
							itoa (bsButtons[i].sFunction, szSel, 10);				
							WritePrivateProfileString(szAppName, szTemp, szSel, 
							                          szProfileName);
						}
					}
	   	   			return 0;

				case IDM_ABOUT:
					DialogBox (hInstance, "About", hwnd, lpfnAboutDlgProc);
		   		   	return 0;

				case IDM_EXIT:
					SendMessage (hwnd, WM_CLOSE, 0, 0);
	   				return 0;
			}
			//
			// Check Editbox
			//
			if (wParam == IDD_EDITBOX && HIWORD (lParam) == EN_KILLFOCUS) {
				GetWindowText (hwndEditBox, szTemp, sizeof (szTemp));
				if (strcmp (szTemp, szMasterPath))
					SendMessage (hwnd, WUM_SETPATH, 0, 0L);
			}
			//
			// Check Drive selection Listbox
			//
			if (wParam == IDD_DRVBOX && 
			    HIWORD (lParam) == LBN_SELCHANGE) {
				SendMessage (hwndDrvBox, LB_GETTEXT,
				             (WORD) SendMessage (hwndDrvBox, LB_GETCURSEL, 
				                                 0, 0L),
				             STRTOLP (szMasterPath)); 

				getcwd (szSel, sizeof (szSel));
				_chdrive (szMasterPath [2] - 'a' + 1);
				FillListBoxes (hwnd);			
			}
			//	
			// Check directory listboxes
			//
			if (wParam >= IDD_1STBOX && 
			    wParam < (WORD) IDD_1STBOX+sLBoxVisible) {

				sLBoxID = TranslateID (wParam);

				switch (HIWORD (lParam)) {
	
			    	case LBN_SELCHANGE:

			        	DlgDirSelect (hwnd, szMasterPath, IDD_DRVBOX);
						strcat (szMasterPath, "\\");

						bDir = FALSE;
						for (i = 0; i < sLBoxCount; i++) {
							if (i < sLBoxID - IDD_1STBOX) {
								GetLBoxText (i, szSel);
								strncat (szMasterPath, szSel+1, 
								         strlen (szSel)-2);
								strcat (szMasterPath, "\\");
	
							} else if (i == sLBoxID - IDD_1STBOX) {
								GetLBoxText (i, szSel);
								if (szSel[0]=='[' && 
								    szSel[strlen(szSel)-1] == ']') {
										strncat (szMasterPath, szSel+1, 
										         strlen (szSel)-2);
										bDir = TRUE;
								} else 
									strcat (szMasterPath, szSel);
							} else
						      SendLBoxMsg (i, LB_RESETCONTENT, 0, 0L);
						}
 						i = wParam - IDD_1STBOX;
						if (bDir) {
							chdir (szMasterPath);
							if (((i != sLBoxVisible - 2) && sLBoxVStart > 0)
  							    || (i == sLBoxVisible - 1)) {
								FillListBoxes (hwnd);
							} else {
								strcpy (szTemp, szMasterPath);
								strcat (szTemp, "\\");
								if (sLBoxID - IDD_1STBOX < sLBoxCount)
									FillListBox (hwnd, szTemp, 
									             sLBoxID - IDD_1STBOX + 1);
							}
						} else
							if ((i != sLBoxVisible - 1) && sLBoxVStart > 0) {
								*strrchr (szMasterPath, '\\') = '\0';
								chdir (szMasterPath);
								i = FillListBoxes (hwnd);
								strcat (szMasterPath, "\\");
								strcat (szMasterPath, szSel);
								SendLBoxMsg (i, LB_SELECTSTRING, -1, 
								             STRTOLP (szSel));
							}

   					SetWindowText (hwndEditBox, szMasterPath);

						if (bDir)
							bExecutable = EnableButtons (hwnd, FALSE, bDir);
						else	
							bExecutable = EnableButtons (hwnd, 
		                                      IsProgram (szPrograms, 
							                             szMasterPath), 
							                  bDir);
	  	   			break;

			      case LBN_DBLCLK:
						if (!bDir) 
							SendMessage (hwnd, WM_COMMAND, IDM_OPEN, 0L);
						break;
				}
			}
			//	
			// Check Buttons
			//
			if (wParam >= IDD_1STBUTTON && 
			    wParam < (WORD) IDD_1STBUTTON+sButtonCount) {
				//
				// Simulate a menu message by sending a WM_COMMAND message
				// to the client window.  
				//
				if (HIWORD (lParam) == BN_CLICKED) {
					i = wParam - IDD_1STBUTTON;
					SendMessage (hwnd, WM_COMMAND, 
					           fsFunctions [bsButtons[i].sFunction].sFunction, 
					           0L);
				}
			}
			break;

		case WM_HSCROLL:

			sNewCount = sLBoxVStart;

			switch (wParam) {
				case SB_TOP:
					ScrollLBoxes (-sLBoxVStart);
					break;

				case SB_BOTTOM:
					i = GetDirDepth ();
					ScrollLBoxes (i - sLBoxVStart - sLBoxVisible);	
					break;

				case SB_LINEDOWN:
					ScrollLBoxes (1);
					break;

				case SB_LINEUP:
					ScrollLBoxes (-1);
					break;

				case SB_PAGEDOWN:
					i = GetDirDepth ();
					ScrollLBoxes (sLBoxVisible);
					break;

				case SB_PAGEUP:
					ScrollLBoxes (-sLBoxVisible);
					break;

				case SB_THUMBPOSITION:
					ScrollLBoxes (LOWORD (lParam) - sNewCount);
					break;
			}			
			if (sLBoxVStart != sNewCount) 
				RefreshListBoxes (hwnd);				
			break;

		case WUM_SETPATH:
			GetWindowText (hwndEditBox, szTemp, sizeof (szTemp));

			_fullpath (szSel, szTemp, sizeof (szSel));

			if (_chdrive ((int) szSel[0] - '@') == -1)
				return 0;

			if (chdir (szSel) == 0) {
				strcpy (szMasterPath, szSel);
				FillListBoxes (hwnd);
			} else {	
				_splitpath (szSel, szDrive, szDir, szName, szExt);
				strlwr (szName);
				*strrchr (szSel, '\\') = '\0';

				if (chdir (szSel) == 0) {
					strcpy (szMasterPath, szSel);
					sLBoxID = FillListBoxes (hwnd);
					strcpy (szSel, szName);
					strcat (szSel, szExt);
					strlwr (szSel);
					SendLBoxMsg (sLBoxID, LB_SELECTSTRING, 
					             -1, STRTOLP (szSel));
				}
			}
			break;

		case WM_DESTROY:
			//
			//Save window position and size in profile
			//
			if (IsZoomed (hwnd)) {
				WritePrivateProfileString (szAppName, "Zoom", "1", 
				                           szProfileName);
			} else if (!IsIconic (hwnd)) {
				WritePrivateProfileString (szAppName, "Zoom", "0", 
				                           szProfileName);
				GetWindowRect (hwnd, &rect);
				itoa (rect.left, szTemp, 10);
				WritePrivateProfileString (szAppName, "x", szTemp, 
				                           szProfileName);
				itoa (rect.top, szTemp, 10);
				WritePrivateProfileString (szAppName, "y", szTemp, 
				                           szProfileName);
				itoa (rect.right - rect.left, szTemp, 10);
				WritePrivateProfileString (szAppName, "cx", szTemp, 
				                           szProfileName);
				itoa (rect.bottom - rect.top, szTemp, 10);
				WritePrivateProfileString (szAppName, "cy", szTemp, 
				                           szProfileName);
			}
			PostQuitMessage (0);
			return 0;
	}
	return DefWindowProc (hwnd, message, wParam, lParam);
}
