/*    (C) Copyright Microsoft Corp. 1991.  All rights reserved.
 *
 *    You have a royalty-free right to use, modify, reproduce and 
 *    distribute the Sample Files (and/or any modified version) in 
 *    any way you find useful, provided that you agree that 
 *    Microsoft has no warranty obligations or liability for any 
 *    Sample Application Files which are modified. 
 */


/** testkeyb.c
 *
 *  DESCRIPTION: 
 *      This app is designed to show how to use the midiKeyB control in
 *      your applications.  It doesn't do a whole lot--so you don't have
 *      to wade through a bunch of code trying to figure out how to use
 *      it.
 *
 *
 **/


/* the includes we need */
    #include <windows.h>
    #include <mmsystem.h>
    #include <stdio.h>
    #include <string.h>
    #include "testkeyb.h"


/* for debuggin' */
    #ifdef DEBUG
        #define D(x)    {x;}
        #define ODS(x)  OutputDebugString(x)
    #else
        #define D(x)
        #define ODS(x)
    #endif


/* I wonder why this isn't defined? */
    #define MAKEWORD(bLow, bHigh)   ((WORD)((bHigh<<8)|bLow))


/* prototypes for good measure */
    long FAR PASCAL tkeybWndProc( HWND, WORD, WORD, LONG );
    BOOL NEAR PASCAL tkeybSetKeyboardParameters( HWND hKeyB );
    BOOL NEAR PASCAL tkeybAllNotesOff( HWND hKeyB );
    void FAR PASCAL tkeybHourGlass( BOOL fHourGlass );


/* globals, no less */
    char        gszAppName[] = "midiKeyB Test Application";
    char        gszShortName[] = "midiKeyB";
    char        gachTemp[ 80 ];             /* temporary stuff...       */
    HANDLE      ghInst;                     /* our instance handle      */
    HWND        ghWnd;                      /* main window handle       */

    HWND        ghKeyB;                     /* midiKeyB control window  */
    HMIDIOUT    ghMidiOut   =   NULL;       /* output MIDI device handle*/

    short       gnDevice    =   MIDIMAPPER; /* current device id        */
    BYTE        gbChannel   =   0;          /* channel number (0 base)  */
    BYTE        gbPatch     =   0;          /* current patch number     */
    BYTE        gbVolume    =   100;        /* current key velocity     */


/* control switches */
    BOOL    gfLabelKeys     = FALSE;        /* put labels on keys?      */
    BOOL    gfHideousColors = FALSE;        /* ...says it all.          */


/** void FAR PASCAL tkeybHourGlass( BOOL fHourGlass )
 *
 *  DESCRIPTION: 
 *      This function changes the cursor to that of the hour glass or
 *      back to the previous cursor.
 *
 *  ARGUMENTS:
 *      BOOL fHourGlass :   TRUE if we need the hour glass.  FALSE if
 *                          we need the arrow back.
 *
 *  RETURN (void):
 *      On return, the cursor will be what was requested.
 *
 *  NOTES:
 *      This function can be called recursively.
 *
 **/

void FAR PASCAL tkeybHourGlass( BOOL fHourGlass )
{
    static HCURSOR  hCursor;
    static WORD     wWaiting = 0;

    if ( fHourGlass )
    {
        if ( !wWaiting )
        {
            hCursor = SetCursor( LoadCursor( NULL, IDC_WAIT ) );
            ShowCursor( TRUE );
        }

        wWaiting++;
    }

    else
    {
        if ( !--wWaiting )
        {
            ShowCursor( FALSE );
            SetCursor( hCursor );
        }
    }
} /* tkeybHourGlass() */


/** BOOL FAR PASCAL tkeybAboutDlgProc( hDlg, wMsg, wParam, lParam )
 *
 *  DESCRIPTION: 
 *      This is the dlg proc for the about box.
 *
 *  NOTES:
 *
 **/

BOOL FAR PASCAL tkeybAboutDlgProc( HWND   hDlg,
                                   WORD   wMsg,
                                   WORD   wParam,
                                   LONG   lParam )
{
    switch ( wMsg )
    {
        case WM_INITDIALOG:
            /* I want the focus on the ok button */
            SetFocus( GetDlgItem( hDlg, IDOK ) );
        break;

        case WM_COMMAND:
            if ( (wParam == IDOK) || (wParam == IDCANCEL) )
            {
                EndDialog( hDlg, TRUE );
                return ( TRUE );
            }
        break;
    }

    return ( FALSE );
} /* tkeybAboutDlgProc() */


/** BOOL FAR PASCAL tkeybConfigDlgProc( hDlg, wMsg, wParam, lParam )
 *
 *  DESCRIPTION: 
 *      This is the dlg proc for the configuration dlg box.
 *
 *  NOTES:
 *
 **/

BOOL FAR PASCAL tkeybConfigDlgProc( HWND   hDlg,
                                    WORD   wMsg,
                                    WORD   wParam,
                                    LONG   lParam )
{
    static short    nVolume;
    HWND            hCtrl;

    switch ( wMsg )
    {
        case WM_INITDIALOG:
        {
            MIDIOUTCAPS moc;
            WORD        wNumDevs;

            /* fill in device combo box */
            hCtrl = GetDlgItem( hDlg, IDD_CFG_DEVICE );

            /* put MIDI Mapper at index 0 */
            wsprintf( gachTemp, "%s", (LPSTR)"MIDI Mapper" );
            SendMessage( hCtrl, CB_ADDSTRING, 0, (LONG)(LPSTR)gachTemp );

            wNumDevs = midiOutGetNumDevs();
            for ( wParam = 0; wParam < wNumDevs; wParam++ )
            {
                SendMessage( hCtrl, CB_ADDSTRING, 0, 
                    midiOutGetDevCaps(wParam, &moc, sizeof(moc)) ?
                    (LONG)(LPSTR)"BOGUS!" : (LONG)(LPSTR)moc.szPname );
            }


            /* this assumes MIDIMAPPER == -1 and is index 0 in combo box */
            SendMessage( hCtrl, CB_SETCURSEL, (gnDevice + 1), 0 );


            /* fill in channel combo box */
            hCtrl = GetDlgItem( hDlg, IDD_CFG_CHANNEL );
            for ( wParam = 1; wParam <= 16; wParam++ )
            {
                wsprintf( gachTemp, "%d", wParam );
                SendMessage( hCtrl, CB_ADDSTRING, 0, (LONG)(LPSTR)gachTemp );
            }
            SendMessage( hCtrl, CB_SETCURSEL, gbChannel, 0 );


            /* set the volume gauge */
            SetDlgItemInt( hDlg, IDD_CFG_VOLTEXT, gbVolume, FALSE );
            hCtrl = GetDlgItem( hDlg, IDD_CFG_VOLUME );
            SetScrollRange( hCtrl, SB_CTL, 0, 127, FALSE );
            SetScrollPos( hCtrl, SB_CTL, gbVolume, FALSE );
            nVolume = (short)gbVolume;


            /* set the patch number */
            SetDlgItemInt( hDlg, IDD_CFG_PATCH, gbPatch, FALSE );


            /* I want the focus on the ok button */
            SetFocus( GetDlgItem( hDlg, IDOK ) );
        }
        break;

        case WM_HSCROLL:
            #define VOL_MIN     0
            #define VOL_MAX     127
            #define VOL_PAGE    15

            hCtrl = HIWORD( lParam );

            switch ( wParam )
            {
                case SB_PAGEDOWN:
                    nVolume += VOL_PAGE;

                    /** fall through **/

                case SB_LINEDOWN:
                    nVolume = min( VOL_MAX, nVolume + 1 );
                break;

                case SB_PAGEUP:
                    nVolume -= VOL_PAGE;

                    /** fall through **/

                case SB_LINEUP:
                    nVolume = max( VOL_MIN, nVolume - 1 );
                break;


                case SB_TOP:
                    nVolume = VOL_MIN;
                break;

                case SB_BOTTOM:
                    nVolume = VOL_MAX;
                break;

                case SB_THUMBPOSITION:
                case SB_THUMBTRACK:
                    nVolume = LOWORD( lParam );
                break;

                default:
                    return ( FALSE );
            }

            SetScrollPos( hCtrl, SB_CTL, nVolume, TRUE );
            SetDlgItemInt( hDlg, IDD_CFG_VOLTEXT, nVolume, FALSE );

            return ( TRUE );
        break;

        case WM_COMMAND:
            switch ( wParam )
            {
                case IDOK:
                    /* get updated configuration stuff from controls */
                    hCtrl = GetDlgItem( hDlg, IDD_CFG_CHANNEL );
                    gbChannel = (BYTE)SendMessage( hCtrl, CB_GETCURSEL, 0, 0 );

                    hCtrl = GetDlgItem( hDlg, IDD_CFG_DEVICE );
                    gnDevice = (BYTE)SendMessage( hCtrl, CB_GETCURSEL, 0, 0 );

                    /* assumes MIDI Mapper is dev id -1--index 0 in combo */
                    gnDevice--;

                    gbVolume = (BYTE)nVolume;

                    gbPatch = (BYTE)GetDlgItemInt( hDlg, IDD_CFG_PATCH, NULL, FALSE );
                    if ( gbPatch > 127 )
                        gbPatch = 0;

                    /* apply changes */
                    tkeybSetKeyboardParameters( ghKeyB );

                    /** fall through **/

                case IDCANCEL:
                    EndDialog( hDlg, TRUE );
                    return ( TRUE );
            }
        break;
    }

    return ( FALSE );
} /* tkeybConfigDlgProc() */


/** long FAR PASCAL tkeybWndProc( HWND, WORD, WORD, LONG )
 *
 *  DESCRIPTION: 
 *      This is just your normal everyday WndProc().
 *
 *  NOTES:
 *
 **/

long FAR PASCAL tkeybWndProc( HWND hWnd, WORD wMsg, WORD wParam, LONG lParam )
{
    FARPROC lpfnDlgProc;

    switch ( wMsg ) 
    {
        case WM_CREATE:
        {
            HMENU hMenu = GetMenu( hWnd );

            CheckMenuItem( hMenu, IDM_TEST_LABELKEYS, MF_BYCOMMAND |
                            (gfLabelKeys ? MF_CHECKED : MF_UNCHECKED) );

            CheckMenuItem( hMenu, IDM_TEST_HIDEOUSCOLORS, MF_BYCOMMAND |
                            (gfHideousColors ? MF_CHECKED : MF_UNCHECKED) );

        }
        break;

        case WM_DESTROY:
            PostQuitMessage( 0 );
        break;

        case WM_COMMAND:
        {
            switch ( wParam )
            {
                case IDM_TEST_DEFAULTKEYB:
                    SendMessage( GetDlgItem( hWnd, IDD_MIDIKEYB ),
                                KEYB_SETLAYOUT, TRUE, MAKELONG( 36, 48 ) );
                break;

                case IDM_TEST_LOWHALFKEYB:
                    SendMessage( GetDlgItem( hWnd, IDD_MIDIKEYB ),
                                KEYB_SETLAYOUT, TRUE, MAKELONG( 64, 0 ) );
                break;

                case IDM_TEST_HIGHHALFKEYB:
                    SendMessage( GetDlgItem( hWnd, IDD_MIDIKEYB ),
                                KEYB_SETLAYOUT, TRUE, MAKELONG( 64, 64 ) );
                break;

                case IDM_TEST_GIANTKEYB:
                    SendMessage( GetDlgItem( hWnd, IDD_MIDIKEYB ),
                                 KEYB_SETLAYOUT, TRUE, MAKELONG( 96, 0 ) );
                break;

                case IDM_TEST_LABELKEYS:
                {
                    HMENU   hMenu = GetMenu( hWnd );
                    HWND    hKeyBWnd = GetDlgItem( hWnd, IDD_MIDIKEYB );

                    gfLabelKeys = !gfLabelKeys;
                    CheckMenuItem( hMenu, IDM_TEST_LABELKEYS, MF_BYCOMMAND |
                                    (gfLabelKeys ? MF_CHECKED : MF_UNCHECKED) );

                    /* set the label style appropriately */
                    lParam = GetWindowLong( hKeyBWnd, GWL_STYLE );

                    if ( gfLabelKeys )
                        lParam |= KEYBS_LABELS;
                    else
                        lParam &= ~KEYBS_LABELS;

                    SetWindowLong( hKeyBWnd, GWL_STYLE, lParam );

                    /* force a repaint on the whole thing */
                    InvalidateRect( hKeyBWnd, NULL, TRUE );
                    UpdateWindow( hKeyBWnd );
                }
                break;

                case IDM_TEST_HIDEOUSCOLORS:
                {
                    HMENU   hMenu = GetMenu( hWnd );
                    HWND    hKeyBWnd = GetDlgItem( hWnd, IDD_MIDIKEYB );

                    gfHideousColors = !gfHideousColors;
                    CheckMenuItem( hMenu, IDM_TEST_HIDEOUSCOLORS, MF_BYCOMMAND |
                                    (gfHideousColors ? MF_CHECKED : MF_UNCHECKED) );

                    /* set the colors--repaint when both are set */
                    SendMessage( hKeyBWnd, KEYB_SETBKCOLOR, FALSE,
                                gfHideousColors ? RGB( 255, 0, 0 ) :
                                                  RGB( 255, 255, 255 ) );

                    SendMessage( hKeyBWnd, KEYB_SETFGCOLOR, TRUE,
                                gfHideousColors ? RGB( 0, 255, 0 ) :
                                                  RGB( 0, 0, 0 ) );
                }
                break;

                case IDM_TEST_ALLNOTESOFF:
                    tkeybHourGlass( TRUE );
                    {
                        if ( ghMidiOut )
                            midiOutReset( ghMidiOut );

                        SendMessage( ghKeyB, KEYB_RESET, TRUE, 0L );
                    }
                    tkeybHourGlass( FALSE );
                break;

                case IDM_TEST_CONFIGURE:
                    /* standard stuff */
                    lpfnDlgProc = MakeProcInstance( tkeybConfigDlgProc, ghInst );
                    DialogBox( ghInst, "DLG_TESTKEYB_CONFIG", ghWnd, lpfnDlgProc );
                    FreeProcInstance( lpfnDlgProc );
                break;

                case IDM_TEST_EXIT:
                    SendMessage( hWnd, WM_CLOSE, 0, 0L );
                break;

                case IDM_HELP_ABOUT:
                    /* standard stuff */
                    lpfnDlgProc = MakeProcInstance( tkeybAboutDlgProc, ghInst );
                    DialogBox( ghInst, "DLG_TESTKEYB_ABOUT", ghWnd, lpfnDlgProc );
                    FreeProcInstance( lpfnDlgProc );
                break;
            }
        }
        break;

        case WM_SIZE:
        {
            RECT    rc;

            /* resize the keyboard control to fit in parent window */
            GetClientRect( hWnd, &rc );
            MoveWindow( GetDlgItem( hWnd, IDD_MIDIKEYB ), 0, 0,
                        rc.right - rc.left, rc.bottom - rc.top, FALSE );
        }
        break;

        case WM_MIDIKEYBKEYDOWN:
            D( wsprintf(gachTemp, "KDOWN: %.4X:%.8lX\r\n", wParam, lParam) );
            ODS( gachTemp );

            /* if the output handle is valid */
            if ( wParam )
                midiOutShortMsg( wParam, lParam );
        break;

        case WM_MIDIKEYBKEYUP:
            D( wsprintf(gachTemp, "  KUP: %.4X:%.8lX\r\n", wParam, lParam) );
            ODS( gachTemp );

            /* if the output handle is valid */
            if ( wParam )
                midiOutShortMsg( wParam, lParam );
        break;

        case WM_CLOSE:
            DestroyWindow( hWnd );
        break;

        default:
            return ( DefWindowProc( hWnd, wMsg, wParam, lParam ) );
    }

    return ( 0L );
} /* tkeybWndProc() */


/** BOOL NEAR PASCAL tkeybSetKeyboardParameters( HWND hKeyB )
 *
 *  DESCRIPTION:
 *      This function sets a bunch of parameters for the keyboard control.
 *
 *  NOTES:
 *
 **/

BOOL NEAR PASCAL tkeybSetKeyboardParameters( HWND hKeyB )
{
    static short    nCurrentDevice = 0;
    WORD            wError;

    /* has the device changed? */
    if ( nCurrentDevice != gnDevice )
    {
        /* do we have an open device?? */
        if ( ghMidiOut )
        {
            tkeybHourGlass( TRUE );
            midiOutReset( ghMidiOut );
            tkeybHourGlass( FALSE );

            if ( ghMidiOut = midiOutClose( ghMidiOut ) )
            {
                MessageBox( ghWnd, "MIDI device doesn't want to close!",
                            gszShortName, MB_OK );
                ghMidiOut = NULL;
            }
        }

        /* attempt to open the device... */
        if ( wError = midiOutOpen( &ghMidiOut, gnDevice, NULL, 0L, 0L ) )
        {
            MessageBox( ghWnd, "Attempt to open MIDI device was most untriumphant!",
                        gszShortName, MB_OK );
            ghMidiOut = NULL;
        }

        /* flag that we now have gnDevice open */
        else nCurrentDevice = gnDevice;

        D( wsprintf( gachTemp, "ID: %d  hMidiOut: %.4Xh\r\n", gnDevice, ghMidiOut ) );
        ODS( gachTemp );
    }

    /* set all of the parameters */
    SendMessage( hKeyB, KEYB_SETHMIDIOUT, ghMidiOut, 0L );
    SendMessage( hKeyB, KEYB_SETCHANNEL, gbChannel, 0L );
    SendMessage( hKeyB, KEYB_SETVELOCITY, gbVolume, 0L );

    /* now send a program change... */
    if ( ghMidiOut )
    {
        if ( midiOutShortMsg( ghMidiOut, MAKEWORD(0xC0 + gbChannel, gbPatch) ) )
        {
            MessageBox( ghWnd, "Program change failed!?!", gszShortName, MB_OK );
        }
    }

    return ( TRUE );
} /* tkeybSetKeyboardParameters() */


/** int PASCAL WinMain( HANDLE, HANDLE, LPSTR, int )
 *
 *  DESCRIPTION: 
 *      This is just your normal everyday WinMain() with two exceptions:
 *      it calls the 'midiKeyBInit()' function to get the midiKeyB
 *      class initialized.  It also crams a midiKeyB control into the
 *      client area of the window.
 *
 *  NOTES:
 *
 **/

int PASCAL WinMain( HANDLE  hInstance,
                    HANDLE  hPrevInstance,
                    LPSTR   lpszCmdLine,
                    int     nCmdShow )
{
    MSG         msg;
    RECT        rc;
    DWORD       dwStyle;

    ghInst = hInstance;

    /* only need to do this for the first instance */
    if ( !hPrevInstance ) 
    {
        WNDCLASS    wc;

        /* initialize the midiKeyB class */
        if ( !midiKeyBInit( hInstance ) )
            return ( FALSE );

        wc.style         = CS_HREDRAW | CS_VREDRAW;
        wc.lpfnWndProc   = tkeybWndProc;
        wc.cbClsExtra    = 0;
        wc.cbWndExtra    = 0;
        wc.hInstance     = hInstance;
        wc.hIcon         = LoadIcon( hInstance, "ICON_TESTKEYB" );
        wc.hCursor       = LoadCursor( NULL, IDC_ARROW );
        wc.hbrBackground = COLOR_WINDOW + 1;
        wc.lpszMenuName  = "MENU_TESTKEYB";
        wc.lpszClassName = gszAppName;

        if ( !RegisterClass( &wc ) )
            return ( FALSE );
    }

    /* open our window */
    ghWnd = CreateWindow( gszAppName, gszAppName,
                          WS_OVERLAPPEDWINDOW,
                          CW_USEDEFAULT, CW_USEDEFAULT,
                          TKEYB_WINDOW_WIDTH, TKEYB_WINDOW_HEIGHT,
                          NULL, NULL, hInstance, NULL );             

    /* was the window created?? */
    if ( !ghWnd )
        return ( FALSE );

    /* set the style to default values */
    dwStyle = WS_VISIBLE | WS_CHILD;

    if ( gfLabelKeys )
        dwStyle |= KEYBS_LABELS;

    /* create the keyboard control in the parent window's client area */
    GetClientRect( ghWnd, &rc );
    ghKeyB = CreateWindow( gszShortName, "zYz", dwStyle,
                           0, 0, rc.right, rc.bottom, ghWnd,
                           IDD_MIDIKEYB, hInstance, NULL );

    /* show it */
    ShowWindow( ghWnd, nCmdShow );

    /** no, we DON'T need an UpdateWindow here **/

    /* set the default keyboard state--open the MIDI Mapper */
    tkeybSetKeyboardParameters( ghKeyB );

    /* the ubiquitous message dispatcher */
    while ( GetMessage( &msg, NULL, 0, 0 ) )
    {
        TranslateMessage( &msg );
        DispatchMessage( &msg );
    }

    /* close current MIDI device if it is open */
    if ( ghMidiOut )
    {   
        /* all notes off so we don't drive people nuts! */
        tkeybHourGlass( TRUE );
        {
            /* this can take a second or two for external synths... */
            midiOutReset( ghMidiOut );
        }
        tkeybHourGlass( FALSE );

        if ( ghMidiOut = midiOutClose( ghMidiOut ) )
        {
            MessageBox( ghWnd, "MIDI device doesn't want to close!",
                        gszShortName, MB_OK );
            ghMidiOut = NULL;
        }
    }

    /* vamoose */
    return ( msg.wParam );
} /* WinMain() */


/** EOF: testkeyb.c **/
