/*
 * DOCUMENT.CPP
 * Component Schmoo Chapter 6
 *
 * Implementation of the CSchmooDoc derivation of CDocument as
 * well as an implementation of CPolylineAdviseSink.
 *
 * Copyright (c)1993 Microsoft Corporation, All Rights Reserved
 *
 * Kraig Brockschmidt, Software Design Engineer
 * Microsoft Systems Developer Relations
 *
 * Internet  :  kraigb@microsoft.com
 * Compuserve:  >INTERNET:kraigb@microsoft.com
 */


#include "coschmoo.h"



/*
 * CSchmooDoc::CSchmooDoc
 * CSchmooDoc::~CSchmooDoc
 *
 * Constructor Parameters:
 *  hInst           HINSTANCE of the application.
 */

CSchmooDoc::CSchmooDoc(HINSTANCE hInst)
    : CDocument(hInst)
    {
    m_uPrevSize=SIZE_RESTORED;
    m_pPL=NULL;
    m_pPLAdv=NULL;
    m_pIPersistStorage=NULL;

    //CHAPTER6MOD
    m_pIAdviseSink=NULL;
    m_dwConn=0;
    //End CHAPTER6MOD
    return;
    }


CSchmooDoc::~CSchmooDoc(void)
    {
    //CHAPTER6MOD
    LPDATAOBJECT        pIDataObject;
    HRESULT             hr;

    //Turn off the advise.
    if (NULL!=m_pPL && 0!=m_dwConn)
        {
        hr=m_pPL->QueryInterface(IID_IDataObject, (LPVOID FAR *)&pIDataObject);

        if (SUCCEEDED(hr))
            pIDataObject->DUnadvise(m_dwConn);
        }

    if (NULL!=m_pIAdviseSink)
        delete m_pIAdviseSink;
    //End CHAPTER6MOD

    if (NULL!=m_pIPersistStorage)
        m_pIPersistStorage->Release();

    if (NULL!=m_pPLAdv)
        delete m_pPLAdv;

    if (NULL!=m_pPL)
        m_pPL->Release();

    CoFreeUnusedLibraries();
    return;
    }






/*
 * CSchmooDoc::FInit
 *
 * Purpose:
 *  Initializes an already created document window.  The client actually
 *  creates the window for us, then passes that here for further
 *  initialization.
 *
 * Parameters:
 *  pDI             LPDOCUMENTINIT containing initialization parameters.
 *
 * Return Value:
 *  BOOL            TRUE if the function succeeded, FALSE otherwise.
 */

BOOL CSchmooDoc::FInit(LPDOCUMENTINIT pDI)
    {
    RECT            rc;
    HRESULT         hr;
    //CHAPTER6MOD
    FORMATETC       fe;
    LPDATAOBJECT    pIDataObject;
    //End CHAPTER6MOD

    //Change the stringtable range to our customization.
    pDI->idsMin=IDS_DOCUMENTMIN;
    pDI->idsMax=IDS_DOCUMENTMAX;

    //Do default initialization
    if (!CDocument::FInit(pDI))
        return FALSE;

    //Create the Polyline Object via COMPOBJ.DLL functions.
    //CHAPTER6MOD
    hr=CoCreateInstance(CLSID_Polyline6, NULL, CLSCTX_INPROC_SERVER
        , IID_IPolyline6, (LPVOID FAR *)&m_pPL);
    //End CHAPTER6MOD

    if (FAILED(hr))
        return FALSE;

    //Initialize the contained Polyline which creates a window.
    GetClientRect(m_hWnd, &rc);
    InflateRect(&rc, -8, -8);

    if (FAILED(m_pPL->Init(m_hWnd, &rc, WS_CHILD | WS_VISIBLE, ID_POLYLINE)))
        return FALSE;


    //Set up an advise on the Polyline.
    m_pPLAdv=new CPolylineAdviseSink((LPVOID)this, (LPUNKNOWN)this);

    if (NULL==m_pPLAdv)
        return FALSE;

    m_pPL->SetAdvise(m_pPLAdv);

    //Get the IPersistStorage interface on the object for loads & saves.
    hr=m_pPL->QueryInterface(IID_IPersistStorage, (LPVOID FAR *)&m_pIPersistStorage);

    if (FAILED(hr))
        return FALSE;

    //CHAPTER6MOD
    /*
     * Create an IAdviseSink and send it to the Polyline's IDataObject
     * with the clipboard format for the Polyline (as in IPOLY6.H).
     */

    //This is a private macro.
    SETDefFormatEtc(fe, m_cf, TYMED_HGLOBAL);

    m_pIAdviseSink=new CImpIAdviseSink((LPVOID)this, (LPUNKNOWN)this);

    if (NULL==m_pIAdviseSink)
        return FALSE;

    //Set up an advise for the Polyline format
    hr=m_pPL->QueryInterface(IID_IDataObject, (LPVOID FAR *)&pIDataObject);

    if (FAILED(hr))
        return FALSE;

    pIDataObject->DAdvise(&fe, ADVF_NODATA, m_pIAdviseSink, &m_dwConn);
    pIDataObject->Release();

    //End CHAPTER6MOD

    return TRUE;
    }




//CHAPTER6MOD
//IUnknown interface for all the interfaces we implement in the document

/*
 * CSchmooDoc::QueryInterface
 * CSchmooDoc::AddRef
 * CSchmooDoc::Release
 *
 * Purpose:
 *  IUnknown members for the CSchmooDoc implementation.
 */

STDMETHODIMP CSchmooDoc::QueryInterface(REFIID riid, LPVOID FAR *ppv)
    {
    *ppv=NULL;

    //The document is the unknown
    if (IsEqualIID(riid, IID_IUnknown))
        *ppv=(LPVOID)this;

    //Return contained interfaces for others.
    if (IsEqualIID(riid, IID_IPolylineAdviseSink6))
        *ppv=(LPVOID)m_pPLAdv;

    if (IsEqualIID(riid, IID_IAdviseSink))
        *ppv=(LPVOID)m_pIAdviseSink;

    /*
     * If we actually assign an interface to ppv we need to AddRef it
     * since we're returning a new pointer.
     */
    if (NULL!=*ppv)
        {
        ((LPUNKNOWN)*ppv)->AddRef();
        return NOERROR;
        }

    return ResultFromScode(S_FALSE);
    }


STDMETHODIMP_(ULONG) CSchmooDoc::AddRef(void)
    {
    return ++m_cRef;
    }


STDMETHODIMP_(ULONG) CSchmooDoc::Release(void)
    {
    /*
     * Since CoSchmoo doesn't use documents like Component Objects, this
     * doesn't do anything except provide a debugging point.
     */
    return --m_cRef;
    }

//End CHAPTER6MOD






/*
 * CSchmooDoc::FMessageHook
 *
 * Purpose:
 *  Processes WM_SIZE for the document so we can resize the Polyline.
 *
 * Parameters:
 *  <WndProc Parameters>
 *  pLRes           LRESULT FAR * in which to store the return value
 *                  for the message.
 *
 * Return Value:
 *  BOOL            TRUE to prevent further processing, FALSE otherwise.
 */

BOOL CSchmooDoc::FMessageHook(HWND hWnd, UINT iMsg, WPARAM wParam
    , LPARAM lParam, LRESULT FAR *pLRes)
    {
    UINT        dx, dy;
    RECT        rc;

    if (WM_SIZE==iMsg)
        {
        //Don't effect the Polyline size to or from minimized state.
        if (SIZE_MINIMIZED!=wParam && SIZE_MINIMIZED !=m_uPrevSize)
            {
            //When we change size, resize any Polyline we hold.
            dx=LOWORD(lParam);
            dy=HIWORD(lParam);

            /*
             * If we are getting WM_SIZE in response to a Polyline
             * notification, then don't resize the Polyline window again.
             */
            if (!m_fNoSize && NULL!=m_pPL)
                {
                //Resize the polyline to fit the new client
                SetRect(&rc, 8, 8, dx-8, dy-8);
                m_pPL->RectSet(&rc, FALSE);

                /*
                 * We consider sizing something that makes the file dirty,
                 * but not until we've finished the create process, which
                 * is why we set fNoDirty to FALSE in WM_CREATE since we
                 * get a WM_SIZE on the first creation.
                 */
                if (!m_fNoDirty)
                    FDirtySet(TRUE);

                SetRect(&rc, 0, 0, dx, dy);

                if (NULL!=m_pAdv)
                    m_pAdv->OnSizeChange((LPCDocument)this, &rc);

                m_fNoDirty=FALSE;
                }
            }

        m_uPrevSize=wParam;
        }

    /*
     * We return FALSE even on WM_SIZE so we can let the default procedure
     * handle maximized MDI child windows appropriately.
     */
    return FALSE;
    }








/*
 * CSchmooDoc::Clear
 *
 * Purpose:
 *  Sets all contents in the document back to defaults with no filename.
 *
 * Paramters:
 *  None
 *
 * Return Value:
 *  None
 */

void CSchmooDoc::Clear(void)
    {
    //Completely reset the polyline
    m_pPL->New();

    CDocument::Clear();
    return;
    }






/*
 * CSchmooDoc::ULoad
 *
 * Purpose:
 *  Loads a given document without any user interface overwriting the
 *  previous contents of the Polyline window.  We do this by opening
 *  the file and telling the Polyline to load itself from that file.
 *
 * Parameters:
 *  fChangeFile     BOOL indicating if we're to update the window title
 *                  and the filename from using this file.
 *  pszFile         LPSTR to the filename to load, NULL for untitled.
 *
 * Return Value:
 *  UINT            An error value from DOCERR_*
 */

UINT CSchmooDoc::ULoad(BOOL fChangeFile, LPSTR pszFile)
    {
    HRESULT             hr;
    //CHAPTER5MOD
    LPSTORAGE           pIStorage;

    if (NULL==pszFile)
        {
        /*
         * As a user of an IPersistStorage we have to provide all objects
         * with an IStorage they can use for incremental access passing
         * that storage to ::InitNew.  Here we create a temporary file
         * that we don't bother holding on to.  If the object doesn't
         * use it, then our ::Release destroys it immediately.
         */

        hr=StgCreateDocfile(NULL, STGM_DIRECT | STGM_READWRITE | STGM_CREATE
            | STGM_DELETEONRELEASE | STGM_SHARE_EXCLUSIVE, 0, &pIStorage);

        if (FAILED(hr))
            return DOCERR_COULDNOTOPEN;

        m_pIPersistStorage->InitNew(pIStorage);
        pIStorage->Release();

        Rename(NULL);
        return DOCERR_NONE;
        }

    /*
     * Open a storage and pass it to the Polyline via IPersistStorage.
     * We do not remain compatible with previous files saved with
     * Component Schmoo.
     */

    hr=StgOpenStorage(pszFile, NULL, STGM_DIRECT | STGM_READ
        | STGM_SHARE_EXCLUSIVE, NULL, 0, &pIStorage);

    if (FAILED(hr))
        return DOCERR_COULDNOTOPEN;

    hr=m_pIPersistStorage->Load(pIStorage);

    pIStorage->Release();
    //End CHAPTER5MOD

    if (FAILED(hr))
        return DOCERR_READFAILURE;

    if (fChangeFile)
        Rename(pszFile);

    //Importing a file makes things dirty
    FDirtySet(!fChangeFile);

    return DOCERR_NONE;
    }







/*
 * CSchmooDoc::USave
 *
 * Purpose:
 *  Writes the file to a known filename, requiring that the user has
 *  previously used FileOpen or FileSaveAs in order to have a filename.
 *
 * Parameters:
 *  uType           UINT indicating the type of file the user requested
 *                  to save in the File Save As dialog.
 *  pszFile         LPSTR under which to save.  If NULL, use the current name.
 *
 * Return Value:
 *  UINT            An error value from DOCERR_*
 */

UINT CSchmooDoc::USave(UINT uType, LPSTR pszFile)
    {
    BOOL                fRename=TRUE;
    HRESULT             hr;
    LPSTORAGE           pIStorage;

    if (NULL==pszFile)
        {
        fRename=FALSE;
        pszFile=m_szFile;
        }

    /*
     * In Component Schmoo, we only deal with one version of data,
     * so all the code in Chapter 2 Schmoo that dealt with 1.0 and
     * 2.0 files has been removed.
     */

    hr=StgCreateDocfile(pszFile, STGM_DIRECT | STGM_READWRITE
        | STGM_CREATE | STGM_SHARE_EXCLUSIVE, 0, &pIStorage);

    if (FAILED(hr))
        return DOCERR_COULDNOTOPEN;

    //Tell the object to save, and also tell it that we're done.
    m_pIPersistStorage->Save(pIStorage, FALSE);
    m_pIPersistStorage->SaveCompleted(pIStorage);

    pIStorage->Release();

    if (FAILED(hr))
        return DOCERR_WRITEFAILURE;

    //Saving makes us clean
    FDirtySet(FALSE);

    if (fRename)
        Rename(pszFile);

    return DOCERR_NONE;
    }






/*
 * CSchmooDoc::Undo
 *
 * Purpose:
 *  Reverses a previous action.
 *
 * Parameters:
 *  None
 *
 * Return Value:
 *  None
 */

void CSchmooDoc::Undo(void)
    {
    m_pPL->Undo();
    return;
    }







/*
 * CSchmooDoc::FClip
 *
 * Purpose:
 *  Places a private format, a metafile, and a bitmap of the display
 *  on the clipboard, optionally implementing Cut by deleting the
 *  data in the current window after rendering.
 *
 * Parameters:
 *  hWndFrame       HWND of the main window
 *  fCut            BOOL indicating cut (TRUE) or copy (FALSE).
 *
 * Return Value:
 *  BOOL            TRUE if successful, FALSE otherwise.
 */

BOOL CSchmooDoc::FClip(HWND hWndFrame, BOOL fCut)
    {
    BOOL            fRet=TRUE;
    HGLOBAL         hMem;
    UINT            i;

    //This array is so we can loop over the formats we provide.
    static UINT     rgcf[3]={0, CF_METAFILEPICT, CF_BITMAP};
    const UINT      cFormats=3;

    if (!OpenClipboard(hWndFrame))
        return FALSE;

    //Clean out whatever junk is in the clipboard.
    EmptyClipboard();

    rgcf[0]=m_cf;

    for (i=0; i < cFormats; i++)
        {
        //Copy private data first.
        hMem=RenderFormat(rgcf[i]);

        if (NULL!=hMem)
            SetClipboardData(rgcf[i], hMem);
        else
            fRet &=FALSE;
        }

    //Free clipboard ownership.
    CloseClipboard();

    //Delete our current data if copying succeeded.
    if (fRet && fCut)
        {
        m_pPL->New();
        FDirtySet(TRUE);
        }

    return fRet;
    }





/*
 * CSchmooDoc::RenderFormat
 *
 * Purpose:
 *  Renders a specific clipboard format into global memory.  We have this
 *  function split out because we'll eventually move to delayed rendering
 *  and this will then be immediately callable from the frame.
 *
 * Parameters:
 *  cf              UINT format to render.
 *
 * Return Value:
 *  HGLOBAL         Global memory handle containing the data.
 */

HGLOBAL CSchmooDoc::RenderFormat(UINT cf)
    {
    //CHAPTER6MOD
    LPDATAOBJECT    pIDataObject;
    FORMATETC       fe;
    STGMEDIUM       stm;
    HRESULT         hr;

    hr=m_pPL->QueryInterface(IID_IDataObject, (LPVOID FAR *)&pIDataObject);

    if (FAILED(hr))
        return NULL;

    //Private macro--not OLE 2.0 defined API
    SETFormatEtc(fe, cf, DVASPECT_CONTENT, NULL
        , TYMED_HGLOBAL | TYMED_GDI | TYMED_MFPICT, 0);

    pIDataObject->GetData(&fe, &stm);
    pIDataObject->Release();

    return stm.hGlobal;
    //End CHAPTER6MOD
    }







/*
 * CSchmooDoc::FQueryPaste
 *
 * Purpose:
 *  Determines if we can paste data from the clipboard.
 *
 * Parameters:
 *  None
 *
 * Return Value:
 *  BOOL            TRUE if data is available, FALSE otherwise.
 */

BOOL CSchmooDoc::FQueryPaste(void)
    {
    return IsClipboardFormatAvailable(m_cf);
    }





/*
 * CSchmooDoc::FPaste
 *
 * Purpose:
 *  Retrieves the private data format from the clipboard and sets it
 *  to the current figure in the editor window.
 *
 *  Note that if this function is called, then the clipboard format
 *  is available because the Paste menu item is only enabled if the
 *  format is present.
 *
 * Parameters:
 *  hWndFrame       HWND of the main window
 *
 * Return Value:
 *  BOOL            TRUE if successful, FALSE otherwise.
 */

BOOL CSchmooDoc::FPaste(HWND hWndFrame)
    {
    HGLOBAL         hMem;
    BOOL            fRet=FALSE;
    //CHAPTER6MOD
    LPDATAOBJECT    pIDataObject;
    HRESULT         hr;
    FORMATETC       fe;
    STGMEDIUM       stm;
    //End CHAPTER6MOD

    if (!OpenClipboard(hWndFrame))
        return FALSE;

    hMem=GetClipboardData(m_cf);
    CloseClipboard();

    if (NULL!=hMem)
        {
        //CHAPTER6MOD
        hr=m_pPL->QueryInterface(IID_IDataObject, (LPVOID FAR *)&pIDataObject);

        if (FAILED(hr))
            {
            GlobalFree(hMem);
            return FALSE;
            }

        SETFormatEtc(fe, m_cf, DVASPECT_CONTENT, NULL, TYMED_HGLOBAL, 0);

        stm.tymed=TYMED_HGLOBAL;
        stm.hGlobal=hMem;

        pIDataObject->SetData(&fe, &stm, TRUE);
        pIDataObject->Release();

        FDirtySet(TRUE);
        //End CHAPTER6MOD
        }

    return TRUE;
    }








/*
 * CSchmooDoc::ColorSet
 *
 * Purpose:
 *  Changes a color used in our contained Polyline.
 *
 * Parameters:
 *  iColor          UINT index of the color to change.
 *  cr              COLORREF new color.
 *
 * Return Value:
 *  COLORREF        Previous color for the given index.
 */

COLORREF CSchmooDoc::ColorSet(UINT iColor, COLORREF cr)
    {
    COLORREF    crRet;

    m_pPL->ColorSet(iColor, cr, &crRet);
    return crRet;
    }





/*
 * CSchmooDoc::ColorGet
 *
 * Purpose:
 *  Retrieves a color currently in use in the Polyline.
 *
 * Parameters:
 *  iColor          UINT index of the color to retrieve.
 *
 * Return Value:
 *  COLORREF        Current color for the given index.
 */

COLORREF CSchmooDoc::ColorGet(UINT iColor)
    {
    COLORREF    crRet;

    m_pPL->ColorGet(iColor, &crRet);
    return crRet;
    }






/*
 * CSchmooDoc::LineStyleSet
 *
 * Purpose:
 *  Changes the line style currently used in the Polyline
 *
 * Parameters:
 *  iStyle          UINT index of the new line style to use.
 *
 * Return Value:
 *  UINT            Previous line style.
 */


UINT CSchmooDoc::LineStyleSet(UINT iStyle)
    {
    UINT    i;

    m_pPL->LineStyleSet(iStyle, &i);
    return i;
    }







/*
 * CSchmooDoc::LineStyleGet
 *
 * Purpose:
 *  Retrieves the line style currently used in the Polyline
 *
 * Parameters:
 *  None
 *
 * Return Value:
 *  UINT            Current line style.
 */


UINT CSchmooDoc::LineStyleGet(void)
    {
    UINT    i;

    m_pPL->LineStyleGet(&i);
    return i;
    }



//CHAPTER6MOD
//CPolylineAdviseSink moved to IADVSINK.CPP
//End CHAPTER6MOD
