/***************************************************************************
*	NAME:  OS.C $Revision: 1.1 $
**	COPYRIGHT:
**	"Copyright (c) 1994, by FORTE
**
**       "This software is furnished under a license and may be used,
**       copied, or disclosed only in accordance with the terms of such
**       license and with the inclusion of the above copyright notice.
**       This software or any other copies thereof may not be provided or
**       otherwise made available to any other person. No title to and
**       ownership of the software is hereby transfered."
****************************************************************************
* $Log: os.c $
* Revision 1.1  1995/02/23 11:06:53  unknown
* Initial revision
* Revision 1.1  1994/09/12 15:17:44  unknown
***************************************************************************/

/**************************************************************************
 *  This module provides an OS indepent API for OS services.  The functions
 *  in this module provide DMA, Interrupt handling, memory allocation,
 *  and semaphores.
 **************************************************************************/

#include <dos.h>
#include <fcntl.h>
#include "iw.h"
#include "iwl.h"
#include "globals.h"

#define INCL_BASE
#define INCL_DOSINFOSEG
#include "os2.h"
#include "os2medef.h"
#include "ssm.h"
#include "audio.h"
#include "audiodd.h"
#include "cdevhlp.h"

#if defined(_MSC_VER) && _MSC_VER == 600
#include <memory.h>
#endif

// #define UCHAR unsigned char
// define USHORT unsigned short
// define ULONG unsigned long

/* local data structures */
#define NUM_CHANNELS 2
#define DMA0_PAGE	0x87		/* chan 0 page register (refresh)*/
#define DMA1_PAGE	0x83		/* chan 1 page register */
#define DMA2_PAGE	0x81		/* chan 2 page register */
#define DMA3_PAGE	0x82		/* chan 3 page register */
#define DMA4_PAGE	0x8F		/* chan 4 page register (unuseable)*/
#define DMA5_PAGE	0x8B		/* chan 5 page register */
#define DMA6_PAGE	0x89		/* chan 6 page register */
#define DMA7_PAGE	0x8A		/* chan 7 page register */

#define DMA1_SNGL	0x0A		/* write single bit register */
#define DMA1_MODE	0x0B		/* write mode register */
#define DMA1_CLRFF	0x0C		/* clear byte ptr flip/flop */
#define DMA2_SNGL	0xD4		/* write single bit register */
#define DMA2_MODE	0xD6		/* write mode register */
#define DMA2_CLRFF	0xD8		/* clear byte ptr flip/flop */

#define OCR1	0x20			/* 8259-1 Operation control register */
#define IMR1	0x21			/* 8259-1 Mask register */
#define OCR2	0xA0			/* 8259-2 Operation control register */
#define IMR2	0xA1			/* 8259-2 Mask register */
#define EOI	0x20

/***************************************************************************

LOCAL DEFINITION:
dma_page - DMA page register map for DMA physical channel numbers

*/
static UCHAR dma_page[] = {
	DMA0_PAGE,
	DMA1_PAGE,
	DMA2_PAGE,
	DMA3_PAGE,
	DMA4_PAGE,
	DMA5_PAGE,
	DMA6_PAGE,
	DMA7_PAGE
};

/***************************************************************************

LOCAL DEFINITION:
DMA_ENTRY - structure used to track status of a virtual DMA channel

DESCRIPTION:
  dma_disable:   disable mask for DMA channel
  dma_enable:    enable mask for DMA channel
  dma_page:      page register of DMA controller
  dma_page_reg:  unused
  dma_addr:      physical address of DMA buffer for transfer
  dma_count:     count of bytes to transfer
  dma_single:    bit mask for single byte DMA transfer
  dma_mode:      mode of DMA transfer
  dma_clrff:     clear DMA controller flip-flop
  dma_channel:   physical DMA channel number to be used for transfer
*/
typedef volatile struct {
	UCHAR	dma_disable;
	UCHAR	dma_enable;
	UCHAR	dma_page;
	USHORT	dma_page_reg;
	USHORT	dma_addr;
	USHORT	dma_count;
	USHORT	dma_single;
	USHORT	dma_mode;
	USHORT	dma_clrff;
	USHORT	dma_channel;
} DMA_ENTRY;

/***************************************************************************

LOCAL DEFINITION:
dma_parms - os layer view of virtual DMA channels

DESCRIPTION:
dma_parms is an array used to track the current state of the virtual
DMA channels used in the kernel.  A virtual DMA channel will be one
of:
  IW_DMA_CHAN_1
  IW_DMA_CHAN_2
  IW_DMA_CHAN_3
*/
static DMA_ENTRY dma_parms[NUM_CHANNELS];

/***************************************************************************

FUNCTION DEFINITION:
os_init_channel - Initialize OS layer data about virtual DMA channel

os_init_channel sets up the information kept by the OS layer about a 
vitual DMA channel.  The OS layer mainly tracks the port addresses and
register values needed to prgram the DMA controller for a given virtual
DMA channel.

NOTE: a virutal DMA channel will be one of:
  IW_DMA_CHAN_1
  IW_DMA_CHAN_2
  IW_DMA_CHAN_3

RETURNS: void
*/
void os_init_channel(
  int channel,      /* virtual DMA channel number */
  int dma_channel)  /* physical DMA channel number */
{
	DMA_ENTRY *dparms = &dma_parms[channel];
	USHORT iochan;

	dparms->dma_channel = dma_channel;
	iochan = dma_channel & 3;
	if (dma_channel <= 3) {
	    dparms->dma_single = DMA1_SNGL;
	    dparms->dma_mode   = DMA1_MODE;
	    dparms->dma_clrff  = DMA1_CLRFF;
	    dparms->dma_addr   = iochan << 1;
	    dparms->dma_count   = dparms->dma_addr | 1;
	} else {
	    dparms->dma_single = DMA2_SNGL;
	    dparms->dma_mode   = DMA2_MODE;
	    dparms->dma_clrff  = DMA2_CLRFF;
	    dparms->dma_addr   = (iochan << 2) | 0xC0;
	    dparms->dma_count   = dparms->dma_addr | 0x02;
	}
	dparms->dma_enable = (UCHAR)iochan;
	dparms->dma_disable = (UCHAR)(iochan | 0x04);
	dparms->dma_page = dma_page[dma_channel];
}

/***************************************************************************

FUNCTION DEFINITION:
os_pgm_dma - program the DMA controller

os_pgm_dma programs the DMA controller for a DMA transfer on a specified
virtual DMA channel.

RETURNS: void 
*/
void os_pgm_dma(
  int chan,     /* virtual DMA channel to program */
  ULONG count,  /* number of bytes to transfer */
  USHORT mode,  /* DMA transfer mode */
  USHORT page,  /* pjysical page number (highest 4 bits of phys. address */
  USHORT addr)  /* physical address of DMA buffer */
{
	DMA_ENTRY *dparms = &dma_parms[chan];
	UCHAR transfer_lsb, transfer_msb;
	UCHAR transfer_addr_lsb, transfer_addr_msb;

	mode |= dparms->dma_channel & 3;
	if (dparms->dma_channel >= 4) {
	    if (count & 1) count++;
	    transfer_lsb = (UCHAR)((count-2)>>1);
	    transfer_msb = (UCHAR)((count-2)>>9);
	    transfer_addr_lsb = (UCHAR)(addr >> 1);
	    transfer_addr_msb = (UCHAR)(addr >> 9);
	    if (page & 1) {
		transfer_addr_msb |= 0x80;
	    }
	} else {
	    transfer_lsb = (UCHAR)(count-1);
	    transfer_msb = (UCHAR)((count-1)>>8);
	    transfer_addr_lsb = (UCHAR)(addr);
	    transfer_addr_msb = (UCHAR)(addr >> 8);
	}

	OS_PUSH_DISABLE();
	OS_OUTPORTB(dparms->dma_single, dparms->dma_disable);	/* disable */
	OS_OUTPORTB(dparms->dma_mode, mode);			/* set mode */
	OS_OUTPORTB(dparms->dma_clrff, 0);			/* clear f/f */
	OS_OUTPORTB(dparms->dma_addr, transfer_addr_lsb);	/* LSB */
	OS_OUTPORTB(dparms->dma_addr, transfer_addr_msb);	/* MSB */
	OS_OUTPORTB(dparms->dma_page, page);			/* page # */
	OS_OUTPORTB(dparms->dma_clrff, 0);			/* clear f/f */
	OS_OUTPORTB(dparms->dma_count, transfer_lsb);		/* LSB count */
	OS_OUTPORTB(dparms->dma_count, transfer_msb);		/* MSB count */
	OS_OUTPORTB(dparms->dma_single, dparms->dma_enable);	/* enable */
	OS_POP_FLAGS();
}

/***************************************************************************

FUNCTION DEFINITION:
os_stop_dma - shutdown DMA transfer activity at the DMA controller

DESCRIPTION:
os_stop_dma stops DMA transfer activity at the DMA controller for a 
specified virtual DMA channel.

RETURNS: void 
*/
void os_stop_dma(
  int channel) /* virtual DMA channel to stop transfer */
{
	DMA_ENTRY *dparms = &dma_parms[channel];

	OS_OUTPORTB(dparms->dma_single, dparms->dma_disable);
}

/***************************************************************************

FUNCTION DEFINITION:
os_dma_count - get number of bytes left to be transferred

DESCRIPTION:
os_dma_count reads the DMA controllers current byte count.  Since the 
controller counts backwards to 0xffff, the number returned is the
number of bytes left to be transferred.

RETURNS: ULONG - number of bytes left to transfer
*/
ULONG os_dma_count(
  int channel) /* virtual DMA channel */
{
	DMA_ENTRY *dparms = &dma_parms[channel];
        ULONG xfer_left;
	UCHAR transfer_lsb;
	UCHAR transfer_msb;

	OS_PUSH_DISABLE();
	OS_OUTPORTB(dparms->dma_clrff, 0);		/* clear f/f */
	transfer_lsb = OS_INPORTB(dparms->dma_count);	/* LSB count */
	transfer_msb = OS_INPORTB(dparms->dma_count);	/* MSB count */
	OS_POP_FLAGS();
        xfer_left = ((USHORT)transfer_lsb + ((USHORT)transfer_msb<<8) + 1) & 0xffff;
	if (dparms->dma_channel >= 4) {	/* 16-bit channel */
	    xfer_left <<= 1;
	}
	return(xfer_left);
}

/***************************************************************************

FUNCTION DEFINITION:
os_file_read - read a number of bytes from a file

os_file_read calls the C runtime library function: DosRead.  This
routine reads up to 64K bytes from a file into a buffer specified by
the caller.

RETURNS: USHORT - number of bytes actually read
                  0 if at EOF before call
*/                  
USHORT os_file_read(
  int handle,           /* file handle to read from */
  void RFAR *io_buffer,  /* buffer to hold data from file */
  USHORT size)          /* number of bytes to read */
{
	USHORT actual;

	DosRead((HFILE)handle, io_buffer, bsize, &actual);
	return(actual);
}

/***************************************************************************

FUNCTION DEFINITION:
os_file_close - close a file

os_file_close calls the C runtime function: DosClose

RETURNS: USHORT - 0 on success
                  EOF if any errors were detected
*/
USHORT os_file_close(
  int handle) /* file handle to close */
{
	DosClose((HFILE)handle);
	return(0);
}

/***************************************************************************

FUNCTION DEFINITION:
os_file_seek - reposition a file pointer

os_file_seek calls the C runtime function: DosChgFilePtr.
Positions the file pointer a number of bytes from the file location given
by the method paremeter.

The method parameter can be one of:
  DOS_SEEK_START   - offset from beginning of file
  DOS_SEEK_END     - offset from the end of the file
  DOS_SEEK_CURRENT - offset from the current position of the file

RETURNS: USHORT - 0 on success
                  non-zero on error
*/
USHORT os_file_seek(
  int handle,    /* file to reposition pointer */
  ULONG foffset, /* number of bytes to change pointer position */
  int method)    /* file position to apply offset to */
{
	ULONG actual;

	DosChgFilePtr((HFILE)handle, foffset, method, (PULONG)&actual);
	return(0);
}

/***************************************************************************

FUNCTION DEFINITION:
os_file_open - open a file for reading and writing

os_file_open calls the C runtime function: DosOpen
This routine opens the file specified for both reading and writing.

RETURNS: int - pointer to opened file
               NULL on error
*/
int os_file_open(
  char RFAR *name) /* path including file name to be opened */
{
	ULONG rc;
	USHORT handle;
	USHORT actionTaken;
 
	rc = DosOpen(name, (PHFILE)&handle, &actionTaken, 0LU,
		0U,
		OPEN_ACTION_OPEN_IF_EXISTS,
		OPEN_FLAGS_NOINHERIT|OPEN_ACCESS_READONLY|OPEN_SHARE_DENYNONE,
		0L);
	if (rc) return(-1);
	return(handle);
}

/***************************************************************************

FUNCTION DEFINITION:
os_file_gets - get a string from a file

os_file_gets is a substitute for the standard C runtime fgets function.
This routine reads a string of characters from a file a places it into
a buffer specified by the caller.  The caller the maximum number of
bytes to be read.  This function stops when it reads either the maximum
number - 1 or a newline.

RETURNS: char RFAR * - pointer to the string read in
                      NULL on EOF or error
*/
char RFAR *os_file_gets(
  int handle,           /* file to read data from */
  char RFAR *io_buffer,  /* buffer for data from file */
  unsigned short size)  /* maximum number of bytes to be read */
{
        USHORT actual;
        long offset;

        /* Read the maximum possible buffer size */
        actual = os_file_read(handle, io_buffer, size-1);
        if (actual == 0) return NULL;
        /* Find the newline (in relation to the end of the string) */
        io_buffer[actual] = '\0';
        for (offset = 0; offset < (long)actual; offset++) {
            if (io_buffer[offset] == '\n') {
                io_buffer[++offset] = '\0';
                break;
            }
	    if (io_buffer[offset] == '\r')
	     io_buffer[offset] = '\n';
        }
        offset -= (long)actual;
        /* Backtrack the file pointer to point to the newline */
        os_file_seek(handle, offset, DOS_SEEK_CURRENT);

        return(io_buffer);
}

/***************************************************************************

FUNCTION DEFINITION:
os_set_irq - set the interrupt vector for an IRQ number 

os_set_irq - sets the interrupt vector for an IRQ number as opposed to 
an interrupt number.  In normal PC usage the IRQ number is mapped to the
interrupt vector table by adding 0x08 to the IRQ number for IRQ's between
0 - 7, and 0x70 for IRQ's from 8 - 15.

This routine converts the IRQ to an interrupt number, sets the vector to
the interrupt fuction specified by the caller, saving the old interrupt
vector in the process, and unmasks the IRQ specified.

RETURNS: void
*/
#ifdef OS2
void os_set_irq(
  int irq,               /* IRQ number to set voector */
  void (RFAR *handler)(void)) /* far pointer to new interrupt function */
#else
void os_set_irq(
  int irq,               /* IRQ number to set voector */
  OS_PINTERRUPT handler) /* far pointer to new interrupt function */
#endif
{
//	DevHelp_SetIRQ((int (pascal near *)())handler, irq, 0); /* not shared */
	DevHlp_SetIRQ((PVOID)handler, irq, 0); /* not shared */
}

/***************************************************************************

FUNCTION DEFINITION:
os_unset_irq - reset the interrupt vector for an IRQ number 

os_unset_irq resets the interrupt vector for an IRQ number to the orginal
value.

NOTE: this routine should only be called after calling os_set_irq

RETURNS: void

SEE ALSO:
os_set_irq
*/
void os_unset_irq(
  int irq) /* IRQ number to reset interrupt vector */
{
//	DevHelp_UnSetIRQ(irq);
	DevHlp_UnSetIRQ(irq);
}

/***************************************************************************

FUNCTION DEFINITION:
os_eoi - send the end-of-interrupt signal to the PIC

RETURNS: void
*/
void os_eoi(
  unsigned short irq) /* IRQ number to send EOI signal */
{
//	DevHelp_EOI(irq);
	DevHlp_EOI(irq);
}

// #define INT_STACK_SIZE	120
// unsigned char os_int_stack[INT_STACK_SIZE];
// unsigned char *os_int_p = os_int_stack;
static unsigned short sav_flags;
static unsigned short disable_count=0;

/***************************************************************************

FUNCTION DEFINITION:
os_push_disable - push the flags register on the stack and disable interrupts

RETURNS: void
*/
void os_push_disable(void)
{
	_asm {
	    cmp disable_count, 0
	    jnz bump
	    pushf
	    cli
	    pop ax
	    mov sav_flags, ax
bump:
	    inc disable_count
//	    mov bx, word ptr os_int_p
//	    mov word ptr [bx], ax
//	    add	os_int_p, 2
	}
}

/***************************************************************************

FUNCTION DEFINITION:
os_popf - pop the flags register off the stack

NOTE: if interrupts were enabled when the flags register was pushed on the
      stack, they will automatically be reenabled.

RETURNS: void
*/
void os_popf(void)
{
	_asm {
//	    sub os_int_p, 2
//	    mov bx, word ptr os_int_p
//	    mov ax, word ptr [bx]
	    sub disable_count,1
	    jnz done
	    mov ax, sav_flags
	    push ax
	    popf
done:
	}
}
