#ifndef lint
static char rcsid[] = "$Id: dbase.build.c,v 1.3 92/11/03 02:43:40 genek Exp $";
#endif

/*
 * dbase.build.c
 *
 *	build the preen.database file with the list of files that
 *	was generated by config.parse.c
 *
 * Gene Kim
 * Purdue University
 * September 27, 1992
 */

#include "../include/config.h"
#include <stdio.h>
#ifdef STDLIBH
#include <stdlib.h>
#include <unistd.h>
#endif
#include <fcntl.h>
#include <sys/param.h>
#include <sys/types.h>
#include <sys/stat.h>
#ifndef XENIX
# include <sys/time.h>
#else
# include <time.h>
#endif 	/* XENIX */
#ifdef DIRENT
# include <dirent.h>
#else
# ifndef XENIX
#  include <sys/dir.h>
# else		/* XENIX */
#  include <sys/ndir.h>
# endif		/* XENIX */
#endif	/* DIRENT */
#if (defined(SYSV) && (SYSV < 3))
# include <limits.h>
#endif	/* SVR2 */
#ifdef STRINGH
#include <string.h>
#else
#include <strings.h>
#endif
#include "../include/list.h"
#include "../include/tripwire.h"

#if defined(SYSV) && (SYSV < 4)
#ifndef HAVE_LSTAT
#  define lstat(x,y) stat(x,y)
#endif
#endif		/* SYSV */

int files_scanned_num = 0;

/* prototypes */
char *mktemp();
static void database_record_write();

#ifndef L_tmpnam
# define L_tmpnam (unsigned int) MAXPATHLEN
#endif

char backupfile[MAXPATHLEN];

/*
 * database_build(struct list **pp_list, int tempdatabaseflag)
 *
 *	take the list of file elements from the list and store all of
 *	the pertinent inode and signature information in the database
 *	file.
 *
 *	if (tempdatabaseflag) is set, then we write to a private file
 *	in the /tmp directory.
 */

void
database_build (pp_list, mode, pp_entry_list)
    struct list **pp_list;
    int mode;
    struct list **pp_entry_list;
{
    struct list_elem *p_fileentry;
    struct list_elem *p_configentry;
    FILE *fpw;
    char database[MAXPATHLEN];
    int entrynum,
        oldumask;
    extern int  errno;


#ifndef XENIX
    struct timezone tzone;
    struct timeval tval;
#else
    long tval;
#endif	/* XENIX */

    fprintf(stderr, "### Phase 3:   %s file information database\n",
	    mode == DBASE_UPDATE ? "Updating" : "Creating");


    /* create the database file
     * 		if we are making the permanent database, then we write
     *		to the specified file.
     *
     *		else, we create a temporary file, and save the name of it.
     */

    /* XXX - we should use open() so we can set the modes */

    oldumask = umask(077);

    if (mode == DBASE_TEMPORARY) {
	char *tmpfilename = (char *) malloc(L_tmpnam);
	(void) strcpy(tmpfilename, TEMPFILE_TEMPLATE);

	if ((char *) mktemp(tmpfilename) == NULL)
	    die_with_err("database_build: mktemp()", (char *) NULL);

	(void) strcpy(tempdatabase_file, tmpfilename);
	(void) strcpy(database, tempdatabase_file);
    }					/* end if temporary database */
    else {			
	sprintf(database, "%s/%s", database_path, database_file);
    }					/* end if non-temporary database */

    /* back up any existing database */
    if (mode == DBASE_UPDATE) {
	FILE *fpin, *fpout;
	char backup[MAXPATHLEN];
        register int ctemp;

	sprintf(backup, "%s.old", database_file);

	/* make sure we stay underneath maximum file length */

	if ((int)(strlen(database_file) + 4) > MAXNAMLEN) {

	    /* tack on .old as well as it fits */
	    (void) strcpy(backup + (MAXNAMLEN - 4), ".old");
	}
	/* so we can reference it later */
	(void) strcpy(backupfile, backup);  /* strlen(./Databases/) == 12 */

	if ((fpin = fopen(database, "r")) == NULL)
	    die_with_err("database_build: Couldn't open database `%s'!\n", database);

	else if ((fpout = fopen(backup, "w")) == NULL)
	    die_with_err("Couldn't open '%s'!\n", backup);

	/* make the backup file */
	while ((ctemp = getc(fpin)) != EOF)
	    putc((char) ctemp, fpout);

	(void) fclose(fpin);
	(void) fclose(fpout);
    }

    /* rebuild the database */
    if ((fpw = fopen(database, "w")) == NULL)
	die_with_err("Hint: Maybe the database directory '%s' doesn't exist?\n",
					database);

    (void) umask(oldumask);


    /* get time information for banner */

#ifndef XENIX
    if (gettimeofday(&tval, &tzone) < 0)
        die_with_err("gettimeofday()", (char *) NULL);
#else
    tval = time((long *) 0);
#endif	/* XENIX */


    /* add a banner to the top of the database file */
    /*		note that the newline comes from date  */
    fprintf(fpw, "# Generated by Tripwire, version %s on %s",
				version_num, ctime((time_t *)&tval));
    fprintf(fpw, "@@dbaseversion %d\n", db_version_num);

    /* we use &filelist as the key */
    if (list_open(pp_list) < 0)
	die_with_err("database_build: list_open() failed!\n", (char *) NULL);

    while ((p_fileentry = list_get(pp_list)) != NULL) {

	struct stat statbuf;
	char *filename, ignorevec[512];

	/* if we're in a debugging mood, print out the entries */

SPDEBUG(10)
printf("--(%3d)--> %s\n", files_scanned_num, p_fileentry->varname);

	/*
	 * if we're in UPDATE mode, we simply copy entries unless
	 * FLAG_UPDATE is set.
	 */

	if (mode == DBASE_UPDATE) {
	    char s[MAXPATHLEN*2];

	    if (! (list_getflag(p_fileentry->varname, pp_list) & FLAG_UPDATE)) {
		sprintf(s, "%s %s", p_fileentry->varname,
						p_fileentry->varvalue);
		fputs(s, fpw);
		continue;
	    }
	}

	/* get the stat information on it */
	filename = p_fileentry->varname;
	if (sscanf(p_fileentry->varvalue, "%d %s", &entrynum, ignorevec) != 2)
           die_with_err("database_build: sscanf() parsing error!\n",
						(char *) NULL);

	if (lstat(filename, &statbuf) < 0) {
	    if (errno == ENOENT) {
		fprintf(stderr,
		    "%s: %s: disappeared.  Skipping...\n", progname, filename);
		continue;
	    }
	    else
	      die_with_err("database_build: lstat()", filename);
	}

	/* pick up NO_OPEN flag if we're in UPDATE mode
	 *
	 * if it is a special file or device, add it to the list, but
	 * make sure we don't open it and read from it!
	 */
	if (mode == DBASE_UPDATE)
	    switch (statbuf.st_mode & S_IFMT) {
	      case S_IFIFO:
	      case S_IFCHR:
	      case S_IFDIR:
	      case S_IFBLK:
#if !defined(SYSV) || (SYSV > 3)
	      case S_IFSOCK:
#endif
		(void) list_setflag(filename, FLAG_NOOPEN, pp_list);
	    }

	database_record_write(fpw, filename, p_fileentry->flag, ignorevec,
					&statbuf, entrynum);

	files_scanned_num++;
    }					/* end while list_read() */

    /* cleanup */
    if (list_close(pp_list) < 0)
      die_with_err("database_build: list_close() failed!\n", (char *) NULL);

    /* print out table of contents in permanent database */
    if (mode != DBASE_TEMPORARY) {
	/* we use &pp_entry_list as the key */
	if (list_open(pp_entry_list) < 0)
	  die_with_err("database_build: list_open() failed!\n", (char *) NULL);

	/* print out the contents */
	while ((p_configentry = list_get(pp_entry_list)) != NULL)
	    fprintf(fpw, "@@contents %s %s\n", p_configentry->varname,
				p_configentry->varvalue);

	/* close the list */
	if (list_close(pp_entry_list) < 0)
	  die_with_err("database_build: list_close() failed!\n", (char *) NULL);
	}

    (void) fclose(fpw);

    return;
}

/*
 * database_record_write(FILE *fpw, char *filename, int flags,
 *                              char *ignorevec, struct stat *statbuf,
 *				int entrynum)
 *
 * 	write out the pertinent information of the specifed file to the
 *	database.
 *
 * 	gather the signatures, and include that in the info going to
 *		to the database.
 *
 *	(entrynum) is the unique entry number tag from tw.config.
 */

static void
database_record_write (fpw, filename, flags, ignorevec, statbuf, entrynum)
    FILE *fpw;
    char *filename;
    int flags;
    char *ignorevec;
    struct stat *statbuf;
    int entrynum;
{
    char 	sigs[NUM_SIGS][SIG_MAX_LEN];
    int		fd, i;
    int		ignoremask;
    char	vec64_a[50];
    char	vec64_c[50];
    char	vec64_m[50];
    char	sigs_concat[1024];
    /* filename, entrynum, ignore, mode, inode, nlinks, uid, gid, size,
     *		access, modify, ctime, {sig0, sig1, ..., sig9}
     */
    static char *format = "%s %ld %s %lo %ld %ld %ld %ld %ld %s %s %s %s\n";

    if (verbosity) {
	fprintf(stderr, "scanning: %s\n", filename);
    }

    /*
     * check for NOOPEN flag (for special files that shouldn't be
     * 		read from, like devices)
     *
     * if it's a symlink, then we collect the stat info on the link
     * 		itself.  remember, we never traverse symlinks!
     *
     * so, we make up null signatures.
     */
    if ((flags & FLAG_NOOPEN) || (flags & FLAG_SYMLINK)) {
	for (i = 0; i < NUM_SIGS; i++) {
	    register char *pc = sigs[i];
	    *pc++ = '0';
	    *pc++ = ' ';
	    *pc++ = '\0';
	}

	goto SKIPPED_SIGS;
    }

    /* descriptor for signature functions */
    if ((fd = open(filename, O_RDONLY)) < 0) {
	/* skip it if we had an error */
	warn_with_err("Trying to open %s for signature\n", filename);
	return;
    }

    /* first find out which signatures we don't need to collect */
    ignoremask = ignore_vec_to_scalar(ignorevec);

    /* collect signatures */
    for (i = 0; i < NUM_SIGS; i++) {
	char *pc = sigs[i];

	/* do we skip this signature? */
	if ((ignoremask & (IGNORE_0 << i)) || (runtimeignore & (IGNORE_0 << i)))
	    (void) strcpy(pc, "0 ");
	else {
	    (*pf_signatures[i])(fd, pc, SIG_MAX_LEN);
	    (void) strcat(pc, " ");
	}
    }

    /* close up the descriptor, since we're done */
    (void) close(fd);

SKIPPED_SIGS:

    /* concatenate all the signature */
    sigs_concat[0] = '\0';
    for (i = 0; i < NUM_SIGS; i++)
	strcat(sigs_concat, sigs[i]);

    /* escape any special characters in the filename */
    filename_escape(filename);

    /* filename, ignore, mode, inode, nlinks, uid, gid, size, access, modify,
     * 		ctime, sig0, sig1, ..., sig9
     */


    fprintf(fpw, format, filename, (long)entrynum, ignorevec,
	(long)statbuf->st_mode, (long)statbuf->st_ino,
	(long)statbuf->st_nlink, (long)statbuf->st_uid,
	(long)statbuf->st_gid, (long)statbuf->st_size,
	ltob64((unsigned long) statbuf->st_atime, vec64_a),
	ltob64((unsigned long) statbuf->st_mtime, vec64_m),
	ltob64((unsigned long) statbuf->st_ctime, vec64_c),
	sigs_concat);

    return;

}
