#ifndef lint
static char rcsid[] = "$Id: preen.c,v 1.2 92/11/03 02:43:57 genek Exp $";
#endif

/*
 * preen.c
 *
 *	preen the filesystems in preen.config against the data stored in
 *	in preen.database.
 *
 * Gene Kim
 * Purdue University
 */

#include "../include/config.h"
#include <stdio.h>
#ifdef STDLIBH
#include <stdlib.h>
#include <unistd.h>
#endif
#include <sys/param.h>
#ifdef STRINGH
#include <string.h>
#else
#include <strings.h>
#endif
#ifdef __STDC__
# include <sys/types.h>
# include <sys/stat.h>
#endif
#if (defined(SYSV) && (SYSV < 3))
# include <limits.h>
#endif	/* SVR2 */
#include "../include/list.h"
#include "../include/tripwire.h"

static struct list *preenlist = NULL;

/* prototypes */
char *mktemp();
static void preen_database_load();

/*
 * preen_gather()
 *
 *	routine that calls all the other functions for preening.
 */

void
preen_gather()
{
    FILE *fp_in;
    struct list *entry_list = (struct list *) NULL;

    /* build the filelist from the preen.config file
     * 		it will create the linked list of files
     */
    configfile_read(&preenlist, &entry_list);

    /* read in the old database */
    preen_database_load();

    /* preen ourselves:
     * 		build a temporary database, then check for diffs
     */
    database_build(&preenlist, DBASE_TEMPORARY, &entry_list);

    /* open it, and begin interp */
    if ((fp_in = fopen(tempdatabase_file, "r")) == NULL)
      die_with_err("preen_gather: fopen(%s)", tempdatabase_file);

    /* build the diff_xxx_lists of ADDED, CHANGED, DELETED */
    preen_interp(fp_in);

    /* now build the report */
    preen_report();
    (void) fclose(fp_in);

    /* remove the temporary database file */
    (void) unlink(tempdatabase_file);

    return;
}

/*
 * preen_update(char *entry)
 *
 *	build the filelist from tw.config file.
 *	check if (updateentry) is an entry
 *		if it is, update all entries with the same entrynum
 *		else if (updateentry) exists
 *			if so, update, w/same entrynum
 *			else append to database, w/entrynum = -1
 */

void
preen_update(entry)
    char *entry;
{
    struct list *entry_list = (struct list *) NULL;

    /* build the filelist from the preen.config file
     * 		it will create the linked list of files
     */
    configfile_read(&preenlist, &entry_list);

    /* read in the old database */
    preen_database_load();

    /* is (updateentry) an entry */
    if (list_isthere(entry, &entry_list)) {
	char s[MAXPATHLEN];
	int entrynum;

	/* get entry number */
	(void) strcpy(s, list_lookup(entry, &entry_list));
	if (sscanf(s, "%d", &entrynum) != 1) {
	    fprintf(stderr, "preen_update: sscanf() parse error!\n");
	    exit(1);
	}

	/* mark all files with this entry number as FLAG_UPDATE */
	database_update_markentries(&filelist, entrynum);

    }
    /* is there even such a file? */
    else if (list_isthere(entry, &preenlist) == 0) {
	fprintf(stderr, "tripwire: update couldn't find entry '%s'! Aborting...\n", entry);
	exit(1);
    }
    /* entry is just a regular file */
    else {
	/* make sure it's in our database */
	if (list_isthere(entry, &filelist) < 0) {
	    fprintf(stderr, "preen_update: entry `%s' not found!\n",
				entry);
	    exit(1);
	}
	list_setflag(entry, FLAG_UPDATE, &filelist);
    }

    /* preen ourselves:
     * 		build a temporary database, then check for diffs
     */
    database_build(&filelist, DBASE_UPDATE, &entry_list);

    /* print banner */
    fputs("###\n", stderr);
    fprintf(stderr,
"### Warning:   Old database file moved to `%s'\n", backupfile);
    fputs("###            in ./Databases.\n", stderr);
	exit(0);

    return;
}

/*
 * void
 * preen_database_load()
 *
 *	load in the old database file.
 */

static void
preen_database_load()
{
    char filename[MAXPATHLEN];
    FILE *fp;
    char s[MAXPATHLEN];
    char key[MAXPATHLEN], value[512];

    if (specified_dbasefile == NULL)
	sprintf(filename, "%s/%s", database_path, database_file);
    else
	(void) strcpy(filename, specified_dbasefile);

    /* check if we are to read from stdin */
    if (strcmp(filename, "-") == 0) {
	fp = stdin;
    }
    /* else open the file */
    else {
	if ((fp = fopen(filename, "r")) == NULL)
	    die_with_err("%s: Couldn't open database file '%s'",
			    filename);
    }

    /* read in entire file */
    while (fgets(s, sizeof(s), fp) != NULL) {

	/* skip comments */
	if (s[0] == '#')
	    continue;
	
	string_split_space(s, key, value);

	/* build table of contents */
	if (strcmp(key, "@@contents") == 0) {
	    continue;
	}
	/* skip database version */
	else if (strcmp(key, "@@dbaseversion") == 0) {
	    continue;
	}
	/* else it's a file */
	else {
	    /* expand any escaped characters */
	    filename_escape_expand(key);

	    list_set(key, value, 0, &filelist);
	}
    }

    /* close the file descriptor */
    if (fp != stdin) {
	(void) fclose(fp);
    }

    return;
}

