#ifndef lint
static char rcsid[] = "$Id: preen.report.c,v 1.3 92/11/03 02:44:02 genek Exp $";
#endif

/*
 * preen.report.c
 *
 *	report generation given the data from preening
 *
 * Gene Kim
 * Purdue University
 */

#include "../include/config.h"
#include <stdio.h>
#ifdef STDLIBH
#include <stdlib.h>
#endif
#include <sys/types.h>
#include <sys/stat.h>
#ifdef STRINGH
#include <string.h>
#else
#include <strings.h>
#endif
#include <time.h>
#ifdef MALLOCH
# include <malloc.h>
#endif
#include "../include/list.h"
#include "../include/tripwire.h"

static void preen_report_changed_enum();
static int preen_change_count();
static char *structstat_fill();
static void pair_print_ss();
static void pair_print_ll();
static void pair_print_llo();

/*
 * preen_report()
 *
 *	report on:
 *		which files have been ADDED
 *		which files have been DELETED
 *		which files have been CHANGED
 *			what attribute changed?
 */

void
preen_report()
{
    struct list_elem *p;
    struct stat statnew, statold;
    char sigsnew[NUM_SIGS][SIG_MAX_LEN], sigsold[NUM_SIGS][SIG_MAX_LEN];
    char *s;
    int unignored;

    fprintf(stderr, "###\n");
    fprintf(stderr, "###\t\t\tTotal files scanned:\t\t%d\n", files_scanned_num);
    fprintf(stderr, "###\t\t\t      Files added:\t\t%d\n", diff_added_num);
    fprintf(stderr, "###\t\t\t      Files deleted:\t\t%d\n", diff_deleted_num);
    fprintf(stderr, "###\t\t\t      Files changed:\t\t%d\n", diff_changed_num);
    fprintf(stderr, "###\n");
    fprintf(stderr, "###\t\t\tAfter applying rules:\n");
    fprintf(stderr, "###\t\t\t      Changes discarded:\t%d\n",
			diff_added_num + diff_deleted_num + diff_changed_num -
	      		(unignored = preen_change_count()) );
    fprintf(stderr, "###\t\t\t      Changes remaining:\t%d\n",
		        unignored + diff_added_num + diff_deleted_num);
    fprintf(stderr, "###\n");

    /****** added ******/

    /* open each of the three lists, using &diff_xxxx_list as keys */
    if (list_open(&diff_added_list) < 0) {
	fprintf(stderr, "preen_report: list_open() failed!\n");
	exit(1);
    }

    /* print out each added file in sequence */
    while ((p = list_get(&diff_added_list)) != NULL) {
	(void) structstat_fill(p->varvalue, &statnew, sigsnew);
	direntry_print(p->varname, statnew, DIFF_ADDED);
    }

    if (list_close(&diff_added_list) < 0) {
	fprintf(stderr, "preen_report: list_close() failed!\n");
	exit(1);
    }

    /****** deleted ******/

    /* now print out the files that were deleted */
    if (list_open(&diff_deleted_list) < 0) {
	fprintf(stderr, "preen_report: list_open() failed!\n");
	exit(1);
    }

    /* print out each added file in sequence */
    while ((p = list_get(&diff_deleted_list)) != NULL) {
	(void) structstat_fill(p->varvalue, &statnew, sigsnew);
	direntry_print(p->varname, statnew, DIFF_DELETED);
    }

    if (list_close(&diff_deleted_list) < 0) {
	fprintf(stderr, "preen_report: list_close() failed!\n");
	exit(1);
    }

    /***** changed ******/

    /*
     * interate through the list
     *		get the ignore vector
     *		foreach each (attribute) {
     *			if (attribute != attribute')
     *				if (!ignored) { flag it; }
     *		}
     */
    /*
    list_print(&diff_changed_list);
    */
    if (list_open(&diff_changed_list) < 0) {
	fprintf(stderr, "preen_report: list_open() failed!\n");
	exit(1);
    }

    /* print out each added file in sequence */
    while ((p = list_get(&diff_changed_list)) != NULL) {

	/* copy the filename and expand any escaped characters */

	/* filename, ignore, mode, inode, nlinks, uid, gid, size, access,
	 * modify, ctime, sig1, sig2
	 */

	/* read in the new value from the changed_list
	 *		throw away the new ignorevector -- we use the old one!
	 */
	(void) structstat_fill(p->varvalue, &statnew, sigsnew);

	/* read in the list1 value form the hash table */
	if ((s = list_lookup(p->varname, &filelist)) == NULL) {
	    fprintf(stderr, "preen_report: list_lookup() failed!\n");
	    exit(1);
	}

	(void) structstat_fill(s, &statold, sigsold);

	/* is this file to be ignored? */
	if (!(list_getflag(p->varname, &diff_changed_list) & FLAG_CHANGED))
	    continue;

	/* print out the report for this file */
	direntry_print(p->varname, statnew, DIFF_CHANGED);

    }

    if (list_close(&diff_changed_list) < 0) {
	fprintf(stderr, "preen_report: list_close() failed!\n");
	exit(1);
    }

    /* enumerate specifics of changed files, if long output specified */
    if (!quiet && unignored != 0) {
	preen_report_changed_enum();
    }

    return;
}

/*
 * preen_report_changed_enum()
 *
 *	enumerate each changed attributed for each of the changed files.
 *	this is treated as yet another pass in the checking process.
 */

static void
preen_report_changed_enum()
{
    struct list_elem *p;
    char *ignorevec;
    char sigsold[NUM_SIGS][SIG_MAX_LEN], sigsnew[NUM_SIGS][SIG_MAX_LEN];
    struct stat statnew, statold;
    char *s;
    char stime1[64], stime2[64];
    int ignoremask;
    int i;
    char label[50];

    (void) fflush(stdout);
    fprintf(stderr, "### Phase 5:   Generating observed/expected pairs for changed files\n");
    fprintf(stderr, "###\n");
    (void) fflush(stderr);

printf("### Attr        Observed (what it is)	      Expected (what it should be)\n");
printf("### =========== ============================= =============================\n");
    /****
    st_atime: Mon Aug 31 16:48:57 1992         Mon Aug 31 14:05:49 1992
    ****/

    /* open the list of changed files */
    if (list_open(&diff_changed_list) < 0) {
	fprintf(stderr, "preen_report: list_open() failed!\n");
	exit(1);
    }

    /* print out each added file in sequence */
    while ((p = list_get(&diff_changed_list)) != NULL) {

	/* filename, ignore, mode, inode, nlinks, uid, gid, size, access,
	 * modify, ctime, sig1, sig2 .. sign
	 */

	/* read in the list2 value from the changed_list
	 *		throw away the new ignorevector -- we use the old one!
	 */
	(void) structstat_fill(p->varvalue, &statnew, sigsnew);

	/* read in the list1 value form the hash table */
	if ((s = list_lookup(p->varname, &filelist)) == NULL) {
	    fprintf(stderr, "preen_report_changed_enum: list_lookup() failed!\n");
	    exit(1);
	}

	ignorevec = structstat_fill(s, &statold, sigsold);

	/* get the ignoremask */
	ignoremask = ignore_vec_to_scalar(ignorevec);

	/* is this file to be ignored? */
	if (!(list_getflag(p->varname, &diff_changed_list) & FLAG_CHANGED))
	    continue;

	printf("%s\n", p->varname);
	/* and then the {expected, received} pairs */

#define STATEQ(x) (statnew.x != statold.x)

	if (!(ignoremask & IGNORE_P))
	    if (STATEQ(st_mode)) {
		pair_print_llo("st_mode:", (long) statnew.st_mode,
			(long) statold.st_mode);
	    }
	
	if (!(ignoremask & IGNORE_I))
	    if (STATEQ(st_ino)) {
		pair_print_ll("st_ino:", (long) statnew.st_ino,
			(long) statold.st_ino);
	    }
	
	if (!(ignoremask & IGNORE_N))
	    if (STATEQ(st_nlink)) {
		pair_print_ll("st_nlink:", (long) statnew.st_nlink,
			(long) statold.st_nlink);
	    }

	if (!(ignoremask & IGNORE_U))
	    if (STATEQ(st_uid)) {
		pair_print_ll("st_uid:", (long) statnew.st_uid,
			(long) statold.st_uid);
	    }

	if (!(ignoremask & IGNORE_G))
	    if (STATEQ(st_gid)) {
		pair_print_ll("st_gid:", (long) statnew.st_gid,
			(long) statold.st_gid);
	    }

	if (!(ignoremask & IGNORE_S))
	    if (STATEQ(st_size)) {
		pair_print_ll("st_size:", (long) statnew.st_size,
			(long) statold.st_size);
	    }

	if (!(ignoremask & IGNORE_A))
	    if (STATEQ(st_atime)) {
		(void) strcpy(stime1, ctime(&statnew.st_atime));
		(void) strcpy(stime2, ctime(&statold.st_atime));
		chop(stime1);
		chop(stime2);
		pair_print_ss("st_atime:", stime1, stime2);
	    }

	if (!(ignoremask & IGNORE_M))
	    if (STATEQ(st_mtime)) {
		(void) strcpy(stime1, ctime(&statnew.st_mtime));
		(void) strcpy(stime2, ctime(&statold.st_mtime));
		chop(stime1);
		chop(stime2);
		pair_print_ss("st_mtime:", stime1, stime2);
	    }

	if (!(ignoremask & IGNORE_C))
	    if (STATEQ(st_ctime)) {
		(void) strcpy(stime1, ctime(&statnew.st_ctime));
		(void) strcpy(stime2, ctime(&statold.st_ctime));
		chop(stime1);
		chop(stime2);
		pair_print_ss("st_ctime:", stime1, stime2);
	    }

	for (i = 0; i < NUM_SIGS; i++) {
	    if (!(runtimeignore & (IGNORE_0 << i)) &&
					!(ignoremask & (IGNORE_0 << i)))
		if (strcmp(sigsnew[i], sigsold[i]) != 0) {
		    (void) sprintf(label, "%s (sig%d):", signames[i], i);
		    pair_print_ss(label, sigsnew[i], sigsold[i]);
		}

	}

	/* separate entries by a space */
	printf("\n");
    }

    if (list_close(&diff_changed_list) < 0) {
	fprintf(stderr, "preen_report_changed_enum: list_close() failed!\n");
	exit(1);
    }
}

/*
 * preen_change_count()
 *
 *	return the number of files that are changed, according to their
 *	ignore vectors.
 */

static int
preen_change_count()
{
    int changed = 0;
    struct list_elem *p;
    char sigsold[NUM_SIGS][SIG_MAX_LEN], sigsnew[NUM_SIGS][SIG_MAX_LEN];
    char vec64_a[50], vec64_m[50], vec64_c[50];
    char trash[512];
    struct stat statnew, statold;
    char *s;
    int ignoremask;
    char ignorevec[512];
    unsigned long mode, ino, nlink, uid, gid, size;
    int entrynum;
    int nfields;

    /***** changed ******/

    /*
     * interate through the list
     *		get the ignore vector
     *		foreach each (attribute) {
     *			if (attribute != attribute')
     *				if (!ignored) { flag it; }
     *		}
     */
    if (list_open(&diff_changed_list) < 0) {
	fprintf(stderr, "preen_report: list_open() failed!\n");
	exit(1);
    }

    /* print out each added file in sequence */
    while ((p = list_get(&diff_changed_list)) != NULL) {

	/* filename, ignore, mode, inode, nlinks, uid, gid, size, access,
	 * modify, ctime, sig1, sig2
	 */

	/* read in the list2 value from the changed_list
	 *		throw away the new ignorevector -- we use the old one!
	 */

	if ((nfields = sscanf(p->varvalue, db_record_format,
		&entrynum, trash,
		&mode, &ino, &nlink, &uid, &gid, &size,
		vec64_a, vec64_m, vec64_c,
		sigsnew[0], sigsnew[1], sigsnew[2], sigsnew[3], sigsnew[4],
		sigsnew[5], sigsnew[6], sigsnew[7], sigsnew[8], sigsnew[9]))
				!= DB_RECORD_FIELDS) {
	    fprintf(stderr, "preen_change_count: illegal database record (nfields == %d).   Aborting...\n", nfields);
	    fprintf(stderr, "	'%s'\n", p->varvalue);
	    exit(1);
	}
        statnew.st_mode = (mode_t)mode;
        statnew.st_ino = (ino_t)ino;
        statnew.st_nlink = (nlink_t)nlink;
        statnew.st_uid = (uid_t)uid;
        statnew.st_gid = (gid_t)gid;
        statnew.st_size = (off_t)size;

	/* convert from base64 to int */
	statnew.st_atime = b64tol(vec64_a);
	statnew.st_mtime = b64tol(vec64_m);
	statnew.st_ctime = b64tol(vec64_c);

	/* read in the list1 value form the hash table */
	if ((s = list_lookup(p->varname, &filelist)) == NULL) {
	    fprintf(stderr, "preen_report: list_lookup() failed!\n");
	    exit(1);
	}

	if (sscanf(s, db_record_format,
		&entrynum, ignorevec,
		&mode, &ino, &nlink, &uid, &gid, &size,
		vec64_a, vec64_m, vec64_c,
		sigsold[0], sigsold[1], sigsold[2], sigsold[3], sigsold[4],
		sigsold[5], sigsold[6], sigsold[7], sigsold[8], sigsold[9])
				!= DB_RECORD_FIELDS) {
	    fprintf(stderr, "preen_change_count: illegal database record! Aborting...\n");
	    fprintf(stderr, "	'%s'\n", s);
	    exit(1);
	}
        statold.st_mode = (mode_t)mode;
        statold.st_ino = (ino_t)ino;
        statold.st_nlink = (nlink_t)nlink;
        statold.st_uid = (uid_t)uid;
        statold.st_gid = (gid_t)gid;
        statold.st_size = (off_t)size;

	/* convert from base64 to int */
	statold.st_atime = b64tol(vec64_a);
	statold.st_mtime = b64tol(vec64_m);
	statold.st_ctime = b64tol(vec64_c);

	/* get the ignoremask */
	ignoremask = ignore_vec_to_scalar(ignorevec);

	/* and then the {expected, received} pairs */

#define FLAGIT() changed++; list_setflag(p->varname, FLAG_CHANGED, &diff_changed_list); continue
#define SIGEQ(x) if (strcmp(sigsnew[(x)], sigsold[(x)]) != 0)

	/* note the pain we go through to avoid dangling else's */
	if (!(ignoremask & IGNORE_P)) { if (STATEQ(st_mode)) {FLAGIT();}}
	if (!(ignoremask & IGNORE_I)) { if (STATEQ(st_ino)) {FLAGIT();}}
	if (!(ignoremask & IGNORE_N)) { if (STATEQ(st_nlink)) {FLAGIT();}}
	if (!(ignoremask & IGNORE_U)) { if (STATEQ(st_uid)) {FLAGIT();}}
	if (!(ignoremask & IGNORE_G)) { if (STATEQ(st_gid)) {FLAGIT();}}
	if (!(ignoremask & IGNORE_S)) { if (STATEQ(st_size)) {FLAGIT();}}
	if (!(ignoremask & IGNORE_A)) { if (STATEQ(st_atime)) {FLAGIT();}}
	if (!(ignoremask & IGNORE_M)) { if (STATEQ(st_mtime)) {FLAGIT();}}
	if (!(ignoremask & IGNORE_C)) { if (STATEQ(st_ctime)) {FLAGIT();}}
	if (!(runtimeignore & IGNORE_0) && !(ignoremask & IGNORE_0))
					{ SIGEQ(0) {FLAGIT();}}
	if (!(runtimeignore & IGNORE_1) && !(ignoremask & IGNORE_1))
					{ SIGEQ(1) {FLAGIT();}}
	if (!(runtimeignore & IGNORE_2) && !(ignoremask & IGNORE_2))
					{ SIGEQ(2) {FLAGIT();}}
	if (!(runtimeignore & IGNORE_3) && !(ignoremask & IGNORE_3))
					{ SIGEQ(3) {FLAGIT();}}
	if (!(runtimeignore & IGNORE_4) && !(ignoremask & IGNORE_4))
					{ SIGEQ(4) {FLAGIT();}}
	if (!(runtimeignore & IGNORE_5) && !(ignoremask & IGNORE_5))
					{ SIGEQ(5) {FLAGIT();}}
	if (!(runtimeignore & IGNORE_6) && !(ignoremask & IGNORE_6))
					{ SIGEQ(6) {FLAGIT();}}
	if (!(runtimeignore & IGNORE_7) && !(ignoremask & IGNORE_7))
					{ SIGEQ(7) {FLAGIT();}}
	if (!(runtimeignore & IGNORE_8) && !(ignoremask & IGNORE_8))
					{ SIGEQ(8) {FLAGIT();}}
	if (!(runtimeignore & IGNORE_9) && !(ignoremask & IGNORE_9))
					{ SIGEQ(9) {FLAGIT();}}

    }

    /* clean up */
    if (list_close(&diff_changed_list) < 0) {
	fprintf(stderr, "preen_report: list_close() failed!\n");
	exit(1);
    }

    return changed;
}

/*
 * structstat_fill(char *string, struct stat *statbuf)
 *
 *	given a string from the database, fill in the statbuf structure.
 *	
 *	return the ignore vector (a static system structure)
 */

static char *
structstat_fill (string, statbuf, sigs)
    char *string;
    struct stat *statbuf;
    char sigs[NUM_SIGS][SIG_MAX_LEN];
{
    char *ignorevec;
    static char structstat_fill_string[512];
    unsigned long        mode, ino, nlink, uid, gid, size;
    int entrynum;
    char vec64_a[50], vec64_m[50], vec64_c[50];

    (void) strcpy(structstat_fill_string, string);
    ignorevec = structstat_fill_string;

    if (sscanf(string, db_record_format,
		&entrynum, ignorevec,
		&mode, &ino, &nlink, &uid, &gid, &size,
		vec64_a, vec64_m, vec64_c,
		sigs[0], sigs[1], sigs[2], sigs[3], sigs[4],
		sigs[5], sigs[6], sigs[7], sigs[8], sigs[9])
				!= DB_RECORD_FIELDS) {
	fprintf(stderr, "structstat_fill: illegal database record! Aborting...\n");
	fprintf(stderr, "	'%s'\n", string);
	exit(1);
    }
    statbuf->st_mode = (mode_t)mode;
    statbuf->st_ino = (ino_t)ino;
    statbuf->st_nlink = (nlink_t)nlink;
    statbuf->st_uid = (uid_t)uid;
    statbuf->st_gid = (gid_t)gid;
    statbuf->st_size = (off_t)size;

    /* convert from base64 to int */
    statbuf->st_atime = b64tol(vec64_a);
    statbuf->st_mtime = b64tol(vec64_m);
    statbuf->st_ctime = b64tol(vec64_c);

    return ignorevec;
}

/*
 * pair_print_ss(char *label, char *s1, char *s2)
 *
 *	print {expected,received} table with strings
 */

static void
pair_print_ss (label, s1, s2)
    char *label;
    char *s1;
    char *s2;
{
    printf("%15s %-30s%-30s\n", label, s1, s2);
    return;
}

/*
 * pair_print_ll(char *label, long l1, long l2)
 *
 *	print {expected,received} table with longs
 */

static void
pair_print_ll (label, l1, l2)
    char *label;
    long l1;
    long l2;
{
    printf("%15s %-30ld%-30ld\n", label, l1, l2);
    return;
}

/*
 * pair_print_llo(char *label, long l1, long l2)
 *
 *	print {expected,received} table with longs in octal
 */

static void
pair_print_llo (label, l1, l2)
    char *label;
    long l1;
    long l2;
{
    printf("%15s %-30lo%-30lo\n", label, l1, l2);
    return;
}

