#ifndef lint
static char rcsid[] = "$Id: utils.c,v 1.27 92/10/20 14:59:01 genek Exp $";
#endif

/*
 * utils.c
 *
 *	miscellaneous utilities for Tripwire
 *
 * Gene Kim
 * Purdue University
 */

#include "../include/config.h"
#include <stdio.h>
#ifdef STDLIBH
#include <stdlib.h>
#endif
#include <ctype.h>
#ifdef STRINGH
#include <string.h>
#else
#include <strings.h>
# if (!defined(strchr) && !defined(index))
#  define strchr(s, c) index(s, c)
# endif
# if (!defined(memcpy) && !defined(bcopy))
#  define memcpy(to, from, n) bcopy(from, to, n)
# endif
#endif
#include <sys/types.h>
#include <sys/stat.h>
#include <pwd.h>
#include <grp.h>
#include <sys/param.h>
#include <ctype.h>
#include <sys/param.h>
#ifndef MAXHOSTNAMELEN
#define MAXHOSTNAMELEN 64
#endif
#ifndef XENIX
# include <sys/time.h>
#else
# include <time.h>
#endif 	/* XENIX */
#ifndef GETHOSTNAME
# include <sys/utsname.h>
#endif
#if (defined(SYSV) && (SYSV < 3))
# include <limits.h>
#endif	/* SVR2 */
#include "../include/list.h"
#include "../include/tripwire.h"

static void print_perm();

#ifndef S_IRGRP
#define S_IRGRP	(S_IREAD >> 3)
#define S_IWGRP (S_IWRITE >> 3)
#define S_IXGRP (S_IEXEC >> 3)
#define S_IROTH (S_IREAD >> 6)
#define S_IWOTH (S_IWRITE >> 6)
#define S_IXOTH (S_IEXEC >> 6)
#endif

void warn_with_err(format, name)
   char *format, *name;
{
    extern int  errno;
    int real_errno = errno;
    char *string;

    if (!name)
      string = format;
    else {
	string = (char *) malloc((unsigned) (strlen(format)+strlen(name)+1));
	if (!string) {
	    fputs("Unexpected malloc() failure in 'warn_with_err'!\n", stderr);
	    exit(-1);
	}
	sprintf(string, format, name);
	errno = real_errno;
    }

    perror(string);
}

void
die_with_err(format, name)
    char *format, *name;
{
    warn_with_err(format, name);
    exit(1);
}

/*
 * filename_hostname_expand(char **ps)
 *
 *	expand any '@'s in the specified string to the hostname.
 *
 *	Ex:   "xxx_@_xxx"  ---> "xxx_mentor.cc.purdue.edu_xxx"
 */

static char hostname[MAXHOSTNAMELEN];

void
filename_hostname_expand(ps)
    char **ps;
{
    char *s = *ps;
    char outpath[MAXPATHLEN];
    char *pc;

    if (! *hostname) {   /* we only need to do this once */
#ifndef GETHOSTNAME
    struct utsname sysinfo;

    if (uname(&sysinfo) < 0)
	die_with_err("filename_hostname_expand: uname()", (char *) NULL);

    (void) strcpy(hostname, sysinfo.nodename);

#else 	/* GETHOSTNAME */

    /* get the hostname */
    if (gethostname(hostname, sizeof(hostname)) < 0)
	die_with_err("filename_hostname_expand: gethostname()", (char *) NULL);

#endif 	/* GETHOSTNAME */
    }

    /* is there a '@' in the filename? */
    if ((pc = strchr(s, '@')) == NULL) {
	return;
    }

    /* copy the first part of the string */
    (void) strncpy(outpath, s, pc-s);

    /* strncpy() doesn't guarantee null-terminated strings! */
    outpath[pc-s] = '\0';

    /* expand the '@' and copy the rest of the string */
    (void) strcat(outpath, hostname);
    (void) strcat(outpath, pc+1);

    /* make our pointer point to the expanded string */
    if ((pc = (char *) malloc((unsigned int) (strlen(outpath) + 1))) == NULL)
	die_with_err("filename_hostname_expand: malloc()", (char *) NULL);

    (void) strcpy(pc, outpath);

    *ps = pc;

    return;
}

/*
 * slash_count(char *pathname)
 *
 *	count the number of slashes in a given pathname.  This is used
 * 	to determine the priority of a given file entry when generating
 * 	the list of files.
 */

int
slash_count (pathname)
    char *pathname;
{
	register int count = 0;
	register char *pc;

	for (pc = pathname; *pc; pc++ )
		if (*pc == '/')
			count++;
	return count;
}

/*
 * string_split_space(char *string, char *s, char *t)
 *
 * 	given (string), place the first word into (s), and the rest of
 *	into (t).
 */

void
string_split_space (string, s, t)
    char *string;
    char *s;
    char *t;
{
    char *sp;

    /*
     * (char *sp) = the first space.  	s = {string[0..(sp-s-1)]}
     *			      		t = {sp[1..end]}
     */

    if ((sp = strchr(string, ' ')) == NULL) {
	fprintf(stderr, "string_split_space: string doesn't contain space!\n");
	exit(1);
	/* XXX - yikes!  this shouldn't be fatal!!!  */
    }

    /* don't forget to null-terminate the string w/strncpy() */
    (void) strncpy(s, string, sp-string);
    s[sp-string] = '\0';

    (void) strcpy(t, sp+1);
    return;
}

/*
 * int
 * string_split_ch(char *string, char *s, char *t, char ch)
 *
 * 	given (string), place the first word into (s), and the rest of
 *	into (t), using (ch) as the field separator.  (ala perl).
 */

int
string_split_ch (string, s, t, ch)
    char *string;
    char *s;
    char *t;
    char ch;
{
    char *sp;

    /*
     * (char *sp) = the first space.  	s = {string[0..(sp-s-1)]}
     *			      		t = {sp[1..end]}
     */

    if ((sp = strchr(string, ch)) == NULL) {
	(void) strcpy(s, string);
	t[0] = '\0';
	return -1;
    }

    /* don't forget to null-terminate the string w/strncpy() */
    (void) strncpy(s, string, sp-string);
    s[sp-string] = '\0';

    (void) strcpy(t, sp+1);
    return 0;
}

/*
 * chop (char *s)
 *
 *	chop off the last character in a string, ala Perl.
 */

void
chop (s)
    char *s;
{
	int slen;

	slen = strlen(s);
	s[slen-1] = '\0';
	return;
}

/*
 * filename_escape_expand(char *filename)
 *
 *	expand \xxx octal characters, metachacters, and known
 *	C escape sequences.
 */

void
filename_escape_expand (filename)
    char *filename;
{
    int i = 0;
    char filetmp[MAXPATHLEN];
    int octal;
    register char *pcin = filename, *pcout = filetmp;

    /*
     * case I:	it's not escaped
     * case II: 	it's a three digit octal number
     * case III:	it's a standard C escape sequence
     *				(\n, \r, \', \", \t, \b, \f)
     *			(from Johnson, Stephen C.,
     *				"Yacc: Yet Another Compiler-Compiler")
     * case IV:	it's one of our metacharacters {@#!|&()= }
     */

    while (*pcin) {

	/* case I: it's not an escape */
	if (*pcin != '\\')
		*pcout++ = *pcin++;

	/* case II: it's a three digit octal number */
	else if (isdigit(*++pcin)) {
	    /* read in the three characters */
	    for (octal = i = 0; i < 3 ; i++, pcin++) {
		octal *= 8;
		
		if (*pcin > '7' || *pcin < '0') {
		    fprintf(stderr,
			    "filename_escape_expand: bogus octal character (%c) in file `%s'!\n",
			    *pcin, filename);
		    exit(1);
		}
		else
		  octal += *pcin-'0';
	    }

	    /* warn of filenames with null's in them */
	    if (octal == 0) {
		fprintf(stderr, "tripwire: warning: null character in file `%s'!\n",  filename);
		exit(1);
	    }

	    *pcout++ = octal & 0xff;
	}

	/* case III: it's a standard C escape sequence */
	/* case IV: it's one of our escape characters */
	else
	    switch(*pcin) {
	    case 'n':		{ *pcout++ = '\n'; break; }
	    case 'r':		{ *pcout++ = '\r'; break; }
	    case 't':		{ *pcout++ = '\t'; break; }
	    case 'b':		{ *pcout++ = '\b'; break; }
	    case 'f':		{ *pcout++ = '\f'; break; }
	    case '\'':		
	    case '"':		
	    case '@':
	    case '!':
	    case '#':
	    case '=':
	    case ' ':
	    case ')':
	    case '(':
	    case '&':
	    case '|':
	    case '\\':
	      /* same as our default case... it's the character itself */
	    default: 		{ *pcout++ = *pcin++; break; }
	  }
    }


    /* null terminate the string */
    *pcout++ = '\0';

    (void) memcpy(filename, filetmp, pcout - filetmp);
    return;
}

/*
 * filename_escape(char *filename)
 *
 *	find any characters that must be escaped in the file name.
 */

void
filename_escape (filename)
    char *filename;
{
    char filetmp[MAXPATHLEN];
    register char *pcin = filename, *pcout = filetmp;
    static char *octal_array[] = {
	"000", "001", "002", "003", "004", "005", "006", "007",
	"010", "011", "012", "013", "014", "015", "016", "017",
	"020", "021", "022", "023", "024", "025", "026", "027",
	"030", "031", "032", "033", "034", "035", "036", "037",
	"040", "041", "042", "043", "044", "045", "046", "047",
	"050", "051", "052", "053", "054", "055", "056", "057",
	"060", "061", "062", "063", "064", "065", "066", "067",
	"070", "071", "072", "073", "074", "075", "076", "077",
	"100", "101", "102", "103", "104", "105", "106", "107",
	"110", "111", "112", "113", "114", "115", "116", "117",
	"120", "121", "122", "123", "124", "125", "126", "127",
	"130", "131", "132", "133", "134", "135", "136", "137",
	"140", "141", "142", "143", "144", "145", "146", "147",
	"150", "151", "152", "153", "154", "155", "156", "157",
	"160", "161", "162", "163", "164", "165", "166", "167",
	"170", "171", "172", "173", "174", "175", "176", "177",
    };
    register char *pccopy;

    /* these only matter if they are the first character */
    if (*pcin == '!' || *pcin == '=' || *pcin == '#')	
	{ *pcout++ = '\\'; *pcout++ = *pcin++; }

    /* these must be replace everywhere in the filename */
    while (*pcin) {
	if (isalnum(*pcin))
	    *pcout++ = *pcin;
	else
	    switch(*pcin) {
	      case '\\':
	      case '\n':
	      case '\r':
	      case '\t':
	      case '\b':
	      case '\f':
	      case '\'':
	      case '\"':
	      case '@':
	      case ' ':
	      case '(':
	      case ')':
	      case '&':
	      case '|':
	      case '#':
		*pcout++ = '\\';
		*pcout++ = *(pccopy = octal_array[*pcin]);
		*pcout++ = *++pccopy;
		*pcout++ = *++pccopy;
		break;
	    default:
		*pcout++ = *pcin;
		break;
	    }
	pcin++;
    }

    /* null terminate the string */
    *pcout++ = '\0';

    (void) memcpy(filename, filetmp, pcout - filetmp);
}

/*
 * lto64(long num, char *vec64)
 *
 *	convert a long integer to a base-64 string.
 *
 *	we return the pointer to the string containing the base-64 number.
 *	the string will always be padded so it is 6 bytes.
 *
 *	REMEMBER: this is also used in the signature routines
 */

static char base64vec[] =
  "./0123456789ABCDEFGHIJKLMNOPQRSTUVWXYZabcdefghijklmnopqrstuvwxyz";

char *
ltob64(num, vec64)
    register unsigned long num;
    char *vec64;
{
    register char *p1 = vec64;
    register int i;


    /* build lsb -> msb */
    for (i = 5; i >= 0; i--) {
        p1[i] = base64vec[num & 0x3f];
	num >>= 6;
    }

    vec64[6] = 0;

    return vec64;
}

/*
 * long
 * b64toi(char *vec)
 *
 *	given a base-64 string, convert to a long.
 */

long
b64tol(vec)
    char *vec;
{
    register char *pc;
    register long num = 0L;

/*  The following takes advantage of the fact that
 *  the ASCII collating sequence has  . / 0 1 2 3...9
 *  all in sequence.  Your milage may vary on other systems.
 *
 *  The string is interpreted as base 64 even if it isn't :-)
 */

    for (pc = vec; *pc; pc++) {
	num <<= 6;
	num += *pc;
	
	if (*pc <= '9') 	
	  num -= '.';
	else if (*pc <= 'Z')
	  num -= '5';  /* '5' == 'A' - 12 */
	else
	  num -= ';';  /* ';' == 'a' - 38 */
    }

    return num;
}

/*
 * direntry_print(char *name, struct stat stabuf))
 *
 *	print out a pretty directory entry for the specified file
 *
 *	this routine was taken from crc_check.c, written by Jon Zeeff
 *	(zeeff@b-tech.ann-arbor.mi.us)
 *
 *	hacked for use in Tripwire by Gene Kim (genek@mentor.cc.purdue.edu).
 */

void
direntry_print (name, statbuf, mode)
    char *name;
    struct stat statbuf;
    int mode;
{
	struct passwd *entry;
	static char owner[20];
	char    a_time[50];

	static int prev_uid = -9999;

	switch(mode) {
	case DIFF_ADDED:
		printf("added:   "); break;
	case DIFF_CHANGED:
		printf("changed: "); break;
	case DIFF_DELETED:
		printf("deleted: "); break;
	}

	if (statbuf.st_uid != prev_uid) {
		entry = (struct passwd *)getpwuid((int) statbuf.st_uid);
		if (entry)
			(void) strcpy(owner, entry->pw_name);
		else
			(void) sprintf(owner, "%d", statbuf.st_uid);
		prev_uid = statbuf.st_uid;
	}
	/*
	if (statbuf.st_gid != prev_gid) {
		group_entry = getgrgid((int) statbuf.st_gid);
		if (group_entry)
			(void) strcpy(group, group_entry->gr_name);
		else
			(void) sprintf(group, "%d", statbuf.st_gid);
		prev_gid = statbuf.st_gid;
	}
	*/

	(void) strcpy(a_time, ctime(&statbuf.st_mtime));
	a_time[24] = '\0';

	print_perm((unsigned long)statbuf.st_mode);

	(void) printf(" %-9.9s %7d %s", owner, statbuf.st_size,
						a_time + 4);
	printf(" %s\n", name);

}

/*	
 *	This routine was taken from crc_check.c, written by Jon Zeeff
 *	(zeeff@b-tech.ann-arbor.mi.us)
 *
 *	hacked for use in Tripwire by Gene Kim (genek@mentor.cc.purdue.edu).
 */

static void
print_perm(perm)
    unsigned long perm;
{

	char    string[20];

	(void) strcpy(string, "----------");

	switch (perm & S_IFMT) {

	case S_IFDIR:
		string[0] = 'd';
		break;

	case S_IFBLK:
		string[0] = 'b';
		break;

	case S_IFCHR:
		string[0] = 'c';
		break;

	case S_IFIFO:
		string[0] = 'p';
		break;
	}
	if (perm & S_IREAD)
		string[1] = 'r';
	if (perm & S_IWRITE)
		string[2] = 'w';
	if (perm & S_ISUID && perm & S_IEXEC)
		string[3] = 's';
	else if (perm & S_IEXEC)
		string[3] = 'x';
	else if (perm & S_ISUID)
		string[3] = 'S';

	if (perm & S_IRGRP)
		string[4] = 'r';
	if (perm & S_IWGRP)
		string[5] = 'w';
	if (perm & S_ISUID && perm & S_IXGRP)
		string[6] = 's';
	else if (perm & S_IXGRP)
		string[6] = 'x';
	else if (perm & S_ISUID)
		string[6] = 'l';

	if (perm & S_IROTH)
		string[7] = 'r';
	if (perm & S_IWOTH)
		string[8] = 'w';
	if (perm & S_ISVTX && perm & S_IXOTH)
		string[9] = 't';
	else if (perm & S_IXOTH)
		string[9] = 'x';
	else if (perm & S_ISVTX)
		string[9] = 'T';

	(void) printf("%s", string);
}
