{$R-}
UNIT GroupFile;
INTERFACE
USES GroupItem, GroupType, GroupExceptions, SysUtils,
  Classes, WinTypes, WinProcs;
TYPE
  TGroupFileObj = Class(TObject)
  private
    fChanged : Boolean;
    fWindowName     : String;
    TGH             : TGroupHeader;
    HdrSize         : Word;
    fGrpFileName    : String;
    F               : File;
    FileIsOpen      : Boolean;
    Buff            : ARRAY[0..255] OF Char;
    FUNCTION fwCheckSum : Word;
    FUNCTION CalcCkSum  : Word;
    FUNCTION PCharFmOffset(Offset : Word; P : PChar; MaxLen :
      Word) : PChar;
    PROCEDURE OpenIf;
    PROCEDURE CloseIf;
    FUNCTION GetNthItem(N : Word; VAR TID : TItemData) : Boolean;
  protected
    FUNCTION GetMinPoint : TPoint;
    PROCEDURE SetMinPoint(CONST NuPoint : TPoint);
    FUNCTION GetNormRect : TRect;
    PROCEDURE SetNormRect(CONST NuRect : TRect);
    PROCEDURE SetWindowName(NuName : String);
    FUNCTION GetChanged : Boolean;
  Public
    Items           : TStringList;
    CONSTRUCTOR Create;
    DESTRUCTOR Destroy; Override;
    PROCEDURE LoadFromFile(Name : String);
    PROCEDURE SaveToFile(Name : String);
    FUNCTION StringFmOffset(Offset : Word) : String;
    PROCEDURE FillBuffer(Offset, Size : Word; Dest : Pointer);
    {read-only properties}
    PROPERTY Changed : Boolean Read GetChanged;
    PROPERTY CmdShow : Word Read TGH.nCmdShow;
    PROPERTY cbGroup : Word Read TGH.cbGroup;
    PROPERTY GrpFileName : String Read fGrpFileName;
    {read-write properties}
    PROPERTY MinPoint : TPoint Read GetMinPoint Write SetMinPoint;
    PROPERTY NormRect : TRect Read GetNormRect Write SetNormRect;
    PROPERTY WindowName : String Read fWindowName Write SetWindowName;
  END;

(**) IMPLEMENTATION (**)
CONST
  Tag_First  = $8000;
  Tag_WorkDr = $8101;
  Tag_HotKey = $8102;
  Tag_RunMin = $8103;
  Tag_Last   = $FFFF;

  CONSTRUCTOR TGroupFileObj.Create;
  BEGIN
    Inherited Create;
    Items := TStringList.Create;
  END;

  DESTRUCTOR TGroupFileObj.Destroy;
  BEGIN
    Items.Free;
    Inherited Destroy;
  END;

  PROCEDURE TGroupFileObj.OpenIf;
  BEGIN
    IF NOT FileIsOpen THEN
      BEGIN
        Assign(F, fGrpFileName);
        Reset(F, 1);
      END;
  END;

  PROCEDURE TGroupFileObj.CloseIf;
  BEGIN
    IF NOT FileIsOpen THEN
      {$I-} Close(F); {$I+}
  END;

  PROCEDURE TGroupFileObj.LoadFromFile(Name : String);

    PROCEDURE ReadFileHeader;
    {First read and verify fixed-size portion of header}
    BEGIN
      HdrSize := SizeOf(TGroupHeader) - SizeOf(rgiItemType);
      BlockRead(F, TGH, HdrSize);
      {If the 'PMCC' ID is not present at start of
       header, raise an exception}
      IF StrLComp(TGH.cIdentifier, 'PMCC', 4) <> 0 THEN
        Raise EGrpError.Create(msg_NotGRPFile);
      {If the checksum is not valid, raise an exception}
      IF CalcCkSum <> 0 THEN
        Raise EGrpError.Create(msg_CheckSumBad);
        {Now calculate actual header size and re-read header}
      HdrSize := SizeOf(TGroupHeader) - SizeOf(rgiItemType) +
        2*TGH.cItems;
      Seek(F, 0);
      BlockRead(F, TGH, HdrSize);
      fWindowName := StringFmOffset(TGH.pName);
    END;

    PROCEDURE ReadItems;
    VAR
      N   : Word;
      TID : TItemData;
    {Read the ItemData information into the Items list}
    BEGIN
      Items.Clear;
      FOR N := 0 TO TGH.cItems - 1 DO
        IF GetNthItem(N, TID) THEN
          Items.AddObject(StringFmOffset(TID.pName),
            TItemDataObj.Create(TID, Self));
    END;

    PROCEDURE ReadTags;
    VAR
      WorkDirPos,
      HoldPos, N : Word;
      TTD        : TTagData;
    BEGIN
      Seek(F, cbGroup);
      BlockRead(F, TTD, 6);
        {First tag must have wID=$8000}
      IF TTD.wID <> Tag_First THEN
        Raise EGrpError.Create(msg_FirstTagBad);
      BlockRead(F, TTD.rgbString, TTD.cb-6);
      REPEAT
        {Read fixed-size portion of tag: actual size in cb}
        BlockRead(F, TTD, 6);
        IF TTD.wID <> Tag_Last THEN
          BEGIN
            {read remainder of tag data}
            WorkDirPos := FilePos(F);
            BlockRead(F, TTD.rgbString, TTD.cb-6);
            CASE TTD.wID OF
              Tag_WorkDr : BEGIN
                HoldPos := FilePos(F);
                WITH Items.Objects[TTD.wItem] AS TItemDataObj DO
                  WorkDir := StringFmOffset(WorkDirPos);
                Seek(F, HoldPos);
              END;
              Tag_HotKey : BEGIN
              WITH Items.Objects[TTD.wItem] AS TItemDataObj DO
                HotKey := TTD.rgbShortCut;
              END;
              Tag_RunMin : BEGIN
                WITH Items.Objects[TTD.wItem] AS TItemDataObj DO
                  Minimized := TRUE;
              END;
              ELSE
                Raise EGrpError.Create(msg_TagBad);
            END;
          END;
      UNTIL TTD.wID = Tag_Last;
      FOR N := 0 TO Items.Count-1 DO
        WITH Items.Objects[N] AS TItemDataObj DO
          Changed := FALSE;
    END;

  BEGIN
    fGrpFileName := Name;
    try
      Assign(F, Name);
      Reset(F, 1);
      FileIsOpen := TRUE;
      ReadFileHeader;
      ReadItems;
      {If the cbGroup field is less than the file size,
       the remainder of the file consists of Windows
       3.1 "tag" items}
      IF cbGroup <> FileSize(F) THEN ReadTags;
    finally
      FileIsOpen := FALSE;
      fChanged := FALSE;
      {$I-} Close(F); {$I+}
    END;
  END;

  PROCEDURE TGroupFileObj.SaveToFile(Name : String);
  {does NOT change fGrpFileName to new name}
  CONST Zero : Byte = 0;
  VAR
    N : Word;

    PROCEDURE WriteAString(vS : String; VAR Loc : Word);
    {Write a string (possibly a property) as a PChar
     to the file F; record the offset of that string
     within the file to the variable Loc}
    VAR S : String;
    BEGIN
      Loc := FilePos(F);
      S := vS;
      BlockWrite(F, S[1], Length(S));
      BlockWrite(F, Zero, 1);
    END;

    PROCEDURE WriteFileHeaderFirst;
    BEGIN
      hdrSize := (SizeOf(TGroupHeader) - SizeOf(TGH.rgiItems)) +
        2*Items.Count;
      {update header}
      WITH TGH DO
        BEGIN
          {set checksum to 0 for now}
          wCheckSum := 0;
          {use ACTUAL number of items}
          cItems := Items.Count;
        END;
      {Write  header}
      BlockWrite(F, TGH, hdrSize);
      {Write Window name}
      WriteAString(fWindowName, TGH.pName);
    END;

    PROCEDURE WriteItemData(I : Word);
    VAR
      TID       : TItemData;
      HoldPos   : Word;
      LocalTIRH : TIconResourceHeader;
    BEGIN
      {update offset in file header}
      TGH.rgiItems[I] := FilePos(F);
      WITH Items.Objects[I] AS TItemDataObj DO
        BEGIN
          Changed := False;
          TID := ActualTID;
          BlockWrite(F, TID, SizeOf(TID));
          WriteAString(ItemName, TID.pName);
          WriteAString(Command, TID.pCommand);
          WriteAString(IconPath, TID.pIconPath);
          TID.pHeader := FilePos(F);
          LocalTIRH := TIRH;
          BlockWrite(F, LocalTIRH, SizeOf(LocalTIRH));
          TID.pANDPlane := FilePos(F);
          BlockWrite(F, AndPlane^, TID.cbAndPlane);
          TID.pXORPlane := FilePos(F);
          BlockWrite(F, XorPlane^, TID.cbXORPlane);
          HoldPos := FilePos(F);
          Seek(F, TGH.rgiItems[I]);
          BlockWrite(F, TID, SizeOf(TID));
          Seek(F, HoldPos);
        END;
    END;

    PROCEDURE WriteFirstTag;
    VAR TTD : TTagData;
    BEGIN
      {first tag}
      WITH TTD DO
        BEGIN
          wID := $8000;
          wItem := $FFFF;
          cb := 10;
          StrPCopy(rgbString, 'PMCC');
        END;
      BlockWrite(F, TTD, 10);
    END;

    PROCEDURE WriteTagData(I : Word);
    VAR TTD : TTagData;
    BEGIN
      WITH Items.Objects[I] AS TItemDataObj, TTD DO
        BEGIN
          wItem := I;
          IF WorkDir <> '' THEN {workdir tag}
            BEGIN
              wID := Tag_WorkDr;
              StrPCopy(rgbString, WorkDir);
              cb := 7 + StrLen(rgbString);
              BlockWrite(F, TTD, cb);
            END;
          IF HotKey <> 0 THEN {hotkey tag}
             BEGIN
               wID := Tag_HotKey;
               cb  := 8;
               rgbShortcut := HotKey;
               BlockWrite(F, TTD, 8);
             END;
          IF Minimized THEN {minimize tag}
            BEGIN
              wID := Tag_RunMin;
              cb  := 6;
              BlockWrite(F, TTD, 6);
            END;
        END;
    END;

    PROCEDURE WriteLastTag;
    VAR TTD : TTagData;
    BEGIN
      WITH TTD DO
        BEGIN
          wID := $FFFF;
          wItem := $FFFF;
          cb := 0;
          BlockWrite(F, TTD, 6);
        END;
    END;

    PROCEDURE WriteUpdatedHeader;
    {Various fields in the header have been updated
     during the writing process; the window name location,
     the location of each itemData element, and cbGroup.
     Now we rewrite the updated header and fix the checksum}
    BEGIN
      Seek(F, 0);
      BlockWrite(F, TGH, hdrSize);
      TGH.wCheckSum := (65536 - CalcCkSum) AND $FFFF;
      Seek(F, 0);
      BlockWrite(F, TGH, hdrSize);
    END;

  BEGIN
    Try
      IF FileIsOpen THEN {$I-} Close(F); {$I+}
      (*fGrpFileName := Name;*)
      Assign(F, Name);
      Rewrite(F, 1);
      FileIsOpen := True;
      WriteFileHeaderFirst;
      {Write item data}
      FOR N := Items.Count-1 DOWNTO 0 DO
        WriteItemData(N);
      TGH.cbGroup := FilePos(F);
      WriteFirstTag;
      FOR N := 0 TO Items.Count-1 DO
        WriteTagData(N);
      {If there were NO tags, chop off the FIRST tag}
      IF FilePos(F) = TGH.cbGroup + 10 THEN
        BEGIN
          Seek(F, TGH.cbGroup);
          Truncate(F);
        END
      ELSE
        WriteLastTag;
      WriteUpdatedHeader;
    Finally
      FileIsOpen := False;
      fChanged := False;
      {$I-} Close(F); {$I+}
    END;
  END;

  FUNCTION TGroupFileObj.CalcCkSum : Word;
  {If the value of TGH.wCheckSum is correct, this function
   returns 0. Otherwise, subtract the result from
   TGH.wCheckSum to get the correct value}
  VAR
    FB    : PWordArray;
    N, FS : Word;
  BEGIN
    Try
      OpenIf;
      FS := FileSize(F);
      Try
        GetMem(FB, FS);
        Seek(F, 0);
        BlockRead(F, FB^, FS);
        Result  := 0;
        FOR N := 0 TO pred(FS DIV 2) DO Inc(Result, FB^[N]);
      Finally
        FreeMem(FB, FS);
      END;
    Finally
      CloseIf;
    END;
  END;

  FUNCTION TGroupFileObj.fwCheckSum : Word;
  BEGIN
    fwCheckSum := TGH.wCheckSum;
  END;

  FUNCTION TGroupFileObj.GetMinPoint : TPoint;
  BEGIN
    GetMinPoint := TGH.ptMin;
  END;

  PROCEDURE TGroupFileObj.SetMinPoint(CONST NuPoint : TPoint);
  BEGIN
    IF (TGH.ptMin.X = NuPoint.X) AND
       (TGH.ptMin.Y = NuPoint.Y) THEN Exit;
    TGH.ptMin := NuPoint;
    fChanged := TRUE;
  END;

  FUNCTION TGroupFileObj.GetNormRect : TRect;
  BEGIN
    GetNormRect := TGH.rcNormal;
  END;

  PROCEDURE TGroupFileObj.SetNormRect(CONST NuRect : TRect);
  BEGIN
    IF EqualRect(TGH.rcNormal, NuRect) THEN Exit;
    TGH.rcNormal := NuRect;
    fChanged := TRUE;
  END;

  FUNCTION TGroupFileObj.PCharFmOffset(Offset : Word; P : PChar;
    MaxLen : Word) : PChar;
      {Reads MaxLen bytes from the file F at the specified offset
       into the PChar P; returns P}
  VAR Actual : Word;
  BEGIN
    Try
      OpenIf;
      Seek(F, Offset);
      BlockRead(F, P^, MaxLen, Actual);
      PCharFmoffset := P
    Finally
      CloseIf;
    END;
  END;

  FUNCTION TGroupFileObj.StringFmOffset(Offset : Word) : String;
  BEGIN
    StringFmOffset := StrPas(
      PCharFmOffset(Offset, Buff, 255)
    );
  END;

  FUNCTION TGroupFileObj.GetNthItem(N : Word; VAR TID : TItemData) :
    Boolean;
    {Valid for N from 0 to TGH.cItems-1.  If Nth item exists,
     reads it into TID and returns TRUE; else FALSE.}
  BEGIN
    IF TGH.rgiItems[N] <> 0 THEN
      BEGIN
        GetNthItem := TRUE;
        Seek(F, TGH.rgiItems[N]);
        BlockRead(F, TID, SizeOf(TID));
      END
    ELSE GetNthItem := FALSE;
  END;

  PROCEDURE TGroupFileObj.FillBuffer(Offset, Size : Word; Dest : Pointer);
  BEGIN
    Try
      OpenIf;
      Seek(F, Offset);
      BlockRead(F, Dest^, Size);
    Finally
      CloseIf;
    END;
  END;

  PROCEDURE TGroupFileObj.SetWindowName(NuName : String);
  BEGIN
    IF NuName = fWindowName THEN Exit;
    fWindowName := NuName;
    fChanged := True;
  END;

  FUNCTION TGroupFileObj.GetChanged : Boolean;
  VAR N : Word;
  BEGIN
    Result := FALSE;
    IF Items.Count = 0 THEN Exit;
    IF fChanged THEN Result := TRUE
    ELSE
      FOR N := 0 TO Items.Count-1 DO
        WITH Items.Objects[N] AS TItemDataObj DO
          IF Changed THEN
            BEGIN
              Result := TRUE;
              Break;
            END;
  END;

END.
