// TWERP - EZTWAIN.DLL demo application
//
// Created	05/11/94	spike
// Revised	06/23/94	spike - rev 1.0 alpha 1
// Revised	07/17/94	spike - rev 1.0 alpha 2 - removed use of Bitmap
// Revised	02/06/95	spike - rev 1.1, added Save As command
// Revised	05/03/95	spike - rev 1.2, added Options menu with PixelTypes

//---------- Includes

#include "eztwain.h"
#include "resource.h"
#include "twerprc.h"

//---------- Global variables

static HANDLE   	hInst;				// current instance
static HWND			hwndMain;			// main window handle

static HANDLE		hdib;				// current DIB, if any
static HPALETTE 	hpal;				// logical palette for same, if any

static unsigned		wPixTypes;			// enabled pixel types
static int			fHideUI;			// hide source U/I
static HCURSOR		hcurWait;			// wait cursor (hourglass)

static char			szAppName[32];
static char			szMessage[128];


//---------- Forward function declarations

BOOL InitInstance(int CmdShow);
long FAR PASCAL MainWndProc(HWND, WORD, WORD, LONG);
BOOL FAR PASCAL AboutDlgProc(HWND, WORD, WORD, LONG);
void ResizeWindow(HWND hwnd, HANDLE hdib);
VOID DiscardImage(VOID);                       
HPALETTE CreateBIPalette (HANDLE hdib);
LPSTR LPBits(LPBITMAPINFOHEADER lpdib);
WORD DibNumColors(VOID FAR *pv);

//---------- Public functions

int PASCAL WinMain (HANDLE hInstance, HANDLE hPrevInstance,
            LPSTR  lpszCmdLine, int CmdShow)
{
    WNDCLASS	wc;
    MSG			msg;

    lpszCmdLine=lpszCmdLine;			// suppress no-use warning

    // Save the instance handle in static variable, which will be used in
    // many subsequence calls from this application to Windows.
    hInst = hInstance;

	LoadString(hInst, IDS_APPNAME, szAppName, sizeof szAppName);

	if (hPrevInstance) {
        // Don't allow multiple instances of this program.
        return FALSE;
    }

    // Define & register main window class
    wc.style         = NULL;							// this class has no style!
    wc.lpfnWndProc   = (WNDPROC)MainWndProc;			// name of window proc
    wc.cbClsExtra    = 0;								// no extra bits
    wc.cbWndExtra    = 0;
    wc.hInstance     = hInst;							// handle to cur instance
    wc.hIcon         = LoadIcon( hInst, "TW_APP_ICO");
    wc.hCursor       = LoadCursor(NULL, IDC_ARROW);
    wc.hbrBackground = CreateSolidBrush(RGB(0, 0, 128));	// use a twerpy background
    wc.lpszMenuName  = "TW_App_Menu";					// class default menu
    wc.lpszClassName = "TW_App_MainWnd";				// class named

    if (!RegisterClass(&wc)) {
        return FALSE;
    }

    if (!InitInstance(CmdShow)) {
            return (FALSE);
    }

    // The main loop of the program.  Get a Message; if it's not an
    // accelerator key, translate it and dispatch it to the application.
    while (GetMessage((LPMSG)&msg, NULL, 0, 0)) {
        if (!TWAIN_MessageHook ((LPMSG)&msg)) {
        	TranslateMessage ((LPMSG)&msg);
        	DispatchMessage ((LPMSG)&msg);
        }
    }
    return (msg.wParam);
} // WinMain




//---------------------------------------------------------------------------

BOOL InitInstance(int CmdShow)
{
    char		WindowTitle[64];
    HMENU		hmenu;
    UINT		uEnable = MF_BYCOMMAND | (TWAIN_IsAvailable() ? MF_ENABLED : MF_GRAYED);

    hcurWait = LoadCursor(NULL, IDC_WAIT);

    LoadString(hInst, IDS_WINDOWTITLE, WindowTitle,  sizeof(WindowTitle));
    hwndMain = CreateWindow("TW_App_MainWnd", WindowTitle, WS_OVERLAPPEDWINDOW,
             0, 0, 500, 350, NULL, NULL, hInst, NULL);

    if (!hwndMain)
        return (FALSE);

	wPixTypes = TWAIN_ANYTYPE;

	// enable or disable the TWAIN menu items
	hmenu = GetMenu(hwndMain);
    EnableMenuItem(hmenu, TW_APP_ACQUIRE, uEnable);
    EnableMenuItem(hmenu, TW_APP_ACQUIRE_TO_CLPB, uEnable);
    EnableMenuItem(hmenu, TW_APP_SELECT_SOURCE, uEnable);
    ShowWindow (hwndMain, CmdShow);
    UpdateWindow (hwndMain);
    return (TRUE);
} // InitInstance



//---------------------------------------------------------------------------

long FAR PASCAL  MainWndProc (HWND hwnd,WORD wMsg,WORD wP, LONG lP)
{
    HDC         	hDC;

	switch (wMsg) {

//-----------------------------------------------------------------
	case WM_INITMENU:
	{
		HMENU hmenu = (HMENU) wP;
		EnableMenuItem(hmenu, TW_APP_SAVEAS, MF_BYCOMMAND | (hdib ? MF_ENABLED : MF_GRAYED));
		// update the pixel type items
		CheckMenuItem(hmenu, TW_APP_BW, MF_BYCOMMAND | ((wPixTypes & TWAIN_BW) ? MF_CHECKED : MF_UNCHECKED));
		CheckMenuItem(hmenu, TW_APP_GRAYSCALE, MF_BYCOMMAND | ((wPixTypes & TWAIN_GRAY) ? MF_CHECKED : MF_UNCHECKED));
		CheckMenuItem(hmenu, TW_APP_RGB, MF_BYCOMMAND | ((wPixTypes & TWAIN_RGB) ? MF_CHECKED : MF_UNCHECKED));
		CheckMenuItem(hmenu, TW_APP_PALETTE, MF_BYCOMMAND | ((wPixTypes & TWAIN_PALETTE) ? MF_CHECKED : MF_UNCHECKED));
		CheckMenuItem(hmenu, TW_APP_ANYPIX, MF_BYCOMMAND | ((wPixTypes == TWAIN_ANYTYPE) ? MF_CHECKED : MF_UNCHECKED));
		CheckMenuItem(hmenu, TW_APP_SHOWUI, MF_BYCOMMAND | (fHideUI ? MF_UNCHECKED : MF_CHECKED));
		CheckMenuItem(hmenu, TW_APP_HIDEUI, MF_BYCOMMAND | (fHideUI ? MF_CHECKED : MF_UNCHECKED));
		return DefWindowProc (hwnd, wMsg, wP, lP);
	}
	
//-----------------------------------------------------------------
	case WM_DESTROY:
    	DiscardImage();
    	PostQuitMessage(0);
		return DefWindowProc (hwnd, wMsg, wP, lP);

//-----------------------------------------------------------------
	case WM_PALETTECHANGED:
		if ((HWND) wP == hwnd)
			return 0;				// this app changed palette: ignore

	// Otherwise, fall through to WM_QUERYNEWPALETTE.

	case WM_QUERYNEWPALETTE: {
		HPALETTE hpalT;
		UINT i;

		if (!hpal)
			return 0;				// no palette in effect

	    hDC = GetDC(hwnd);
    	hpalT = SelectPalette (hDC, hpal, FALSE);

		i = RealizePalette(hDC); // i == #entries that changed

	    SelectPalette (hDC, hpalT, FALSE);
    	ReleaseDC(hwnd, hDC);


    	// If any palette entries changed, repaint the window.

		if (i > 0) {
        	InvalidateRect(hwnd, NULL, TRUE);
        }

    	return i;
    } // WM_QUERYNEWPALETTE, WM_PALETTECHANGED

//-----------------------------------------------------------------
	case WM_PAINT: {
    	PAINTSTRUCT 	ps;
		hDC = BeginPaint(hwnd, &ps);
		if (hDC) {
			if (hpal) {
				SelectPalette (hDC, hpal, FALSE);
				RealizePalette (hDC);
			}   /* if */
			if (hdib) {
				LPBITMAPINFOHEADER lpbmi = (LPBITMAPINFOHEADER)GlobalLock(hdib);
				if (lpbmi) {
					HCURSOR hcurSave = SetCursor(hcurWait);
					int w = (int)lpbmi->biWidth;
					int h = (int)lpbmi->biHeight;
					StretchDIBits(hDC,
							  	0, 0,			// x,y destination
							  	w, h,			// pixel width & height in destination
							  	0, 0,			// x,y source
							  	w, h,			// width & height in source
							  	LPBits(lpbmi), // pointer to 'bits' - the pixels
							  	(LPBITMAPINFO)lpbmi,
							  	DIB_RGB_COLORS,	// DIB palette is (presumably?) RGB entries
							  	SRCCOPY);		// raster operation (copy)
					SetCursor(hcurSave);
					GlobalUnlock(hdib);
				}
			}
		}
        EndPaint(hwnd, &ps);
        break;
	} // WM_PAINT


//-----------------------------------------------------------------
	 case WM_COMMAND:

        switch (wP) {

			case TW_APP_ACQUIRE: {
                // free up current image and palette if any
				DiscardImage();
				// clear the window during/before Acquire dialog
				InvalidateRect(hwnd, NULL, TRUE);
				TWAIN_SetHideUI(fHideUI);
				hdib = TWAIN_AcquireNative(hwnd, wPixTypes);
				if (hdib) {
					// compute or guess a palette to use for display
				    hpal = CreateBIPalette (hdib);
				    // size the window to just contain the image
					ResizeWindow(hwnd, hdib);        
					// force repaint of window
					InvalidateRect(hwnd, NULL, TRUE);
				}
				break;
			} // case TW_APP_ACQUIRE

			case TW_APP_ACQUIRE_TO_CLPB: {
				if (!TWAIN_AcquireToClipboard(hwnd, TWAIN_ANYTYPE)) {
					LoadString(hInst, IDS_ERR_NOIMAGE, szMessage, sizeof szMessage);
					MessageBox(hwnd, szMessage, szAppName, MB_ICONINFORMATION | MB_OK);
				}
				break;
			}

			case TW_APP_SELECT_SOURCE:

				TWAIN_SelectImageSource(hwnd);
				break;


			case TW_APP_SAVEAS:
			{
				int result = TWAIN_WriteNativeToFilename(hdib, NULL);
				//	-1	user cancelled File Save dialog
				//	-2	could not create or open file for writing
				//	-3	(weird) unable to access DIB
				//	-4	writing to .BMP failed, maybe output device is full?
				if (result < -1) {
					LoadString(hInst, IDS_ERR_WRITE, szMessage, sizeof szMessage);
					MessageBox(hwnd, szMessage, szAppName, MB_ICONINFORMATION | MB_OK);
				}
				break;
			}

			case TW_APP_QUIT:
				DestroyWindow(hwnd);
				break;

//-----------------------------------------------------------------
            case TW_APP_ABOUT:
		    {
                // bring up the about dialog box
                FARPROC lpProcAbout = MakeProcInstance((FARPROC)AboutDlgProc, hInst);
	            DialogBox (hInst, "TW_APP_ABOUTBOX", hwnd, lpProcAbout);
	            FreeProcInstance(lpProcAbout);
	            break;
	        }

//-----------------------------------------------------------------
            case TW_APP_BW:
            	wPixTypes ^= TWAIN_BW;
	            break;
	        case TW_APP_GRAYSCALE:
	        	wPixTypes ^= TWAIN_GRAY;
	        	break;
	        case TW_APP_RGB:
	        	wPixTypes ^= TWAIN_RGB;
	        	break;
	        case TW_APP_PALETTE:
	        	wPixTypes ^= TWAIN_PALETTE;
	        	break;
	        case TW_APP_ANYPIX:
	        	wPixTypes = TWAIN_ANYTYPE;
	        	break;

//-----------------------------------------------------------------

			case TW_APP_SHOWUI:
			case TW_APP_HIDEUI:
				fHideUI = (wP == TW_APP_HIDEUI);
				break;

//-----------------------------------------------------------------
		  default:
 	         break;
        }	/* switch */
        break;

	default:
		return DefWindowProc (hwnd, wMsg, wP, lP);
	}	/* switch */
	return 0L ;
}	// MainWndProc



//---------------------------------------------------------------------------

void ResizeWindow(HWND hwnd, HANDLE hdib)
{
	HDC	hDC = GetDC(hwnd);
	if (hDC) {
		LPBITMAPINFOHEADER lpbmi = (LPBITMAPINFOHEADER)GlobalLock(hdib);
		if (lpbmi) {
		    char 		Buffer [80];
    		int 		Min, Width, TextLength;
            RECT		Rectangle;
    		Rectangle.left = 0;
    		Rectangle.top  = 0;
    		Rectangle.right  = (int)lpbmi->biWidth;
    		Rectangle.bottom = (int)lpbmi->biHeight;

			// AdjustWindowRect fails if you have a multiline menu.
		    AdjustWindowRect (&Rectangle,  WS_OVERLAPPEDWINDOW, TRUE);

			// set min window width to that of the caption
    		TextLength = GetWindowTextLength (hwnd);
    		GetWindowText (hwnd, Buffer, TextLength);
    		Min = LOWORD(GetTextExtent(hDC, (LPCSTR)Buffer, TextLength));
    		Min += GetSystemMetrics(SM_CXMIN);

    		if ((Width = Rectangle.right - Rectangle.left) < Min) {
				Width = Min;
    		}

    		SetWindowPos (hwnd, (HWND)NULL, 0, 0,
            		      Width,
						  Rectangle.bottom - Rectangle.top,
						  SWP_NOMOVE | SWP_NOZORDER);

			GlobalUnlock(hdib);
		}

    	ReleaseDC(hwnd, hDC);
    }
} // end ResizeWindow


//---------------------------------------------------------------------------

BOOL FAR PASCAL AboutDlgProc(HWND hDlg, WORD message, WORD wP, LONG lParam)
// Call-back function (dialog procedure) for 'About' box.
// On OK or Cancel, closes the dialog.
{
	int nVer = TWAIN_EasyVersion();
	char szBuff[100];

	lParam=lParam;								// suppress no-use warning
	switch (message){
		case WM_INITDIALOG:						// message: initialize dialog box
			LoadString(hInst, IDS_EZT_VER, szMessage, sizeof szMessage);
			wsprintf(szBuff, szMessage, nVer/100, nVer % 100);
			SetDlgItemText(hDlg, IDEZT_VER, szBuff);    
			LoadString(hInst, (TWAIN_IsAvailable() ? IDS_TWAIN_PRESENT : IDS_NO_TWAIN),
			           szBuff, sizeof szBuff);
			SetDlgItemText(hDlg, IDTWAIN_AVAIL, szBuff); 
			return (TRUE);

		case WM_COMMAND:						// message: received a command
			if ((wP == IDOK) || (wP == IDCANCEL)) {
												// OK button or System menu close
				EndDialog(hDlg, TRUE);			// Exits the dialog box
				return (TRUE);
			}
			break;
		default:
			break;
	}

    return (FALSE);            // Didn't process a message
} // AboutDlgProc


//---------------------------------------------------------------------------

VOID DiscardImage (VOID)
// delete/free global palette, and dib, as necessary.
{

	if (hpal) {
		DeleteObject(hpal);
		hpal = NULL;
	}
	if (hdib) {
		TWAIN_FreeNative(hdib);
		hdib = NULL;
	}
} // DiscardImage



//---------------------------------------------------------------------------

LPSTR LPBits(LPBITMAPINFOHEADER lpdib)
// Given a pointer to a locked DIB, return a pointer to the actual bits (pixels)
{
    DWORD dwColorTableSize = (DWORD)(DibNumColors(lpdib) * sizeof(RGBQUAD));
    LPSTR lpBits = (LPSTR)lpdib + lpdib->biSize + dwColorTableSize;

    return lpBits;
} // end LPBits



//---------------------------------------------------------------------------

WORD DibNumColors (VOID FAR *pv)
// given a pointer to a locked DIB, return the number of palette entries: 0,2,16, or 256
{
    int                   Bits;
    LPBITMAPINFOHEADER    lpbi;
    LPBITMAPCOREHEADER    lpbc;

    lpbi = ((LPBITMAPINFOHEADER)pv);
    lpbc = ((LPBITMAPCOREHEADER)pv);

    /*    With the BITMAPINFO format headers, the size of the palette
     *    is in biClrUsed, whereas in the BITMAPCORE - style headers, it
     *    is dependent on the bits per pixel ( = 2 raised to the power of
     *    bits/pixel).
     */
    if (lpbi->biSize != sizeof(BITMAPCOREHEADER)){
        if (lpbi->biClrUsed != 0)
            return (WORD)lpbi->biClrUsed;
        Bits = lpbi->biBitCount;
    } else {
		Bits = lpbc->bcBitCount;
	}
    return (1 << Bits) & 0x01ff;	// up to 8 bits, 2 ^ Bits - otherwise, 0.
}



//---------------------------------------------------------------------------

HPALETTE CreateBIPalette (HANDLE hdib)
// given a pointer to a locked DIB, returns a handle to a plausible logical
// palette to be used for rendering the DIB. For 24-bit dibs, returns NULL.
{
	LPBITMAPINFOHEADER	lpbmi = (LPBITMAPINFOHEADER)GlobalLock(hdib);
    LOGPALETTE          *pPal;
    HPALETTE            hPalette = NULL;
    WORD                nColors, nEntries;
    WORD                i;

    if (!lpbmi)
        return NULL;

    nColors = DibNumColors(lpbmi);				// size of DIB palette
    nEntries = (nColors ? nColors : 256);		// size of palette to create
    // if nColors == 0, it means the DIB is 24 bit (normally) and so has
    // no explicit palette - we will have to make one up.

	// allocate a logical palette structure
	pPal = (LOGPALETTE*)LocalAlloc(LPTR,sizeof(LOGPALETTE) + nEntries *
                          sizeof(PALETTEENTRY));
	if (!pPal) {
		GlobalUnlock(hdib);					// alloc failed
		return (HPALETTE)NULL;
	}

	pPal->palNumEntries = nEntries;
	pPal->palVersion    = 0x300;			// Windows 3.0 or later

	if (nColors) {
		// Fill in the palette entries from the DIB color table
    	// Get a pointer to the color table
		RGBQUAD FAR *pRgb = (RGBQUAD FAR *)((LPSTR)lpbmi + (WORD)lpbmi->biSize);

		// copy from DIB palette (triples, by the way) into the LOGPALETTE
		for (i = 0; i < nEntries; i++) {
			pPal->palPalEntry[i].peRed   = pRgb[i].rgbRed;
			pPal->palPalEntry[i].peGreen = pRgb[i].rgbGreen;
			pPal->palPalEntry[i].peBlue  = pRgb[i].rgbBlue;
			pPal->palPalEntry[i].peFlags = (BYTE)0;
		} // for

	} else {
		// nColors = 0, must be a full-color DIB: no explicit palette.
		// Synthesize a 256-color palette
    	BYTE Red, Green, Blue;
		Red = Green = Blue = 0;

		// Generate 256 (= 8*8*4) RGB combinations to fill the palette
		// Note - this is pointless if in a >256-color vid mode, 
		// and is also not very effective in 16-color mode.
		// And, in 256-color modes, Windoze does a truly rotten job of
		// mapping from 24 bits to 8.
		for (i = 0; i < pPal->palNumEntries; i++) {
			pPal->palPalEntry[i].peRed   = Red;
			pPal->palPalEntry[i].peGreen = Green;
			pPal->palPalEntry[i].peBlue  = Blue;
			pPal->palPalEntry[i].peFlags = (BYTE)0;

			if (!(Red += 32))
				if (!(Green += 32))
					Blue += 64;
		} // for

	}
	// create a logical palette
	hPalette = CreatePalette(pPal);
	LocalFree((HANDLE)pPal);

	GlobalUnlock(hdib);
    return hPalette;
}



