unit DynaForm;

(*******************************************************************************
DynaForm Form Template for dynamic forms.
 1995 Objective Software Technology Pty Ltd.
Publishers of ABC for Delphi and other fine Delphi tools.
Email 100035.2441@compuserve.com
Fax +61 6 2732100
Phone +61 6 2732100
********************************************************************************

Purpose:

This template allows AUTOMATIC configuration of user changes to forms,
without needing any additional code to save/restore changed properties.

It automatically saves/restores all published form properties and owned
components at runtime using a resource file external to the exe program.
So for example, all published properties such as size, position, color, fonts etc
are automatically saved.  In addition, any additional components created at runtime
are also saved and then restored.

How to Use:

1. Open this form file.

2. Save it as a template using Options|Gallery|{form templates tab}|{Add Button}

3. When creating a new form using the template, edit the constant named
   DefaultResFile that is defined at the start of the implementation section.
   A unique resource file must be provided for each form class.

4. The resource file(s) will then be automatically restored/saved each time the
   form(s) are created/destroyed at runtime.

5. No dialogs are currently provided for customising form properties.
   You will need to design these to your own requirements.

Public Properties Defined in This Template:

    ResFile: TFileName
      Set in the OnCreate event to specify the name of the file to restore/save properties.
    NoRestore: Boolean
      Set to true in the OnCreate event to disable restore action.
    NoSave: Boolean
      Set to true to disable save action.
    OnRestore: TNotifyEvent
      This event occurs immediately after the form has been restored from the resource file.
      Set in the OnCreate event to specify the event procedure to execute.
    OnSave: TNotifyEvent
      This event occurs immediately before the form is saved.

Caveats/Restrictions:

1. This file is distributed as limited freeware.
   If you use it, please retain the copyright message in your code.
   Please notify us of any suggestions or changes that may be of use to other
   developers so we can incorporate them in a future distribution.

2. If you want to use multiple instances of a form class,
   use code in the OnCreate event to assign a unique name to the resource file
   for each instance, or disable the form saving with the NoSave property.

3. If the form's OnCreate event changes any published properties of the form,
   or if it modifies any components owned by the form, then it should be executed
   again as the OnRestore event.  To do this, assign the OnChange event to OnRestore
   within the OnCreate procedure, i.e. OnRestore := OnCreate(self);

4. Data in these resource files is streamed using the same mechanism Delphi
   uses to read/write DFM files.  The files are written in binary format and
   cannot be edited as text.

5. When designing and debugging a form in the Delphi IDE, it is necessary to
   disable the save function or delete the resource file each time the form is
   edited.  Otherwise design changes appear to be lost at runtime, when the old
   resource is restored.

6. Avoid using the file suffix *.DFM for resource files otherwise your
   delphi project could be inadvertantly overwritten.
*******************************************************************************)

interface

uses
  SysUtils, WinTypes, WinProcs, Messages, Classes, Graphics, Controls,
  Forms, Dialogs;

type
  TDynaForm = class(TForm)
  private
    { Private declarations }
    FResFile: TFileName;
    FNoRestore: Boolean;
    FNoSave: Boolean;
    FOnRestore: TNotifyEvent;
    FOnSave: TNotifyEvent;
    procedure SaveToFile(AResFile: TFileName);
    procedure RestoreFromFile(AResFile: TFileName);
    function GetDefaultResFile: TFileName;
  public
    constructor Create(AOwner: TComponent); override;
    destructor Destroy; override;
    { The following public properties can be set in code to change the filing
      behaviour - for example in OnCreate}
    property ResFile: TFileName read FResFile write FResFile;           {also defaults to the constant declared below}
    property NoRestore: Boolean read FNoRestore write FNoRestore;       {must be set in OnCreate event}
    property NoSave: Boolean read FNoSave write FNoSave;
    property OnRestore: TNotifyEvent read FOnRestore write FOnRestore;  {must be set in OnCreate event}
    property OnSave: TNotifyEvent read FOnSave write FOnSave;
  end;

var
  DynaForm: TDynaForm;

implementation

{$R *.DFM}

const
{
 Edit this constant to set a static file name for instances of this form.
 e.g. UNIT1.DFR
 If no name is specified at runtime, the template will
 Create/Read/Write a file of this name in the exe directory
 }
  DefaultResFile = '';

constructor TDynaForm.Create(AOwner: TComponent);
{
Create form instance from EXE and then restore saved property values.

First does inherited Form Create and OnCreate event;
Set the NoRestore property to True in OnCreate to disable these functions:
 - rereads the published form and component properties from a Res File;
 - executes the OnRestore event if assigned.
}
begin
  inherited Create(AOwner);
  if not NoRestore then begin
    if ResFile = '' then ResFile := GetDefaultResFile;
    if ResFile <> '' then begin
      RestoreFromFile(ResFile);
      if Assigned(FOnRestore) then FOnRestore(self);
    end;
  end;
end;

destructor TDynaForm.Destroy;
{
Save the published form and component properties to a Res File
before destroying.
Set NoSave to true to disable this function.
}
begin
  if Assigned(FOnSave) then FOnSave(self);
  if not NoSave then begin
    if ResFile = '' then ResFile := GetDefaultResFile;
    if ResFile <> '' then SaveToFile(ResFile);
  end;
  inherited Destroy;
end;

procedure TDynaForm.SaveToFile(AResFile: TFileName);
var
  Stream: TFileStream;
  FormName: string;
begin
  FormName := Copy(ClassName, 2, 99);
  Stream := TFileStream.Create(AResFile, fmCreate);
  try
    Stream.WriteComponentRes(FormName, Self);
  finally
    Stream.Free;
  end;
end;

procedure TDynaForm.RestoreFromFile(AResFile: TFileName);
var
  Stream: TFileStream;
  I: integer;
begin
  try
    Stream := TFileStream.Create(AResFile, fmOpenRead);
    try
      {delete all components}
      for I := ComponentCount - 1 downto 0 do
        Components[I].Free;
      Stream.ReadComponentRes(Self);
    finally
      Stream.Free;
    end;
  except
    on EFOpenError do {nothing};
  end;
end;

function TDynaForm.GetDefaultResFile: TFileName;
begin
  if DefaultResFile = '' then begin
    ShowMessage('DefaultResFile constant not defined');
    Result := '';
  end
  else
    Result := ExtractFilePath(Application.ExeName) + DefaultResFile;
end;

end.
