unit ExecFile;

interface

uses
  Windows, Messages, SysUtils, Classes, Graphics, Controls, Forms, Dialogs, FileCtrl, Utils;

type
  TdsExecFile = class(TComponent)
  private
	 { Private declarations }
		FFileName		: String;								{ Full file name to be executed }
		FFilePath		: String;								{ Path to working directory }
		FWait				: Boolean;								{ Flag to make execution wait for completion }
		FParameters		: String;								{ Parameters to pass on in command line }
		Procedure SetFileName( AFileName : String );		{ Set the file name }
		Procedure SetFilePath( AFilePath : String );		{ Set the working directory }
		Procedure SetWait( AWait : Boolean );				{ Set the wait flag }
		Procedure SetParameters( AParameters : String );{ Set command line parameters to pass on }
  protected
	 { Protected declarations }
		StartUpInfo : TStartupInfo;							{ Process start up information record }
		ProcessInfo	: TProcessInformation;					{ New process information record }
  public
	 { Public declarations }
  published
	 { Published declarations }
		Property FileName : String read FFileName write SetFileName;
		Property FilePath : String read FFilePath write SetFilePath;
		Property Parameters : String read FParameters write SetParameters;
		Property Wait : Boolean read FWait write SetWait;
		Procedure Execute;
  end;

	EdsExecFileGeneralFailure	= Class( Exception );
	EdsExecFileNoFileName		= Class( Exception );
	EdsExecFileFileNoExist		= Class( Exception );
	EdsExecFilePathNoExist		= Class( Exception );

procedure Register;


implementation


procedure Register;
begin
  RegisterComponents('Apollo', [TdsExecFile]);
end;


procedure TdsExecFile.Execute;
Var
	ExeName	: String;
Begin
	{ Check that the path and filename exist }
	if Trim(FFileName) = '' then
		Raise EdsExecFileNoFileName.Create('Filename must be specified');

	ExeName := FFileName;

	{ If the execution file does not exist, then try adding the path, if that fails then you're stuffed }
	if Not( FileExists( ExeName ) ) then
		if Not( FileExists( FFilePath + ExeName ) ) then
			Raise EdsExecFileFileNoExist.Create( FFileName + ' does not exist' )
		else
			ExeName := FFilePath + ExeName;

	{ Check that the specified path exists }
	if Not( DirectoryExists( FFilePath ) ) then
		Raise EdsExecFilePathNoExist.Create( FFilePath + ' does not exist' );

	{ Append the parameters to the executable name }
	ExeName := ExeName + ' ' + FParameters;

	{ Initialise the StartUpInfo record, which handles the creation of a new main window for a process }
	FillChar( StartUpInfo, SizeOf( StartUpInfo ), #0 );
	StartUpInfo.cb := SizeOf( StartUpInfo );
	StartUpInfo.dwFlags := STARTF_USESHOWWINDOW;
	StartUpInfo.wShowWindow := SW_SHOWMAXIMIZED;

	if Not CreateProcess( 	nil,							{ Not used - no parameters passed as yet }
									PChar( ExeName ),			{ Command Line - used to allow 16 bit execution & parameters }
									nil,							{ No Process security attributes }
									nil,							{ No Thread security attributes }
									false,						{ No handle inheritance }
									CREATE_NEW_CONSOLE or NORMAL_PRIORITY_CLASS,	{ Creation flags }
									nil,							{ No new environment block }
									PChar( FFilePath ),		{ No pointer to the current directory }
									StartUpInfo,				{ Pointer to the StartUpInfo block }
									ProcessInfo					{ Pointer to the ProcessInformation block }
								) then
	Begin
		raise EdsExecFileGeneralFailure.Create('CreateProcess failed');
	End;

	If ( FWait ) then
	Begin
		{ Wait for ever, or until the process finishes }
		WaitForSingleObject( ProcessInfo.hProcess, INFINITE );
		{ Get the exit code of the process and create an exception on an error - if I knew the types!
		GetExitCodeProcess( ProcessInfo.hProcess, Result );	}
	End;
End;


Procedure TdsExecFile.SetFileName( AFileName : String );
Begin
	FFileName := AFileName;
End;


Procedure TdsExecFile.SetFilePath( AFilePath : String );
Begin
	{ Ensure the the path always ends with a slash }
	If RightStr( AFilePath, 1 ) <> '\' then
		AFilePath := AFilePath + '\';
	FFilePath := AFilePath;
End;


Procedure TdsExecFile.SetWait( AWait : Boolean );
Begin
	FWait := AWait;
End;


Procedure TdsExecFile.SetParameters( AParameters : String );
Begin
	FParameters := AParameters;
End;


end.
