{===EZDSLSKP==========================================================

Part of the Delphi Structures Library--the skip list.

EZDSLSKP is Copyright (c) 1993, 1995 by  Julian M. Bucknall

VERSION HISTORY
18Jun95 JMB 1.00 conversion of EZStrucs to Delphi
======================================================================}
{ Copyright (c) 1993, 1995, Julian M. Bucknall. All Rights Reserved   }

unit EZDSLSkp;

{Declare the compiler defines}
{$I EZDSLDEF.INC}

{------Changeable compiler switches-----------------------------------}
{$A+   Word align variables }
{$F+   Force Far calls }
{$K+   Use smart callbacks
{$N+   Allow coprocessor instructions }
{$P+   Open parameters enabled }
{$Q+   Integer overflow checking }
{$R+   Range checking }
{$S+   Stack checking }
{$T-   @ operator is NOT typed }
{$U-   Non Pentium safe FDIV }
{$Z-   No automatic word-sized enumerations}
{---------------------------------------------------------------------}

interface

uses
  SysUtils,
  WinTypes,
  WinProcs,
  Classes,
  EZDSLCts,
  EZDSLBse,
  EZDSLSup,
  EZRndStm;

type
  TSkipList = class(TAbstractContainer)
    {-Skip linked list object}
    private
      BF, AL    : PNode;
      CurLevels : integer;
      RandGen   : PRandStream;
      skNewNodeLevel : integer;

    protected
      procedure acDisposeNode(aNode : PNode); override;
      function  acNewNode(aData : pointer) : PNode; override;

    public
      constructor Create(DataOwner : boolean); override;
      constructor Clone(Source : TAbstractContainer;
                        DataOwner : boolean; NewCompare : TCompareFunc); override;
      destructor Destroy; override;

      function  Delete(Cursor : TListCursor) : TListCursor;
      procedure Empty; override;
      function  Erase(Cursor : TListCursor) : TListCursor;
      function  Examine(Cursor : TListCursor) : pointer;
      procedure Insert(var Cursor : TListCursor; aData : pointer);
      function  IsAfterLast(Cursor : TListCursor) : boolean;
      function  IsBeforeFirst(Cursor : TListCursor) : boolean;
      function  Iterate(Action : TIterator; Backwards : boolean;
                        ExtraData : pointer) : pointer;
      procedure Join(List : TSkipList);
      function  Next(Cursor : TListCursor) : TListCursor;
      function  Prev(Cursor : TListCursor) : TListCursor;
      function  Replace(Cursor : TListCursor; aData : pointer) : pointer;
      function  Search(var Cursor : TListCursor; aData : pointer) : boolean;
      function  SetBeforeFirst : TListCursor;
      function  SetAfterLast : TListCursor;
      function  Split(Cursor : TListCursor) : TSkipList;
  end;

implementation

{-An iterator for cloning a skip list}
function SkipListClone(SL : TAbstractContainer;
                       aData : pointer;
                       NSL : pointer) : boolean; far;
  var
    NewList : TSkipList absolute NSL;
    NewData : pointer;
    Dummy   : TListCursor;
  begin
    Result := true;
    with NewList do
      begin
        if IsDataOwner then
             NewData := DupData(aData)
        else NewData := aData;
        try
          Insert(Dummy, NewData);
        except
          DisposeData(NewData);
          raise;
        end;
      end;
  end;

{=TSkipList===========================================================
A skip linked list

This is a special type of linked list of data objects. Compared with
TList and TDList, this implementation uses nodes of varying sizes. The
nodes have between 1 and 16 (skMaxLevels) of forward pointers, the
higher ones skipping over nodes with less forward pointers. This means
much faster search times, but slightly slower list update times (ie
insert and delete). Can cope with searching long lists without too
much degradation. Compared with a red-black binary search tree, this
type of data structure will consume more memory, will have faster
insert times, slower (?) delete times, and will have comparable
(amortised) search times.

Reference
  Scheiner: Skip Lists (DDJ January 1994)
======================================================================}
constructor TSkipList.Create(DataOwner : boolean);
  var
    Level : integer;
  begin
    {Note: we cannot use a NodeStore as the nodes have different
           sizes, so set NodeSize to 0.}
    NodeSize := 0;
    inherited Create(DataOwner);
    RandGen := CreateRandStream(0);
    skNewNodeLevel := skMaxLevels;
    BF := acNewNode(nil);
    FCount := 0;
    skNewNodeLevel := 1;
    AL := acNewNode(nil);
    FCount := 0;
    for Level := 0 to pred(skMaxLevels) do
      BF^.FwLink[Level] := AL;
    BF^.BkLink:= BF;
    AL^.FwLink[0] := AL;
    AL^.BkLink:= BF;
    CurLevels := 1;
  end;
{--------}
constructor TSkipList.Clone(Source : TAbstractContainer;
                            DataOwner : boolean;
                            NewCompare : TCompareFunc);
  var
    OldList : TSkipList absolute Source;
    Dummy   : pointer;
  begin
    Create(DataOwner);
    Compare := NewCompare;
    DupData := OldList.DupData;
    DisposeData := OldList.DisposeData;

    if not (Source is TSkipList) then
      RaiseError(escBadSource);

    if OldList.IsEmpty then Exit;

    SetAfterLast;
    Dummy := OldList.Iterate(SkipListClone, false, Self);
  end;
{--------}
destructor TSkipList.Destroy;
  begin
    if Assigned(RandGen) then
      DestroyRandStream(RandGen);
    inherited Destroy;
  end;
{--------}
function TSkipList.Delete(Cursor : TListCursor) : TListCursor;
  var
    aData     : pointer;
    Walker    : PNode;
    NextStep  : TListCursor;
    TempNode  : PNode;
    Level     : integer;
    CompResult: integer;
    PrevLink  : array [0..pred(skMaxLevels)] of PNode;
  begin
    {$IFDEF DEBUG}
    Assert((not IsBeforeFirst(Cursor)) and (not IsAfterLast(Cursor)), ascDeleteEdges);
    {$ENDIF}
    aData := Examine(Cursor);
    Walker := PNode(SetBeforeFirst);
    for Level := pred(CurLevels) downto 0 do
      begin
        NextStep := TListCursor(Walker^.FwLink[Level]);
        if IsAfterLast(NextStep) then
             CompResult := -1
        else CompResult := Compare(aData, Examine(NextStep));
        while (CompResult > 0) do
          begin
            Walker := PNode(NextStep);
            NextStep := TListCursor(Walker^.FwLink[Level]);
            if IsAfterLast(NextStep) then
                 CompResult := -1
            else CompResult := Compare(aData, Examine(NextStep));
          end;
        PrevLink[Level] := Walker;
      end;
    with PNode(Cursor)^ do
      begin
        TempNode := FwLink[0];
        TempNode^.BkLink := BkLink;
        PrevLink[0]^.FwLink[0] := FwLink[0];
        for Level := 1 to pred(Lvls) do
          PrevLink[Level]^.FwLink[Level] := FwLink[Level];
      end;
    acDisposeNode(PNode(Cursor));
    Delete := TListCursor(TempNode);
  end;
{--------}
procedure TSkipList.Empty;
  var
    Temp,
    Cursor : TListCursor;
    Level  : integer;
  begin
    {Note: it will be faster to delete nodes from first principles
           rather than repeatedly call the Erase method.}
    if not IsEmpty then
      begin
        Cursor := Next(SetBeforeFirst);
        while not IsAfterLast(Cursor) do
          begin
            Temp := Cursor;
            Cursor := Next(Cursor);
            if IsDataOwner then
              DisposeData(Examine(Temp));
            acDisposeNode(PNode(Temp));
          end;
      end;
    if InDone then
      begin
        if Assigned(BF) then
          acDisposeNode(BF);
        if Assigned(AL) then
          acDisposeNode(AL);
      end
    else
      begin
        {patch everything up again}
        for Level := 0 to pred(skMaxLevels) do
          BF^.FwLink[Level] := AL;
        AL^.BkLink:= BF;
        CurLevels := 1;
        FCount := 0;
      end;
  end;
{--------}
function TSkipList.Erase(Cursor : TListCursor) : TListCursor;
  var
    Data : pointer;
  begin
    {Note: Delete requires the Data field so dispose the data
           afterwards}
    Data := Examine(Cursor);
    Erase := Delete(Cursor);
    if IsDataOwner then
      DisposeData(Data);
  end;
{--------}
function  TSkipList.Examine(Cursor : TListCursor) : pointer;
  begin
    {$IFDEF DEBUG}
    Assert((not IsBeforeFirst(Cursor)) and (not IsAfterLast(Cursor)), ascExamineEdges);
    {$ENDIF}
    Examine := PNode(Cursor)^.Data;
  end;
{--------}
procedure TSkipList.Insert(var Cursor : TListCursor; aData : pointer);
  var
    Walker    : PNode;
    NextStep  : TListCursor;
    TempNode  : PNode;
    Level     : integer;
    CompResult: integer;
    PrevLink  : array [0..pred(skMaxLevels)] of PNode;
  begin
    Walker := PNode(SetBeforeFirst);
    {note: the following for loop is executed at least once
           because CurLevels >= 1}
    for Level := pred(CurLevels) downto 0 do
      begin
        NextStep := TListCursor(Walker^.FwLink[Level]);
        if IsAfterLast(NextStep) then
             CompResult := -1
        else CompResult := Compare(aData, Examine(NextStep));
        while (CompResult > 0) do
          begin
            Walker := PNode(NextStep);
            NextStep := TListCursor(Walker^.FwLink[Level]);
            if IsAfterLast(NextStep) then
                 CompResult := -1
            else CompResult := Compare(aData, Examine(NextStep));
          end;
        PrevLink[Level] := Walker;
      end;
    if (CompResult = 0) then
      RaiseError(escInsertDup);
    Level := 1;
    while (Level < skMaxLevels) and (rsRandom(RandGen, 4) = 0) do
      inc(Level);
    if (Level > CurLevels) then
      begin
        PrevLink[CurLevels] := BF;
        inc(CurLevels);
        Level := CurLevels;
      end;
    skNewNodeLevel := Level;
    TempNode := acNewNode(aData);
    for Level := pred(skNewNodeLevel) downto 0 do
      with PrevLink[Level]^ do
        begin
          TempNode^.FwLink[Level] := FwLink[Level];
          FwLink[Level] := TempNode;
        end;
    with TempNode^.FwLink[0]^ do
      begin
        TempNode^.BkLink := BkLink;
        BkLink := TempNode;
      end;
    Cursor := TListCursor(TempNode);
  end;
{--------}
function  TSkipList.IsAfterLast(Cursor : TListCursor) : boolean;
  begin
    IsAfterLast := (PNode(Cursor) = AL);
  end;
{--------}
function  TSkipList.IsBeforeFirst(Cursor : TListCursor) : boolean;
  begin
    IsBeforeFirst := (PNode(Cursor) = BF);
  end;
{--------}
function  TSkipList.Iterate(Action : TIterator; Backwards : boolean;
                            ExtraData : pointer) : pointer;
  var
    Walker : TListCursor;
  begin
    if Backwards then
      begin
        Walker := Prev(SetAfterLast);
        while not IsBeforeFirst(Walker) do
          if Action(@Self, Examine(Walker), ExtraData) then
            Walker := Prev(Walker)
          else
            begin
              Result := Examine(Walker);
              Exit;
            end;
      end
    else
      begin
        Walker := Next(SetBeforeFirst);
        while not IsAfterLast(Walker) do
          if Action(@Self, Examine(Walker), ExtraData) then
            Walker := Next(Walker)
          else
            begin
              Result := Examine(Walker);
              Exit;
            end;
      end;
  end;
{--------}
procedure TSkipList.Join(List : TSkipList);
  var
    Dummy,
    Walker : TListCursor;
    Data   : pointer;
  begin
    {$IFDEF DEBUG}
    Assert(List.IsDataOwner = IsDataOwner, ascCannotJoinData);
    {$ENDIF}
    if not Assigned(List) then Exit;

    if not List.IsEmpty then
      begin
        Walker := List.Next(List.SetBeforeFirst);
        while not List.IsAfterLast(Walker) do
          begin
            Data := List.Examine(Walker);
            Walker := List.Delete(Walker);
            Insert(Dummy, Data);
          end;
      end;
    List.Free;
  end;
{--------}
function  TSkipList.Next(Cursor : TListCursor) : TListCursor;
  begin
    {$IFDEF DEBUG}
    Assert(not IsAfterLast(Cursor), ascAlreadyAtEnd);
    {$ENDIF}
    Next := TListCursor(PNode(Cursor)^.FwLink[0]);
  end;
{--------}
function  TSkipList.Prev(Cursor : TListCursor) : TListCursor;
  begin
    {$IFDEF DEBUG}
    Assert(not IsBeforeFirst(Cursor), ascAlreadyAtStart);
    {$ENDIF}
    Prev := TLIstCursor(PNode(Cursor)^.BkLink);
  end;
{--------}
function  TSkipList.Replace(Cursor : TListCursor; aData : pointer) : pointer;
  begin
    {$IFDEF DEBUG}
    Assert((not IsBeforeFirst(Cursor)) and (not IsAfterLast(Cursor)), ascReplaceEdges);
    {$ENDIF}
    Replace := Examine(Cursor);
    Cursor := Delete(Cursor);
    Insert(Cursor, aData);
  end;
{--------}
function  TSkipList.Search(var Cursor : TListCursor; aData : pointer) : boolean;
  var
    Walker    : PNode;
    NextStep  : TListCursor;
    Level     : integer;
    CompResult: integer;
  begin
    Walker := PNode(SetBeforeFirst);
    for Level := pred(CurLevels) downto 0 do
      begin
        NextStep := TListCursor(Walker^.FwLink[Level]);
        if IsAfterLast(NextStep) then
             CompResult := -1
        else CompResult := Compare(aData, Examine(NextStep));
        while (CompResult > 0) do
          begin
            Walker := PNode(NextStep);
            NextStep := TListCursor(Walker^.FwLink[Level]);
            if IsAfterLast(NextStep) then
                 CompResult := -1
            else CompResult := Compare(aData, Examine(NextStep));
          end;
      end;
    Cursor := NextStep;
    Search := (CompResult = 0);
  end;
{--------}
function  TSkipList.SetBeforeFirst : TListCursor;
  begin
    SetBeforeFirst := TListCursor(BF);
  end;
{--------}
function  TSkipList.SetAfterLast : TListCursor;
  begin
    SetAfterLast := TListCursor(AL);
  end;
{--------}
function  TSkipList.Split(Cursor : TListCursor) : TSkipList;
  var
    NewList   : TSkipList;
    Dummy,
    NextCursor: TListCursor;
    Data      : pointer;
  begin
    {$IFDEF DEBUG}
    Assert((not IsBeforeFirst(Cursor)) and (not IsAfterLast(Cursor)), ascSplitEdges);
    {$ENDIF}
    NewList := TSkipList(TAbstractContainerClass(ClassType).Create(IsDataOwner));
    NewList.Compare := Compare;
    NewList.DupData := DupData;
    NewList.DisposeData := DisposeData;
    Result := NewList;

    while not IsAfterLast(Cursor) do
      begin
        Data := Examine(Cursor);
        Cursor := Delete(Cursor);
        NewList.Insert(Dummy, Data);
      end;
  end;
{--------}
procedure TSkipList.acDisposeNode(aNode : PNode);
  begin
    {$IFDEF DEBUG}
    Assert(Assigned(aNode), ascFreeNilNode);
    {$ENDIF}
    SafeFreeMem(aNode, aNode^.Size);
    dec(FCount);
  end;
{--------}
function  TSkipList.acNewNode(aData : pointer) : PNode;
  var
    Node : PNode;
    NodeBytes : word;
  begin
    {Note: we must override the default node allocation as the nodes
           vary in size. The object variable NewNodeLevel is the
           number of forward links we must reserve.}
    {$IFDEF DEBUG}
    Assert((0 < skNewNodeLevel) and (skNewNodeLevel <= skMaxLevels), ascBadSkipLevel);
    {$ENDIF}
    {$IFDEF MaxIs64KItems}
    if (Count = $FFFF) then
      RaiseError(escTooManyItems);
    {$ENDIF}
    {Note: the formula below translates to this table for 16-bit Delphi
       SkipLevel:  1  2  3  4  5  6  7  8  9 10 11 12 13 14 15 16
        NodeSize: 16 20 24 28 32 36 40 44 48 52 56 60 64 68 72 76 }
    NodeBytes := ((skNewNodeLevel+2) * sizeof(pointer)) + (sizeof(word) * 2);
    SafeGetMem(Result, NodeBytes);
    with Result^ do
      begin
        Data := aData;
        Size := NodeBytes;
        Lvls := skNewNodeLevel;
      end;
    inc(FCount);
  end;
{---------------------------------------------------------------------}

end.

