{ Copyright 1995 Tempest Software.  All Rights Reserved.

  Permission is granted to use this software for private,
  non-commercial purposes only.  This software is released
  as is, without warranty of any kind, either expressed or
  implied.  If you wish to use this software in a product for
  which you will receive remuneration, please contact the author.

  Please email comments to 'comments@tempest-sw.com'.

  Ray Lischner
  Tempest Software
  26 July 1995

  $Log: huge.pas $
  Revision 1.3  1995/07/26 13:24:50  LISCH
  Added FastHugeOffset, which reduces run-time by another 1/3.
  Revision 1.2  1995/07/09 17:58:28  LISCH
  Added HugeDec. Improved HugeInc and HugeOffset.
  Rewrote Insert & Delete to use optimized block moves.
  Revision 1.1  1995/06/07 15:05:54  LISCH
  Initial revision
}

unit Huge;

interface

  uses Classes, WinTypes;

  { Class THugeList is similar to TList, but the indexes are of type LongInt,
    rather than Integer.  The size of the list is limited only by the amount
    of virtual memory that can be allocated from Windows.

    Each item is a Pointer.  THugeList never dereferences this pointer, and
    it can be any value you want it to be.  When items are deleted from the
    list and when the list is destroyed, these items are not freed or
    dereferenced.

    List references are always range checked, raising an exception
    for index violations.

    Every effort has been made to ensure that THugeList follows identical
    semantics as TList, but without reference to the TList sources.  Thus,
    the author may have introduced discrepancies.  Please bring them to the
    attention of the author by sending email to "bugs@tempest-sw.com".
  }
  type
    THugeList = class
    private
      fHandle: THandle;             { Global memory handle for fList }
      fList: Pointer;               { Pointer to the base of the list }
      fCount: LongInt;              { Number of items in the list }
      fCapacity: LongInt;           { Number of available slots in the list }
    protected
      function GetPointer(Index: LongInt): Pointer;
      procedure ReAllocList(NewCapacity: LongInt);
      procedure CheckIndex(Index: LongInt);
      function ExpandSize: LongInt;
      function GetItem(Index: LongInt): Pointer;
      procedure SetItem(Index: LongInt; Item: Pointer);
      procedure SetCapacity(NewCapacity: LongInt);
      property Handle: THandle read fHandle;
    public
      destructor Destroy; override;
      function Add(Item: Pointer): LongInt;
      property Capacity: LongInt read fCapacity write SetCapacity;
      procedure Clear;
      property Count: LongInt read fCount;
      procedure Delete(Index: LongInt);
      procedure Exchange(I1, I2: LongInt);
      procedure Expand;
      function First: Pointer;
      function IndexOf(Item: Pointer): LongInt;
      procedure Insert(Index: LongInt; Item: Pointer);
      function Last: Pointer;
      procedure Move(CurIndex, NewIndex: LongInt);
      procedure Pack;
      function Remove(Item: Pointer): LongInt;

      property Items[Index: LongInt]: Pointer read GetItem write SetItem; default;
    end;

  { HugeInc() and HugeOffset() add an offset to a far pointer.
    The offset can be greater than 64K, and the pointer's segment
    is properly updated.  Both a procedure and a function are
    provided; they do the same thing, but return the result
    differently.  Choose whichever is more convenient.
  }
  procedure HugeInc(var HugePtr: Pointer; Amount: LongInt);
  procedure HugeDec(var HugePtr: Pointer; Amount: LongInt);
  function  HugeOffset(HugePtr: Pointer; Amount: LongInt): Pointer;

  procedure Register;

implementation

  uses SysUtils, WinProcs;

  {$R HugeRes.Res}
  {$I HugeRes.Inc}

  procedure __AHSHIFT; far; external 'KERNEL' index 113;

  procedure HugeInc(var HugePtr: Pointer; Amount: LongInt); assembler;
  asm
    mov ax, Amount.Word[0]  { Store Amount in DX:AX }
    mov dx, Amount.Word[2]
    les bx, HugePtr         { Get the reference to HugePtr }
    add ax, es:[bx]         { Add the offset parts }
    adc dx, 0               { Propagate carry to the high word of Amount }
    mov cx, OFFSET __AHSHIFT
    shl dx, cl              { Shift high word of Amount for segment }
    add es:[bx+2], dx       { Increment the segment of HugePtr }
    mov es:[bx], ax
  end;

  procedure HugeDec(var HugePtr: Pointer; Amount: LongInt); assembler;
  asm
    les bx, HugePtr         { Store HugePtr ptr in es:[bx] }
    mov ax, es:[bx]
    sub ax, Amount.Word[0]  { Subtract the offset parts }
    mov dx, Amount.Word[2]
    adc dx, 0               { Propagate carry to the high word of Amount }
    mov cx, OFFSET __AHSHIFT
    shl dx, cl              { Shift high word of Amount for segment }
    sub es:[bx+2], dx
    mov es:[bx], ax
  end;

  function HugeOffset(HugePtr: Pointer; Amount: LongInt): Pointer; assembler;
  asm
    mov ax, Amount.Word[0]  { Store Amount in DX:AX }
    mov dx, Amount.Word[2]
    add ax, HugePtr.Word[0] { Add the offset parts }
    adc dx, 0               { Propagate carry to the high word of Amount }
    mov cx, OFFSET __AHSHIFT
    shl dx, cl              { Shift high word of Amount for segment }
    add dx, HugePtr.Word[2] { Increment the segment of HugePtr }
  end;

  { Having written the generic routines, HugeOffset, etc., we do not
    use them to THugeList.  Instead, THugeList relies on the fact that
    Windows always returns a segment-aligned pointer from GlobalAlloc.
    Thus, we know the memory base has a zero offset, so we can further
    optimize our pointer references.  This optimization cuts off about
    1/3 of the run-time on a Pentium 75.

    I tried further optimization by making FastHugeOffset into an inline
    assembler function, but that wasn't any faster.  Hence, let's stick
    with the readability of Pascal.
  }
  function FastHugeOffset(Segment: Word; Offset: LongInt): Pointer;
  begin
    Result := Ptr(Segment + LongRec(Offset).HI shl Ofs(__AHSHIFT), LongRec(Offset).Lo);
  end;

  {
    Copy memory from Src to Dst, copying Size units from
    Base+Src to Base+Dst.  The reason we separate the Base pointer
    from the Dst and Src indexes is to we can easily compare
    the indexes to see what direction we are copying, without
    having to muck about with normalizing segments.
    Offsets and the size are in longwords, for our convenience,
    but we need to convert them to short words, in case we are
    running on a 286 machine, which does not support the movsd
    instruction, just movsw.

    I'm not a fan of writing assembly language, so most of the code
    is Pascal, and I only dip into assembly for the low-level
    copying code, in DoMove, plus setting the direction flag,
    which we do with inline assembler procedures.
  }
  procedure cld; inline ($fc);  { clear direction flag }
  procedure std; inline ($fd);  { set direction flag }

  procedure DoMove(SrcPtr, DstPtr: Pointer; Size: Word); assembler;
  asm
    push ds
    lds si, SrcPtr
    les di, DstPtr
    mov cx, Size.Word[0]
    rep movsw
    pop ds
  end;

  procedure HugeMove(Base: Pointer; Dst, Src, Size: LongInt); far;
    { When computing how many words to move, we need to see
      how far from a segment boundary SrcPtr and DstPtr are,
      since the MOVS instruction won't cross a segment boundary
      properly.  Since we know that the list is allocated on
      a segment boundary, and we store long words, we know that
      SrcPtr and DstPtr will always be on longword boundaries.

      It is actually a little tricky to determine how many words
      we can copy before hitting a segment boundary.  It depends on
      whether we are copying up or down.  When copying down, the
      difference between $10000 and the pointer offset determines
      how many bytes we can copy in that segment.  If the segment
      offset is zero, then truncating $10000 to a word yields zero,
      in which case we copy the maximum number of words, $7FFF.
    }
    function ComputeDownMoveSize(SrcOfs, DstOfs: Word): Word;
    begin
      if SrcOfs > DstOfs then
        Result := Word($10000 - SrcOfs) div 2
      else
        Result := Word($10000 - DstOfs) div 2;
      if Result = 0 then
        Result := $7FFF;
    end;
    function ComputeUpMoveSize(SrcOfs, DstOfs: Word): Word;
    begin
      if SrcOfs = $FFFF then
        Result := DstOfs div 2
      else if DstOfs = $FFFF then
        Result := SrcOfs div 2
      else if SrcOfs > DstOfs then
        Result := DstOfs div 2 + 1
      else
        Result := SrcOfs div 2 + 1;
    end;

  var
    SrcPtr, DstPtr: Pointer;
    MoveSize: Word;
  begin
    SrcPtr := HugeOffset(Base, Src * SizeOf(Pointer));
    DstPtr := HugeOffset(Base, Dst * SizeOf(Pointer));

    { Convert longword size to words. }
    Size := Size * (SizeOf(LongInt) div SizeOf(Word));

    if Src < Dst then
    begin
      { start from the far end and work towards the front }
      std;
      HugeInc(SrcPtr, (Size-1) * SizeOf(Word));
      HugeInc(DstPtr, (Size-1) * SizeOf(Word));

      while Size > 0 do
      begin
        MoveSize := ComputeUpMoveSize(Word(SrcPtr), Word(DstPtr));
        if MoveSize > Size then
          MoveSize := Word(Size);
        DoMove(SrcPtr, DstPtr, MoveSize);
        Dec(Size, MoveSize);
        HugeDec(SrcPtr, MoveSize * SizeOf(Word));
        HugeDec(DstPtr, MoveSize * SizeOf(Word));
      end
    end
    else
    begin
      cld;
      while Size > 0 do
      begin
        MoveSize := ComputeDownMoveSize(Word(SrcPtr), Word(DstPtr));
        if MoveSize > Size then
          MoveSize := Word(Size);
        DoMove(SrcPtr, DstPtr, MoveSize);
        Dec(Size, MoveSize);
        HugeInc(SrcPtr, MoveSize * SizeOf(Word));
        HugeInc(DstPtr, MoveSize * SizeOf(Word));
      end;
    end;
  end;

  { Destroy a THugeList by unlocking and freeing the list's memory }
  destructor THugeList.Destroy;
  begin
    if Handle <> 0 then
    begin
      GlobalUnlock(Handle);
      GlobalFree(Handle);
    end;
    inherited Destroy;
  end;

  { Reallocate the list, copying the old list values into
    the new list.  Reset all the fields for the new list.
  }
  procedure THugeList.ReAllocList(NewCapacity: LongInt);
  var
    NewHandle: THandle;
  begin
    if Handle = 0 then
      NewHandle := GlobalAlloc(GPTR, NewCapacity*SizeOf(Pointer))
    else
      NewHandle := GlobalReAlloc(Handle, NewCapacity*SizeOf(Pointer), GMEM_ZEROINIT);

    if NewHandle = 0 then
      OutOfMemoryError;
    try
      fList := GlobalLock(NewHandle);
      if fList = nil then
        OutOfMemoryError;

      { If the list shrunk, then update Count for the smaller size. }
      if NewCapacity < Count then
        fCount := NewCapacity;

      { If Windows allocated even more memory, then see how much
        we really have.  The user might not have requested that
        much, but why waste memory?
      }
      fCapacity := GlobalSize(NewHandle) div SizeOf(Pointer);

      { And remember all the new information }
      fHandle := NewHandle;
    except
      { Oops. Something went wrong, make sure we don't leave the
        new memory dangling.  If something goes really, really
        wrong, and the new memory has already been stored in Self,
        then leave it alone.
      }
      if Handle <> NewHandle then
      begin
        GlobalUnlock(NewHandle);
        GlobalFree(NewHandle);
      end;
      raise;
    end;
  end;

  { Return the new capacity of the list when we expand it.
    Here we emulate TList, which expands the list by 4, 8, or
    16 elements, depending on the number of items already in
    the list.
  }
  function THugeList.ExpandSize: LongInt;
  begin
    if Capacity > 8 then
      ExpandSize := Capacity + 16
    else if Capacity > 4 then
      ExpandSize := Capacity + 8
    else
      ExpandSize := Capacity + 4
  end;

  { Check a list index and raise an exception if it is not valid }
  procedure THugeList.CheckIndex(Index: LongInt);
  begin
    if (Index < 0) or (Index >= Count) then
      raise EListError.CreateResFmt(IDS_INDEXERROR, [Index])
  end;

  { Return an item of the list, raising an exception for an invalid index }
  function THugeList.GetItem(Index: LongInt): Pointer;
  var
    Ptr: ^Pointer;
  begin
    CheckIndex(Index);
    Ptr := FastHugeOffset(LongRec(fList).Hi, Index * sizeof(Pointer));
    GetItem := Ptr^;
  end;

  { Get an item without checking for range errors.  Return a pointer
    to the slot in the list, so the pointer value can be changed.
    This function is available for use by derived classes, but is
    not used by THugeList itself.  Instead, THugeList calls
    FastHugeOffset() directly, which improves performance.
  }
  function THugeList.GetPointer(Index: LongInt): Pointer;
  begin
    GetPointer := FastHugeOffset(LongRec(fList).Hi, Index * sizeof(Pointer));
  end;

  { Set a list item. }
  procedure THugeList.SetItem(Index: LongInt; Item: Pointer);
  var
    Ptr: ^Pointer;
  begin
    CheckIndex(Index);
    Ptr := FastHugeOffset(LongRec(fList).Hi, Index * sizeof(Pointer));
    Ptr^ := Item;
  end;

  { Set the new capacity of the list.  If the list shrinks,
    then adjust Count.
  }
  procedure THugeList.SetCapacity(NewCapacity: LongInt);
  begin
    if Capacity <> NewCapacity then
      ReAllocList(NewCapacity);
  end;

  { Add Item to the end of the list }
  function THugeList.Add(Item: Pointer): LongInt;
  begin
    Insert(Count, Item);
    Add := Count-1;
  end;

  { Clear all the items in the list. }
  procedure THugeList.Clear;
  begin
    if Handle <> 0 then
    begin
      GlobalUnlock(Handle);
      GlobalFree(Handle);
      fHandle := 0;
      fList := nil;
    end;
    fCount := 0;
    fCapacity := 0;
  end;

  { Delete the item at Index, shifting down all other items
    with higher indexes.
  }
  procedure THugeList.Delete(Index: LongInt);
  begin
    CheckIndex(Index);
    Dec(fCount);
    HugeMove(fList, Index, Index+1, Count-Index);
  end;

  { Exchange the items at indexes I1 and I2. }
  procedure THugeList.Exchange(I1, I2: LongInt);
  var
    Tmp: Pointer;
    P, Q: ^Pointer;
  begin
    CheckIndex(I1);
    CheckIndex(I2);
    P := FastHugeOffset(LongRec(fList).Hi, I1 * sizeof(Pointer));
    Q := FastHugeOffset(LongRec(fList).Hi, I2 * sizeof(Pointer));
    Tmp := P^;
    P^ := Q^;
    Q^ := Tmp;
  end;

  { Expand the list's capacity to make room for more items.
    This is done automatically when new items are added,
    so there is little reason for the user to call Expand
    explicitly.
  }
  procedure THugeList.Expand;
  begin
    Capacity := ExpandSize;
  end;

  { Return the first item in the list }
  function THugeList.First: Pointer;
  begin
    First := Items[0];
  end;

  { Return the index of the first occurrence of Item.
    Return -1 for not found.
  }
  function THugeList.IndexOf(Item: Pointer): LongInt;
  type
    MetaPointer = ^Pointer;
  var
    i: LongInt;
    Ptr: Pointer;
  begin
    IndexOf := -1;
    for i := 0 to Count-1 do
    begin
      Ptr := FastHugeOffset(LongRec(fList).Hi, i * SizeOf(Pointer));
      if MetaPointer(Ptr)^ = Item then
      begin
        IndexOf := i;
        Break;
      end;
    end;
  end;

  { Insert Item at position, Index.  Slide all other items
    over to make room.  The user can insert to any valid index,
    or to one past the end of the list, thereby appending an
    item to the list.  In the latter case, adjust the capacity
    if needed.
  }
  procedure THugeList.Insert(Index: LongInt; Item: Pointer);
  var
    i: LongInt;
    Ptr: ^Pointer;
    NextPtr: ^Pointer;
  begin
    if (Index < 0) or (Index > Count) then
      raise EListError.CreateResFmt(IDS_INDEXERROR, [Index]);
    if Count >= Capacity then
      Expand;

    { Make room for the inserted item. }
    Ptr := FastHugeOffset(LongRec(fList).Hi, Index * SizeOf(Pointer));
    HugeMove(Ptr, 1, 0, Count-Index);

    Ptr^ := Item;
    Inc(fCount);
  end;

  { Return the last item in the list.  Raise an exception
    if the list is empty.
  }
  function THugeList.Last: Pointer;
  begin
    Last := Items[Count - 1];
  end;

  { Move an item from CurIndex to NewIndex.  Delete the
    item as NewIndex and insert a copy of the item from
    CurIndex to NewIndex.
  }
  procedure THugeList.Move(CurIndex, NewIndex: LongInt);
  begin
    Items[NewIndex] := Items[CurIndex];
  end;

  { Pack the list by removing nil slots.  After packing
    Count might be smaller.  After each loop iteration,
    we keep invariant:
      Items[k] <> nil for all k <= i
    Thus, when we reach the end of the loop, then we know we have
    packed the list.  We march through the list, using the I index.
    Whenever Items[i] = nil, we collect a maximal string of nil slots,
    and then shift down the remaining items, adjusting Count to match.
  }
  procedure THugeList.Pack;
  var
    i, j, k: LongInt;
    P, Q: ^Pointer;
  begin
    { Instead of a for loop, use a while loop, so we use the
      current value of Count for each iteration, since Count
      changes during the loop.
    }
    i := 0;
    P := fList;
    while i < Count do
    begin
      if P^ <> nil then
      begin
        Inc(i);
        P := FastHugeOffset(LongRec(fList).Hi, i*SizeOf(Pointer));
      end
      else
      begin
        { Collect a run of nil slots }
        for j := i+1 to Count-1 do
        begin
          P := FastHugeOffset(LongRec(fList).Hi, j * SizeOf(Pointer));
          if P^ <> nil then
            Break;
        end;
        { We only need to shift slots if there is a non-nil value.
          If all the remaining slots are nil, then we are done.
        }
        if P^ = nil then
        begin
          fCount := i;
          Break;
        end;

        { Now shift the slots; setting the newly vacated slots to nil, just
          in case we accidentally try to refer to one of them.
          Stop when we get to a nil slot.
        }
        k := i;
        while j < Count do
        begin
          P := FastHugeOffset(LongRec(fList).Hi, k * SizeOf(Pointer));
          Q := FastHugeOffset(LongRec(fList).Hi, j * SizeOf(Pointer));
          P^ := Q^;
          { Check after assigning to P^, so the check for nil at
            the top of the loop is true.  A small inefficiency for
            greater programming ease and maintainability.
          }
          if Q^ = nil then
            Break;
          Q^ := nil;
          Inc(k);
          Inc(j);
        end;
        { Adjust Count by the number of nil slots removed }
        Dec(fCount, j-k);
        { Set the loop counter to the next nil slot }
        i := k;
      end;
    end;
  end;

  { Remove Item and return its index. }
  function THugeList.Remove(Item: Pointer): LongInt;
  var
    Index: LongInt;
  begin
    Index := IndexOf(Item);
    if Index = -1 then
      raise EListError.CreateRes(IDS_ITEMNOTFOUND);
    Delete(Index);
    Remove := Index;
  end;

  { The THugeList class is not persistent, so there is nothing to register.
    Perhaps THugeList should be persistent, but it doesn't seem important
    until we have Huge components, too.
  }
  procedure Register;
  begin
  end;

end.
