unit LGeneral;

interface
uses
  SysUtils, WinTypes, WinProcs, Controls, ShellAPI, Dialogs,
  Graphics, IniFiles;

type
  TPString = class(TObject)
  private
    pStr : PString;
  public
    constructor Create( NewPString : PString );
    destructor Destroy; override;
  end;

{ --INIfile Objects -- }

    TLchIniFile = class(TIniFile)
    private
      sLchINIDir: String;
    public
      constructor Create;
      function  ReadRect( const sSect: String ): TRect;
      procedure WriteRect( const sSect: String;
                           const Rect: TRect );
    end;

    TWskIniFile = class(TIniFile)
    private
      sWskDLLDir, sWskINIDir: String;
    public
      constructor Create( iniLch: TLchIniFile );
    end;

const
  kMaxCommandLine = 255;
  kMaxCommandBufSize = kMaxCommandLine + 3;  { + 2byte length + null byte }

{ INI files }
  ksWinSockINI     : String = 'TrumpWsk.INI';
  ksLaunchConfig   : String = 'Launch Config';
  ksLaunchINI      : String = 'Launch.INI';
{ Contents:  Launcher INI file: TrumpWsk.INI }
  ksLaunchWndSect  : String = 'LaunchWindow';
    ksCorner         : String = 'Corner';
        ksNone         : String = 'None';
        ksTopLeft      : String = 'TopLeft';
        ksTopRight     : String = 'TopRight';
        ksBottomLeft   : String = 'BottomLeft';
        ksBottomRight  : String = 'BottomRight';
    kasCorners: array [ False..True, False..True ] of String =
    ( ('TopLeft', 'TopRight'), ('BottomLeft', 'BottomRight') );
      kNone        = 0;
      kTopLeft     = 1;
      kTopRight    = 2;
      kBottomLeft  = 3;
      kBottomRight = 4;
    kaiCorners: array [ False..True, False..True ] of Integer =
    ( (kTopLeft, kTopRight), (kBottomLeft, kBottomRight) );
    ksOrientation    : String = 'Orientation';
      ksSideways     : String = 'Sideways';
      ksUpAndDown    : String = 'UpAndDown';
      kSideways  = 0;
      kUpAndDown = 1;
    ksbBkgdBitmap : String = 'bBkgdBitmap';
    kssBkgdBmpFile: String = 'sBkgdBitmapFile';
  ksConfigWndSect  : String = 'ConfigWindow';
    ksbNoDialer    : String = 'NoDialer';
  ksLaunchPgmSect  : String = 'Programs';
  ksLaunchPgmIdent : String = 'Program';
  kiBorder = 8;
  kiInnerBorder = 8;
     ksHintDelim : String = '<:';
  ksHintEndDelim : String = ':>';
      ksUntitled : String = 'Untitled';
  kiIconRowHeight = 36;
  kiIconColWidth  = 36;
  kiIconHeight    = 32;
  kiIconWidth     = 32;

  kiComPortDefault = 0;
  kiBaudDefault = 2;
  ksDialerSect : String = 'Trumpet Winsock';
    ksWskDLLDirIdent : String = 'WskDLLDir';
    ksWskINIDirIdent : String = 'WskINIDir';
    ksAreaCode : String = 'AreaCode';
      ksDefaultAC    : String = '203';
    ksPhone    : String = 'Phone';
      ksDefaultPhone : String = '323-4448';
    ksPrefix   : String = 'Prefix';
    ksDialOne  : String = 'DialOne';
    ksBlockCallWait: String = 'BlockCallWaiting';
    ksBlockCallWaitIdent: String = 'BlockCallWaitingString';
    ksBlockCallWaitDigits: String = '*70,';

{ Contents:  Trumpet Winsock INI file: TrumpWsk.INI }
  ksWMainSect : String = 'Trumpet Winsock';
    ksPort     : String = 'slip-port';
    ksBaudRate : String = 'slip-baudrate';
    ksGateway  : String = 'Gateway';

  ksWVarSect  : String = 'default vars';
    ksPhoneNo   : String = '$number';
    ksUserid    : String = '$username';
    ksPassword  : String = '$password';

  kBAD = 0;
  kBAT = 1;
  kCOM = 2;
  kEXE = 3;
  kPIF = 4;

  kSuccess           = 0;
  kFileNotFound      = 1;
  kFileNotExecutable = 2;
  kFileHasNoSuchIcon = 3;
  kFakeIcon          = 4;
  kIconError         = 5;

function HitTest( ctl: TControl; XPos, YPos: Integer ):Boolean;
function GetFileType( var sFileName: String ):Integer;
function GetFileIcon( sFileName : String; var hNewIcon : hIcon ):Boolean;
function GetThisFileIcon(      sFileName : String;
                           var hNewIcon : hIcon;
                               iIconIdx : Integer ):Integer;
function GetWinDir( bSystem: Boolean ): String;
function MakePath( const sDir, sFile: String ): String;
function Trim( const str, sDelim: String;
                     var  sRight: String; var iOff: Integer ): String;

implementation

{ -------------- String Object --------------  }
constructor TPString.Create( NewPString : PString );
begin
  inherited Create;
  pStr := NewPString;
end;

destructor TPString.Destroy;
begin
  if Self <> nil then
    DisposeStr( pStr );
  inherited Destroy;
end;

{ -------------- General Purpose --------------  }
function HitTest( ctl: TControl; XPos, YPos: Integer ):Boolean;
begin
  { assume failure }
  Result := False;
  with ctl do
  begin
    if  (XPos >= Left) and (XPos < Left+Width ) and
        (YPos >= Top ) and (YPos < Top +Height) then
        Result := True;
  end;
end;

function GetFileType( var sFileName: String ):Integer;
var
  sExt: String;
begin
  sExt := ExtractFileExt( sFileName );
  if sExt = '.BAT' then
    Result := kBAT
  else if sExt = '.COM' then
    Result := kCOM
  else if sExt = '.PIF' then
    Result := kPIF
  else if sExt = '.EXE' then
    Result := kEXE
  else
    Result := kBAD;
end;

function GetPath: String;
var
  sCurr: String;
begin
  GetDir( 0, sCurr );
  Result := sCurr + ';' + GetWinDir(False) + ';' + GetWinDir(True);
end;

function GetFileIcon(  sFileName : String; var hNewIcon : hIcon ):Boolean;
var
  iErr: Integer;
  sErr: String;
begin
  Result := False;
  iErr := GetThisFileIcon( sFileName, hNewIcon, 0 );
  case iErr of
  kFileNotFound:
    sErr := 'File not found.  Check path.';
  kFileNotExecutable:
    sErr := 'File must be executable (either EXE, COM, BAT or PIF).';
  kFileHasNoSuchIcon:
    sErr := 'No icons found.  Check path.';
  kSuccess:
    begin
      Result := True;
      Exit;
    end;
  end;
  MessageDlg( sErr, mtWarning, [mbOK], 0 );
end;

function GetThisFileIcon(      sFileName : String;
                           var hNewIcon : hIcon;
                               iIconIdx : Integer ):Integer;
var
  pFileName : PChar;
  sFN : String;
  iFileType : Integer;
begin
  Result := kSuccess;
  pFileName := StrAlloc( kMaxCommandBufSize );
  try
    sFileName := UpperCase( sFileName );
    { if no extension --- add .exe extension }
    sFN := ExtractFileName( sFileName );
    if EmptyStr = ExtractFileExt( sFN ) then
      sFileName := sFileName + '.EXE';

    iFileType := GetFileType( sFileName );
    sFN := FileSearch( sFileName, GetPath );
    if sFN = EmptyStr then
      Result := kFileNotFound
    else begin
      case iFileType of
      kEXE:
        StrPCopy( pFileName, sFileName );
      kCOM, kBAT, kPIF:
        StrPCopy( pFileName, 'club.ico' );
      else
        Result := kFileNotExecutable;
      end;
      if (Result <> kFileNotExecutable) then
      begin
        { assume failure }
        Result := kFileHasNoSuchIcon;
        hNewIcon := ExtractIcon( HInstance, pFileName, iIconIdx );
        if (hNewIcon >= hIcon(32)) then
          Result := kSuccess;
      end;
    end;
  finally
    StrDispose( pFileName );
  end;
end;

{$IFDEF SAMMMMMM }
function GetFileIcon(  sFileName : String; var hNewIcon : hIcon ):Boolean;
var
  pFileName : PChar;
  sFN : String;
  iFileType : Integer;
begin
  Result := False;
  pFileName := StrAlloc( kMaxCommandBufSize );
  try
    sFileName := UpperCase( sFileName );
    { if no extension --- add .exe extension }
    sFN := ExtractFileName( sFileName );
    if EmptyStr = ExtractFileExt( sFN ) then
      sFileName := sFileName + '.EXE';

    iFileType := GetFileType( sFileName );
    sFN := FileSearch( sFileName, GetPath );
    if sFN = EmptyStr then
      MessageDlg( 'File not found.  Check path.', mtWarning, [mbOK], 0 )
    else
    begin
      case iFileType of
      kEXE:
        StrPCopy( pFileName, sFileName );
      kCOM, kBAT, kPIF:
        StrPCopy( pFileName, 'club.ico' );
      else
        hNewIcon := hIcon(1);
      end;
      if (hNewIcon <> hIcon(1)) then
          hNewIcon := ExtractIcon( HInstance, pFileName, 0 );
      if (hNewIcon = hIcon(1)) then
        MessageDlg( 'File must be .EXE, .COM, BAT or .PIF', mtWarning, [mbOK], 0 )
      else if (hNewIcon = hIcon(0)) then
        MessageDlg( 'No icons found.  Check path.', mtWarning, [mbOK], 0 )
      else
        Result := True;
    end;
  finally
    StrDispose( pFileName );
  end;
end;
{$ENDIF SAMMMMMM }

function GetWinDir( bSystem: Boolean ): String;
var
  pBuf : PChar;
  iSize: Word;
begin
  Result := EmptyStr;
  try
    GetMem( pBuf, 200);
    {Get the windows directory and size of the "string"}
    if bSystem then
      iSize := GetSystemDirectory(  pBuf, 200)
    else
      iSize := GetWindowsDirectory( pBuf, 200);
    {Convert the directory to a string}
    Result := StrPas( pBuf );
    Dispose( pBuf );
  except
    on EOutOfMemory do
      MessageDlg( 'Not enough memory in the heap', mtWarning, [mbOk], 0 );
  end;
end;

function MakePath( const sDir, sFile: String ): String;
var
  sMid : string;
begin
  if ( '\' <> Copy( sDir, Length(sDir), 1 )) then
    sMid := '\'
  else
    sMid := '';
  Result := sDir + sMid + sFile;
end;

function Trim( const str, sDelim: String;
                     var  sRight: String; var iOff: Integer ): String;
begin
  Result := str;
  sRight := EmptyStr;
  iOff := Pos( sDelim, str );
  if iOff > 0 then
  begin
     Result := Copy( str, 1, iOff-1 );
     sRight := Copy( str, iOff + Length(sDelim), Length( str ) );
  end;
end;

{ --INIfile Objects -- }
constructor TLchIniFile.Create;
begin
  sLchINIDir := '.\' + ksLaunchINI;
  if not FileExists( sLchINIDir ) then
     sLchINIDir := ksLaunchINI;
  inherited Create( sLchINIDir );
end;

function TLchIniFile.ReadRect( const sSect: String ): TRect;
begin
  Result.Top    := ReadInteger( sSect, 'Top', 100 );
  Result.Left   := ReadInteger( sSect, 'Left', 100 );
  Result.Bottom := ReadInteger( sSect, 'Bottom', 200 );
  Result.Right  := ReadInteger( sSect, 'Right', 200 );
end;

procedure TLchIniFile.WriteRect( const sSect: String;
                                 const Rect: TRect );
begin
  WriteInteger( sSect, 'Top',    Rect.Top );
  WriteInteger( sSect, 'Left',   Rect.Left );
  WriteInteger( sSect, 'Bottom', Rect.Bottom );
  WriteInteger( sSect, 'Right',  Rect.Right );
end;

constructor TWskIniFile.Create( iniLch: TLchIniFile );
begin
  sWskDLLDir := GetWinDir( False );
  sWskINIDir := sWskDLLDir;
  { Use Windows directory if Launch INI file does not direct us elsewhere }
  sWskDLLDir := iniLch.ReadString( ksDialerSect, ksWskDLLDirIdent, sWskDLLDir );
  sWskINIDir := iniLch.ReadString( ksDialerSect, ksWskINIDirIdent, sWskINIDir );
  inherited Create( MakePath( sWskINIDir, ksWinSockINI ) );
end;

end.
