unit LProgram;

interface

uses
  SysUtils,
  WinTypes, { for hIcon }
  WinProcs, { for DrawIcon }
  Graphics, { for TBitmap }
  Classes, Controls,Dialogs, IniFiles,
  LConfig,  { TfrmConfig }
  LButton;  { TLnchButton }

{$DEFINE DEBUGX}
type
  TLnchProgram = class( TObject )
  private
    { Private declarations }
    sIniLine : String;
    sPgmIdent : String;
    sCmdLine : String;
    sExecutable : String;
    sArgs : String;
    sHint : String;
    bVisible : Boolean;
    iBtnNo: Integer;  { set in GetPgmIdent, used in Add }
    btn: TLnchButton;
    function DedupToolbarINI:Boolean;
    function GetPgmIdent:Integer;
    procedure WriteIniLine;
    procedure SetVisible( bNewVis: Boolean );
    procedure SetCmdLine( sNew: String );
    procedure SetHint( sNew: String );
  public
    { Public declarations }
    constructor CreateLine( sINIline: String );
    constructor Init( const sNewPgmIdent:String;
                      const  sNewCmdLine:String;
                      const     sNewHint:String;
                      const  bNewVisible:Boolean );
    destructor Destroy; override;
    function UpdateButton( bVerbose: Boolean ):Integer;
    function CreateButton( AOwner: TComponent; bVerbose: Boolean ):Boolean;
    procedure RemoveProgramFromToolbarINI;
    function Update( bVerbose: Boolean ): Integer;
    function OkToAdd( bDupCheck: Boolean ): Boolean;
    function  Add( const sProgramList: TStringList;
                  const     bDupCheck: Boolean ):Boolean;
    function ParseIniValue( const sIniValue:String ):Boolean;
    function ParseIniLine(  const sIniLine:String ):Boolean;
    function BtnHitTest( const x, y: Integer ):Boolean;
    property CmdLine : String read sCmdLine write SetCmdLine;
    property    Hint : String read sHint write SetHint;
    property Visible : Boolean read bVisible write SetVisible;
    function Reposition( const bSideways: Boolean;
                         const iVisBtnNo:Integer ):Boolean;
  end;


implementation

uses
  Launcher, { TfrmLauncher }
  Lgeneral;

{ Create program Object from INI file line
  or from Dropped file Cmd Line (with prepended = sign) }
constructor TLnchProgram.CreateLine( sINIline: String );
begin
  inherited Create;
  ParseIniLine( sINIline );
end;

{ Create program Object from component parts. }
constructor TLnchProgram.Init( const sNewPgmIdent:String;
                               const  sNewCmdLine:String;
                               const     sNewHint:String;
                               const  bNewVisible:Boolean );
var
  iOff: Integer;
  sRight: String;
begin
  inherited Create;
  sPgmIdent := sNewPgmIdent;  { Don't allow user to specify yet }
  sCmdLine  := sNewCmdLine;
  sExecutable := Trim( sCmdLine, ' ', sRight, iOff );
  sHint     := sNewHint;
  bVisible  := bNewVisible;
  if sHint = EmptyStr then
  begin
     sHint := ExtractFileName( sCmdLine );
     sHint := Trim( sHint, '.', sRight, iOff );
  end;
end;

destructor TLnchProgram.Destroy;
begin
  if btn <> nil then
     btn.Destroy;
  inherited Destroy;
end;

function TLnchProgram.DedupToolbarINI:Boolean;
var
  wResp : Word;
  iIniIndex : Integer;
  sList : TStringList;
  pgmTemp: TLnchProgram;
  ini : TLchIniFile;
begin
  wResp := mrYes;  { assume dup OK or no dup will be found }
  sList := TStringList.Create;
  try
    ini := TLchIniFile.Create;
    try
      { Create empty program Object }
      pgmTemp := TLnchProgram.Create;
      try
        ini.ReadSectionValues( ksLaunchPgmSect, sList );
        for iIniIndex := 0 to sList.Count-1 do
        begin
          if pgmTemp.ParseIniLine( sList.Strings[ iIniIndex ] ) then
          begin
          { Check executable (uppercase) and args (mixed case) separately }
            if    ( Uppercase(sExecutable) = Uppercase(pgmTemp.sExecutable) )
              and ( sArgs       = pgmTemp.sArgs ) then
            begin
              wResp := MessageDlg( 'Program is already on toolbar.  Add Again?'
                                 + #13#10 + pgmTemp.sCmdLine,
                                    mtConfirmation, [mbYes, mbNo], 0 );
              break;
            end;  { end if }
          end;  { end if }
        end;  { end for }
      finally
        pgmTemp.Destroy;
      end;
    finally
      ini.Free;
    end;
  finally
    sList.Free;
  end;
  if wResp = mrYes then
    Result := True
  else
    Result := False;
end;

{ Find first available PgmIdent
  starting with 'Program0', 'Program1', 'Program2', etc.
  Returns first available integer value
  Self.sPgmIdent will contain the appropriate string }
function TLnchProgram.GetPgmIdent:Integer;
var
  sVal: String;
  ini : TLchIniFile;
begin
  ini := TLchIniFile.Create;
  try
    Result := 0;
    while True do begin
      sPgmIdent := ksLaunchPgmIdent + IntToStr( Result );
      sVal := ini.ReadString( ksLaunchPgmSect, sPgmIdent, 'xx' );
      if ( 'xx' = sVal ) then
        break;
      Inc( Result );
    end;
  finally
    ini.Free;
  end;
end;

procedure TLnchProgram.WriteIniLine;
var
  sVisFlag : String;
  ini : TLchIniFile;
begin
  ini := TLchIniFile.Create;
  try
    if bVisible then sVisFlag := 'True'
                else sVisFlag := 'False';
    sIniLine := sCmdLine + ksHintDelim + sHint + ksHintEndDelim + sVisFlag;
    ini.WriteString( ksLaunchPgmSect, sPgmIdent, sIniLine );
    sIniLine := sPgmIdent + '=' + sIniLine;
  finally
    ini.Free;
  end;
end;

procedure TLnchProgram.RemoveProgramFromToolbarINI;
var
  ini : TIniFile;
  sList : TStringList;
  iIniIndex : Integer;
begin
  ini := TIniFile.Create( ksLaunchINI );
  sList := TStringList.Create;
  try
    ini.ReadSectionValues( ksLaunchPgmSect, sList );
{$IFDEF DEBUG }
    writeln( sIniLine );
{$ENDIF DEBUG }
    iIniIndex := sList.IndexOf( sIniLine );
    if (iIniIndex > 0) and (iIniIndex < sList.Count) then
    begin
      sList.Delete( iIniIndex );
      ini.EraseSection( ksLaunchPgmSect );
      for iIniIndex := 0 to sList.Count - 1 do
      begin
{$IFDEF DEBUG }
        writeln( sList.Strings[ iIniIndex ] );
{$ENDIF DEBUG }
        if ParseIniLine( sList.Strings[ iIniIndex ] ) then
        begin
           sPgmIdent := ksLaunchPgmIdent + IntToStr( iIniIndex );
           WriteIniLine;
        end;
      end;
    end;
  finally
    sList.Free;
  end;
end;

{ return kSuccess:   if Icon loaded OK
         kFakeIcon:  if error with this command and fake icon loaded
         kIconError: if any other error occurred and no icon loaded
}
function TLnchProgram.UpdateButton( bVerbose: Boolean ):Integer;
var
  hNewIcon : hIcon;
  bGoodIcon: Boolean;
begin
  Result := kSuccess;
  StrPCopy( btn.CmdLine, sCmdLine );
  btn.Hint := Hint;
  with btn do
  begin
    if bVerbose then
      bGoodIcon := GetFileIcon( sExecutable, hNewIcon )
    else
      bGoodIcon := ( kSuccess = GetThisFileIcon( sExecutable, hNewIcon, 0 ) );

{$IFDEF STRICT }
    if not bGoodIcon then
      Result := kIconError;
{$ELSE STRICT }
    { if something wrong with this executable, use fake icon }
    if not bGoodIcon then
    begin
      { assume fake error will also be unavailable }
      Result := kIconError;
      { try to extract fake icon from Program Manager }
      if ( kSuccess = GetThisFileIcon( 'Progman', hNewIcon, 2 ) ) then
        Result := kFakeIcon;
    end;
{$ENDIF STRICT }

  { if icon was good or we are allowed to use fake icon, draw icon }
    if Result <> kIconError then
    begin
      if hIcon <> 0 then
         DeleteObject( hIcon );
      hIcon := hNewIcon;
      Glyph := nil;
      Glyph.Height := 32;
      Glyph.Width := 32;
      if not DrawIcon( Glyph.Canvas.Handle, 0, 0, hNewIcon ) then
        Result := kIconError;
    end;
  end;
end;

function TLnchProgram.CreateButton( AOwner: TComponent;
                                    bVerbose: Boolean  ):Boolean;
var
  hNewIcon : hIcon;
begin
  Result := False;
  btn := TLnchButton.Create( AOwner );
  try
    with btn do begin
      Height := kiIconRowHeight;
      Width  := kiIconRowHeight;
      Glyph.Height := kiIconHeight;
      Glyph.Width  := kiIconHeight;
      ShowHint := True;
      Hint := Self.Hint;
    end;
    Result := ( kIconError <> UpdateButton( bVerbose ) );
  finally
    if not Result then
    begin
      btn.Free;
      btn := nil;
    end;
  end;
end;

function TLnchProgram.Update( bVerbose: Boolean ): Integer;
begin
  { We know how to write ourselves to the INI file }
  Result := UpdateButton( bVerbose );
  if Result <> kIconError then
    WriteIniLine;
end;

function TLnchProgram.OkToAdd( bDupCheck: Boolean ): Boolean;
begin
  { Assume no dup check or dup check will succeed }
  Result := True;
  if bDupCheck then
    Result := DedupToolbarINI;
  if Result then
    Result := CreateButton( frmLauncher, bDupCheck );
end;

function TLnchProgram.Add( const sProgramList: TStringList;
                           const    bDupCheck: Boolean ):Boolean;
begin
  Result := OkToAdd( bDupCheck );
  { Add or update INI line for this program.}
  if Result then
  begin
    { We know how to write ourselves to the INI file }
    iBtnNo := GetPgmIdent;
    WriteIniLine;
   { insert new or replacement TLnchProgram object }
   {sProgramList.InsertObject( GetPgmIdent, sCmdLine, Self ); }
    sProgramList.InsertObject( 0, sCmdLine, Self );
    { Create a real button and add it to the toolbar }
    frmLauncher.InsertControl( btn );
  end;
end;

function TLnchProgram.ParseIniValue( const sIniValue:String ):Boolean;
var
  s: String;
begin
  s := '=' + sIniValue;
  Result := ParseIniLine( s );
end;

function TLnchProgram.ParseIniLine( const sIniLine:String ):Boolean;
var
  iIdentLen, iSpaceOff, iHintOff, iHintEndOff, iDotOff: Integer;
  sVisFlag: String;
begin
  { assume invalid Ini Line }
  Result := False;
  sPgmIdent := Trim( sIniLine, '=', sCmdLine, iIdentLen );
  if (iIdentLen > 0) and (iIdentLen < kMaxCommandLine) then
  begin
    sCmdLine    := Trim( sCmdLine, ksHintDelim, sHint, iHintOff );
    sExecutable := Trim( sCmdLine, ' ', sArgs, iSpaceOff );
    sExecutable := UpperCase( sExecutable );
    { If no hint specified, use entire value of this INI line }
    if iHintOff <= 0 then
    begin
      sHint := ExtractFileName( sExecutable );
      sHint := Trim( sHint, '.', sVisFlag, iDotOff );
      bVisible := True;
    { else strip away the hint }
    end else
    begin
      sHint := Trim( sHint, ksHintEndDelim, sVisFlag, iHintEndOff );
      bVisible := (Copy( sVisFlag, 1, 1 ) = 'T');
    end;
    Result := True;
  end;
end;

function TLnchProgram.BtnHitTest( const x, y: Integer ):Boolean;
begin
  Result := False;
  if (btn <> nil) then
    Result := HitTest( btn as TControl, x, y);
end;

{ Warning!  Do not change Visible property from Reposition
  because writing to Visible calls SetButtonPositions
  which calls Reposition.  We wouldn't want to recurse, now. }
function TLnchProgram.Reposition( const bSideways: Boolean;
                                  const iVisBtnNo:Integer ):Boolean;
begin
  btn.Reposition( bVisible, bSideways, iVisBtnNo );
  Result := bVisible;
end;

procedure TLnchProgram.SetVisible( bNewVis : Boolean );
begin
  bVisible := bNewVis;
  frmLauncher.SetButtonPositions;
end;

procedure TLnchProgram.SetCmdLine( sNew: String );
begin
  sCmdLine := sNew;
end;

procedure TLnchProgram.SetHint( sNew: String );
begin
  sHint := sNew;
end;

end.

