unit Launcher;

interface

uses
  SysUtils, WinTypes, WinProcs, Messages, Classes, Graphics, Controls,
  Forms, Dialogs, IniFiles, Buttons, WinCRT, ExtCtrls, ShellAPI,
  LGeneral,
  LProgram,  { for TLnchProgram }
  LPgmList,  { for TPgmList }
  LConfig, StdCtrls;   { frmConfig }

{$DEFINE DEBUGX}
type
  TfrmLauncher = class(TForm)
    btnConfig: TSpeedButton;
    Bevel1: TBevel;
    imgBkgd: TImage;
    procedure FormCreate(Sender: TObject);
    procedure FormDestroy(Sender: TObject);
    procedure btnConfigClick(Sender: TObject);
    procedure FormDragOver(Sender, Source: TObject; X, Y: Integer;
      State: TDragState; var Accept: Boolean);
    procedure FormMouseMove(Sender: TObject; Shift: TShiftState; X,
      Y: Integer);
    procedure FormCloseQuery(Sender: TObject; var CanClose: Boolean);
    procedure FormPaint(Sender: TObject);
  private
    { Private declarations }
    bDraggable : Boolean;
    sProgramList : TPgmList;  { Objects in list are TLnchProgram }
    frmConfig: TfrmConfig;
    LaunchRect : TRect;
    iniLch: TLchIniFile;
    iniWsk: TWskIniFile;
    procedure WMNCLButtonUp(var M: TWMNCLButtonUp); message wm_NCLButtonUp;
    function  InDraggableArea( XPos, YPos:SmallInt ):Boolean;
    procedure WMNCHitTest(var M: TWMNCHitTest); message wm_NCHitTest;
    procedure WMDropFiles(var M: TWMDropFiles ); message wm_DropFiles;
    procedure RecallOldPositionINI;
    procedure ResumeOldPosition;
    procedure OnAppMessage(var Msg: TMsg; var Handled: Boolean);
    function     GetProgramCount: Integer;
    function     GetProgram(  iIndex: Integer ): TLnchProgram;
    function  LookupProgram( sCmdLine: String ): TLnchProgram;
  public
    { Public declarations }
    iCorner, iOrientation : Integer;
    bBitmap: Boolean;
    procedure FormDragOn;
    procedure FormDragOff;
    procedure SavePositionINI;
    property GetProgramList : TPgmList read sProgramList;
    property ProgramObjects[ iIndex : Integer ] : TLnchProgram
                             read GetProgram;
    property Programs[ sCmdLine: String ] : TLnchProgram
                       read LookupProgram;
    property ProgramObjCount : Integer read GetProgramCount;

    procedure ReleaseConfigForm;
    procedure AdjustWidth( iVisibleBtnCount: Integer );
    procedure SetButtonPositions;
    procedure DeleteProgram( sCmdLine: String );
    function AddProgram( const sCmdLine, sHint: String;
                         bVisible: Boolean ): Boolean;
    procedure UpdateConfigPgmList( sCmdLine: String );
    procedure SetPgmString( pgm: TLnchProgram );
  end;

var
  frmLauncher: TfrmLauncher;
  MessageID: Word;

implementation

uses
  LDropOp;

{$R *.DFM}

{ -------------- Launcher Form --------------  }
function TfrmLauncher.GetProgramCount: Integer;
begin
  Result := sProgramList.ProgramCount;
end;

function TfrmLauncher.GetProgram( iIndex: Integer ): TLnchProgram;
begin
  Result := sProgramList.ProgramObjects[ iIndex ];
end;

function TfrmLauncher.LookupProgram( sCmdLine: String ): TLnchProgram;
begin
  Result := sProgramList.Programs[ sCmdLine ];
end;

procedure TfrmLauncher.OnAppMessage(var Msg: TMsg; var Handled: Boolean);
begin
  { If it's the special message, then focus this window }
  If Msg.Message = MessageID then
    SetFocus;
end;

procedure TfrmLauncher.SetPgmString( pgm: TLnchProgram );
begin
  with sProgramList do
    Strings[ IndexOfObject( pgm ) ] := pgm.CmdLine;
end;

procedure TfrmLauncher.UpdateConfigPgmList( sCmdLine: String );
begin
  if frmConfig <> nil then
    frmConfig.lstPrograms.Items.Insert( 0, sCmdLine );
end;

procedure TfrmLauncher.FormDragOn;
begin
  bDraggable := True;
{$IFDEF DEBUG}
  Color := clBtnShadow;
{$ENDIF DEBUG}
end;

procedure TfrmLauncher.FormDragOff;
begin
  bDraggable := False;
{$IFDEF DEBUG}
  Color := clBtnFace;
{$ENDIF DEBUG}
end;

procedure TfrmLauncher.WMNCLButtonUp(var M: TWMNCLButtonUp);
begin
  inherited;                    { call the inherited message handler }
{  FormDragOff; }
  SavePositionINI;
end;

function TfrmLauncher.InDraggableArea( XPos, YPos:SmallInt ):Boolean;
var
  ii : Integer;
begin
  Result := False;
{$IFDEF DEBUG}
  writeln( ' XPos: ', XPos, ', YPos ', YPos );
{$ENDIF DEBUG}
  { Don't Allow window dragging on First HitTest }
  if not bDraggable then
    Exit
  else
  begin
    if HitTest( btnConfig as TControl, XPos-Left, YPos-Top) then
      Exit;
    with sProgramList do
    begin
      for ii := 0 to Count - 1 do
      begin
        { Leave function if mouse is over a button }
        if ProgramObjects[ ii ].BtnHitTest( XPos-Left, YPos-Top ) then
           Exit;
      end;
    end;
  end;
  if (XPos < Left) or (XPos > Width + Left) or (YPos < Top) or (YPos > Top + Height) then
     Exit;
  { Return true if mouse is not over a button }
{  SetCaptureControl( Self ); }
  Result := True;
end;

procedure TfrmLauncher.WMNCHitTest(var M: TWMNCHitTest);
begin
  inherited;                    { call the inherited message handler }
  if InDraggableArea( M.XPos, M.YPos ) then
  begin
    if  M.Result = htClient then    { is the click in the client area?   }
        M.Result := htCaption;      { if so, make Windows think it's     }
                                    { on the caption bar.                }
  end;
end;

{ Delete from INI file and program list }
procedure TfrmLauncher.DeleteProgram( sCmdLine: String );
begin
  with sProgramList do
  begin
    Delete( IndexOf( sCmdLine ) );
  end;
  SetButtonPositions;
end;

function TfrmLauncher.AddProgram( const sCmdLine, sHint: String;
                                   bVisible: Boolean ): Boolean;
var
   NewProgram : TLnchProgram;
begin
   NewProgram :=
     TLnchProgram.Init( EmptyStr, { no Identifying name given for INI line }
                        sCmdLine, { given Cmd Line }
                        sHint,    { given Hint }
                        bVisible  { make visible button  }
                      );
  Result := NewProgram.Add( sProgramList, True );
  SetButtonPositions;
end;

procedure TfrmLauncher.WMDropFiles(var M: TWMDropFiles );
var
  DropOp : TDropOperation;
begin
  inherited;
  DropOp := TDropOperation.Create( M.Drop );
  sProgramList.AddList( DropOp.DroppedFileNames, True );
  SetButtonPositions;
  DropOp.Destroy;
end;

procedure TfrmLauncher.AdjustWidth( iVisibleBtnCount: Integer );
var
  iRows, iCols: Integer;
begin
  { Move static items }
  if iOrientation = kSideways then
  begin
    iRows := 0;
    iCols := iVisibleBtnCount;
  end else
  begin
    iRows := iVisibleBtnCount;
    iCols := 0;
  end;
  btnConfig.Left := kiBorder
                  + iCols*( kiInnerBorder + kiIconColWidth );
  btnConfig.Top := kiBorder
                 + iRows*( kiInnerBorder + kiIconRowHeight );

  Width := kiBorder + kiBorder - kiInnerBorder
         + (iCols+1)*( kiInnerBorder + kiIconColWidth );
  if ( Left + Width > Screen.Width ) then
       Left := Screen.Width - Width;

  Height := kiBorder + kiBorder - kiInnerBorder
         + (iRows+1)*( kiInnerBorder + kiIconRowHeight );
  if ( Top + Height > Screen.Height ) then
       Top := Screen.Height - Height;
end;

{ Warning!!  Do not set TLnchProgram.Visible property from
  within this procedure!!  This procedure is called when
  that property goes from invisible to Visible. }
procedure TfrmLauncher.SetButtonPositions;
var
  ii, iVisBtnNo: Integer;
  bSideways: Boolean;
begin
  ResumeOldPosition;
  iVisBtnNo := 0;
  for ii := 0 to ProgramObjCount - 1 do
  begin
    bSideways := (iOrientation = kSideways);
    if ProgramObjects[ ii ].Reposition( bSideways, iVisBtnNo ) then
       Inc( iVisBtnNo );
  end;
  AdjustWidth( iVisBtnNo );
end;

procedure TfrmLauncher.SavePositionINI;
begin
  iniLch.WriteRect( ksLaunchWndSect, BoundsRect );
  LaunchRect := BoundsRect;
end;

procedure TfrmLauncher.RecallOldPositionINI;
var
  sCorner, sOrientation, sBkgdBmpFile: String;
begin
  LaunchRect := iniLch.ReadRect( ksLaunchWndSect );
  if iniLch <> nil then
  begin
    sCorner := iniLch.ReadString( ksLaunchWndSect, ksCorner, ksTopRight );
    if sCorner = ksTopLeft then
       iCorner := kTopLeft
    else if sCorner = ksTopRight then
       iCorner := kTopRight
    else if sCorner = ksBottomLeft then
       iCorner := kBottomLeft
    else if sCorner = ksBottomRight then
       iCorner := kBottomRight
    else
       iCorner := kNone;

    sOrientation := iniLch.ReadString( ksLaunchWndSect,
                                       ksOrientation,
                                       ksSideways );
    if sOrientation = ksUpAndDown then
       iOrientation := kUpAndDown
    else
       iOrientation := kSideways;
  end;
  bBitmap := iniLch.ReadBool( ksLaunchWndSect, ksbBkgdBitmap, True );
  sBkgdBmpFile := iniLch.ReadString( ksLaunchWndSect, kssBkgdBmpFile, EmptyStr);
  if sBkgdBmpFile <> EmptyStr then
  begin
    imgBkgd.Picture.LoadFromFile( sBkgdBmpFile );
  end;
end;

procedure TfrmLauncher.ResumeOldPosition;
begin
  if iCorner = kNone then
  begin
    Top  := LaunchRect.Top;
    Left := LaunchRect.Left;
  end else
  begin
    case iCorner of
      kTopLeft, kTopRight:
        Top  := 0;
      kBottomLeft,kBottomRight:
        Top  := Screen.Height - kiIconRowHeight;
    end;
    case iCorner of
      kTopLeft, kBottomLeft:
        Left := 0;
      kTopRight,kBottomRight:
        Left := Screen.Width;
    end;
  end;
end;

procedure TfrmLauncher.FormCreate(Sender: TObject);
var
  sIniPgmList : TStringList;
  iVisibleBtnCount : Integer;
begin
{$IFDEF DEBUG}
  WindowOrg := Point(0, 400);
  WindowSize := Point(640, 160);
  InitWinCRT;
{$ENDIF DEBUG}
  { Assign the OnMessage event handler for Application }
  Application.OnMessage := OnAppMessage;

  iniLch := TLchIniFile.Create;
  iniWsk := TWskIniFile.Create( iniLch );

  RecallOldPositionINI;
  ResumeOldPosition;
  DragAcceptFiles( Handle, True );

  AdjustWidth( 1 );
  sProgramList := TPgmList.Create( iniLch, btnConfig );
  SetButtonPositions;
  FormDragOn;
end;

procedure TfrmLauncher.FormDestroy(Sender: TObject);
var
  ii : Integer;
begin
  sProgramList.DestroyList;
  iniLch.Free;
  iniWsk.Free;
{$IFDEF DEBUG}
  DoneWinCRT;
{$ENDIF DEBUG}
end;

procedure TfrmLauncher.ReleaseConfigForm;
begin
  frmConfig.Release;
  frmConfig := nil;
end;

procedure TfrmLauncher.btnConfigClick(Sender: TObject);
begin
  SavePositionINI;
  if frmConfig = nil then
    Application.CreateForm(TfrmConfig, frmConfig);
  frmConfig.Show;
end;

procedure TfrmLauncher.FormDragOver(Sender, Source: TObject; X, Y: Integer;
  State: TDragState; var Accept: Boolean);
begin
  if (dsDragMove = State) or (dsDragEnter = State) then
    Accept := True
  else
    Accept := False;
end;

procedure TfrmLauncher.FormMouseMove(Sender: TObject; Shift: TShiftState;
  X, Y: Integer);
begin
  SetCaptureControl( Self );
  if (X < 0) or (X > Width) or (Y < 0) or (Y > Height) then
    FormDragOff
  else
    FormDragOn;
end;

procedure TfrmLauncher.FormCloseQuery(Sender: TObject;
  var CanClose: Boolean);
begin
  CanClose := (mrYes = MessageDlg( 'Are you sure?',
                                   mtConfirmation, [mbYes,mbNo], 0) );
end;

procedure TfrmLauncher.FormPaint(Sender: TObject);
var
  x, y, lWidth, lHeight: LongInt;
  bmp: TBitMap;
begin
  if bBitmap then
  begin
    bmp := imgBkgd.Picture.Bitmap;
    lWidth := bmp.Width;
    lHeight := bmp.Height;

    y := 0;
    while y < Height do
    begin
      x := 0;
      while x < Width do
      begin
        Canvas.Draw( x, y, bmp );
        Inc( x, lWidth );
      end;
      Inc( Y, lHeight );
    end;
  end else
  begin
    Canvas.Brush.Color := clBtnFace;
    Canvas.Rectangle( 0, 0, Width, Height );
  end;
end;

end.
