unit Killmain;

{
  This sample project is the Delphi version of KillDLL, a utility
  coming with E! for Windows, a programming editor from MainSoft .
  This code has been ported from BP7 + OWL to Delphi in order to
  evaluate the compatibilty of both products and the differences
  in code generation.

  Although the programs are functionally equivalent, the Delphi
  version is much bigger (about 15 times) than its BP7 counterpart.
  The source code is smaller though (about 50 lines less).

  This code is based on information given in Undocumented Windows
  by Andrew Schulman. Unlike other similar utilities, it displays
  only the DLL modules. The usage count is displayed along with
  the DLL names.

  Written and uploaded by MainSoft sarl.
  Public Domain Software.
}

interface

uses
  SysUtils, WinTypes, WinProcs,  Classes,
  Forms, Dialogs, ToolHelp, StdCtrls, ExtCtrls, Controls;

type
  TKill = class(TForm)
    Panel1: TPanel;
    Header: TPanel;
    lbDLLList: TListBox;
    btnUnload: TButton;
    btnRefresh: TButton;
    btnExit: TButton;
    Confirm: TCheckBox;
    Label1: TLabel;
    Label2: TLabel;

    procedure FormCreate(Sender: TObject);
    procedure FormDestroy(Sender: TObject);
    function  IsDLL(Module : THandle) : boolean;
    procedure LoadListBox;
    procedure btnRefreshClick(Sender: TObject);
    procedure btnUnloadClick(Sender: TObject);
    procedure btnExitClick(Sender: TObject);
    procedure lbDLLListClick(Sender: TObject);
  end;

const
{Maximum number of listed DLLs}

  MAXDLL = 300;

var
  Kill: TKill;

  ModuleCount : integer;
  ModuleList  : array[0..MAXDLL - 1] of PModuleEntry;


implementation

{$R *.DFM}

function GetMem(var P; Size : word) : boolean;
var
  Pt : Pointer absolute P;
begin
  System.GetMem(Pt, Size);
  GetMem := (Pt <> nil);
end;

procedure TKill.FormCreate(Sender: TObject);

begin
  ModuleCount := 0;
  LoadListBox;
end;

procedure TKill.FormDestroy(Sender: TObject);

var
  i : integer;

begin
  for i := 0 to Pred(ModuleCount) do
    FreeMem(ModuleList[i], SizeOf(TModuleEntry));
end;

function TKill.IsDLL(Module : THandle) : boolean;
{-Return TRUE if the specified module is a DLL}

const
  FlagOffset = 12;
  LibFlag    = $8000;

var
  PW : PWord;

  function HandleToSel(h : THandle) : word;
  var
    dummy : word;
  begin
    dummy := LoWord(GetVersion);
    if Hi(dummy) >= 10 then
      h := h or $0001
    else
      Dec(h);
    HandleToSel := h;
  end;

begin
  PW := PWord(Ptr(HandleToSel(Module), FlagOffset));
  IsDLL := PW^ and LibFlag <> 0;
end;

procedure TKill.LoadListbox;

var
  E        : TModuleEntry;
  ME       : PModuleEntry;
  i        : integer;
  CountStr : string;

  procedure AddEntry;
  {-Add a new entry to the list of DLLs}
  var
    dummystr : string;
    usagestr : string;
    j        : integer;
  begin
    try
      if (ModuleCount < MAXDLL)
      and IsDLL(E.hModule)
      and GetMem(ME, SizeOf(TModuleEntry)) then begin
        ME^ := E;
        ModuleList[ModuleCount] := ME;
        Inc(ModuleCount);
        Str(GetModuleUsage(ME^.hModule), usagestr);
        dummystr := StrPas(ME^.szModule) + ' (' + usagestr + ')';
        lbDLLList.Items.Add(dummystr);
      end;
    except
      on EOutOfMemory do begin
        for j := 0 to Pred(ModuleCount) do
          FreeMem(ModuleList[i], SizeOf(TModuleEntry));
        MessageDlg('Out of memory. Aborting.',
                    mtError,
                    [mbAbort],
                    0);
      end;
    end;
  end;

begin
  lbDLLList.Clear;
 {-Clear module list if not empty}
  for i := 0 to Pred(ModuleCount) do
    FreeMem(ModuleList[i], SizeOf(TModuleEntry));
  ModuleCount := 0;
  E.dwSize := SizeOf(TModuleEntry);
 {-Enumerate modules}
  if ModuleFirst(@E) then begin
    AddEntry;
    while ModuleNext(@E) do
      AddEntry;
  end;
 {-Display count}
  Str(ModuleCount, CountStr);
  Header.Caption := CountStr + ' ' + Header.Caption;
end;

procedure TKill.btnRefreshClick(Sender: TObject);

begin
  LoadListBox;
end;

procedure TKill.btnUnloadClick(Sender: TObject);

const
 MAXSEL = 100;

var
  h        : THandle;
  Msg      : string;
  index    : integer;
  bOk      : boolean;

begin
  if lbDLLList.SelCount <> 0 then for index := 0 to Pred(lbDLLList.Items.Count) do begin
    if not lbDLLList.Selected[index] then
   {-Module not selected. Skip}
      continue;
    bOk := false;
    if Confirm.Checked then begin {-Prompt user}
      Msg := 'Are you really sure that you want to unload '
            + StrPas(ModuleList[index]^.szModule)
            + '?';
      bOk := MessageDlg(Msg, mtConfirmation, mbOkCancel, 0) = mrOK;
    end;
    if (not Confirm.Checked) or bOk then begin
      h := ModuleList[index]^.hModule;
      repeat
        FreeLibrary(h);
      until GetModuleUsage(h) = 0;
    end;
  end;
  LoadListbox; {-Refresh list}
end;

procedure TKill.btnExitClick(Sender: TObject);

begin
  Application.Terminate;
 {-The module list will freed in FormDestroy}
end;

procedure TKill.lbDLLListClick(Sender: TObject);
{-The Unload button is not enabled until a module
  has been selected}
begin
  btnUnLoad.Enabled := true;
end;

end.
