unit Dynarray;

interface

uses
  SysUtils, WinTypes, WinProcs, Messages, Classes, Graphics, Controls,
  Forms, Dialogs, ExtCtrls;

const
  ALLOCATE_FAILED = 'Memory Allocation Failed';
  OUT_OF_BOUNDS = 'Array Reference Out of Bounds';

type

  EAllocateFailed = class( Exception );
  EOutOfBounds = class( Exception );

  Plongint = ^longint;

  TDynamicArray = class( TObject )
  private
     stream: TMemoryStream;
     nAllocated: word;
     nSize1, nSize2: word;
  protected
     procedure SetValue( n: word; nVal: longint );
     function GetValue( n: word ): longint;
     procedure SetValue2D( n1, n2: word; nVal: longint );
     function GetValue2D( n1, n2: word ): longint;
  public
     constructor Create( n: word );
     constructor Create2D( n1, n2: word );
     destructor Destroy; override;
     procedure Dim( n: word );
     procedure Dim2D( n1, n2: word );
     procedure ReDim( n: word );
     property Value[n: word]: longint read GetValue write SetValue;
     property Value2D[n1, n2: word]: longint read GetValue2D write SetValue2D;
  end;

implementation

(*************************************************************
Construction/destruction.
*************************************************************)
constructor TDynamicArray.Create( n: word );
begin
  stream := TMemoryStream.Create;
  Dim( n );
end;

constructor TDynamicArray.Create2D( n1, n2: word );
begin
  nSize1 := n1;
  nSize2 := n2;
  Create( n1 * n2 );
end;

destructor TDynamicArray.Destroy;
begin
  stream.Free;
  inherited Destroy;
end;

(*************************************************************
Plumbing.
*************************************************************)
procedure TDynamicArray.SetValue( n: word; nVal: longint );
var
  pVal: PLongint;
  pValChar: PChar;
begin
  if n > nAllocated then
     raise EOutOfBounds.Create( OUT_OF_BOUNDS )
  else
     begin
        pValChar := stream.Memory;
        pValChar := pValChar + ( ( n - 1 ) * SizeOf( longint ) );
        pVal := PLongint( pValChar );
        pVal^ := nVal;
     end;
end;

function TDynamicArray.GetValue( n: word ): longint;
var
  pVal: PLongint;
  pValChar: PChar;
begin
  if n > nAllocated then
     raise EOutOfBounds.Create( OUT_OF_BOUNDS )
  else
     begin
        pValChar := stream.Memory;
        pValChar := pValChar + ( ( n - 1 ) * SizeOf( longint ) );
        pVal := PLongint( pValChar );
        Result := pVal^;
     end;
end;

procedure TDynamicArray.SetValue2D( n1, n2: word; nVal: longint );
begin
  SetValue( ( n1 - 1 ) * nSize2 + n2, nVal );
end;

function TDynamicArray.GetValue2D( n1, n2: word ): longint;
begin
  Result := GetValue( ( n1 - 1 ) * nSize2 + n2 );
end;

(*************************************************************
Dimension the array.
*************************************************************)
procedure TDynamicArray.Dim( n: word );
var
  i: word;
begin
try
  stream.SetSize( n * SizeOf( longint ) );
  nAllocated := n;
  for i := 1 to n do
     Value[i] := 0;
except
  raise EAllocateFailed.Create( ALLOCATE_FAILED );
  nAllocated := 0;
end;
end;

procedure TDynamicArray.Dim2D( n1, n2: word );
begin
  nSize1 := n1;
  nSize2 := n2;
  Dim( n1 * n2 );
end;

(*************************************************************
Re-dimension and keep old values in tact.
*************************************************************)
procedure TDynamicArray.ReDim( n: word );
var
  streamTemp: TMemoryStream;
  nMove: word;
  pTemp, pStream: PChar;
  i: integer;
begin
  if nAllocated = 0 then
     Dim( n )
  else
     begin
        streamTemp := TMemoryStream.Create;
try
        streamTemp.SetSize( nAllocated * SizeOf( longint ) );
        pTemp := streamTemp.Memory;
        pStream := stream.Memory;
        System.Move( pStream[0], pTemp[0], nAllocated * SizeOf( longint ) );
        stream.SetSize( n );
        if n > 0 then
           begin
              if n > nAllocated then
                 nMove := nAllocated
              else
                 nMove := n;
              System.Move( pTemp[0], pStream[0], nMove * SizeOf( longint ) );
           end;
        nAllocated := n;
except
        raise EAllocateFailed.Create( ALLOCATE_FAILED );
end;
        streamTemp.Free;
     end;
end;

end.
