{*******************************************************}
{                                                       }
{       Intelligent DB Navigator                        }
{                                                       }
{       Copyright (c) 1995 Rohit Gupta                  }
{                                                       }
{*******************************************************}

unit RgNav;

interface

uses SysUtils, WinTypes, WinProcs, Messages, Classes, Controls, Forms,
  Graphics, Menus, StdCtrls, ExtCtrls, DB, DBTables, Mask, Buttons, DBConsts,
  DBCtrls, RgUseful;

const
  InitRepeatPause = 400;  { pause before repeat timer (ms) }
  RepeatPause     = 100;  { pause before hint window displays (ms)}
  SpaceSize       =  5;   { size of space between special buttons }

  SHntOfs         = 60000;
  SCapOfs         = 60100;
  SKeyOfs         = 60200;
  SInsertQuestion = 60310;
  SDeleteQuestion = 60311;
  SEditQuestion   = 60312;
  SPostQuestion   = 60317;
  SCancelQuestion = 60318;

type
  TNavButton = class;
  TNavDataLink = class;

  TNavGlyph   = (ngEnabled, ngDisabled);
  TAllNavBtn  = (nbFirst,  nbPrior,  nbNext,  nbLast,   nbKey,
                 nbSearch, nbTable,  nbForm,  nbPrint,  nbRefresh,
                 nbInsert, nbDelete, nbEdit,  nbExtra1, nbExtra2,
                 nbExtra3, nbHint,   nbPost,  nbCancel);
  TNormNavBtn = nbFirst..nbExtra3;
  TEditNavBtn = nbHint..nbCancel;
  TNavColors  = (ncBlack, ncBlue, ncRed);
  TBtnSize    = (X1,X2,X3,X4,X5);

  TAllBtnSet   = set of TAllNavBtn;
  TNormBtnSet  = set of TNormNavBtn;
  TNavBtnStyle = set of (nsAllowTimer, nsFocusRect);

  ENavClick = procedure (Sender: TObject; Button: TAllNavBtn) of object;

{ TRGNavigator }

  TRGNavigator = class (TCustomPanel)
  private
    FDataLink: TNavDataLink;
    FVisibleButtons: TAllBtnSet;
    VisibleCopy:     TAllBtnSet;
    FHints: TStrings;
    EditBtnWidth,
    ButtonWidth: Integer;
    MinBtnSize: TPoint;
    FOnNavClick: ENavClick;
    FocusedButton: TAllNavBtn;
    FConfirmDelete,
    FConfirmInsert,
    FConfirmEdit,
    FConfirmPost,
    FConfirmCancel: Boolean;
    FMenu: TMenuITem;

    FScrlColor,
    FFuncColor,
    FCtrlColor,
    FToolColor: TNavColors;

    FKeySize,
    FEx1Size,
    FEx2Size,
    FEx3Size:   TBtnSize;

    IndexList:  TStringList;
    KeyNumber:  Integer;

    function  GetDataSource: TDataSource;
    procedure SetDataSource(Value: TDataSource);

    procedure SetButtonColor (I : TAllNavBtn);
    procedure SetScrlColor (Value : TNavColors);
    procedure SetFuncColor (Value : TNavColors);
    procedure SetCtrlColor (Value : TNavColors);
    procedure SetToolColor (Value : TNavColors);

    procedure SetBtnSize   (var Target, Value : TBtnSize);
    procedure SetKeySize   (Value : TBtnSize);
    procedure SetEx1Size   (Value : TBtnSize);
    procedure SetEx2Size   (Value : TBtnSize);
    procedure SetEx3Size   (Value : TBtnSize);

    function  GetCapt1 : string;
    function  GetCapt2 : string;
    function  GetCapt3 : string;
    procedure SetCaption (Idx : TAllNavBtn; Value : string);
    procedure SetCapt1 (Value : string);
    procedure SetCapt2 (Value : string);
    procedure SetCapt3 (Value : string);

    procedure SetMenu (Value : TMenuItem);

    procedure InitButtons;
    procedure InitHints;
    procedure Click(Sender: TObject);
    procedure BtnMouseDown (Sender: TObject; Button: TMouseButton;
                            Shift: TShiftState; X, Y: Integer);
    function  GetVisible : TNormBtnSet;
    procedure SetVisible(Value: TNormBtnSet);
    procedure AdjustSize (var W: Integer; var H: Integer);
    procedure SetHints(Value: TStrings);
    procedure WMSize(var Message: TWMSize);  message WM_SIZE;
    procedure WMSetFocus(var Message: TWMSetFocus); message WM_SETFOCUS;
    procedure WMKillFocus(var Message: TWMKillFocus); message WM_KILLFOCUS;
    procedure WMGetDlgCode(var Message: TWMGetDlgCode); message WM_GETDLGCODE;
    procedure CMEnabledChanged(var Message: TMessage); message CM_ENABLEDCHANGED;
    procedure SetKeyCaption;
  protected
    procedure DataChanged;
    procedure EditingChanged;
    procedure ActiveChanged;
    procedure Loaded; override;
    procedure KeyDown(var Key: Word; Shift: TShiftState); override;
    procedure Notification(AComponent: TComponent; Operation: TOperation); override;
  public
    Buttons: array[TAllNavBtn] of TNavButton;
    constructor Create(AOwner: TComponent); override;
    destructor Destroy; override;
    procedure SetBounds(ALeft, ATop, AWidth, AHeight: Integer); override;
    procedure BtnClick(Index: TAllNavBtn);
  published
    property DataSource: TDataSource read GetDataSource write SetDataSource;
    property VisibleButtons: TNormBtnSet read GetVisible write SetVisible
             default [nbFirst,  nbPrior,  nbNext,   nbLast,   nbKey,
                      nbSearch, nbTable,  {nbForm,} nbPrint,  nbRefresh,
                      nbInsert, nbDelete, nbEdit,   nbExtra1, nbExtra2, nbExtra3];
    property Align;
    property DragCursor;
    property DragMode;
    property Enabled;
    property Ctl3D;
    property Hints: TStrings read FHints write SetHints;
    property ParentCtl3D;
    property ParentShowHint;
    property PopupMenu;
    property ConfirmDelete: Boolean read FConfirmDelete write FConfirmDelete default True;
    property ConfirmInsert: Boolean read FConfirmInsert write FConfirmInsert default True;
    property ConfirmEdit:   Boolean read FConfirmEdit   write FConfirmEdit   default True;
    property ConfirmPost:   Boolean read FConfirmPost   write FConfirmPost   default True;
    property ConfirmCancel: Boolean read FConfirmCancel write FConfirmCancel default True;
    property ShowHint;
    property TabOrder;
    property TabStop;
    property Visible;
    property OnClick: ENavClick read FOnNavClick write FOnNavClick;
    property OnDblClick;
    property OnDragDrop;
    property OnDragOver;
    property OnEndDrag;
    property OnEnter;
    property OnExit;
    property OnResize;
    property ColorScroll:   TNavColors read FScrlColor write SetScrlColor;
    property ColorFunc:     TNavColors read FFuncColor write SetFuncColor;
    property ColorCtrl:     TNavColors read FCtrlColor write SetCtrlColor;
    property ColorTool:     TNavColors read FToolColor write SetToolColor;
    property SizeOfKey:     TBtnSize   read FKeySize   write SetKeySize default X3;
    property SizeOfExtra1:  TBtnSize   read FEx1Size   write SetEx1Size default X3;
    property SizeOfExtra2:  TBtnSize   read FEx2Size   write SetEx2Size default X3;
    property SizeOfExtra3:  TBtnSize   read FEx3Size   write SetEx3Size default X3;
    property Menu:          TMenuItem  read FMenu      write SetMenu;
    property CaptionExtra1: string     read GetCapt1   write SetCapt1;
    property CaptionExtra2: string     read GetCapt2   write SetCapt2;
    property CaptionExtra3: string     read GetCapt3   write SetCapt3;
  end;

{ TNavButton }

  TNavButton = class(TSpeedButton)
  private
    FIndex: TAllNavBtn;
    FNavStyle: TNavBtnStyle;
    FRepeatTimer: TTimer;
    procedure TimerExpired(Sender: TObject);
  protected
    procedure Paint; override;
    procedure MouseDown(Button: TMouseButton; Shift: TShiftState;
      X, Y: Integer); override;
    procedure MouseUp(Button: TMouseButton; Shift: TShiftState;
      X, Y: Integer); override;
  public
    destructor Destroy; override;
    property NavStyle: TNavBtnStyle read FNavStyle write FNavStyle;
    property Index : TAllNavBtn read FIndex write FIndex;
  end;

{ TNavDataLink }

  TNavDataLink = class(TDataLink)
  private
    FNavigator: TRGNavigator;
  protected
    procedure EditingChanged; override;
    procedure DataSetChanged; override;
    procedure ActiveChanged; override;
  public
    constructor Create(ANav: TRGNavigator);
    destructor Destroy; override;
  end;

procedure RegisterRGNavigator;

implementation

uses Dialogs;

{$R RGNAV}

{ TRGNavigator }

const
  BtnStateName : array[TNavGlyph] of PChar = ('EN', 'DI');

  BtnTypeName  : array[TAllNavBtn] of string[10]
               = ('First', 'Prior',  'Next',   'Last',    'Key',    'Search',
                  'Table', 'Form',   'Print',  'Refresh', 'Insert', 'Delete',
                  'Edit',  'Extra1', 'Extra2', 'Extra3',  'Hint',   'Post',
                  'Cancel');

constructor TRGNavigator.Create(AOwner: TComponent);
begin
  inherited Create(AOwner);
  ControlStyle := ControlStyle - [csAcceptsControls, csSetCaption] + [csFramed, csOpaque];
  FDataLink := TNavDataLink.Create(Self);
  FVisibleButtons := [nbFirst,  nbPrior,  nbNext,   nbLast,    nbKey,
                      nbSearch, nbTable, {nbForm,}  nbPrint,  nbRefresh,
                      nbInsert, nbDelete, nbEdit,   nbExtra1, nbExtra2,
                      nbExtra3];
  FKeySize := X4;
  FEx1Size := X3;
  FEx2Size := X3;
  FEx3Size := X3;
  FHints   := TStringList.Create;
  InitButtons;
  BevelOuter := bvNone;
  BevelInner := bvNone;
  Width      := 241;
  Height     := 25;
  ButtonWidth    := 0;
  FocusedButton  := nbFirst;
  FConfirmDelete := True;
  FConfirmInsert := True;
  FConfirmEdit   := True;
  FConfirmPost   := True;
  FConfirmCancel := True;
  IndexList      := TStringList.Create;
end;

destructor TRGNavigator.Destroy;
begin
  FDataLink.Free;
  FDataLink := nil;
  inherited Destroy;
end;

procedure TRGNavigator.SetButtonColor (I : TAllNavBtn);
  procedure SetIt (Col : TNavColors);
  begin
    with Buttons[I].Font
    do case Col of
            ncBlue : Color := clBlue;
            ncRed  : Color := clRed;
            else     Color := clBlack;
       end;
  end;

var
  Cl : char;
  ResName: array[0..40] of Char;
begin
  case I of
       nbFirst,nbPrior,
       nbNext,nbLast,nbKey       : begin
                                     Cl := char(ord(FScrlColor)+$30);
                                     SetIt (FScrlColor);
                                   end;
       nbRefresh,nbSearch,
       nbTable,nbForm,nbPrint    : begin
                                     Cl := char(ord(FFuncColor)+$30);
                                     SetIt (FFuncColor);
                                   end;
       nbInsert,nbDelete,
       nbEdit,nbCancel,nbPost    : begin
                                     Cl := char(ord(FCtrlColor)+$30);
                                     SetIt (FCtrlColor);
                                   end;
       nbExtra1,nbExtra2,
       nbExtra3,nbHint           : begin
                                     Cl := char(ord(FToolColor)+$30);
                                     SetIt (FToolColor);
                                   end;
       else                        Cl := '0';
  end;
  Buttons[I].Glyph.Handle := LoadBitmap(HInstance,
                          StrFmt(ResName, '%srgn_%s', [Cl,BtnTypeName[I]]));
end;

procedure TRGNavigator.SetScrlColor (Value : TNavColors);
begin
  if Value = FScrlColor
  then exit;
  FScrlColor := Value;
  SetButtonColor (nbFirst);
  SetButtonColor (nbPrior);
  SetButtonColor (nbNext);
  SetButtonColor (nbLast);
  SetButtonColor (nbKey);
end;

procedure TRGNavigator.SetFuncColor (Value : TNavColors);
begin
  if Value = FFuncColor
  then exit;
  FFuncColor := Value;
  SetButtonColor (nbSearch);
  SetButtonColor (nbTable);
  SetButtonColor (nbForm);
  SetButtonColor (nbPrint);
  SetButtonColor (nbRefresh);
end;

procedure TRGNavigator.SetCtrlColor (Value : TNavColors);
begin
  if Value = FCtrlColor
  then exit;
  FCtrlColor := Value;
  SetButtonColor (nbInsert);
  SetButtonColor (nbDelete);
  SetButtonColor (nbEdit);
  SetButtonColor (nbCancel);
  SetButtonColor (nbPost);
end;

procedure TRGNavigator.SetToolColor (Value : TNavColors);
begin
  if Value = FToolColor
  then exit;
  FToolColor := Value;
  SetButtonColor (nbExtra1);
  SetButtonColor (nbExtra2);
  SetButtonColor (nbExtra3);
  SetButtonColor (nbHint);
end;

procedure TRGNavigator.SetBtnSize (var Target, Value : TBtnSize);
var
  W, H : Integer;
begin
  if Value = Target
  then exit;
  Target := Value;
  W := Width;
  H := Height;
  AdjustSize (W,H);
end;

procedure TRGNavigator.SetKeySize (Value : TBtnSize);
begin SetBtnSize (FKeySize,Value); end;

procedure TRGNavigator.SetEx1Size (Value : TBtnSize);
begin SetBtnSize (FEx1Size,Value); end;

procedure TRGNavigator.SetEx2Size (Value : TBtnSize);
begin SetBtnSize (FEx2Size,Value); end;

procedure TRGNavigator.SetEx3Size (Value : TBtnSize);
begin SetBtnSize (FEx3Size,Value); end;

function TRGNavigator.GetCapt1 : string;
begin GetCapt1 := Buttons [nbExtra1].Caption; end;

function TRGNavigator.GetCapt2 : string;
begin GetCapt2 := Buttons [nbExtra2].Caption; end;

function TRGNavigator.GetCapt3 : string;
begin GetCapt3 := Buttons [nbExtra3].Caption; end;

procedure TRGNavigator.SetCaption (Idx : TAllNavBtn; Value : string);
var
  I : Integer;
begin
  if Buttons [Idx].Caption = Value
  then Exit;
  Buttons [Idx].Caption := Value;
  if FMenu = nil
  then exit;
  with FMenu
  do for I := 0 to Count-1
     do if Items [I].Tag = ord(Idx)
        then begin
             Items [I].Caption := Value;
             exit;
        end;
end;

procedure TRGNavigator.SetCapt1 (Value : string);
begin SetCaption (nbExtra1,Value); end;

procedure TRGNavigator.SetCapt2 (Value : string);
begin SetCaption (nbExtra2,Value); end;

procedure TRGNavigator.SetCapt3 (Value : string);
begin SetCaption (nbExtra3,Value); end;

procedure TRGNavigator.SetMenu (Value : TMenuItem);
  procedure InsertIt (const Nam, Cap, Key, Hnt : string; Tg : integer);
  var
    Item : TMenuItem;
  begin
    Item := TMenuITem.Create (FMenu.Owner);
    if not Assigned (Item)
    then exit;
    with Item
    do begin
       Name     := Nam+'Menu';
       Caption  := Cap;
       ShortCut := TextToShortCut(Key);
       Hint     := Hnt;
       OnClick  := Self.Click;
       Tag      := Tg;
    end;
    FMenu.Add (Item);
  end;

  procedure SetOnClick;  { Delphi loses above Onclick }
  var
    I : Integer;
  begin
    with FMenu
    do for I := 0 to Count-1
       do with Items [I]
          do if Caption <> '-'
             then OnClick := Self.Click;
  end;

var
  I : TAllNavBtn;
  J : Integer;
  S : String;
begin
  if Value = FMenu
  then exit;
  FMenu := Value;
  if not assigned (FMenu)
  then exit;

  if not (csDesigning in ComponentState)
  then begin
       SetOnClick;
       Exit;
  end;

  with FMenu      { Check FirstMenu, if exists, then abort }
  do if Count > 0
  then begin
       S := BtnTypeName [nbFirst] + 'Menu';
       for J := 0 to Count-1
       do if Items [J].Name = S
          then exit;
  end;

  for I := Low(TAllNavBtn) to High(TAllNavBtn)
  do if Buttons [I].Enabled
     and (I <> nbHint)
     then begin
          case I of
               nbSearch,
               nbInsert,
               nbExtra1,
               nbPost : InsertIt (BtnTypeName[I]+'_','-','','',0);
          end;
          case I of
               nbExtra1,
               nbExtra2,
               nbExtra3 : S := Buttons [I].Caption;
               else       S := LoadStr (SCapOfs+ord(I));

          end;
          InsertIt (BtnTypeName[I],S,LoadStr (SKeyOfs+ord(I)),Buttons[I].Hint,ord(I));
     end;
end;

procedure TRGNavigator.InitButtons;
var
  I: TAllNavBtn;
  Btn: TNavButton;
  X: Integer;
begin
  MinBtnSize := Point(20, 18);
  X := 0;
  for I := Low(Buttons) to High(Buttons) do
  begin
    Btn := TNavButton.Create (Self);
    Btn.Index := I;
    Btn.Visible := I in FVisibleButtons;
    Btn.Enabled := True;
    Btn.SetBounds (X, 0, MinBtnSize.X, MinBtnSize.Y);
    Btn.NumGlyphs := 2;
    Btn.OnClick := Click;
    Btn.OnMouseDown := BtnMouseDown;
    Btn.Parent := Self;
    Buttons[I] := Btn;
    SetButtonColor (I);
    X := X + MinBtnSize.X;
    case I of
         nbHint   : Buttons [I].Margin := 0;
         nbKey,
         nbExtra1,
         nbExtra2,
         nbExtra3 : begin
                      if I = nbKey
                      then Buttons[I].Caption := 'Primary'
                      else Buttons[I].Caption := BtnTypeName [I];
                      Buttons[I].Margin  := 3;
                    end;
    end;
  end;
  InitHints;
  Buttons[nbPrior].NavStyle := Buttons[nbPrior].NavStyle + [nsAllowTimer];
  Buttons[nbNext]. NavStyle := Buttons[nbNext]. NavStyle + [nsAllowTimer];
end;

procedure TRGNavigator.InitHints;
var
  I: Integer;
  J: TAllNavBtn;
begin
  for J := Low(Buttons) to High(Buttons)
  do Buttons[J].Hint := LoadStr (SHntOfs+ord(J));
  J := Low(Buttons);
  for I := 0 to (FHints.Count - 1) do
  begin
    if FHints.Strings[I] <> ''
    then Buttons[J].Hint := FHints.Strings[I];
    if J = High(Buttons)
    then Exit;
    Inc(J);
  end;
end;

procedure TRGNavigator.SetHints(Value: TStrings);
begin
  FHints.Assign(Value);
  InitHints;
end;

procedure TRGNavigator.Notification(AComponent: TComponent;
  Operation: TOperation);
begin
  inherited Notification(AComponent, Operation);
  if (Operation = opRemove) and (FDataLink <> nil) and
    (AComponent = DataSource) then DataSource := nil;
end;

function TRGNavigator.GetVisible : TNormBtnSet;
begin
  Result := FVisibleButtons * [low(TNormNavBtn)..high(TNormNavBtn)];
end;

procedure TRGNavigator.SetVisible(Value: TNormBtnSet);
var
  I: TAllNavBtn;
  W, H: Integer;
begin
  W := Width;
  H := Height;
  if (nbTable in FVisibleButtons) and (nbForm in Value)
  then Value := Value - [nbTable];
  if (nbForm in FVisibleButtons) and (nbTable in Value)
  then Value := Value - [nbForm];
  FVisibleButtons := Value;
  for I := Low(Buttons) to High(Buttons)
  do begin
     Buttons[I].Visible := I in FVisibleButtons;
     if assigned (FMenu)
     then FMenu.Items [ord(I)].Enabled := Buttons[I].Visible;
  end;
  AdjustSize (W, H);
  if (W <> Width) or (H <> Height)
  then inherited SetBounds (Left, Top, W, H);
  Invalidate;
end;

procedure TRGNavigator.AdjustSize (var W: Integer; var H: Integer);
var
  HintSize : Integer;
  function GetButtonSIze (I : TAllNavBtn) : Integer;
  begin
      case I of
           nbKey    : Result := ord(FKeySize);
           nbExtra1 : Result := ord(FEx1Size);
           nbExtra2 : Result := ord(FEx2Size);
           nbExtra3 : Result := ord(FEx3Size);
           nbHint   : if FDataLink.Editing
                      then begin
                           if HintSize <> 0
                           then Result := HintSize
                           else begin
                                Result := ((W-EditBtnWidth*2) div EditBtnWidth)-2;
                                HintSize := Result;  { For Later }
                           end;
                      end
                      else Result := 20;
           else       Result := 0;
      end;
      Inc (Result);
  end;

var
  Count: Integer;
  MinW: Integer;
  I: TAllNavBtn;
  LastBtn: TAllNavBtn;
  BWidth,
  Space, Temp, Remain,
  X, VisibleBtns,
  Extra : Integer;

begin
  if (csLoading in ComponentState) then Exit;
  if Buttons[nbFirst] = nil then Exit;

  HintSize := 0;
  Count := 0;
  VisibleBtns := 0;
  LastBtn := High(Buttons);
  for I := Low(Buttons) to High(Buttons) do
  begin
    if Buttons[I].Visible then
    begin
      Inc (Count,GetButtonSize(I));
      Inc (VisibleBtns);
      LastBtn := I;
    end;
  end;
  if Count = 0 then Inc(Count);

  MinW := Count * (MinBtnSize.X - 1){ + 1};
  if W < MinW
  then W := MinW;
  if H < MinBtnSize.Y
  then H := MinBtnSize.Y;

  ButtonWidth := ((W - 1) div Count);
  Temp := (Count * ButtonWidth){ + 1};   { Space Required }
  Extra := 0;
  while W-Temp > VisibleBtns           { If more than # butons }
  do begin                             { Then distribute it }
     Inc (Extra);
     Inc (Temp,VisibleBtns);
  end;
(*
buttons[nbFirst].Caption := inttostr (W);
buttons[nbLast].Caption := inttostr (MinW);
buttons[nbKey].Caption := inttostr (Temp);
*)
(*
  if Align = alNone   { Align if Required }  not really needed
  then W := Temp;
*)
  X := 0;
  Remain := W - Temp;
  Temp := Count div 2;
  for I := Low(Buttons) to High(Buttons) do
  begin
    if Buttons[I].Visible
    then begin
         Space := 0;
         if Remain <> 0
         then begin
              Dec (Remain);
              Space := 1;
         end;
         BWidth := ButtonWidth * GetButtonSize (I) + Extra;
         Buttons[I].SetBounds (X, 0, BWidth + Space, Height);
         Inc (X, BWidth + Space);
         LastBtn := I;
    end
    else Buttons[I].SetBounds (Width + 1, 0, ButtonWidth, Height);
  end;
  if not FDataLink.Editing
  then EditBtnWidth := ButtonWidth;
end;

procedure TRGNavigator.SetBounds(ALeft, ATop, AWidth, AHeight: Integer);
var
  W, H: Integer;
begin
  W := AWidth;
  H := AHeight;
  AdjustSize (W, H);
  inherited SetBounds (ALeft, ATop, W, H);
end;

procedure TRGNavigator.WMSize(var Message: TWMSize);
var
  W, H: Integer;
begin
  inherited;

  { check for minimum size }
  W := Width;
  H := Height;
  AdjustSize (W, H);
  if (W <> Width) or (H <> Height) then
    inherited SetBounds(Left, Top, W, H);
  Message.Result := 0;
end;

procedure TRGNavigator.Click(Sender: TObject);
begin
  if Sender is TNavButton
  then BtnClick (TNavButton (Sender).Index)
  else BtnClick (TAllNavBtn(TMenuItem(Sender).Tag));
end;

procedure TRGNavigator.BtnMouseDown(Sender: TObject; Button: TMouseButton;
  Shift: TShiftState; X, Y: Integer);
var
  OldFocus: TAllNavBtn;
  Form: TForm;
begin
  OldFocus := FocusedButton;
  FocusedButton := TNavButton (Sender).Index;
  if TabStop and (GetFocus <> Handle) and CanFocus then
  begin
    SetFocus;
    if (GetFocus <> Handle) then
      Exit;
  end
  else if TabStop and (GetFocus = Handle) and (OldFocus <> FocusedButton) then
  begin
    Buttons[OldFocus].Invalidate;
    Buttons[FocusedButton].Invalidate;
  end;
end;

procedure TRGNavigator.BtnClick(Index: TAllNavBtn);
begin
  if (DataSource <> nil) and (DataSource.State <> dsInactive) then
  begin
    with DataSource.DataSet do
    begin
      case Index of
        nbPrior  : Prior;
        nbNext   : Next;
        nbFirst  : First;
        nbLast   : Last;
        nbRefresh: Refresh;
        nbInsert : if not FConfirmInsert or
                     (MessageDlg (LoadStr(SInsertQuestion),
                      mtConfirmation, mbOKCancel, 0) <> idCancel)
                   then Insert;
        nbDelete : if not FConfirmDelete or
                     (MessageDlg (LoadStr(SDeleteQuestion),
                      mtConfirmation, mbOKCancel, 0) <> idCancel)
                   then Delete;
        nbEdit   : if not FConfirmEdit or
                     (MessageDlg (LoadStr(SEditQuestion),
                      mtConfirmation, mbOKCancel, 0) <> idCancel)
                   then Edit;
        nbCancel : if not FConfirmCancel or
                     (MessageDlg (LoadStr(SCancelQuestion),
                      mtConfirmation, [mbYes,mbNo], 0) = idYes)
                   then Cancel;
        nbPost   : if not FConfirmPost or
                     (MessageDlg (LoadStr(SPostQuestion),
                      mtConfirmation, mbOKCancel, 0) <> idCancel)
                   then Post;
        nbKey    : begin
                     if KeyNumber < IndexList.Count-1
                     then inc (KeyNumber)
                     else KeyNumber := 0;
                     SetKeyCaption;
                   end;
      end;
    end;
  end;
  if (not (csDesigning in ComponentState))
  and Assigned(FOnNavClick)
  then FOnNavClick(Self, Index);
end;

procedure TRGNavigator.WMSetFocus(var Message: TWMSetFocus);
begin
  Buttons[FocusedButton].Invalidate;
end;

procedure TRGNavigator.WMKillFocus(var Message: TWMKillFocus);
begin
  Buttons[FocusedButton].Invalidate;
end;

procedure TRGNavigator.KeyDown(var Key: Word; Shift: TShiftState);
var
  NewFocus: TAllNavBtn;
  OldFocus: TAllNavBtn;
begin
  OldFocus := FocusedButton;
  case Key of
    VK_RIGHT:
      begin
        NewFocus := FocusedButton;
        repeat
          if NewFocus < High(Buttons) then
            NewFocus := Succ(NewFocus);
        until (NewFocus = High(Buttons)) or (Buttons[NewFocus].Visible);
        if NewFocus <> FocusedButton then
        begin
          FocusedButton := NewFocus;
          Buttons[OldFocus].Invalidate;
          Buttons[FocusedButton].Invalidate;
        end;
      end;
    VK_LEFT:
      begin
        NewFocus := FocusedButton;
        repeat
          if NewFocus > Low(Buttons) then
            NewFocus := Pred(NewFocus);
        until (NewFocus = Low(Buttons)) or (Buttons[NewFocus].Visible);
        if NewFocus <> FocusedButton then
        begin
          FocusedButton := NewFocus;
          Buttons[OldFocus].Invalidate;
          Buttons[FocusedButton].Invalidate;
        end;
      end;
    VK_SPACE:
      begin
        if Buttons[FocusedButton].Enabled then
          Buttons[FocusedButton].Click;
      end;
  end;
end;

procedure TRGNavigator.WMGetDlgCode(var Message: TWMGetDlgCode);
begin
  Message.Result := DLGC_WANTARROWS;
end;

procedure TRGNavigator.DataChanged;
var
test,
  UpEnable, DnEnable: Boolean;
begin
  UpEnable := Enabled and FDataLink.Active and not FDataLink.DataSet.BOF;
  DnEnable := Enabled and FDataLink.Active and not FDataLink.DataSet.EOF;
  Buttons[nbFirst].Enabled := UpEnable;
  Buttons[nbPrior].Enabled := UpEnable;
  Buttons[nbNext].Enabled := DnEnable;
  Buttons[nbLast].Enabled := DnEnable;
  Buttons[nbDelete].Enabled := Enabled and FDataLink.Active and
    FDataLink.DataSet.CanModify and
    not (FDataLink.DataSet.BOF and FDataLink.DataSet.EOF);
end;

procedure TRGNavigator.EditingChanged;
var
  CanModify: Boolean;
  I        : TAllNavBtn;
  W, H     : Integer;
begin
  CanModify := Enabled and FDataLink.Active and FDataLink.DataSet.CanModify;
  if CanModify
  then if FDataLink.Editing
       then begin
            VisibleCopy := FVisibleButtons;
            FVisibleButtons := [LOW(TEditNavBtn)..HIGH(TEditNavBtn)];
            W := Width;
            H := Height;
            for I := LOW(TNormNavBtn) to HIGH(TNormNavBtn)
            do begin
               Buttons[I].Visible := false;
               if assigned (FMenu)
               then FMenu.Items [ord(I)].Enabled := false;
            end;
            for I := LOW(TEditNavBtn) to HIGH(TEditNavBtn)
            do begin
               Buttons[I].Visible := true;
               if assigned (FMenu)
               then FMenu.Items [ord(I)].Enabled := true;
            end;
            AdjustSize (W, H);
            if (W <> Width) or (H <> Height)
            then inherited SetBounds (Left, Top, W, H);
            Invalidate;
       end
       else begin
            SetVisible (VisibleCopy);
       end;
(*
  Buttons[nbInsert].Enabled := CanModify;
  Buttons[nbEdit].Enabled := CanModify and not FDataLink.Editing;
  Buttons[nbPost].Enabled := CanModify and FDataLink.Editing;
  Buttons[nbCancel].Enabled := CanModify and FDataLink.Editing;
  Buttons[nbRefresh].Enabled := not (FDataLink.DataSet is TQuery);
*)
end;

procedure TRGNavigator.ActiveChanged;
var
  I: TAllNavBtn;
begin
(*
  if not (Enabled and FDataLink.Active) then
    for I := Low(Buttons) to High(Buttons) do
      Buttons[I].Enabled := False
  else
  begin
    DataChanged;
    EditingChanged;
  end;
*)
end;

procedure TRGNavigator.CMEnabledChanged(var Message: TMessage);
begin
  inherited;
  if not (csLoading in ComponentState) then
    ActiveChanged;
end;

procedure TRGNavigator.SetDataSource(Value: TDataSource);
begin
  FDataLink.DataSource := Value;
  if not (csLoading in ComponentState)
  then ActiveChanged;
end;

function TRGNavigator.GetDataSource: TDataSource;
begin
  Result := FDataLink.DataSource;
end;

procedure TRGNavigator.SetKeyCaption;
begin
  with IndexList
  do if (Count > 0) and (KeyNumber < Count)
     then begin
          Buttons [nbKey].Caption := WordCase (IndexList.Strings[KeyNumber]);
          TTable(FDataLink.DataSet).IndexName := IndexList.Strings[KeyNumber];
     end;
end;

procedure TRGNavigator.Loaded;
var
  W, H: Integer;
begin
  inherited Loaded;
  W := Width;
  H := Height;
  AdjustSize (W, H);
  if (W <> Width) or (H <> Height) then
    inherited SetBounds (Left, Top, W, H);
  InitHints;

  if (assigned (FDataLink))  { Get Index Names }
  then begin
    with FDataLink
    do if DataSet is TTable
    then with DataSet as TTable
         do begin
            GetIndexNames (IndexList);
            SetKeyCaption;
         end;
  end;

  ActiveChanged;
end;

{TNavButton}

destructor TNavButton.Destroy;
begin
  if FRepeatTimer <> nil then
    FRepeatTimer.Free;
  inherited Destroy;
end;

procedure TNavButton.MouseDown(Button: TMouseButton; Shift: TShiftState;
  X, Y: Integer);
begin
  inherited MouseDown (Button, Shift, X, Y);
  if nsAllowTimer in FNavStyle then
  begin
    if FRepeatTimer = nil then
      FRepeatTimer := TTimer.Create(Self);

    FRepeatTimer.OnTimer := TimerExpired;
    FRepeatTimer.Interval := InitRepeatPause;
    FRepeatTimer.Enabled  := True;
  end;
end;

procedure TNavButton.MouseUp(Button: TMouseButton; Shift: TShiftState;
                                  X, Y: Integer);
begin
  inherited MouseUp (Button, Shift, X, Y);
  if FRepeatTimer <> nil then
    FRepeatTimer.Enabled  := False;
end;

procedure TNavButton.TimerExpired(Sender: TObject);
begin
  FRepeatTimer.Interval := RepeatPause;
  if (FState = bsDown) and MouseCapture then
  begin
    try
      Click;
    except
      FRepeatTimer.Enabled := False;
      raise;
    end;
  end;
end;

procedure TNavButton.Paint;
var
  R: TRect;
begin
  inherited Paint;
  if (GetFocus = Parent.Handle) and
     (FIndex = TRGNavigator (Parent).FocusedButton) then
  begin
    R := Bounds(0, 0, Width, Height);
    InflateRect(R, -3, -3);
    if FState = bsDown then
      OffsetRect(R, 1, 1);
    DrawFocusRect(Canvas.Handle, R);
  end;
end;

{ TNavDataLink }

constructor TNavDataLink.Create(ANav: TRGNavigator);
begin
  inherited Create;
  FNavigator := ANav;
end;

destructor TNavDataLink.Destroy;
begin
  FNavigator := nil;
  inherited Destroy;
end;

procedure TNavDataLink.EditingChanged;
begin
  if FNavigator <> nil then FNavigator.EditingChanged;
end;

procedure TNavDataLink.DataSetChanged;
begin
  if FNavigator <> nil then FNavigator.DataChanged;
end;

procedure TNavDataLink.ActiveChanged;
begin
  if FNavigator <> nil then FNavigator.ActiveChanged;
end;

{ Register }

procedure RegisterRGNavigator;
begin
  RegisterComponents('RG', [TRGNavigator]);
end;

end.
