unit Rotater;

{

  This demo shows how to scale and rotate a Delphi bitmap. It also
  demonstrates palette management and printing techniques. These seem
  to be common questions on the CIS Delphi forum.

  The printing example is a modified version of the code contained in
  the online file "manual.txt", courtesy of Borland Inc.

  The rotation code is courtesy of P. H. Kahler. The original assembler
  code is contained in the file "rotate.zip" in the Delphi forum.

  WARNING! These routines only work with 256 color bitmaps! Another color
  depth might cause a crash (I haven't tried it out).

  Also, there are still some unresolved problems with palettes, as you'll
  see if you print a bitmap or use the load/save option.

  Modification history...

  1. 20-25 April 95, Scott (Scooter) Stephenson ( initial code )
  2. Could be you: who wants to rewrite the rotation routine in assembler?
  3. This could be you too: who wants to rewrite as a Delphi VCL?
  
  To run this program, be sure to "build all".

}

interface

uses
  rotation, { this is the library with the rotation routine }
  printers,
  SysUtils, WinTypes, WinProcs, Messages, Classes, Graphics, Controls,
  Forms, Dialogs, StdCtrls, ExtCtrls;

type
  TMainWindow = class(TForm)
    SourceImage: TImage;
    DerivedImage: TImage;
    Panel1: TPanel;
    Label1: TLabel;
    ButtonRotateRight: TButton;
    ButtonRotateLeft: TButton;
    ButtonZoom: TButton;
    ButtonPan: TButton;
    PanelClock: TPanel;
    Timer1: TTimer;
    PanelMemoryFree: TPanel;
    PanelAngle: TPanel;
    PanelZoom: TPanel;
    ButtonPrint: TButton;
    ButtonLoadNewBitmap: TButton;
    OpenDialog1: TOpenDialog;
    ButtonLoadSave: TButton;
    procedure FormCreate(Sender: TObject);
    procedure ButtonRotateRightClick(Sender: TObject);
    procedure ButtonRotateLeftClick(Sender: TObject);
    procedure ButtonZoomClick(Sender: TObject);
    procedure ButtonPanClick(Sender: TObject);
    procedure Timer1Timer(Sender: TObject);
    procedure ButtonPrintClick(Sender: TObject);
    procedure ButtonLoadNewBitmapClick(Sender: TObject);
    procedure FormResize(Sender: TObject);
    procedure ButtonLoadSaveClick(Sender: TObject);
  private
    { Private declarations }
  public
    { Public declarations }
  zoomFactor,
  RotationAngle: real;
  procedure rotateAndShow;
  end;

var
  MainWindow: TMainWindow;

implementation

{$R *.DFM}

procedure TMainWindow.FormCreate(Sender: TObject);
var
  NewBitmap: TBitmap; { temporary variable to hold the bitmap }
begin
{ initialize variables }
zoomFactor:= 1;
rotationAngle:= 0;
NewBitmap        := TBitmap.Create;        { create a destination bitmap }
NewBitmap.Width  := DerivedImage.Width;    { assign a height and width... }
NewBitmap.Height := DerivedImage.Height;
DerivedImage.Picture.Graphic := NewBitmap; { assign bitmap to image component }
rotateAndShow;
end;

procedure tMainWindow.rotateAndShow;
var
  s: string;
begin
Screen.Cursor := crHourglass;   { Show hourglass cursor }
try
  ScaleAndRotate( sourceImage.Picture.bitmap,
                  derivedImage.Picture.bitmap,
                  zoomFactor,
                  rotationAngle,
                  sourceImage.Picture.bitmap.width div 2,
                  sourceImage.Picture.bitmap.height div 2);
  derivedImage.refresh;
  { show angle of rotation and zoom factor }
  str(rotationAngle:3:0,s);
  panelAngle.caption:= ' Angle : ' + s;
  str(zoomFactor:3:1,s);
  panelZoom.caption:= ' Zoom : ' + s;
  { show memory status, to see if any has been lost }
  panelMemoryFree.caption:= inttostr(memavail div 1024) + ' K free';
finally
  Screen.Cursor := crDefault;    { Always restore cursor to normal }
  end;
end;

procedure TMainWindow.ButtonRotateRightClick(Sender: TObject);
begin
rotationAngle:= rotationAngle + 10;
rotateAndShow;
end;

procedure TMainWindow.ButtonRotateLeftClick(Sender: TObject);
begin
rotationAngle:= rotationAngle - 10;
rotateAndShow;
end;

procedure TMainWindow.ButtonZoomClick(Sender: TObject);
begin
if zoomFactor > 0.5 then zoomFactor:= zoomFactor - 0.5;
rotateAndShow;
end;

procedure TMainWindow.ButtonPanClick(Sender: TObject);
begin
zoomFactor:= zoomFactor + 0.5;
rotateAndShow;
end;

procedure TMainWindow.Timer1Timer(Sender: TObject);
begin
{ I just added a clock so you could see how long the rotation took. }
panelClock.caption := TimeToStr(Time);
end;


procedure TMainWindow.ButtonPrintClick(Sender: TObject);
{ This method shows how to stretch a bitmap onto a target dc without
  screwing up the palette (which happens when Delphi uses the Windows
  "stretchBlt" API function. I decided to use the printer dc to
  demonstrate this routine, primarily because so many people on
  the Delphi forum were expressing confusion over how to print
  a bitmap without screwing up the palette. However, you could just
  as easily stretchDraw the bitmap to the screen. }
begin
Screen.Cursor := crHourglass;   { Show hourglass cursor }
try
  printer.begindoc;
  { draw the derived image so that it fills the whole page on the printer}
  with derivedImage.Picture do stretchDrawBitmap(Bitmap,
                                                 Printer.Canvas.Handle,
                                                 0,0,
                                                 printer.PageWidth,
                                                 printer.PageHeight);
  printer.enddoc;
finally
  Screen.Cursor := crDefault;   { Always restore to normal }
  end;
end;

procedure TMainWindow.ButtonLoadNewBitmapClick(Sender: TObject);
begin
{ This routine demonstrates that bitmaps with
  different palettes don't really pose a problem. }
if openDialog1.execute then
     sourceImage.Picture.bitmap.loadFromFile(opendialog1.filename);
derivedImage.refresh;
end;

procedure TMainWindow.FormResize(Sender: TObject);
{ note: this ensures that the bitmap grows when the window is resized }
begin
derivedImage.Picture.Bitmap.width := derivedImage.width;
derivedImage.Picture.Bitmap.height:= derivedImage.height;
rotateAndShow;
end;

procedure TMainWindow.ButtonLoadSaveClick(Sender: TObject);
begin
{ This method demonstrates that the rotated bitmap can be saved
  to disk. Then, we load it again to show that the palette and
  other image data were correctly processed }
if MessageDlg('This routine will save the rotated bitmap to C:\test.bmp. ' +
              'Then it will load the bitmap back into the program.' +
              'I''m afraid this isn''t totally debugged.' +
              'Dou want to try?',
             mtConfirmation, [mbYes, mbNo], 0) = mrYes then
  begin
  derivedImage.Picture.Bitmap.savetofile('c:\test.bmp');
  derivedImage.Picture.Bitmap.loadfromfile('c:\test.bmp');
  end;
end;

end.
