unit rotation;

{

  This unit rotates bitmaps. It was derived from a program by Paul Kahler.
  Paul's original program can be found on Compuserve's Delphi forum; look
  for "rotate.zip". But, be warned that the original is in assembler.

  You can use these routines in your programs by including this unit in
  your main program's "uses" clause. Fair warning: I haven't included
  any safety checks, so make sure you send valid bitmaps to these routines!

  WARNING! These routines only work with 256 color bitmaps! Another color
  depth might cause a crash (I haven't tried it out).

  Also, there are still some problems with palettes that have to be solved.

 Modification history...
  1. 20-25 April 95, Scott Stephenson ( initial code )

}

interface

uses WinTypes,winprocs,graphics;

procedure scaleAndRotate(baseBitmap,
                         destBitmap: TBitmap;
                         zoomFactor,
                         rotationAngle     : real;
                         baseCenterx,
                         baseCentery : integer);

procedure StretchDrawBitmap(Bitmap: TBitmap; dc: hdc; X, Y, wide, high: longint);

implementation

(**************************************************
  bitmap rotation code
***************************************************)

Var   ddx,
      ddy,
      d2x,
      d2y,
      i,
      j            :real;
      baseBmpBits,            { these point to the bitmap's bits }
      destBmpBits  : pointer;

Procedure PreparetoRotate
                (x,            { x center of source rotation  }
                 y,            { y center of source rotation  }
                 x1,           { x center of dest bitmap }
                 y1,           { y center of dest bitmap }
                 scale,        { scale }
                 rot: real);   { angle of rotation }

begin
{ the following lines of code calculate a 'right' and 'down' vector used
  for scanning the source bitmap. I use quotes because these directions
  depend on the rotation. For example, with a rotation, 'right' could mean
  up and to the left while 'down' means up and to the right. Since the
  destination image (screen) is scanned left-right/top-bottom, the bitmap
  needs to be scanned in arbitrary directions to get a rotation. }

     rot:= rot * pi / 180;
     ddx:= scale * cos(rot);
     ddy:= scale * sin(rot);
     d2x:= scale * cos(rot + pi /2);
     d2y:= scale * sin(rot + pi /2);

{ Since we want to rotate around the CENTER of the screen and not the upper
  left corner, we need to move 1/2 centerx pixels 'left' and 1/2 centery 'up' in the bitmap.}

     i:= x - ddx * x1  - d2x * y1;
     j:= y - ddy * x1  - d2y * y1;

end;


{

  This is the routine that applies Paul's "magic numbers" to achieve
  rotation and scaling. Notice that I improved speed by converting
  reals to large integers, so that the loops can use integer math
  exclusively.

  This routine is just begging to be re-written in assembler. Paul's
  original code did 30+ rotations per second on a 64K bitmap. It sure
  would be nice to see that kind of speed here.

  Maybe the Win 95 version of Delphi will help a bit. Working with
  32 bit addresses would greatly simplify this code by eliminating
  the overhead of performing segment arithmetic.

  }

function PerformRotation( baseBitmap,
                          destBitmap     : tbitmap;
                          zoomFactor,
                          rotationAngle  : real;
                          baseCenterx,
                          baseCentery    : integer): boolean;

var pbytefrom,                { address of the source pixel }
    pbyteto        : ^byte;   { address of the destination pixel }
    baseWidth32,              { baseline bitmap width, 32 bit boundary }
    destWidth32,              { destination bitmap width, 32 bit boundary }
    baseOffset,               { address of the baseline bitmap }
    destOffset,               { address of the destination bitmap }
    i1,                       { these are magic numbers...}
    j1,
    tempi,
    tempj,
    tempddx,
    tempddy,
    tempd2x,
    tempd2y,
    destCenterx,              { x center of rotation of destination }
    destCentery,              { y center of rotation of destination }
    basePixelx,               { the x-coordinate of the source pixel }
    basePixely,               { the y-coordinate of the source pixel }
    x,y            : longint; { the loop variables }
    destbitshi,               { these are used for calculating offset }
    destbitslo,               { pointers into bitmap... }
    basebitshi,
    basebitslo     : word;

const factor  = 1000; { used to avoid real numbers, 1000 seems accurate enough }

type long = record { used to calculate pointers in huge memory blocks }
            lo,hi: word;
            end;

begin

{ adjust our x increments to align on 32 bit word boundary }
baseWidth32:= baseBitmap.width;
destWidth32:= destBitmap.width;
while baseWidth32 mod 4 <> 0 do inc(baseWidth32);
while destWidth32 mod 4 <> 0 do inc(destWidth32);

{ DIB bitmaps are upside down, correct for that }
baseCentery:= baseBitmap.height - baseCentery;

{ assume that rotation will occur around the center of
  the destination bitmap - you could make these parameters if
  you wanted }
destCenterx:= destBitmap.width  div 2;
destCentery:= destBitmap.height div 2;

{ calculate rotation vectors }
prepareToRotate(baseCenterx,
                baseCentery,
                destcenterx * 1.0,
                destcentery * 1.0,
                zoomFactor,
                round(rotationangle));

{ apply a multiplication factor to avoid using real numbers in loop }
tempi  := round(i   * factor);
tempj  := round(j   * factor);
tempddx:= round(ddx * factor);
tempddy:= round(ddy * factor);
tempd2y:= round(d2y * factor);
tempd2x:= round(d2x * factor);

{ calculate the segment offsets outside of the loop }
destbitshi:= long(destBmpBits).hi;
destbitslo:= long(destBmpBits).lo;
basebitshi:= long(baseBmpBits).hi;
basebitslo:= long(baseBmpBits).lo;

{ here's the cosmic bitmap rotation loop,
  it's begging for optimization in assembler }

for y:= 0 to destBitmap.height - 1 do
  begin { the vertical loop }
  i1:= tempi;
  j1:= tempj;
  destOffset:= y * destWidth32; { this is the temporary y, or row offset to dest bitmap }
  for x:= 0 to destBitmap.width - 1 do
      begin
      { calculate the dest address }
      pbyteto:= ptr(destbitshi + long(destOffset).hi * selectorinc,
                    destbitslo + long(destOffset).lo);
      { calculate source pixel }
      inc(i1,tempddx);
      inc(j1,tempddy);
      { remove our multiplication factor }
      basePixelx := i1 div factor;
      basePixely := j1 div factor;
      { must check source rect, or we get tiled copies of original;
        also make sure we are inside source address space }
      if  (basePixelx > -1) and
          (basePixely > -1) and
          (basePixelx < baseBitmap.width) and
          (basePixely < baseBitmap.height)
      then
        begin
       { get address of source pixel }
        baseOffset:= basePixelx + basePixely * baseWidth32;
        pbyteFrom := ptr(basebitshi + long(baseOffset).hi * selectorinc,
                         basebitslo + long(baseOffset).lo);
        pbyteto^  := pbytefrom^;
        end
      else pbyteto^:= 255; { put a white pixel at destination }
      inc(destOffset);     { move to the next pixel }
      end;
  inc(tempi, tempd2x);
  inc(tempj, tempd2y);
  end; { the vertical loop }
end;

{
  This is the routine called by your main program. The basic algorithm is...

  1. get bits of source
  2. get bits of dest
  3. manipulate bits
  4. set bits of dest
  5. free bits of source
  6. free bits of dest
}
procedure scaleAndRotate(baseBitmap,
                         destBitmap     : TBitmap;
                         zoomFactor,
                         rotationAngle  : real;
                         baseCenterx,
                         baseCentery    : integer);
  var
    info1,
    info2      : PBitmapInfo;
    infoSize1,
    infoSize2  : Integer;
    ImageSize1,
    imageSize2 : Longint;
begin
{ by putting the whole routine in a resource protection block, we make
  sure we don't lose big chunks of memory if things go wrong. }
try
{
  This next step causes a problem when creating a new dest bitmap.
  What really should be done here is to fill the destination palette
  with the source palette. Assigning the destination bitmap the
  source bitmap's palette's handle is bad ju-ju, but for the moment,
  I've grimaced and accepted it as a workable hack.
}

destBitmap.palette:= baseBitmap.palette; { ouch, how embarassing! }

{ get the base bitmap's bits }
with baseBitmap do
  begin
  GetDIBSizes(Handle, InfoSize1, ImageSize1);
  Info1 := MemAlloc(InfoSize1);
  baseBmpBits := MemAlloc(ImageSize1);
  GetDIB(Handle, baseBitmap.Palette, info1^, baseBmpBits^);
  end;
{ get the destination bitmap's bits }
with destBitmap do
  begin
  GetDIBSizes(Handle, InfoSize2, ImageSize2);
  Info2 := MemAlloc(InfoSize2);
  destBmpBits := MemAlloc(ImageSize2);
  GetDIB(Handle, destBitmap.Palette, Info2^, destBmpBits^);
  end;

PerformRotation( baseBitmap,
                 destBitmap,
                 zoomFactor,
                 rotationAngle,
                 baseCenterx,
                 baseCentery);

with Info2^.bmiHeader do
  SetDIBitsToDevice(destBitmap.Canvas.Handle,
                    0, 0,
                    destbitmap.Width, destbitmap.Height,
                    0, 0,
                    0,
                    biHeight,
                    destbmpbits,
                    info2^,
                    dib_rgb_colors);

{ no matter what happens, be sure to free allocated memory }
finally
  { free source bitmap's memory }
  with baseBitmap do
    begin
    FreeMem(baseBmpBits, ImageSize1);
    FreeMem(Info1, InfoSize1);
    end;
  { free destination bitmap's memory }
  with destBitmap do
    begin
    FreeMem(DestBmpBits, ImageSize2);
    FreeMem(Info2, InfoSize2);
    end;
  end;
end;

{ This routine prints a bitmap into a target rectangle,
  designated by x/y coordinates ( top/left ) and wide/high
  ( width/height )
}
procedure StretchDrawBitmap(Bitmap: TBitmap; dc: hdc; X, Y, wide, high: longint);
  var
    Info: PBitmapInfo;
    InfoSize: Integer;
    Image: Pointer;
    ImageSize: Longint;
begin
with Bitmap do
  begin
  GetDIBSizes(Handle, InfoSize, ImageSize);
  Info := MemAlloc(InfoSize);
    try
      Image := MemAlloc(ImageSize);
      try
        GetDIB(Handle, bitmap.Palette, Info^, Image^);
        { Notice that StretchDIBits takes care of banding.
          Also, be warned that it really works better with printers --
          some display adapters can't handle it! }
        with Info^.bmiHeader do
          StretchDIBits(dc, X, Y, Wide,
          High, 0, 0, biWidth, biHeight, Image, Info^,
          DIB_rgb_COLORS, SRCCOPY);
      finally
      FreeMem(Image, ImageSize);
      end;
    finally
    FreeMem(Info, InfoSize);
    end;
  end;
end;

begin
end.
