unit Copymain;
(*
    File: COPYMAIN.PAS for Project COPYTEST.DPR
    Nov. '95
*)

interface

uses
  SysUtils, WinTypes, WinProcs, Messages, Classes, Graphics, Controls,
  Forms, Dialogs, StdCtrls, Buttons, FileCtrl, Gauges, ExtCtrls,
  MMSystem, Options, FileTran;

const
  Ndefault = 5; {copy times}

type
  TCopyForm = class(TForm)
    FileListBox1: TFileListBox;
    CopyBtn: TBitBtn;
    ExitBtn: TBitBtn;
    FileListBox2: TFileListBox;
    Panel1: TPanel;
    DirectoryListBox1: TDirectoryListBox;
    DestBox: TGroupBox;
    SourceBox: TGroupBox;
    DirLabel1: TLabel;
    DirLabel2: TLabel;
    MoveBtn: TBitBtn;
    SDriveBtn: TBitBtn;
    Drive2Btn: TBitBtn;
    DirectoryListBox2: TDirectoryListBox;
    OptionBtn: TBitBtn;
    StatPanel: TPanel;
    CopiesBtn: TBitBtn;
    Gauge1: TGauge;
    SwapBtn: TBitBtn;
    procedure ExitBtnClick(Sender: TObject);
    procedure FileListBox1Click(Sender: TObject);
    procedure FormCreate(Sender: TObject);
    procedure FileListBox2DragDrop(Sender, Source: TObject; X, Y: Integer);
    procedure FileListBox2DragOver(Sender, Source: TObject; X, Y: Integer;
      State: TDragState; var Accept: Boolean);
    procedure DirectoryListBox2Change(Sender: TObject);
    procedure SDriveBtnClick(Sender: TObject);
    procedure FormKeyDown(Sender: TObject; var Key: Word;
      Shift: TShiftState);
    procedure FormKeyUp(Sender: TObject; var Key: Word;
      Shift: TShiftState);
    procedure MoveBtnClick(Sender: TObject);
    procedure CopyMove;
    procedure OptionBtnClick(Sender: TObject);
    procedure FileListBox2Click(Sender: TObject);
    procedure FileListBox2DblClick(Sender: TObject);
    procedure FormClose(Sender: TObject; var Action: TCloseAction);
    procedure SwapBtnClick(Sender: TObject);
    procedure FormCloseQuery(Sender: TObject; var CanClose: Boolean);
    procedure DirectoryListBox1Change(Sender: TObject);
    procedure CopiesBtnClick(Sender: TObject);
    procedure CheckResult;
    procedure DisableButtons;
    procedure EnableButtons;
    procedure FileListBox1MouseDown(Sender: TObject; Button: TMouseButton;
      Shift: TShiftState; X, Y: Integer);
  private
    { Private declarations }
    FT: TFileTransfer;
    FD: TSearchRec;       {copy file data }
    ElapsedTime: LongInt; {Time to copy file}
    CntrlDown: boolean;   {for drag/drop}
    aMove: boolean;       {True if Move}
    confirm: boolean;     {True if confirm copy}
    cancel: boolean;      {for copy test}
    ProcessActive: boolean; {for copy test}
    N: Integer;           {copy times}
  public
    { Public declarations }
  end;


var
  CopyForm: TCopyForm;

implementation

{$R *.DFM}

function AddBackSlash(const FDir: string): string;
begin
  if (length(FDir) < 255) and (FDir[length(FDir)] <> '\') then
    Result := Fdir+'\'
  else
    Result := FDir;
end;

function GetFileInfo(const FN: string; var F: TSearchRec): boolean;
{-Returns True if file is found}
begin
  Result := FindFirst(FN, faAnyFile, F) = 0;
  FindClose(F);
  if not Result then
    MessageBeep(MB_ICONASTERISK);
end;

function ShowFileStats(const F: TSearchRec): string;
{-Return formatted string w/ file info}
const
  DateTimeFormat = 'mmm d, yy  hh:mm:ss';
var
  AttrStr, S: string;
begin
  try
    S := FormatDateTime(DateTimeFormat,
      FileDateToDateTime(F.Time));
  except {illegal date or time}
    S := 'BAD FILE DATE';
  end;
 { Formatting for attribute string }
  with F do
  begin
    AttrStr := '----';
    if Attr and faReadOnly <> 0 then
      attrstr[1] := 'R';
    if Attr and faArchive <> 0 then
      attrstr[2] := 'A';
    if Attr and faSysFile <> 0 then
      attrstr[3] := 'S';
    if Attr and faHidden <> 0 then
      attrstr[4] := 'H';
  end;
  Result := Format(' %13s  Size: %6s   Date: %s  %s',
    [F.Name, FormatFloat(',##########0', F.Size), S, AttrStr]);
end;

function CTLabel(const N: Integer): string;
begin
  Result := 'Copy '+IntToStr(N)+' &Times'
end;


{ TCopyForm methods}

procedure TCopyForm.ExitBtnClick(Sender: TObject);
begin
  Close
end;

procedure TCopyForm.FormClose(Sender: TObject; var Action: TCloseAction);
begin
  FT.Free;
end;

procedure TCopyForm.FormCloseQuery(Sender: TObject; var CanClose: Boolean);
begin
  if ProcessActive then
  begin
    CanClose := False;
    MessageBeep(MB_ICONASTERISK);
  end
end;

procedure TCopyForm.FormCreate(Sender: TObject);
{-Init stuff}
const
  CML = ' Command Line Param <';
var
  oldMode: Word;
begin
  CntrlDown := False;
  CopyBtn.Enabled := False;
  CopiesBtn.Enabled := False;
  MoveBtn.Enabled := False;
  ProcessActive := False;
  Panel1.Caption := 'Select Source and Target Directories';
  FT := TFileTransfer.Create; {Create a TFileTransfer instance}
  oldMode := SetErrorMode(SEM_FAILCRITICALERRORS);
  if ParamCount > 0 then
  try
    DirectoryListBox1.Directory := ParamStr(1);
  except
    ShowMessage('Unable to set Source Directory to'+
    CML+ParamStr(1)+'>');
  end;
  if ParamCount > 1 then
  try
    DirectoryListBox2.Directory := ParamStr(2);
  except
    ShowMessage('Unable to set Destination Directory to'+
    CML+ParamStr(2)+'>');
  end;
  SetErrorMode(oldMode);
  DirectoryListBox1Change(Sender);
  DirectoryListBox2Change(Sender);
  {get N}
  N := Ndefault;
  if ParamCount > 2 then
  try
    N := StrToInt(ParamStr(3));
  except
    ShowMessage('Unable to convert'+CML+ParamStr(3)+'> to a number');
  end;
  CopiesBtn.Caption := CTLabel(N);
end;

procedure TCopyForm.CheckResult;
{-Check result of copy}
var
  Speed: Real;
  ix: integer;
  S: string;
  RD: TSearchRec;       { file data }
begin
  if not FT.Completed then
  begin
    Panel1.Caption := 'Copy or move of file '+FD.Name+' was not completed';
    FileListBox2.Update;
  end
  else if not GetFileInfo(AddBackSlash(
    DirectoryListBox2.Directory)+FD.Name, RD) then
    Panel1.Caption := 'File was NOT found.'
  else
  begin
    if ElapsedTime <= 0 then {err}
      S := 'Can''t determine time or rate'
    else
    begin
      Speed := RD.Size/ElapsedTime;
      S := Format(' Elapsed time: %s ms.   %4.1n KB/sec.',
      [FormatFloat(',##########0', ElapsedTime), Speed]);
    end;
    StatPanel.Caption := S;
    Panel1.Caption := ShowFileStats(RD);
    if aMove then {show its absence }
      FileListBox1.Update;

    {show its presence }
    FileListBox2.Update;
    with FileListBox2 do
      for ix := 0 to Items.Count-1 do
      begin
        if FD.Name = UpperCase(FileListBox2.Items[ix]) then
        begin
          TopIndex := ix;
          ItemIndex := ix;
          break;
        end;
      end;
  end
end;

procedure TCopyForm.SDriveBtnClick(Sender: TObject);
{-Change drive }
var
  Dir: string;
begin
  if (Sender as TBitBtn).Tag = 1 then {Dest drive button was pressed}
    Dir := DirectoryListBox2.Directory
  else
    Dir := DirectoryListBox1.Directory;
  if SelectDirectory(Dir, [sdAllowCreate, sdPerformCreate], 0) then
    if (Sender as TBitBtn).Tag = 1 then {Dest drive button was pressed}
      DirectoryListBox2.Directory := Dir
    else
      DirectoryListBox1.Directory := Dir
end;

procedure TCopyForm.OptionBtnClick(Sender: TObject);
begin
  OptionsDlg.ShowModal;
end;

procedure TCopyForm.SwapBtnClick(Sender: TObject);
{-Swap source and dest}
var
  Dir: string;
begin
  Dir := DirectoryListBox1.Directory;
  DirectoryListBox1.Directory := DirectoryListBox2.Directory;
  DirectoryListBox2.Directory := Dir;
end;

procedure TCopyForm.FileListBox1Click(Sender: TObject);
{-Get name of file to copy/move}
begin
  with FileListBox1 do
    if ItemIndex >= 0 then
    if GetFileInfo(AddBackSlash(Directory)+Items[ItemIndex], FD) then
    begin
      Panel1.Caption := ShowFileStats(FD);
      if DirectoryListBox1.Directory <> DirectoryListBox2.Directory then
      begin
        CopyBtn.Enabled := True;
        CopiesBtn.Enabled := True;
        MoveBtn.Enabled := True;
      end
    end
    else
      Panel1.Caption := 'File '+Items[ItemIndex]+' was NOT found.';
end;

procedure TCopyForm.FileListBox2Click(Sender: TObject);
var
  RD: TSearchRec;       { file data }
begin
  with FileListBox2 do
    if ItemIndex >= 0 then
    begin
      if GetFileInfo(AddBackSlash(Directory)+Items[ItemIndex], RD) then
      begin
        Panel1.Caption := ShowFileStats(RD);
        StatPanel.Caption := '';
      end
    end
end;

procedure TCopyForm.FileListBox2DblClick(Sender: TObject);
{-Delete a file}
var
  FN: string;

  function Deleted: boolean;
  begin
    Result := DeleteFile(FN);
    if Result then
    begin
      Panel1.Caption := 'File '+FN+' deleted.';
      FileListBox2.UpDate;
    end
  end;

begin
  with FileListBox2 do
    if ItemIndex >= 0 then
    begin
      StatPanel.Caption := '';
      FN := AddBackSlash(Directory)+Items[ItemIndex];
      if (MessageDlg('Delete file '+FN+'?',
        mtConfirmation, mbOkCancel , 0) = mrOk) then {let's do it}
      begin
        if not Deleted then
        begin
          if FT.HasAttr(FN, faReadOnly) and
            (MessageDlg('File '+FN+' is Read-Only.'#13'Delete anyway?',
            mtConfirmation, mbOkCancel , 0) = mrOk) then {let's try again}
              if (FileSetAttr(FN, faArchive) = 0) and Deleted then
                exit;
          Panel1.Caption := 'File '+FN+' was not deleted.';
          MessageBeep(MB_ICONASTERISK);
        end;
      end
    end
end;

procedure TCopyForm.DirectoryListBox2Change(Sender: TObject);
begin
  DirLabel2.Caption := DirectoryListBox2.Directory;
  DestBox.Caption := 'Drive '+UpCase(DirectoryListBox2.Directory[1])+
  '  Destination Path:';
  if DirectoryListBox1.Directory <> DirectoryListBox2.Directory then
  begin
    if FileListBox1.ItemIndex >= 0 then
    begin
      CopyBtn.Enabled := True;
      CopiesBtn.Enabled := True;
      MoveBtn.Enabled := True;
    end
    else
      Panel1.Caption := 'Select a source file';
  end
  else
  begin
    CopyBtn.Enabled := False;
    CopiesBtn.Enabled := False;
    MoveBtn.Enabled := False;
  end
end;

procedure TCopyForm.DirectoryListBox1Change(Sender: TObject);
begin
  DirLabel1.Caption := DirectoryListBox1.Directory;
  SourceBox.Caption := 'Drive '+UpCase(DirectoryListBox1.Directory[1])+
  '  Source Path:';
  DirectoryListBox2Change(Sender);
end;

procedure TCopyForm.CopyMove;
{-Handles both copying and moving files}
var
  Sourc, Dest, M: string;
  StartTime: LongInt;
begin {CopyMove}
  Dest := DirectoryListBox2.Directory;
  Sourc := AddBackSlash(DirectoryListBox1.Directory)+FD.Name;

  {Get source file info}
  if not GetFileInfo(Sourc, FD) then
  begin
    Screen.Cursor := crDefault;
    Panel1.Caption := 'Source file not found';
    exit;
  end;

  if aMove then {Move}
    M := 'Move'
  else
    M := 'Copy';
  M := Format('%s %s'#13'to %s?',[M, Sourc, Dest]);
  if confirm then
    Screen.Cursor := crDefault;
  if not(confirm) or
    (MessageDlg(M, mtConfirmation, mbOkCancel , 0) = mrOk) then {let's do it}
  begin
    with FT do
      try
        if confirm then
          Screen.Cursor := crHourGlass;

        {Destination file options}
        with OptionsDlg do
        begin
          if Source.Checked then
            UseSourceTimeStamp
          else if Current.Checked then
            UseCurrentTime
          else if Preset.Checked then
            UseTimeStamp(HourBar.Position, MinuteBar.Position);
          { Read Only Option }
          if SetROon.Checked then
            ReadOnlyAction := ForceOn
          else if SetROoff.Checked then
            ReadOnlyAction := ForceOff
          else
            ReadOnlyAction := NoChange;
        end;

        try
          if aMove then {Move button was pressed}
          begin
            Panel1.Caption := 'Moving File: '+FD.Name;
            Application.ProcessMessages;
            StartTime := timeGetTime;
            MoveFile(Sourc, Dest);
            ElapsedTime := timeGetTime;
          end
          else
          begin
            Panel1.Caption := 'Copying File: '+FD.Name;
            Application.ProcessMessages;
            StartTime := timeGetTime;
            CopyFile(Sourc, Dest);
            ElapsedTime := timeGetTime;
        end;
        except
          on E:EFileError do
          begin
            ElapsedTime := timeGetTime;
            Screen.Cursor := crDefault;
            Panel1.Caption := '';
            MessageDlg(E.Message, mtError, [mbCancel], 0);
          end
        end;
      finally
        Screen.Cursor := crDefault;
        dec(ElapsedTime, StartTime);  {calc elapsed time}
      end;
  end;
end;

procedure TCopyForm.DisableButtons;
begin
  MoveBtn.Enabled := False;
  CopyBtn.Enabled := False;
  CopiesBtn.Enabled := False;
  ExitBtn.Enabled := False;
  Panel1.Caption := '';
  StatPanel.Caption := '';
end;

procedure TCopyForm.EnableButtons;
begin
  if (FileListBox1.ItemIndex >= 0) and
     (FileListBox1.Directory <> FileListBox2.Directory) then
  begin
    CopyBtn.Enabled := True;
    CopiesBtn.Enabled := True;
    MoveBtn.Enabled := True;
  end;
  ExitBtn.Enabled := True;
end;

procedure TCopyForm.MoveBtnClick(Sender: TObject);
begin
  DisableButtons;
  Screen.Cursor := crHourGlass;
  aMove := (Sender as TBitBtn).Tag = 1; {0 is Copy}
  confirm := False;
  CopyMove;
  CheckResult;
  EnableButtons;
end;

{ drag / drop stuff }

procedure TCopyForm.FileListBox1MouseDown(Sender: TObject;
  Button: TMouseButton; Shift: TShiftState; X, Y: Integer);
begin
  if Button = mbLeft then
    TFileListBox(Sender).BeginDrag(False);
end;

procedure TCopyForm.FormKeyDown(Sender: TObject; var Key: Word;
  Shift: TShiftState);
begin
  if GetKeyState(VK_CONTROL) < 0 then
    CntrlDown := True;
end;

procedure TCopyForm.FormKeyUp(Sender: TObject; var Key: Word;
  Shift: TShiftState);
begin
  if GetKeyState(VK_CONTROL) < 0 then
    CntrlDown := False
end;

procedure TCopyForm.FileListBox2DragOver(Sender, Source: TObject; X,
  Y: Integer; State: TDragState; var Accept: Boolean);
begin
  Accept := (Source is TFileListBox);
end;

procedure TCopyForm.FileListBox2DragDrop(Sender, Source: TObject; X,
  Y: Integer);
begin
  if Source is TFileListBox then
  begin
    DisableButtons;
    Screen.Cursor := crHourGlass;
    aMove := not CntrlDown; {Copy is with it down}
    confirm := True; {to make sure before...}
    CopyMove;
    CntrlDown := False; { clear it here in case blocked by dialog}
    CheckResult;
    EnableButtons;
  end;
end;

{ The copy test }

procedure TCopyForm.CopiesBtnClick(Sender: TObject);
{-Test copying a file N times, then do some stats}
var
  test: integer;
  sum, mintime, maxtime, meantime: Longint;
  over, under, rate: Real;
begin
  if ProcessActive then {stop it}
  begin
    cancel := True;
    exit;
  end;
  cancel := False;
  ProcessActive := True;
  DisableButtons;
  CopiesBtn.Enabled := True;
  CopiesBtn.Caption := 'CANCEL';
  CopiesBtn.Font.Color := clRed;
  Gauge1.Visible := True;
  Gauge1.Progress := 0;
  aMove := False;
  confirm := False;
  mintime := MaxLongInt;
  maxtime := 0;
  sum := 0;
  {Do Tests}
  for test := 1 to N do
  begin
    CopyMove;
    Gauge1.Progress := test*(100 div N);
    if not FT.Completed or cancel then
      break;
    StatPanel.Caption := Format('Test %d   %d ms.',
    [test, ElapsedTime]);
    Application.ProcessMessages;
    if ElapsedTime < mintime then
       mintime := ElapsedTime;
    if ElapsedTime > maxtime then
       maxtime := ElapsedTime;
    inc(sum, ElapsedTime);
  end;
  ProcessActive := False;
  CopiesBtn.Caption := CTLabel(N);
  CopiesBtn.Font.Color := clBlack;
  if not FT.Completed then
    Panel1.Caption := 'Copy was not successful'
  else if cancel then
    Panel1.Caption := 'Copy cancelled'
  else
  begin
    CheckResult;
    {Make some statistical stuff}
    meantime := sum div N;
    if meantime > 0 then
    begin
      over := 100.0 * (maxtime-meantime)/meantime;
      under := 100.0 * (mintime-meantime)/meantime;
      rate := FD.Size/meantime;
      StatPanel.Caption := Format
      ('Avg: %s ms.  Min: %3.1n %%  Max: +%3.1n %%   %4.1n KB/sec.',
      [FormatFloat(',##########0', meantime), under, over, rate]);
    end
  end;
  EnableButtons;
  Gauge1.Visible := False;
end;

end.
