unit FileTran;
{
  File Transfer class TFileTransfer:
  Copy or move a file
}

interface

uses WinProcs, SysUtils, WinTypes, Classes, Consts, Forms,
     LZExpand;

type
  EFileError = class(EInOutError);

  EMemAlloc = class(EOutOfMemory);

  TimeStampTypes = (UseSource, UseCurrent, UsePreset);
  AttribActions = (NoChange, ForceOn, ForceOff);

  { File Transfer class }
  TFileTransfer = class
  private
    CopyBuffer: Pointer; { buffer for copying }
    TimeStamp: Longint;
    TotalBytesCopied: Longint;
    PresetTime: Word;
    TimeStampChoice: TimeStampTypes;
    Success : boolean;
    procedure UpdateTimeStamp(const Source, Dest: Integer);
  public
    ReadOnlyAction : AttribActions;
    constructor Create;
    destructor Destroy; override;
    procedure CopyFile(const FileName, DestDir: TFileName);
    procedure MoveFile(const FileName, DestDir: TFileName);
    procedure UseSourceTimeStamp;
    procedure UseCurrentTime;
    procedure UseTimeStamp(const Hour, Minute: word);
    function GetBytesCopied: Longint;
    function Completed: boolean;
    class function HasAttr(const FileName: string; Attr: Word): Boolean;
    class function MakeDest(const sfile, dpath: TFileName): TFileName;
    class function GetFileAttr(const FileName: string): Word;
    class procedure SetFileAttr(const FileName: string; const Attr: Word);
  end;

implementation

const
  STMemError = 'Can''t allocate memory for copy buffers';
  {File error messages}
  STCantMove = 'Cannot move file %s';
  STDOSerror = #13'DOS Error: %d';
  STAccessError = #13'Access Denied';
  STOpenError = 'Unable to open %s'+STDOSerror;
  STCreateError = 'Unable to create %s'+STDOSerror;
  STOpenAccessError = 'Unable to open %s'+STAccessError;
  STCreateAccessError = 'Unable to create %s'+STAccessError;
  STCopyError = 'Error copying file %s to'#13'%s';
  STGetAttribError = 'Can''t get file attributes for %s';
  STSetAttribError = 'Can''t set file attributes for %s';
  STDeleteError = 'Can''t delete file %s';


constructor TFileTransfer.Create;
begin
  TimeStampChoice := UseSource;
  PresetTime := $0800;  {01:00:00}
  TotalBytesCopied := 0;
  Success := False;
  ReadOnlyAction := NoChange;
  if LZStart < 0 then
    raise EMemAlloc.Create(STMemError);
end;

procedure TFileTransfer.UseSourceTimeStamp;
begin
  TimeStampChoice := UseSource
end;

procedure TFileTransfer.UseCurrentTime;
begin
  TimeStampChoice := UseCurrent
end;

procedure TFileTransfer.UseTimeStamp(const Hour, Minute: word);
begin
  TimeStampChoice := UsePreset;
  PresetTime := Hour shl 11 + Minute shl 5;
end;

function TFileTransfer.GetBytesCopied: Longint;
begin
  Result := TotalBytesCopied;
end;

function TFileTransfer.Completed: boolean;
begin
  Result := Success;
end;

destructor TFileTransfer.Destroy;
begin
  LZDone;
end;

class function TFileTransfer.HasAttr(const FileName: string; Attr: Word): Boolean;
begin
  Result := (FileGetAttr(FileName) and Attr) = Attr;
end;

class function TFileTransfer.MakeDest(const sfile, dpath: TFileName): TFileName;
{-Returns string w/ full Dest path}
begin
  Result := ExpandFileName(dpath); { expand the destination path }
  if HasAttr(Result, faDirectory) then { if destination is a directory... }
  begin
    if Result[length(Result)] <> '\' then
      Result := Result+'\';
    Result := Result + ExtractFileName(sfile); { ...clone file name }
  end
end;

class function TFileTransfer.GetFileAttr(const FileName: string): Word;
begin
  Result := FileGetAttr(FileName); {get file attributes}
  if Result < 0 then
    raise EFileError.CreateFmt(STGetAttribError, [FileName]);
end;

class procedure TFileTransfer.SetFileAttr
  (const FileName: string; const Attr: Word);
begin
  if FileSetAttr(FileName, Attr) <> 0 then
    raise EFileError.CreateFmt(STSetAttribError, [FileName]);
end;

procedure TFileTransfer.UpdateTimeStamp(const Source, Dest: Integer);
{-Update Timestamp based on selection; assumes files are open}
begin
  case TimeStampChoice of
    UseCurrent:
      TimeStamp := DateTimeToFileDate(Now); { get current time }
    UsePreset:
    begin
      TimeStamp := FileGetDate(Source); { get source's timestamp }
      TimeStamp := TimeStamp and $FFFF0000;
      TimeStamp := TimeStamp or PresetTime;
    end;
    UseSource:
      TimeStamp := FileGetDate(Source); { get source's timestamp }
  end;
  FileSetDate(Dest, TimeStamp); { set timestamp }
end;

procedure TFileTransfer.CopyFile(const FileName, DestDir: TFileName);
{ CopyFile method copies the file passed in FileName to the directory
  specified in DestDir.
  Uses LZExpand functions.
  Raises an exception on fault: source open, dest create, CopyLZFile
}
const
  AccessErr = -5;
var
  Source, Dest: Integer; { handles }
  Destination: TFileName; { holder for expanded destination name }
  LZerr: Longint;
  created: boolean;
  oldMode: Word;
  err: Integer;
  errmsg: string;
  Attr: Word;

begin
  Success := False;
  TotalBytesCopied := 0;
  created := False;

  Destination := MakeDest(FileName, DestDir);

  Source := FileOpen(FileName, fmShareDenyWrite); { open source file }
  if Source < 0 then
  begin
    if Source = AccessErr then
      errmsg := Format(STOpenAccessError, [FileName])
    else
      errmsg := Format(STOpenError, [FileName, Source]);
    raise EFileError.Create(errmsg);
  end;

  try
    Dest := FileCreate(Destination); { create output file; overwrite existing }
    if Dest < 0 then
    begin
      if Dest = AccessErr then
        errmsg := Format(STCreateAccessError, [Destination])
      else
        errmsg := Format(STCreateError, [Destination, Dest]);
      raise EFileError.Create(errmsg);
    end;
    created := True;

    try
      { copy the file }
      LZerr := CopyLZFile(Source, Dest);
      { if a negative value is returned raise an exception }
      if LZerr < 0 then
      begin
        case LZerr of
        lzerror_BadInHandle:
          errmsg := 'Unable to read source';
        lzerror_BadOutHandle:
          errmsg := 'Unable to write destination';
        lzerror_GlobAlloc:
          errmsg := 'Insufficient memory';
        lzerror_Read:
          errmsg := 'File format error';
        lzerror_Write:
          errmsg := 'Insufficient space';
        else
          errmsg := Format('CopyLZFile error: %d', [LZerr]);
        end; {case}
        errmsg := Format(STCopyError, [FileName, Destination])+#13+errmsg;
        raise EFileError.Create(errmsg);
      end;
      TotalBytesCopied := LZerr;
      { with default timestamp = Source's, skip update }
      if TimeStampChoice <> UseSource then
        UpdateTimeStamp(Source, Dest);
      Success := True;
    finally
      FileClose(Dest); { close the destination file }
      {Check result}
      if not(Success) and created then
        DeleteFile(Destination); { ... delete the partial file }
    end;

    {Update destination file attributes, if need to }
    Attr := GetFileAttr(FileName); {get source's attributes}
    if (Attr > 0) or (ReadOnlyAction <> NoChange) then
    begin
      if ReadOnlyAction = ForceOff then
        Attr := Attr and not faReadOnly
      else if ReadOnlyAction = ForceOn then
        Attr := Attr or faReadOnly;
      SetFileAttr(Destination, Attr);
    end;

  finally
    FileClose(Source); { close the source file }
  end;
end;

procedure TFileTransfer.MoveFile(const FileName, DestDir: TFileName);
{ MoveFile method moves the file passed in FileName to the directory
  specified in DestDir. Tries to just rename the file.
  If that fails, tries to copy the file and delete the original.
  Raises an exception if the source file cannot be changed from
  read-only, and therefore cannot be deleted after copying
  (such as attempt to move a file from a CDROM).
}
var
  Destination: TFileName;
  Dest: Integer;
  Attr: Word;
  ResetReadOnly: boolean;
begin
  TotalBytesCopied := 0;
  Success := False;
  ResetReadOnly := False;
  { expand the destination path }
  Destination := MakeDest(FileName, DestDir);
  if RenameFile(FileName, Destination) then { try just renaming }
  begin
    if TimeStampChoice <> UseSource then {fix up timestamp}
    begin
      Dest := FileOpen(Destination, fmShareExclusive); { open file }
      if Dest < 0 then {unlikely but...}
        raise EFileError.CreateFmt('Unable to open moved file %s',[Destination]);
      UpdateTimeStamp(Dest, Dest);
      FileClose(Dest); { close the destination file }
    end;
    {Update destination file attributes, if need to}
    if ReadOnlyAction <> NoChange then
    begin
      Attr := GetFileAttr(Destination); {get attributes}
      if ReadOnlyAction = ForceOn then
        Attr := Attr or faReadOnly
      else if ReadOnlyAction = ForceOff then
        Attr := Attr and not faReadOnly;
      SetFileAttr(Destination, Attr);
    end;
    Success := True
  end
  else
  begin   { Move by copying }
    { if it's read-only we wouldn't be able to delete it }
    if HasAttr(FileName, faReadOnly) then
    begin
      Attr := GetFileAttr(FileName); {get current}
      if FileSetAttr(FileName, Attr and not faReadOnly) <> 0 then {can't change it}
        raise EFileError.CreateFmt(STCantMove, [FileName]);
      { Restore it in case of exception during copy}
      SetFileAttr(FileName, Attr);
      ResetReadOnly := True;
    end;
    CopyFile(FileName, ExpandFileName(DestDir)); { copy it over to destination...}
    if ResetReadOnly then
      SetFileAttr(FileName, Attr and not faReadOnly);
    if not DeleteFile(FileName) then { ...and delete the original }
      raise EFileError.CreateFmt(STDeleteError, [FileName]);
  end;
end;

end.
