;*****************************************************************************
;
; COPYTILE
;
; Copyright (c) 1991-1995 Ted Gruber Software.  All rights reserved.
;
; Prototype
;
;    void copytile (int tile_number, int column, int row, int page);
;
; Description
;
;    Copy a 16x16 tile from a fixed position to the specified row and column.
;    Tiles must be aligned on video memory doublewords (i.e., multiples of 16
;    horizontal pixels).
;
;    This is a specialized version of fg_transfer() that can be used in the
;    QuickFire demo program. The source and destination video addresses for
;    each tile are computed in advance (see the Addr256 macro below). While
;    it is not a general version of such a function, it should be fairly easy
;    to tailor this function to your specific application's needs.
;
; Parameters
;
;    tile_number  The tile number to transfer, between 0 and 255.
;
;    column       The source column number, between 0 and 21.
;
;    row          The source row number, between 0 and 14.
;
;    page         The "page" defining the region for the destination row
;                 and column. It must be 0 or 1.
;
; Return value
;
;    none
;
; Restrictions
;
;    This function is designed to work with in real mode with the medium or
;    large memory models only.
;
;    This function supports mode X only (Fastgraph's modes 20 to 23).
;
;    The calling program must have called fg_resize(352,727) to change the
;    video page dimensions to 352x727. The upper 352x240 area is considered
;    "page 0" and the 352x240 area that follows is considered "page 1".
;
;*****************************************************************************

arg1      equ     [bp+6]        ; address of tile_number parameter
arg2      equ     [bp+8]        ; address of column parameter
arg3      equ     [bp+10]       ; address of row parameter
arg4      equ     [bp+12]       ; address of page parameter

CT_TEXT   SEGMENT byte public 'CODE'
          ASSUME  cs:CT_TEXT

_copytile PROC    far
          PUBLIC  _copytile

          push    bp            ; save caller's BP register
          mov     bp,sp         ; make BP point to argument list
          push    di            ; save caller's DI register
          push    si            ; save caller's SI register
          cld                   ; set the direction flag to forward

          mov     ax,0A000h
          mov     es,ax         ; point ES to start of video memory

          mov     dx,03CEh      ; port address for graphics controller index
          mov     ax,0008h      ; use latches and ignore CPU data
          out     dx,ax         ; send the request

          mov     dl,0C4h       ; port address for sequence controller index
          mov     ax,0F02h      ; mask to enable all four bit planes
          out     dx,ax         ; send the request

          push    ds            ; save caller's data segment
          push    es
          pop     ds            ; point DS to start of video memory

          mov     ax,arg1       ; AX = source tile number
          lea     bx,srce256    ; point BX to address table for source tiles
          shl     ax,1          ; convert tile number to table offset
          add     bx,ax         ; point BX to entry for specified tile
          mov     si,cs:[bx]    ; DS:SI = video address for source tile

          mov     ax,arg3       ; AL = destination row number
          mov     ah,22         ; AH = number of tiles per row
          mul     ah
          add     ax,arg2       ; AX = destination tile number
          lea     bx,dest256    ; assume we'll transfer from upper page
          mov     dx,arg4       ; DX = page number (0=upper, 1=lower)
          neg     dx            ; DX = 0 if upper page, FFFFh if lower
          and     dx,22*15*2    ; DX = offset to upper or lower table
          add     bx,dx         ; point BX to address table for dest tiles
          shl     ax,1          ; convert tile number to table offset
          add     bx,ax         ; point BX to entry for specified tile
          mov     di,cs:[bx]    ; ES:DI = video address for destination tile

          xor     ch,ch         ; zero CH
          mov     ax,352/4-4    ; AX = offset to first tile in next row
          REPT    16
          mov     cl,4          ; four bytes represent 16 pixels
          rep     movsb         ; transfer all 16 pixels of this row
          add     si,ax         ; point source address to next row
          add     di,ax         ; point destination address to next row
          ENDM

          pop     ds            ; restore original data segment
          mov     dx,03CEh      ; port address for graphics controller index
          mov     ax,0FF08h     ; set all bits in the bit mask register
          out     dx,ax         ; send the request

          xor     ax,ax         ; in case copytile was called as a function
          pop     si            ; restore caller's SI register
          pop     di            ; restore caller's DI register
          pop     bp            ; restore caller's BP register
          ret                   ; return to the caller

_copytile ENDP

;-----------------------------------------------------------------------------

Addr256   MACRO   start, columns, rows

address   =       start

          REPT    rows

          REPT    columns
          dw      address
address   =       address + 4
          ENDM

address   =       (address - 4*columns) + 352/4*16
          ENDM

          ENDM

;-----------------------------------------------------------------------------

srce256:  Addr256 352/4*480,20,12   ; source tile addresses
dest256:  Addr256 0,22,15           ; destination tile addresses (upper page)
          Addr256 352/4*240,22,15   ; destination tile addresses (lower page)

CT_TEXT   ENDS
          END
