/****************************************************************************\
*                                                                            *
*  TGSDEMO.CPP                                                               *
*                                                                            *
*  This program displays a scaled image (a logo) against a background of     *
*  scrolling clouds. The background and logo images are stored in PCX files  *
*  with identical palettes that do not use the Windows system colors.        *
*                                                                            *
*  TGSDEMO is based on an original program by Michael Miller.                *
*                                                                            *
\****************************************************************************/

#include <fgwin.h>

long WINAPI _export WindowProc(HWND,UINT,UINT,LONG);
void move_background();

char datapath[128];

/****************************************************************************\
*                                                                            *
*  Virtual Buffer class and function declarations                            *
*                                                                            *
\****************************************************************************/

class VB {
private:
   int hvb;                             // virtual buffer handle
public:
   VB(int, int, int init=FALSE);        // constructor function
   ~VB();                               // destructor function
   int handle();                        // member function (get VB handle)
};

VB::VB(int width, int height, int init) // constructor
{
   if (init) fg_vbinit();
   hvb = fg_vballoc(width,height);
   fg_vbopen(hvb);
   fg_vbcolors();
}

VB::~VB()                               // destructor
{
   fg_vbclose();
   fg_vbfree(hvb);
   fg_vbfin();
}

int VB::handle(void)                    // member function to get the VB handle
{
   return(hvb);
}

VB *workVB;                             // global pointer to workspace VB
VB *backVB;                             // global pointer to background VB

/****************************************************************************\
*                                                                            *
*  Image Buffer class and function declarations                              *
*                                                                            *
\****************************************************************************/

class IMAGE
{
private:
   int image_width, image_height;
   int center_x, yloc;
   char *imagedata;
   char *scaledata;
public:
   IMAGE(char *pcxfile, int width, int height, int ypos);
   ~IMAGE();
   void show(int);
};

// constructor
IMAGE::IMAGE(char *pcxfile, int width, int height, int ypos)
{
   char filename[128];

   lstrcpy(filename,datapath);
   lstrcat(filename,pcxfile);

   // assign image_width and image_height
   image_width  = width;
   image_height = height;
   yloc = ypos;

   // find page center coordinates
   center_x = (320 - image_width) / 2;

   // dynamically allocate memory for imagedata and scaling
   imagedata = new char [image_width * image_height];
   scaledata = new char [image_width * image_height];

   // load the actual image from the PCX file
   fg_vbopen(workVB->handle());
   fg_showpcx(filename,8);

   // retrieve the image as a 256-color bitmap
   fg_move(0,image_height-1);
   fg_getimage(imagedata,image_width,image_height);
};

// destructor
IMAGE::~IMAGE()
{
   delete[] imagedata;
   delete[] scaledata;
}

// member function to display the scaled object
void IMAGE::show(int scale)
{
   if (scale <= 0)
      return;

   // image is scaled down
   else if (scale < 10)
   {
      int dwidth  = (image_width  * scale) / 10;
      int dheight = (image_height * scale) / 10;
      int xloc = (320 - dwidth) / 2;
      fg_scale(imagedata,scaledata,image_width,image_height,dwidth,dheight);
      fg_move(xloc,yloc);
      fg_drwimage(scaledata,dwidth,dheight);
   }

   // image is displayed at full size
   else
   {
      fg_move(center_x,yloc);
      fg_drwimage(imagedata,image_width,image_height);
   }
   return;
}

/****************************************************************************\
*                                                                            *
*  WinMain                                                                   *
*                                                                            *
\****************************************************************************/

int PASCAL WinMain (HINSTANCE hInstance, HINSTANCE hPrevInstance,
                    LPSTR, int nCmdShow)
{
   HWND window;
   MSG  msg;
   WNDCLASS wndclass;
   static char szAppName[] = "FGtgsdemo";
   char filename[128];
   register int i, j;

   if (!hPrevInstance)
   {
      wndclass.style         = CS_HREDRAW | CS_VREDRAW;
      wndclass.lpfnWndProc   = WindowProc;
      wndclass.cbClsExtra    = 0;
      wndclass.cbWndExtra    = 0;
      wndclass.hInstance     = hInstance;
      wndclass.hIcon         = LoadIcon (NULL, IDI_APPLICATION);
      wndclass.hCursor       = LoadCursor (NULL, IDC_ARROW);
      wndclass.hbrBackground = (HBRUSH)GetStockObject (WHITE_BRUSH);
      wndclass.lpszMenuName  = NULL;
      wndclass.lpszClassName = szAppName;
      RegisterClass(&wndclass);
   }

   window = CreateWindow(
      szAppName,               // window class name
      "Scaling and Scrolling", // window caption
      WS_OVERLAPPEDWINDOW,     // window style
      CW_USEDEFAULT,           // initial x position
      CW_USEDEFAULT,           // initial y position
      320,                     // initial x size
      200,                     // initial y size
      NULL,                    // parent window handle
      NULL,                    // window menu handle
      hInstance,               // program instance handle
      NULL);                   // creation parameters

   VB workspace(320,200,TRUE); // 320x200 workspace virtual buffer object
   VB background(1600,200);    // 1600x200 background virtual buffer object
   workVB = &workspace;        // global pointer to workspace VB object
   backVB = &background;       // global pointer to background VB object

   ShowWindow(window,nCmdShow);
   UpdateWindow(window);

   GetModuleFileName(hInstance,datapath,127);
   for (i = lstrlen(datapath); i >= 0; i--)
   {
      if (datapath[i] == '\\')
      {
         datapath[i+1] = 0;
         break;
      }
   }
   lstrcpy(filename,datapath);
   lstrcat(filename,"SKY.PCX");

   fg_vbopen(backVB->handle());
   fg_showpcx(filename,8);

   // copy the first 640 columns onto the next 640 columns, in
   // reverse order (make a mirror image)
   for (i = 639; i >= 0; i--)
   {
      j = 1279 - i;
      fg_vbcopy(i,i,0,199,j,199,backVB->handle(),backVB->handle());
   }

   // make another copy of first 320 columns
   fg_vbcopy(0,319,0,199,1280,199,backVB->handle(),backVB->handle());

   // work buffer is currently open for pasting
   fg_vbopen(workVB->handle());

   // handle messags while scrolling background
   while (TRUE)
   {
      if (PeekMessage(&msg,NULL,0,0,PM_REMOVE))
      {
         if (msg.message == WM_QUIT)
            break;
         else
         {
            TranslateMessage(&msg);
            DispatchMessage(&msg);
         }
      }
      else
         move_background();
   }
   return msg.wParam;
}

/****************************************************************************\
*                                                                            *
*  WindowProc                                                                *
*                                                                            *
*  Window procedure to handle messages sent to the window.                   *
*                                                                            *
\****************************************************************************/

HDC hdc;
HPALETTE hpal;
UINT cxClient, cyClient;

long WINAPI _export WindowProc(HWND window, UINT message,
                               UINT wParam, LONG lParam)
{
   PAINTSTRUCT ps;

   switch (message)
   {
      case WM_CREATE:
         hdc = GetDC(window);
         fg_setdc(hdc);
         hpal = fg_defpal();
         fg_realize(hpal);
         return 0;

      case WM_PAINT:
         BeginPaint(window,&ps);
         fg_vbscale(0,319,0,199,0,319,0,199);
         EndPaint(window,&ps);
         return 0;

      case WM_SETFOCUS:
         fg_realize(hpal);
         InvalidateRect(window,NULL,TRUE);
         return 0;

      case WM_SIZE:
         cxClient = LOWORD(lParam);
         cyClient = HIWORD(lParam);
         return 0;

      case WM_DESTROY:
         DeleteObject(hpal);
         ReleaseDC(window,hdc);
         PostQuitMessage(0);
         return 0;
   }
   return DefWindowProc(window,message,wParam,lParam);
}

/****************************************************************************\
*                                                                            *
*  move_background                                                           *
*                                                                            *
*  Build the next frame and display it in the client window.                 *
*                                                                            *
\****************************************************************************/

void move_background()
{
   static IMAGE logo1 = IMAGE("LOGO1.PCX",264,40,160);
   static IMAGE logo2 = IMAGE("LOGO2.PCX",264,40,80);
   static int hBack = backVB->handle();
   static int hWork = workVB->handle();
   static int x = 0;

   // increment storage virtual buffer starting x coordinate
   x += 8;
   if (x >= 1280) x = 0;

   // copy part of background image to workspace buffer
   fg_vbcopy(x,x+319,0,199,0,199,hBack,hWork);

   // add the logo
   if (x <= 240)
      logo2.show(x/10);
   else
   {
      logo1.show((x-240)/10);
      logo2.show(10);
   }

   // display the frame in the client window
   fg_vbscale(0,319,0,199,0,cxClient,0,cyClient);
}
