LISTING 1================================================================

/* SIERP.C -- (C) 1990 by Dick Oliver, R1 Box 5140, Morrisville, VT  05661
   A program to "draw" and "paint" Sierpinksi's Triangle, defined by a
   "seed" (or "parent") shape and three transformations of that shape
   ("children").  You may copy, modify, and recompile this source code
   as you please, as long as you do not sell it or any product produced
   from any part of it.  The author makes no claims as to readability or
   suitability for a particular task, but I'll be happy to give advice
   and assistance. */

#include <stdio.h>   /* For getch() */
#include <math.h>    /* For cos() and sin() */
#include <graph.h>   /* For graphics calls */

#include "sierp.h"  /* You can change this for other template definitions */

int seedx[NPOINTS] = {SEEDX},   /* The "parent" polygon */
    seedy[NPOINTS] = {SEEDY};

      /* The tranformations which define the "children" */
float movex[NTRANS] = {MOVEX},  /* Displacement */
      movey[NTRANS] = {MOVEY},
      sizex[NTRANS] = {SIZEX},  /* Size change */
      sizey[NTRANS] = {SIZEY},
      spinx[NTRANS] = {SPINX},  /* Rotation */
      spiny[NTRANS] = {SPINY},

      /* The transformation matrix T, computed from the above variables */
      Ta[NTRANS], Tb[NTRANS], Tc[NTRANS], Td[NTRANS];

/* Function prototypes */
void draw(float a, float b, float c, float d, float mx, float my, int iter);
void paint(int mx, int my);

void main(void)
{   int t;
    _setvideomode(_VRES16COLOR);    /* Initialize the screen */
    _clearscreen(_GCLEARSCREEN);

    /* Compute a,b,c,d from the move, size, and spin variables */
    for (t = 0; t < NTRANS; t++)
    {   Ta[t] =   sizex[t] * cos(spinx[t]);
    	Tb[t] = - sizey[t] * sin(spiny[t]);
    	Tc[t] =   sizex[t] * sin(spinx[t]);
    	Td[t] =   sizey[t] * cos(spiny[t]);
    }

    /* Invoke draw with an initial transformation to move the triangle
       to the center of the screen, unchanged in size or rotation */
    draw(1.0, 0.0, 0.0, 1.0, (float) CENTERX, (float) CENTERY, NLEVELS);
    _settextposition(30,0);
    _outtext("Press any key to paint.");
    getch();
    _clearscreen(_GCLEARSCREEN);

    /* Invoke paint, specifying the center of the screen */
    paint(CENTERX, CENTERY);
    _settextposition(30,0);
    _outtext("Press any key to exit.");
    getch();

    _setvideomode(_DEFAULTMODE); /* Go back to text mode and exit */
}


/* This recursive routine draws one "parent" polygon, then calls itself
   to draw the "children" using the transformations defined above */

void draw(float a, float b, float c, float d, float mx, float my, int iter)
{   int t;
    iter--;   /* Count one more level of drawing depth */
    { 	/* Use a,b,c,d,mx,my to transform the polygon */
    	float x1, y1;  /* Point on the parent */
    	int p, x2[NTRANS], y2[NTRANS]; /* Points on the child */
    	for (p = 0; p < NPOINTS; p++)
    	{   x1 = seedx[p];
	    y1 = seedy[p];
	    x2[p] = a * x1 + b * y1 + mx;
	    y2[p] = c * x1 + d * y1 + my;
    	}
    	/* Now draw the new polygon on the screen */
    	_moveto(x2[NPOINTS - 1], y2[NPOINTS - 1]);
    	for (p = 0; p < NPOINTS; p++) _lineto(x2[p], y2[p]);
    }
    if (iter < 0) return;  /* If we're at the deepest level, back out */

    /* Do a recursive call for each "child" of the polygon we just drew */
    for (t = 0; t < NTRANS; t++)
    {   draw(Ta[t] * a + Tc[t] * b,
	     Tb[t] * a + Td[t] * b,
	     Ta[t] * c + Tc[t] * d,
	     Tb[t] * c + Td[t] * d,
	     movex[t] * a + movey[t] * b + mx,
	     movex[t] * c + movey[t] * d + my,
	     iter);
    }
}


/* This routine uses "random iteration" to paint the fractal dot-by-dot.
   The resulting shape will be the same as if we called draw with a
   huge value for the number of levels */

void paint(int mx, int my)
{   int t;
    unsigned long ct = 0;    /* Counter for number of dots painted so far */
    float x1 = 0.0, y1 = 0.0, x2, y2; /* Current and next dot */

    /* Keep going until a key is pressed or we reach the COUNT limit */
    while(!kbhit() && (++ct < COUNT))
    {   t = rand() % NTRANS;  /* Pick one of the transformations at random */

    	/* Then move from a dot on the "whole" to the corresponding dot
    	   on some transformed "part" */
    	x2 = x1 * Ta[t] + y1 * Tb[t] + movex[t];
    	y2 = x1 * Tc[t] + y1 * Td[t] + movey[t];
    	x1 = x2, y1 = y2;

    	/* Skip the first few dots--it takes a while to "find" the fractal */
    	if (ct > 8) _setpixel((int) x2 + mx, (int) y2 + my);
    }
}
