/*
 *                          The AVRIL System
 *
 *  AVRIL stands for A Virtual Reality Interface Library.  It's a library
 *  of routines for doing fast, polygon-based rendering on a variety of
 *  platforms.  It will also provide support for device i/o, simulation,
 *  user interface, interaction detection and more.
 *
 *  It's designed to be easy to use, portable, and fast.
 *
 */

/* Copyright 1994 by Bernie Roehl */

/*
   You may use this code for your own non-commercial projects without
   paying any fees or royalties.  "Non-commercial", in this context,
   means that the software you write is given away for free to anyone
   who wants it.
   
   Commercial use, including shareware, requires a licensing
   fee and a specific written agreement with the author.

   All programs created using this software (both commercial and
   non-commercial) must acknowledge the use of the AVRIL library,
   both in the documentation and in a banner screen at the start or
   end of the program.

   For more information, contact Bernie Roehl (broehl@uwaterloo.ca).

*/

/* This is version 1.0 -- July, 1994 */

/* Three data types are used in AVRIL:

        vrl_Scalar  -- a measure of virtual distance
        vrl_Factor  -- a multiplication factor, usually in the range -1 to +1
        vrl_Angle   -- measured in 65536th's of a degree

   In the floating-point version of the code, they're all floats.  In
   the fixed-point version, they're all stored in a 32-bit word; they
   use differing numbers of bits for the integer and fractional parts of
   their values.
 */

#include <stdio.h>   /* FILE */
#include <mem.h>     /* memcpy() */

#define VRL_USE_FIXED_POINT 1

#define VRL_VERSION 1

#ifdef VRL_USE_FIXED_POINT
typedef long vrl_Scalar;  /* 32.0 */
typedef long vrl_Factor;  /* 3.29 */
#define VRL_UNITY 536870912L  /* 2**29 */
typedef long vrl_Angle;   /* 16.16 */
#define VRL_ANGLECONVERSION 65536  /* 2**16 */
vrl_Factor vrl_ScalarDivide(vrl_Scalar a, vrl_Scalar b);
vrl_Scalar vrl_ScalarMultDiv(vrl_Scalar a, vrl_Scalar b, vrl_Scalar c);
vrl_Scalar vrl_FactorMultiply(vrl_Factor a, vrl_Scalar b);
#define vrl_ScalarRound(a) (a)

#else  /* floating-point */
typedef float vrl_Scalar;
typedef float vrl_Factor;
#define VRL_UNITY 1.0
typedef float vrl_Angle;
#define VRL_ANGLECONVERSION 1
#define vrl_ScalarDivide(a, b) ((a) / (b))
#define vrl_ScalarMultDiv(a, b, c) (((a) * (b)) / (c))
#define vrl_FactorMultiply(a, b) ((a) * (b))
vrl_Scalar vrl_ScalarRound(vrl_Scalar a);
#endif

/* Additional types */

typedef unsigned long vrl_Time;   /* a time in milliseconds */
typedef short vrl_Boolean;        /* zero or non-zero */

/*
   Conversion routines.  You should always use these, to ensure your code
   will work in both the floating-point and fixed-point implementations.
 */

#define float2scalar(f) ((vrl_Scalar) (f))
#define scalar2float(s) ((float) (s))
#define float2factor(f) ((vrl_Factor) ((f) * VRL_UNITY))
#define factor2float(a) (((float) (a)) / VRL_UNITY)
#define float2angle(f) ((vrl_Angle) ((f) * VRL_ANGLECONVERSION))
#define angle2float(a) (((float) (a)) / VRL_ANGLECONVERSION)

/* Math functions */

void vrl_MathInit(void);     /* initializes math routines */
vrl_Factor vrl_Sine(vrl_Angle angle);
vrl_Factor vrl_Cosine(vrl_Angle angle);

/* Vector functions */

typedef vrl_Scalar vrl_Vector[3];

#define X 0
#define Y 1
#define Z 2
#define XROT 3
#define YROT 4
#define ZROT 5

void vrl_VectorCreate(vrl_Vector result, vrl_Scalar x, vrl_Scalar y, vrl_Scalar z);
void vrl_VectorAdd(vrl_Vector result, vrl_Vector v1, vrl_Vector v2);
void vrl_VectorSub(vrl_Vector result, vrl_Vector v1, vrl_Vector v2);
void vrl_VectorNegate(vrl_Vector v);
vrl_Factor vrl_VectorDotproduct(vrl_Vector v1, vrl_Vector v2);
vrl_Scalar vrl_VectorCrossproduct(vrl_Vector result, vrl_Vector v1, vrl_Vector v2);
vrl_Scalar vrl_VectorMagnitude(vrl_Vector v);
void vrl_VectorNormalize(vrl_Vector v);
vrl_Scalar vrl_VectorDistance(vrl_Vector v1, vrl_Vector v2);
void vrl_VectorScale(vrl_Vector v, vrl_Scalar newmag);
void vrl_VectorRescale(vrl_Vector v, vrl_Scalar newmag);

#define vrl_VectorCopy(destination, source) memcpy((destination), (source), sizeof(vrl_Vector))
#define vrl_VectorPrint(out, str, v) fprintf(out, "%s%f,%f,%f", (str), scalar2float((v)[X]), scalar2float((v)[Y]), scalar2float((v)[Z]))
#define vrl_VectorEqual(v1, v2) ((v1)[X] == (v2)[X] && (v1)[Y] == (v2)[Y] && (v1)[Z] == (v2)[Z])
#define vrl_VectorZero(v) ((v)[X] = (v)[Y] = (v)[Z] = 0)

extern vrl_Vector vrl_VectorNULL;  /* the [0,0,0] vector */

/* note that normalized vectors actually have vrl_Factors as elements */

/* Matrix functions */

typedef vrl_Scalar vrl_Matrix[4][3];

/* The rotational elements of a matrix are actually vrl_Factors, not
   vrl_Scalars; only the translational elements are vrl_Scalars. */

void vrl_MatrixIdentity(vrl_Matrix m);
void vrl_MatrixMultiply(vrl_Matrix result, vrl_Matrix m1, vrl_Matrix m2);
void vrl_MatrixInverse(vrl_Matrix result, vrl_Matrix m);
void vrl_MatrixRotX(vrl_Matrix m, vrl_Angle angle, vrl_Boolean leftside);
void vrl_MatrixRotY(vrl_Matrix m, vrl_Angle angle, vrl_Boolean leftside);
void vrl_MatrixRotZ(vrl_Matrix m, vrl_Angle angle, vrl_Boolean leftside);
void vrl_MatrixRotVector(vrl_Matrix m, vrl_Angle angle, vrl_Vector vector, vrl_Boolean leftside);
void vrl_MatrixResetRotations(vrl_Matrix m);
void vrl_MatrixGetBasis(vrl_Vector v, vrl_Matrix m, int axis);
void vrl_MatrixTranslate(vrl_Matrix result, vrl_Vector v, vrl_Boolean leftside);
void vrl_MatrixGetRotations(vrl_Matrix m, vrl_Angle *rx, vrl_Angle *ry, vrl_Angle *rz);

#define vrl_MatrixCopy(result, m) memcpy((result), (m), sizeof(vrl_Matrix))
#define vrl_MatrixGetTranslation(v, m) vrl_VectorCopy((v), (m)[3])
#define vrl_MatrixSetTranslation(m, v) vrl_VectorCopy((m)[3], (v))

/* Transformation functions */

void vrl_Transform(vrl_Vector result, vrl_Matrix m, vrl_Vector v);
vrl_Scalar vrl_TransformX(vrl_Matrix m, vrl_Vector v);
vrl_Scalar vrl_TransformY(vrl_Matrix m, vrl_Vector v);
vrl_Scalar vrl_TransformZ(vrl_Matrix m, vrl_Vector v);

/* Data types */

typedef struct _vrl_world vrl_World;
typedef struct _vrl_object vrl_Object;
typedef struct _vrl_shape vrl_Shape;
typedef struct _vrl_rep vrl_Rep;
typedef struct _vrl_facet vrl_Facet;
typedef struct _vrl_surface vrl_Surface;
typedef struct _vrl_light vrl_Light;
typedef struct _vrl_camera vrl_Camera;
typedef struct _vrl_device vrl_Device;
typedef struct _vrl_device_channel vrl_DeviceChannel;
typedef struct _vrl_serial_port vrl_SerialPort;
typedef struct _vrl_surface_map vrl_Surfacemap;
typedef unsigned char vrl_Palette[256][3];

typedef enum
		{
		VRL_COORD_LOCAL = 0, VRL_COORD_PARENT,
		VRL_COORD_WORLD, VRL_COORD_OBJREL
		} vrl_CoordFrame;

/* Worlds */

struct _vrl_world
	{
	vrl_Object *objects;      /* tree of objects */
	vrl_Light *lights;        /* linked list of lights */
	vrl_Camera *cameras;      /* linked list of cameras */
	vrl_Camera *camera;       /* current camera */
	vrl_Factor ambient;       /* ambient light level */
	vrl_Scalar scale;         /* millimeters per unit of virtual space */
	vrl_Scalar movestep;      /* default movement step size */
	vrl_Angle rotstep;        /* default rotation step size */
	vrl_Boolean movement_mode : 1;    /* non-zero if we can fly by looking up */
	vrl_Boolean set_palette: 1;       /* set if this world has a new palette */
	vrl_Boolean screenclear : 1;      /* set if we should clear the screen */
	vrl_Boolean horizon : 1;          /* set if we should draw a horizon   */
	long horizoncolors[10];   /* entry 0 is ground, entry n is sky */
	int nhorizoncolors;       /* number of colors used in horizoncolors[] */
	vrl_Palette palette;      /* the palette, if one is used */
	};

vrl_World *vrl_WorldInit(vrl_World *world);
void vrl_WorldAddLight(vrl_Light *light);
void vrl_WorldRemoveLight(vrl_Light *light);
vrl_Light *vrl_WorldFindLight(char *name);
void vrl_WorldAddCamera(vrl_Camera *camera);
void vrl_WorldRemoveCamera(vrl_Camera *camera);
vrl_Camera *vrl_WorldFindCamera(char *name);
void vrl_WorldAddObject(vrl_Object *obj);
void vrl_WorldRemoveObject(vrl_Object *obj);
vrl_Object *vrl_WorldFindObject(char *name);
int vrl_WorldCountObjects(void);
int vrl_WorldCountFacets(void);
int vrl_WorldCountLights(void);
int vrl_WorldCountCameras(void);
void vrl_WorldGetBounds(vrl_Vector v1, vrl_Vector v2);
void vrl_WorldGetCenter(vrl_Vector v);
vrl_Scalar vrl_WorldGetSize(void);

#define vrl_WorldCreate() vrl_WorldInit(malloc(sizeof(vrl_World)))
#define vrl_WorldSetScreenClear(n) (vrl_current_world->screenclear = (n) ? 1 : 0)
#define vrl_WorldGetScreenClear() (vrl_current_world->screenclear)
#define vrl_WorldToggleScreenClear() (vrl_current_world->screenclear = !vrl_current_world->screenclear)
#define vrl_WorldSetHorizon(n) (vrl_current_world->horizon = (n) ? 1 : 0)
#define vrl_WorldGetHorizon() (vrl_current_world->horizon)
#define vrl_WorldToggleHorizon() (vrl_current_world->horizon = !vrl_current_world->horizon)
#define vrl_WorldSetMovementMode(n) (vrl_current_world->movement_mode = (n) ? 1 : 0)
#define vrl_WorldGetMovementMode() (vrl_current_world->movement_mode)
#define vrl_WorldToggleMovementMode() (vrl_current_world->movement_mode = !vrl_current_world->movement_mode)
#define vrl_WorldSetMovestep(distance) (vrl_current_world->movestep = (distance))
#define vrl_WorldGetMovestep() (vrl_current_world->movestep)
#define vrl_WorldSetTurnstep(angle) (vrl_current_world->rotstep = (angle))
#define vrl_WorldGetTurnstep() (vrl_current_world->rotstep)
#define vrl_WorldSetScale(s) (vrl_current_world->scale = (s))
#define vrl_WorldGetScale() (vrl_current_world->scale)
#define vrl_WorldSetAmbient(amb) (vrl_current_world->ambient = (amb))
#define vrl_WorldGetAmbient() (vrl_current_world->ambient)
#define vrl_WorldSetGroundColor(color) (vrl_current_world->horizoncolors[0] = (color))
#define vrl_WorldGetGroundColor() (vrl_current_world->horizoncolors[0])
#define vrl_WorldSetSkyColor(color) (vrl_current_world->horizoncolors[vrl_current_world->nhorizoncolors-1] = (color))
#define vrl_WorldGetSkyColor() (vrl_current_world->horizoncolors[vrl_current_world->nhorizoncolors-1])
#define vrl_WorldGetPalette() (vrl_current_world->palette)
#define vrl_WorldUsePalette() (vrl_current_world->set_palette = 1)
#define vrl_WorldGetLights() (vrl_current_world->lights)
#define vrl_WorldGetCameras() (vrl_current_world->cameras)
#define vrl_WorldGetObjectTree() (vrl_current_world->objects)
#define vrl_WorldSetCamera(cam) (vrl_current_world->camera = (cam))
#define vrl_WorldGetCamera() (vrl_current_world->camera)
#define vrl_WorldUpdate() vrl_ObjectUpdate(vrl_WorldGetObjectTree())

extern vrl_World *vrl_current_world;   /* the currently active world */

#define vrl_WorldSetCurrent(world) (vrl_current_world = (world))
#define vrl_WorldGetCurrent() (vrl_current_world)

/* Objects */

struct _vrl_object
	{
	vrl_Shape *shape;           /* geometry information */
	vrl_Surfacemap *surfmap;    /* array of pointers to surface descriptors */
	vrl_Matrix localmat;        /* transformation matrix relative to our parent */
	vrl_Matrix globalmat;       /* transformation matrix relative to the world */
	vrl_Object *parent;         /* pointer to our parent in the hierarchy */
	vrl_Object *children;       /* pointer to our children */
	vrl_Object *siblings;       /* pointers to our siblings */
	vrl_Vector minbound, maxbound;  /* bounding box (world coords) */
	vrl_Boolean fixed : 1;              /* set if object is immobile */
	vrl_Boolean moved : 1;              /* set when our local matrix has changed */
	vrl_Boolean rotate_box : 1;         /* set if bounding box should rotate */
	vrl_Boolean highlight : 1;          /* set if object is highlighted */
	vrl_Boolean invisible : 1;          /* set if object is invisible */
	unsigned char layer;        /* the layer we're on (0 for all, 1-255) */
	vrl_Object *contents;       /* points to objects contained by this one (not used) */
	vrl_Rep *forced_rep;        /* if not NULL, forces a rep to be used */
	char *name;                 /* name of the object (may be NULL) */
	void *applic_data;          /* pointer to application-specific data */
	int (*function)(vrl_Object *obj);  /* object function */
	vrl_Object *next;           /* points to next object on a list */
	};

vrl_Object *vrl_ObjectInit(vrl_Object *obj);
vrl_Object *vrl_ObjectCreate(vrl_Shape *shape);
vrl_Object *vrl_ObjectCopy(vrl_Object *obj);
void vrl_ObjectDestroy(vrl_Object *object);
void vrl_ObjectMove(vrl_Object *obj, vrl_Scalar x, vrl_Scalar y, vrl_Scalar z);
void vrl_ObjectRelMove(vrl_Object *obj, vrl_Scalar x, vrl_Scalar y, vrl_Scalar z);
void vrl_ObjectRotVector(vrl_Object *obj, vrl_Angle angle, vrl_Vector vector);
void vrl_ObjectRotReset(vrl_Object *obj);
void vrl_ObjectRotate(vrl_Object *obj, vrl_Angle angle, int axis, vrl_CoordFrame frame, vrl_Object *relative_to);
void vrl_ObjectTranslate(vrl_Object *obj, vrl_Vector v, vrl_CoordFrame frame, vrl_Object *relative_to);
vrl_Object *vrl_ObjectAttach(vrl_Object *obj, vrl_Object *newparent);
vrl_Object *vrl_ObjectDetach(vrl_Object *obj);
vrl_Object *vrl_ObjectUpdate(vrl_Object *object);
void vrl_ObjectTraverse(vrl_Object *object, int (*function)(vrl_Object *obj));
void vrl_ObjectMakeFixed(vrl_Object *object);
void vrl_ObjectMakeMovable(vrl_Object *object);
vrl_Object *vrl_ObjectFindRoot(vrl_Object *obj);
vrl_Scalar vrl_ObjectComputeDistance(vrl_Object *obj1, vrl_Object *obj2);

#define vrl_ObjectRotX(obj, angle) vrl_ObjectRotate((obj), (angle), X, VRL_COORD_PARENT, NULL)
#define vrl_ObjectRotY(obj, angle) vrl_ObjectRotate((obj), (angle), Y, VRL_COORD_PARENT, NULL)
#define vrl_ObjectRotZ(obj, angle) vrl_ObjectRotate((obj), (angle), Z, VRL_COORD_PARENT, NULL)
#define vrl_ObjectVectorMove(obj, v) vrl_ObjectMove((obj), (v)[X], (v)[Y], (v)[Z])
#define vrl_ObjectVectorRelMove(obj, v) vrl_ObjectTranslate((obj), (v), VRL_COORD_PARENT, NULL)
#define vrl_ObjectSetLayer(object, lay) ((obj)->layer = (lay))
#define vrl_ObjectGetLayer(object) ((object)->layer)
#define vrl_ObjectSetShape(object, shp) ((object)->shape = (shp))
#define vrl_ObjectGetShape(object) ((object)->shape)
#define vrl_ObjectSetSurfacemap(object, map) ((object)->surfmap = (map))
#define vrl_ObjectGetSurfacemap(object) ((object)->surfmap)
#define vrl_ObjectSetHighlight(object, high) ((object)->highlight = (high))
#define vrl_ObjectGetHighlight(object) ((object)->highlight)
#define vrl_ObjectToggleHighlight(object) ((object)->highlight = !(object)->highlight)
#define vrl_ObjectSetVisibility(object, vis) ((object)->invisible = ((vis) ? 1 : 0))
#define vrl_ObjectGetVisibility(object) ((object)->invisible)
#define vrl_ObjectToggleVisibility(object) ((object)->invisible = !(object)->invisible)
#define vrl_ObjectIsFixed(object) ((object)->fixed)
#define vrl_ObjectGetMinbounds(object, v) vrl_VectorCopy((v), (object)->minbound)
#define vrl_ObjectGetMaxbounds(object, v) vrl_VectorCopy((v), (object)->maxbound)
#define vrl_ObjectSetRep(object, r) ((object)->forced_rep = (r))
#define vrl_ObjectGetRep(object) ((object)->forced_rep)
#define vrl_ObjectGetX(object) ((object)->globalmat[3][X])
#define vrl_ObjectGetY(object) ((object)->globalmat[3][Y])
#define vrl_ObjectGetZ(object) ((object)->globalmat[3][Z])
#define vrl_ObjectGetLocation(object, v) vrl_VectorCopy((v), (object)->globalmat[3])
#define vrl_ObjectGetRotations(object, rx, ry, rz) vrl_MatrixGetRotations((object)->globalmat, (rx), (ry), (rz))
#define vrl_ObjectSetName(obj, str) ((obj)->name = (str))
#define vrl_ObjectGetName(obj) ((obj)->name)
#define vrl_ObjectSetApplicationData(obj, data) ((obj)->applic_data = (data))
#define vrl_ObjectGetApplicationData(obj) ((obj)->applic_data)
#define vrl_ObjectSetFunction(obj, fn) ((obj)->function = (fn))
#define vrl_ObjectGetFunction(obj) ((obj)->function)
#define vrl_ObjectGetForwardVector(object, v) vrl_MatrixGetBasis((v), (object)->globalmat, Z)
#define vrl_ObjectGetRightVector(object, v) vrl_MatrixGetBasis((v), (object)->globalmat, X)
#define vrl_ObjectGetUpVector(object, v) vrl_MatrixGetBasis((v), (object)->globalmat, Y)

/* Shapes */

struct _vrl_shape
	{
	vrl_Vector center;  /* center of bounding sphere */
	vrl_Scalar radius;  /* radius of bounding sphere */
	vrl_Vector minbound, maxbound;  /* bounding box */
	vrl_Surfacemap *default_surfacemap;  /* default surface map for this shape */
	vrl_Rep *replist;  /* linked list of representations */
	char *name;        /* name of this shape */
	vrl_Shape *next;   /* shapes are kept in a linked list */
	};

vrl_Shape *vrl_ShapeInit(vrl_Shape *shape);
void vrl_ShapeComputeBounds(vrl_Shape *shape);
void vrl_ShapeRescale(vrl_Shape *shape, float sx, float sy, float sz);
void vrl_ShapeOffset(vrl_Shape *shape, vrl_Scalar tx, vrl_Scalar ty, vrl_Scalar tz);
void vrl_ShapeUpdate(vrl_Shape *shape);
void vrl_ShapeTransform(vrl_Matrix mat, vrl_Shape *shape);
vrl_Rep *vrl_ShapeGetRep(vrl_Shape *shape, vrl_Scalar size);
void vrl_ShapeAddRep(vrl_Shape *shape, vrl_Rep *rep, vrl_Scalar size);
void vrl_ShapeTraverseReps(vrl_Shape *shape, int (*function)(vrl_Rep *rep));       /* NEEDED? */
int vrl_ShapeCountReps(vrl_Shape *shape);
vrl_Shape *vrl_ShapeGetList(void);
vrl_Shape *vrl_ShapeFind(char *name);

#define vrl_ShapeCreate() vrl_ShapeInit(malloc(sizeof(vrl_Shape)))
#define vrl_ShapeGetRadius(shape) ((shape)->radius)
#define vrl_ShapeGetCenter(shape, v) vrl_VectorCopy((v), (shape)->center)
#define vrl_ShapeGetMinbounds(shape, v) vrl_VectorCopy((v), (shape)->minbound)
#define vrl_ShapeGetMaxbounds(shape, v) vrl_VectorCopy((v), (shape)->maxbound)
#define vrl_ShapeGetSurfacemap(shape) ((shape)->default_surfacemap)
#define vrl_ShapeSetSurfacemap(shape, map) ((shape)->default_surfacemap = (map))
#define vrl_ShapeGetName(shape) ((shape)->name)
#define vrl_ShapeSetName(shape, str) ((shape)->name = (str))
#define vrl_ShapeGetNext(shape) ((shape)->next)
#define vrl_ShapeGetFirstRep(shape) ((shape)->replist)

/* Representations */

typedef enum
		{
		VRL_SORT_NONE = 0, VRL_SORT_FARTHEST, VRL_SORT_NEAREST,
		VRL_SORT_AVERAGE, VRL_SORT_OTHER
		} vrl_SortingType;

struct _vrl_rep
	{
	vrl_Scalar size;    /* (radius/distance) value at which to use this rep */
	vrl_Rep *next;      /* next less-detailed rep */
	vrl_SortingType sorttype;  /* type of poly sorting to do on this rep */
	int nvertices;      /* number of vertices (and normals if present) */
	vrl_Vector *vertices;   /* array of vertices */
	vrl_Vector *normals;    /* array of vertex normals; can be NULL */
	vrl_Facet *facets;  /* facets are kept in a linked list */
	};

vrl_Rep *vrl_RepInit(vrl_Rep *rep, int nvertices, vrl_Boolean has_normals);
void vrl_RepAddFacet(vrl_Rep *rep, vrl_Facet *facet);
void vrl_RepTraverseVertices(vrl_Rep *rep, int (*function)(vrl_Vector *vertex, vrl_Vector *normal));
void vrl_RepTraverseFacets(vrl_Rep *rep, int (*function)(vrl_Facet *facet));
vrl_Facet *vrl_RepGetFacet(vrl_Rep *rep, int n);
vrl_Facet *vrl_RepFindFacet(vrl_Rep *rep, unsigned int id);
int vrl_RepCountFacets(vrl_Rep *rep);

#define vrl_RepCreate(nv, norm) vrl_RepInit(malloc(sizeof(vrl_Rep)), (nv), (norm))
#define vrl_RepSetSorting(rep, type) ((rep)->sorttype = (type))
#define vrl_RepGetSorting(rep) ((rep)->sorttype)
#define vrl_RepGetSize(rep) ((rep)->size)
#define vrl_RepCountVertices(rep) ((rep)->nvertices)
#define vrl_RepGetVertex(rep, n, v) vrl_VectorCopy((v), (rep)->vertices[n])
#define vrl_RepSetVertex(rep, n, v) vrl_VectorCopy((rep)->vertices[n], (v))
#define vrl_RepHasNormals(rep) ((rep)->normals)
#define vrl_RepGetNormal(rep, n, v) vrl_VectorCopy((v), (rep)->normals[n])
#define vrl_RepGetNext(rep) ((rep)->next)

/* Facets */

struct _vrl_facet
	{
	int surface;            /* index into object's surface array */
	vrl_Vector normal;      /* perpendicular to facet, left-hand rule */
	unsigned int id;        /* identifier for this facet */
	vrl_Boolean highlight : 1;      /* facet is highlighted */
	vrl_Boolean interior  : 1;      /* facet is on the interior of an object (not used) */
	vrl_Facet *details;     /* points to linked list of detail facets (not used) */
	vrl_Facet *nearside, *farside;  /* only farside is used */
	int npoints;  /* number of points in the facet */
	int *points;  /* indices into array of vertices of the facet points */
	};

vrl_Facet *vrl_FacetInit(vrl_Facet *facet, int npts);
void vrl_FacetTraverse(vrl_Facet *facet, int (*function)(vrl_Facet *f));
void vrl_FacetComputeNormal(vrl_Facet *facet, vrl_Vector *vertices);

#define vrl_FacetCreate(npts) vrl_FacetInit(malloc(sizeof(vrl_Facet)), (npts))
#define vrl_FacetSetSurfnum(facet, n) ((facet)->surface = (n))
#define vrl_FacetGetSurfnum(facet) ((facet)->surface)
#define vrl_FacetCountPoints(facet) ((facet)->npoints)
#define vrl_FacetGetPoint(facet, n) ((facet)->points[n])
#define vrl_FacetGetVertex(rep, facet, n, v) vrl_VectorCopy((v), (rep)->vertices[(facet)->points[n]])
#define vrl_FacetSetHighlight(facet, high) ((facet)->highlight = ((high) ? 1 : 0))
#define vrl_FacetGetHighlight(facet) ((facet)->highlight)
#define vrl_FacetToggleHighlight(facet) ((facet)->highlight = !(facet)->highlight)
#define vrl_FacetSetInterior(facet, inter) ((facet)->interior = ((inter) ? 1 : 0))
#define vrl_FacetGetInterior(facet) ((facet)->interior)
#define vrl_FacetToggleInterior(facet) ((facet)->interior = !(facet)->interior)
#define vrl_FacetSetId(facet, n) ((facet)->id = (n))
#define vrl_FacetGetId(facet) ((facet)->id)
#define vrl_FacetGetNormal(facet, v) vrl_VectorCopy((v), (facet)->normal)
#define vrl_FacetSetPoint(facet, n, m) ((facet)->points[n] = (m))
#define vrl_FacetGetPoint(facet, n) ((facet)->points[n])

/* Surfaces */

typedef enum
		{
		VRL_SURF_SIMPLE = 0, VRL_SURF_FLAT, VRL_SURF_METAL,
		VRL_SURF_GLASS, VRL_SURF_GOURAUD
		} vrl_SurfaceType;

struct _vrl_surface
	{
	vrl_SurfaceType type;
	unsigned char hue;
	unsigned char brightness;
	};

vrl_Surface *vrl_SurfaceFromDesc(unsigned int desc, vrl_Surface *surf);
unsigned int vrl_SurfaceToDesc(vrl_Surface *surf);
vrl_Surface *vrl_SurfaceInit(vrl_Surface *surf, unsigned char hue);

#define vrl_SurfaceCreate(hue) vrl_SurfaceInit(malloc(sizeof(vrl_Surface)), hue)
#define vrl_SurfaceSetType(surf, t) ((surf)->type = (t))
#define vrl_SurfaceGetType(surf) ((surf)->type)
#define vrl_SurfaceSetHue(surf, h) ((surf)->hue = (h))
#define vrl_SurfaceGetHue(surf) ((surf)->hue)
#define vrl_SurfaceSetBrightness(surf, b) ((surf)->brightness = (b))
#define vrl_SurfaceGetBrightness(surf) ((surf)->brightness)

struct _vrl_surface_map
	{
	int nentries;
	vrl_Surface **entries;
	};

vrl_Surfacemap *vrl_SurfacemapCreate(int n);

#define vrl_SurfacemapSetSurface(map, surfnum, surf) ((map)->entries[surfnum] = (surf))
#define vrl_SurfacemapGetSurface(map, surfnum) ((map)->entries[surfnum])

/* Lights */

typedef enum
	{
	VRL_LIGHT_AMBIENT = 0, VRL_LIGHT_DIRECTIONAL, VRL_LIGHT_POINTSOURCE
	} vrl_LightingType;

struct _vrl_light
	{
	vrl_LightingType type;  /* type of light source */
	vrl_Boolean on : 1;     /* set if the light is on */
	vrl_Factor intensity;   /* how bright the light is */
	vrl_Object *object;     /* the object this light is associated with, if any */
	vrl_Light *next;        /* lights are kept in a linked list */
	char *name;             /* name of this light */
	void *applic_data;      /* pointer to application-specific data */
	int (*function)(vrl_Light *light);  /* light function */
	};

vrl_Light *vrl_LightInit(vrl_Light *light);
vrl_Light *vrl_LightCreate(void);
void vrl_LightDestroy(vrl_Light *light);

#define vrl_LightSetType(light, ltype) ((light)->type = (ltype))
#define vrl_LightGetType(light) ((light)->type)
#define vrl_LightOn(light) ((light)->on = 1)
#define vrl_LightOff(light) ((light)->on = 0)
#define vrl_LightToggle(light) ((light)->on = !((light)->on))
#define vrl_LightIsOn(light) ((light)->on)
#define vrl_LightSetIntensity(light, inten) ((light)->intensity = (inten))
#define vrl_LightGetIntensity(light) ((light)->intensity)
#define vrl_LightAssociate(light, obj) ((light)->object = (obj))
#define vrl_LightDisAssociate(light) ((light)->object = NULL)
#define vrl_LightGetObject(light) ((light)->object)
#define vrl_LightSetName(light, str) ((light)->name = (str))
#define vrl_LightGetName(light) ((light)->name)
#define vrl_LightGetNext(light) ((light)->next)
#define vrl_LightSetApplicationData(light, data) ((light)->applic_data = (data))
#define vrl_LightGetApplicationData(light) ((light)->applic_data)

#define vrl_LightMove(light, x, y, z) vrl_ObjectMove((light)->object, x, y, z)
#define vrl_LightRelMove(light, x, y, z) vrl_ObjectRelMove((light)->object, (x), (y), (z))
#define vrl_LightVectorMove(light, v) vrl_ObjectVectorMove((light)->object, (v))
#define vrl_LightVectorRelMove(light, v) vrl_ObjectVectorRelMove((light)->object, (v))
#define vrl_LightRotX(light, angle) vrl_ObjectRotX((light)->object, (angle))
#define vrl_LightRotY(light, angle) vrl_ObjectRotY((light)->object, (angle))
#define vrl_LightRotZ(light, angle) vrl_ObjectRotZ((light)->object, (angle))
#define vrl_LightRotVector(light, angle, vector) vrl_ObjectRotVector((light)->object, (angle), (vector))
#define vrl_LightRotReset(light) vrl_ObjectRotReset((light)->object)
#define vrl_LightRotate(light, angle, axis, frame, relative_to) vrl_ObjectRotate((light)->obj, (angle), (axis), (frame), (relative_to))
#define vrl_LightTranslate(light, v, axis, frame, relative_to) vrl_ObjectTranslate((light)->obj, (v), (frame), (relative_to))
#define vrl_LightAttach(light, newparent) vrl_ObjectAttach((light)->object, (newparent))
#define vrl_LightDetach(light) vrl_ObjectDetach((light)->object)

#define vrl_LightGetLocation(light, v) vrl_ObjectGetLocation((light)->object, (v))
#define vrl_LightGetRotations(light, rx, ry, rz) vrl_ObjectGetRotations((light)->object, (rx), (ry), (rz))

#define vrl_LightGetX(light) ((light)->object->globalmat[3][X])
#define vrl_LightGetY(light) ((light)->object->globalmat[3][Y])
#define vrl_LightGetZ(light) ((light)->object->globalmat[3][Z])

/* Cameras */

struct _vrl_camera
	{
	vrl_Scalar hither;   /* distance to near clipping plane */
	vrl_Scalar yon;      /* distance to far culling plane */
	float zoom;          /* zoom factor (1/tan(FOV/2)) */
	float aspect;        /* aspect ratio */
	vrl_Boolean ortho : 1;    /* set if we want orthographic projection (not used) */
	vrl_Scalar orthodist;     /* apparent "distance" for orthographic projection (not used) */
	vrl_Object *object;       /* the object this camera is attached to */
	unsigned char need_updating;  /* set when zoom or aspect is changed */
	/* these next four are only used internally, for object culling */
	vrl_Factor aright, cright, btop, ctop;
	vrl_Camera *next;       /* cameras are kept in a linked list */
	char *name;             /* name of this camera */
	void *applic_data;      /* pointer to application-specific data */
	};

vrl_Camera *vrl_CameraInit(vrl_Camera *camera);
vrl_Camera *vrl_CameraCreate(void);
void vrl_CameraDestroy(vrl_Camera *camera);

#define vrl_CameraSetZoom(camera, zf) (((camera)->zoom = (zf)), (camera)->need_updating = 1)
#define vrl_CameraGetZoom(camera) ((camera)->zoom)
#define vrl_CameraSetAspect(camera, asp) (((camera)->aspect = (asp)), (camera)->need_updating = 1)
#define vrl_CameraGetAspect(camera) ((camera)->aspect)
#define vrl_CameraSetHither(camera, h) ((camera)->hither = (h))
#define vrl_CameraGetHither(camera) ((camera)->hither)
#define vrl_CameraSetYon(camera, y) ((camera)->yon = (y))
#define vrl_CameraGetYon(camera) ((camera)->yon)
#define vrl_CameraAssociate(camera, obj) ((camera)->object = (obj))
#define vrl_CameraGetObject(camera) ((camera)->object)
#define vrl_CameraSetName(camera, str) ((camera)->name = (str))
#define vrl_CameraGetName(camera) ((camera)->name)
#define vrl_CameraGetNext(camera) ((camera)->next)
#define vrl_CameraSetApplicationData(cam, data) ((cam)->applic_data = (data))
#define vrl_CameraGetApplicationData(cam) ((cam)->applic_data)

#define vrl_CameraMove(camera, x, y, z) vrl_ObjectMove((camera)->object, x, y, z)
#define vrl_CameraRelMove(camera, x, y, z) vrl_ObjectRelMove((camera)->object, (x), (y), (z))
#define vrl_CameraVectorMove(camera, v) vrl_ObjectVectorMove((camera)->object, (v))
#define vrl_CameraVectorRelMove(camera, v) vrl_ObjectVectorRelMove((camera)->object, (v))
#define vrl_CameraRotX(camera, angle) vrl_ObjectRotX((camera)->object, (angle))
#define vrl_CameraRotY(camera, angle) vrl_ObjectRotY((camera)->object, (angle))
#define vrl_CameraRotZ(camera, angle) vrl_ObjectRotZ((camera)->object, (angle))
#define vrl_CameraRotVector(camera, angle, vector) vrl_ObjectRotVector((camera)->object, (angle), (vector))
#define vrl_CameraRotReset(camera) vrl_ObjectRotReset((camera)->object)
#define vrl_CameraRotate(camera, angle, axis, frame, relative_to) vrl_ObjectRotate((camera)->obj, (angle), (axis), (frame), (relative_to))
#define vrl_CameraTranslate(camera, v, axis, frame, relative_to) vrl_ObjectTranslate((camera)->obj, (v), (frame), (relative_to))
#define vrl_CameraAttach(camera, newparent) vrl_ObjectAttach((camera)->object, (newparent))
#define vrl_CameraDetach(camera) vrl_ObjectDetach((camera)->object)

#define vrl_CameraGetX(camera) ((camera)->object->globalmat[3][X])
#define vrl_CameraGetY(camera) ((camera)->object->globalmat[3][Y])
#define vrl_CameraGetZ(camera) ((camera)->object->globalmat[3][Z])

#define vrl_CameraGetLocation(camera, v) vrl_ObjectGetLocation((camera)->object, (v))
#define vrl_CameraGetRotations(camera, rx, ry, rz) vrl_ObjectGetRotations((camera)->object, (rx), (ry), (rz))

#define vrl_CameraGetForwardVector(camera, v) vrl_ObjectGetForwardVector((camera)->object, (v))
#define vrl_CameraGetRightVector(camera, v) vrl_ObjectGetRightVector((camera)->object, (v))
#define vrl_CameraGetUpVector(camera, v) vrl_ObjectGetUpVector((camera)->object, (v))

/* Rendering functions */

typedef struct
	{
	vrl_Boolean memory : 1;       /* set if the renderer ran out of memory */
	vrl_Boolean objects : 1;      /* set if there were too many objects  */
	vrl_Boolean facets : 1;       /* set if there were too many facets   */
	} vrl_RenderStatus;

vrl_Boolean vrl_RenderInit(int maxvert, int maxf, int maxobjs, int maxlights, unsigned int mempoolsize);
void vrl_RenderQuit(void);
void vrl_RenderBegin(vrl_Camera *camera, vrl_Light *lights);
void vrl_RenderSetAmbient(vrl_Factor amb);
void vrl_RenderHorizon(void);
vrl_RenderStatus *vrl_RenderObjlist(vrl_Object *objects);

void vrl_RenderMonitorInit(int x, int y);  /* monitor the specified point */
vrl_Boolean vrl_RenderMonitorRead(vrl_Object **obj, vrl_Facet **facet, int *vertnum);  /* returns zero if none */

/* Layer support */

void vrl_LayerOn(int n);
void vrl_LayerOff(int n);
void vrl_LayerToggle(int n);
vrl_Boolean vrl_LayerIsOn(int n);
void vrl_LayerAllOn(void);

/* File i/o routines */

void vrl_FileSetLoadpath(char *path);
char *vrl_FileFixupFilename(char *fname);
vrl_Shape *vrl_ReadPLG(FILE *in);
int vrl_WritePLG(vrl_Shape *shape, FILE *out);
void vrl_SetReadPLGscale(float x, float y, float z);
void vrl_SetReadPLGoffset(float x, float y, float z);
int vrl_ReadWLD(FILE *in);
void vrl_ReadWLDfeature(int argc, char *argv[], char *rawtext);
vrl_Object *vrl_ReadFIG(FILE *in, vrl_Object *parent, char *rootname);
void vrl_SetReadFIGpartArray(vrl_Object **ptr, int maxparts);
void vrl_SetReadFIGscale(float x, float y, float z);
vrl_Object *vrl_ReadObjectPLG(FILE *in);

vrl_Object *vrl_ObjectLoadPLGfile(char *filename);
int vrl_ObjectSavePLGfile(vrl_Object *object, char *filename);
vrl_Object *vrl_ReadFIGfile(char *filename);

/* Display routines */

typedef struct _vrl_outvertex vrl_OutputVertex;

struct _vrl_outvertex
	{
	vrl_OutputVertex *next;   /* linked list */
	long coords[3];           /* X, Y screen coordinates and Z-depth */
	unsigned int intensity;   /* used for Gouraud shading */
	unsigned char outcode;    /* used for XY clipping */
	/* future versions of AVRIL may have additional info per vertex */
	};

typedef struct
	{
	int version;          /* driver version number */
	char *driver;         /* points to driver name */
	int zbuffer;          /* 0 = none, 1 = software, 2 = hardware */
	int gouraud;          /* 0 = none, 1 = software, 2 = hardware */
	} vrl_DisplayCap;

typedef struct  /* used for passing data to display driver */
	{
	int iparm1, iparm2, iparm3, iparm4;
	long lparm1;
	void *pparm1, *pparm2;
	} vrl_DisplayData;

typedef enum
		{
		VRL_DISPLAY_INIT = 0, VRL_DISPLAY_QUIT, VRL_DISPLAY_GET_CAPABILITIES, VRL_DISPLAY_CLEAR,
		VRL_DISPLAY_POINT, VRL_DISPLAY_LINE, VRL_DISPLAY_BOX, VRL_DISPLAY_POLY,
		VRL_DISPLAY_TEXT, VRL_DISPLAY_USE_Z_BUFFER, VRL_DISPLAY_CLEAR_Z_BUFFER,
		VRL_DISPLAY_SET_WINDOW, VRL_DISPLAY_GET_WINDOW,
		VRL_DISPLAY_SET_DRAW_PAGE, VRL_DISPLAY_GET_DRAW_PAGE,
		VRL_DISPLAY_SET_VIEW_PAGE, VRL_DISPLAY_GET_VIEW_PAGE,
		VRL_DISPLAY_GET_WIDTH, VRL_DISPLAY_GET_HEIGHT, VRL_DISPLAY_GET_DEPTH,
		VRL_DISPLAY_GET_NPAGES, VRL_DISPLAY_GET_TEXT_HEIGHT, VRL_DISPLAY_GET_TEXT_WIDTH,
		VRL_DISPLAY_UPDATE, VRL_DISPLAY_SET_PALETTE, VRL_DISPLAY_GET_PALETTE,
		VRL_DISPLAY_READ_SCANLINE, VRL_DISPLAY_WRITE_SCANLINE,
		VRL_DISPLAY_SET_WIREFRAME_COLOR, VRL_DISPLAY_GET_WIREFRAME_COLOR,
		VRL_DISPLAY_SET_HIGHLIGHT_COLOR, VRL_DISPLAY_GET_HIGHLIGHT_COLOR,
		VRL_DISPLAY_SET_DRAW_MODE, VRL_DISPLAY_GET_DRAW_MODE
		} vrl_DisplayCommand;

typedef long vrl_DisplayDriverFunction(vrl_DisplayCommand cmd, vrl_DisplayData *data);

void vrl_DisplaySetDriver(vrl_DisplayDriverFunction fn, unsigned int default_mode);
char *vrl_DisplayGetName(void);
int vrl_DisplayGetVersion(void);
int vrl_DisplayInit(unsigned int mode);
void vrl_DisplayQuit(void);
vrl_DisplayCap *vrl_DisplayGetCapabilities(void);
void vrl_DisplayClear(long color);
void vrl_DisplayPoint(int x, int y, long color);
void vrl_DisplayLine(int x1, int y1, int x2, int y2, long color);
void vrl_DisplayBox(int x1, int y1, int x2, int y2, long color);
void vrl_DisplayPoly(vrl_OutputVertex *vertices, vrl_Surface *surf, unsigned int intensity, int highlight, int xyclip);
void vrl_DisplayText(int x, int y, long color, char *message);
void vrl_DisplayUseZbuffer(int flag);
void vrl_DisplayClearZbuffer(void);
void vrl_DisplaySetWindow(int x1, int y1, int x2, int y2);
void vrl_DisplayGetWindow(int *x1, int *y1, int *x2, int *y2);
void vrl_DisplaySetDrawPage(int page);
int vrl_DisplayGetDrawPage(void);
void vrl_DisplaySetViewPage(int page);
int vrl_DisplayGetViewPage(void);
int vrl_DisplayGetWidth(void);
int vrl_DisplayGetHeight(void);
int vrl_DisplayGetDepth(void);
int vrl_DisplayGetNpages(void);
int vrl_DisplayGetTextHeight(char *string);
int vrl_DisplayGetTextWidth(char *string);
void vrl_DisplayUpdate(void);  /* forces ram->framebuffer copy if applicable */
void vrl_DisplaySetPalette(int start, int end, vrl_Palette palette);  /* updates the palette */
void vrl_DisplayGetPalette(int start, int end, vrl_Palette palette);  /* updates the palette */
void vrl_DisplayReadScanline(int n, unsigned char *buffer);
void vrl_DisplayWriteScanline(int n, unsigned char *buffer);
long vrl_DisplaySetWireframeColor(long color);
long vrl_DisplayGetWireframeColor(void);
long vrl_DisplaySetHighlightColor(long color);
long vrl_DisplayGetHighlightColor(void);
void vrl_DisplaySetDrawMode(int wireframe);
int vrl_DisplayGetDrawMode(void);

/* Timer routines */

vrl_Boolean vrl_TimerInit(void);
void vrl_TimerQuit(void);
vrl_Time vrl_TimerRead(void);
void vrl_TimerDelay(vrl_Time time);
#define vrl_TimerGetTickRate() (1000)

/* Mouse routines */

vrl_Boolean vrl_MouseInit(void);
void vrl_MouseQuit(void);
vrl_Boolean vrl_MouseReset(void);
vrl_Boolean vrl_MouseRead(int *x, int *y, unsigned int *buttons);
void vrl_MouseCursorHide(void);
void vrl_MouseCursorShow(void);
void vrl_MouseGetLimits(int *min_x, int *max_x, int *min_y, int *max_y);
void vrl_MouseSetUsage(int u);
int vrl_MouseGetUsage(void);
void vrl_MouseSetPointer(void *u);
void *vrl_MouseGetPointer(void);

/* Keyboard routines */

vrl_Boolean vrl_KeyboardCheck(void);
unsigned int vrl_KeyboardRead(void);

/* System routines */

vrl_Boolean vrl_SystemStartup(void);
void vrl_SystemRun(void);
vrl_RenderStatus *vrl_SystemRender(vrl_Object *list);
vrl_Time vrl_SystemGetRenderTime(void);
vrl_Time vrl_SystemGetFrameRate(void);
void vrl_SystemCommandLine(int argc, char *argv[]);

void vrl_SystemRequestRefresh(void);
vrl_Boolean vrl_SystemQueryRefresh(void);

void vrl_SystemStartRunning(void);
void vrl_SystemStopRunning(void);
vrl_Boolean vrl_SystemIsRunning(void);

/* Routines defined by application */

void vrl_ApplicationDrawUnder(void);
void vrl_ApplicationDrawOver(vrl_RenderStatus *stat);
void vrl_ApplicationInit(void);
void vrl_ApplicationKey(unsigned int c);
void vrl_ApplicationMouseUp(int x, int y, unsigned int buttons);

/* User interface routines */

void vrl_UserInterfaceBox(int w, int h, int *x, int *y);
void vrl_UserInterfacePopText(char *text[]);
void vrl_UserInterfacePopMsg(char *msg);
int vrl_UserInterfacePopMenu(char *text[]);
int vrl_UserInterfaceMenuDispatch(char *text[], int (**funcs)(void));
unsigned vrl_UserInterfacePopPrompt(char *prompt, char *buff, int n);
int vrl_UserInterfaceDismiss(void);

void vrl_UserInterfaceDrawCompass(vrl_Camera *camera, int compass_x, int compass_y, int compass_armlen);
void vrl_UserInterfaceDropText(int x, int y, int color, char *text);

/* Pseudo-tasking routines */

vrl_Boolean vrl_TaskCreate(void (*function)(void), void *data, vrl_Time period);
void vrl_TaskRun(void);
void *vrl_TaskGetData(void);
vrl_Time vrl_TaskGetElapsed(void);
vrl_Time vrl_TaskGetTimeNow(void);

/* Routines for creating primitive shapes */

vrl_Shape *vrl_PrimitiveBox(vrl_Scalar width, vrl_Scalar height, vrl_Scalar depth, vrl_Surfacemap *map);
vrl_Shape *vrl_PrimitiveCone(vrl_Scalar radius, vrl_Scalar height, int nsides, vrl_Surfacemap *map);
vrl_Shape *vrl_PrimitiveCylinder(vrl_Scalar bottom_radius, vrl_Scalar top_radius, vrl_Scalar height, int nsides, vrl_Surfacemap *map);
vrl_Shape *vrl_PrimitivePrism(vrl_Scalar width, vrl_Scalar height, vrl_Scalar depth, vrl_Surfacemap *map);
vrl_Shape *vrl_PrimitiveSphere(vrl_Scalar radius, int vsides, int hsides, vrl_Surfacemap *map);

/* Device support */

typedef enum
		{
		VRL_DEVICE_INIT = 0, VRL_DEVICE_QUIT, VRL_DEVICE_POLL, VRL_DEVICE_RESET
		} vrl_DeviceCommand;

typedef int vrl_DeviceDriverFunction(vrl_DeviceCommand cmd, vrl_Device *device);
typedef int vrl_DeviceOutputFunction(vrl_Device *device, int parm1, vrl_Scalar parm2);

typedef enum
	{
	VRL_MOTION_NONE = 0, VRL_MOTION_RELATIVE, VRL_MOTION_ABSOLUTE
	} vrl_DeviceMotionMode;

typedef int vrl_Buttonmap[][2];

/* In the following struct, fields marked with a (^) can be set by the
   device function; not all of them need to be, since reasonable defaults
   are already set for each.  Fields marked with a (^^) must be set for
   every device. */

struct _vrl_device
	{
	int version;                /* device struct version number (^) */
	char *desc;                 /* user-readable device description (^^) */
	vrl_DeviceDriverFunction *fn;   /* the driver function itself */
	vrl_SerialPort *port;       /* pointer to serial port */
	vrl_DeviceMotionMode rotation_mode;     /* rotation mode for this device (^) */
	vrl_DeviceMotionMode translation_mode;  /* translation mode for this device (^) */
	int mode;                   /* mode of operation (^) */
	vrl_Time period;            /* milliseconds between reads (^) */
	vrl_Time lastread;          /* timer value when we last read */
	int nbuttons;               /* number of buttons the device has (^) */
	unsigned long buttons;      /* current button status */
	unsigned long bchanged;     /* which buttons have changed state */
	vrl_Buttonmap *buttonmap;   /* button mapping table for 2D devices */
	int noutput_channels;       /* number of output channels (^) */
	vrl_DeviceOutputFunction *outfunc;  /* function to call to generate output (^) */
	int nchannels;              /* number of input channels the device has (^^) */
	vrl_DeviceChannel *channels;   /* pointer to array of channels (^^) */
	void *localdata;            /* data used only by the driver (^) */
	vrl_Device *next;           /* devices are kept in a linked list */
	};

vrl_Device *vrl_DeviceOpen(vrl_DeviceDriverFunction fn, vrl_SerialPort *port);
void vrl_DeviceClose(vrl_Device *device);
int vrl_DevicePoll(vrl_Device *device);
int vrl_DeviceReset(vrl_Device *device);
void vrl_DevicePollAll(void);
void vrl_DeviceCloseAll(void);
void vrl_DeviceOutput(vrl_Device *device, int channel, vrl_Scalar value);
vrl_Device *vrl_DeviceGetFirst(void);

#define vrl_DeviceGetNext(dev) ((dev)->next)
#define vrl_DeviceGetDesc(device) ((device)->desc)
#define vrl_DeviceSetPeriod(device, per) ((device)->period = (per))
#define vrl_DeviceGetPeriod(device) ((device)->period)
#define vrl_DeviceGetMode(device) ((device)->mode)
#define vrl_DeviceSetMode(device, m) ((device)->mode = (m))
#define vrl_DeviceGetPort(device) ((device)->port)
#define vrl_DeviceGetNchannels(device) ((device)->nchannels)
#define vrl_DeviceGetNButtons(device) ((device)->nbuttons)
#define vrl_DeviceGetButtons(device) ((device)->buttons)
#define vrl_DeviceGetChangedButtons(device) ((device)->bchanged)
#define vrl_DeviceGetNOutputChannels(device) ((device)->noutput_channels)
#define vrl_DeviceGetRotationMode(device) ((device)->rotation_mode)
#define vrl_DeviceGetTranslationMode(device) ((device)->translation_mode)

#define vrl_DeviceGetButtonmap(device) ((device)->buttonmap)
#define vrl_DeviceSetButtonmap(device, map) ((device)->buttonmap = (map))

struct _vrl_device_channel
	{
	long centerpoint;        /* value of center point in raw device coords */
	long deadzone;           /* minimum acceptable value in device coords */
	long range;              /* maximum absolute value relative to zero */
	vrl_Scalar scale;        /* maximum returned value */
	vrl_Boolean accumulate : 1;      /* if set, accumulate values */
	vrl_Boolean changed : 1;         /* set if rawvalue has changed */
	long rawvalue;           /* current value in raw device coordinates */
	long oldvalue;           /* previous value in raw device coordinates */
	vrl_Scalar value;        /* current value, clipped, centered and deadzoned and accumulated */
	};

#define vrl_DeviceGetCenter(device, channel) ((device)->channels[channel].centerpoint)
#define vrl_DeviceGetDeadzone(device, channel) ((device)->channels[channel].deadzone)
#define vrl_DeviceSetDeadzone(device, channel, value) ((device)->channels[channel].deadzone = (value))
#define vrl_DeviceGetScale(device, channel) ((device)->channels[channel].scale)
#define vrl_DeviceSetScale(device, channel, value) ((device)->channels[channel].scale = (value))
#define vrl_DeviceGetAccumulate(device, channel) ((device)->channels[channel].accumulate)
#define vrl_DeviceSetAccumulate(device, channel, value) ((device)->channels[channel].accumulate = (value))
#define vrl_DeviceGetChanged(device, channel) ((device)->channels[channel].changed)
#define vrl_DeviceGetRawValue(device, channel) ((device)->channels[channel].rawvalue)
#define vrl_DeviceGetValue(device, channel) ((device)->channels[channel].value)

/* Serial input/output support */

typedef enum
		{
		VRL_PARITY_EVEN = 0, VRL_PARITY_ODD, VRL_PARITY_NONE
		} vrl_ParityType;

struct _vrl_serial_port {
	unsigned int address;  /* hardware address */
	int irq;               /* IRQ level (*not* interrupt number!) */
	int buffsize;          /* size of the buffer */
	char *buffer;          /* pointer to the buffer */
	int in, out;           /* offsets into buffer */
	vrl_SerialPort *next;  /* serial ports are kept in a linked list */
	};

vrl_SerialPort *vrl_SerialOpen(unsigned int address, int irq, unsigned int buffsize);
void vrl_SerialClose(vrl_SerialPort *port);
void vrl_SerialCloseAll(void);
void vrl_SerialSetParameters(vrl_SerialPort *port, unsigned int baud, vrl_ParityType parity, int databits, int stopbits);
vrl_Boolean vrl_SerialCheck(vrl_SerialPort *port);
unsigned int vrl_SerialGetc(vrl_SerialPort *port);
void vrl_SerialPutc(unsigned int c, vrl_SerialPort *port);
void vrl_SerialPutString(unsigned char *s, vrl_SerialPort *p);
void vrl_SerialFlush(vrl_SerialPort *p);
void vrl_SerialSetDTR(vrl_SerialPort *port, vrl_Boolean value);
void vrl_SerialSetRTS(vrl_SerialPort *port, vrl_Boolean value);

typedef struct _device_packet_buffer
	{
	int buffsize;           /* size of a packet */
	int ind;                /* current index into buffer */
	unsigned char *buffer;  /* pointer to the buffer itself */
	} vrl_DevicePacketBuffer;

vrl_DevicePacketBuffer *vrl_DeviceCreatePacketBuffer(int buffsize);
void vrl_DeviceDestroyPacketBuffer(vrl_DevicePacketBuffer *buff);
vrl_Boolean vrl_DeviceGetPacket(vrl_SerialPort *port, vrl_DevicePacketBuffer *buff);
#define vrl_DevicePacketGetBuffer(buff) (((vrl_DevicePacketBuffer *)buff)->buffer)

/* Configuration file support */

void vrl_ConfigSetCompassDisplay(vrl_Boolean flag);
vrl_Boolean vrl_ConfigGetCompassDisplay(void);
void vrl_ConfigToggleCompassDisplay(void);
void vrl_ConfigSetPositionDisplay(vrl_Boolean flag);
vrl_Boolean vrl_ConfigGetPositionDisplay(void);
void vrl_ConfigTogglePositionDisplay(void);
void vrl_ConfigSetFramerateDisplay(vrl_Boolean flag);
vrl_Boolean vrl_ConfigGetFramerateDisplay(void);
void vrl_ConfigToggleFramerateDisplay(void);
char *vrl_ConfigGetDisplayDriverName(void);
vrl_DisplayDriverFunction *vrl_ConfigGetDisplayDriver(void);
unsigned int vrl_ConfigGetDisplayMode(void);
vrl_Device *vrl_ConfigFindDevice(char *name);
void vrl_ConfigureAllDevices(void);

int vrl_ReadCFG(FILE *in);
int vrl_ReadCFGProcessLine(char *buff);
void vrl_ConfigStartup(char *filename);

/* PCX file support */

vrl_Boolean vrl_ReadPCX(FILE *in);
vrl_Boolean vrl_WritePCX(FILE *out);

/*
   These #defines are provided for the sake of backward compatability to the
   pre-release version of AVRIL; they should no longer be used.
 */

#define Scalar vrl_Scalar
#define Factor vrl_Factor
#define Angle vrl_Angle
#define Vector vrl_Vector
#define Matrix vrl_Matrix

#define vrl_NewObject(shape) vrl_ObjectCreate(shape)
#define vrl_NewWorld() vrl_WorldCreate()
#define vrl_NewShape() vrl_ShapeCreate()
#define vrl_NewSurface(hue) vrl_SurfaceCreate(hue)
#define vrl_NewSurfacemap(n) vrl_SurfacemapCreate(n)
#define vrl_NewLight() vrl_LightCreate()
#define vrl_NewCamera() vrl_CameraCreate()

/* End of avril.h */
