/* Read an AVRIL-style CFG file */

/* Written by Bernie Roehl, July 1994 */

/* Copyright 1994 by Bernie Roehl */

/* You may use this code for your own non-commercial projects without
   paying any fees or royalties.  "Non-commercial", in this context,
   means that the software you write is given away for free to anyone
   who wants it.
   
   Commercial use, including shareware, requires a licensing
   fee and a specific written agreement with the author.

   All programs created using this software (both commercial and
   non-commercial) must acknowledge the use of the AVRIL library,
   both in the documentation and in a banner screen at the start or
   end of the program.

   For more information, contact Bernie Roehl (broehl@uwaterloo.ca).

*/

#include <stdlib.h>
#include <string.h>
#include <ctype.h>

#include "avril.h"
#include "avrildrv.h"

typedef struct _driver_details DriverDetails;
typedef struct _driver_parms ChannelParms;

struct _driver_parms
	{
	int channel;               /* channel number */
	vrl_Scalar scale;          /* scaling factor */
	vrl_Scalar deadzone;       /* dead zone */
	ChannelParms *next;
	};

struct _driver_details
	{
	char *name;             /* name of this driver (e.g. "headtracker") */
	char *drivername;       /* type of driver (e.g. "joystick") */
	vrl_Device *device;     /* pointer to the device itself (NULL until opened) */
	unsigned int mode;      /* mode to set for the driver */
	unsigned int address;   /* serial port address */
	int irq;                /* serial port IRQ number */
	int buffsize;           /* serial port buffer size */
	ChannelParms *parms;    /* channel-specific parameters */
	DriverDetails *next;    /* linked list */
	};

/* These next two should really be in avrildrv.h */

extern vrl_DisplayDriverFunction vrl_DisplayDriverRVD;
extern vrl_DisplayDriverFunction vrl_DisplayDriverWGT;

static int show_compass = 0, show_position = 0, show_framerate = 0;

static char *display_drivername = "rvd";
static vrl_DisplayDriverFunction *display_driver = vrl_DisplayDriverRVD;
static unsigned int display_mode = 0;

static DriverDetails *driverlist = NULL;

static int getline(char *buff, int maxbytes, FILE *in)
	{
	char *p;
	if (fgets(buff, maxbytes, in) == NULL) return 0;
	if ((p = strchr(buff, '\n')) != NULL) *p = '\0';
	return 1;
	}

static int lookup(char *token, char *table[], int n)
	{
	int i;
	for (i = 0; i < n; ++i)
		if (!stricmp(token, table[i]))
			return i;
	return -1;
	}

static int parse(char *buff, char *seps, char *argv[])
	{
	char *p;
	int argc = 0;
	if ((p = strchr(buff, '#')) != NULL) *p = '\0';
	p = strtok(buff, seps);
	while (p)
		{
		argv[argc++] = p;
		p = strtok(NULL, seps);
		}
	return argc;
	}

static char *statement_table[] =
	{
	"version", "loadpath", "include",
	"device", "devconfig", "displaydriver", "sounddriver",
	"compass", "framerate", "position",
	"stereoparams", "stereoleft", "stereoright", "stereomode"
	};

static enum statements
	{
	st_version, st_loadpath, st_include,
	st_device, st_devconfig, st_displaydriver, st_sounddriver,
	st_compass, st_framerate, st_position,
	st_stereoparams, st_stereoleft, st_stereoright, st_stereomode,
	ncmds
	};

static void driver_setup(int argc, char *argv[])
	{
	DriverDetails *drvr;
	drvr = (DriverDetails *) malloc(sizeof(DriverDetails));
	if (drvr == NULL) return;
	drvr->name = "No Name";  drvr->drivername = "No Driver Name";
	drvr->device = NULL;
	drvr->mode = drvr->address = drvr->irq = drvr->buffsize = 0;
	drvr->next = driverlist;  driverlist = drvr;
	drvr->parms = NULL;
	switch (argc)
		{
		default:
		case 7: drvr->buffsize = strtoul(argv[6], NULL, 0);
		case 6: drvr->irq = strtoul(argv[5], NULL, 0);
		case 5: drvr->address = strtoul(argv[4], NULL, 0);
		case 4: drvr->mode = strtoul(argv[3], NULL, 0);
		case 3: drvr->drivername = strdup(argv[2]);
		case 2: drvr->name = strdup(argv[1]);
		break;
		}
	}

int vrl_ReadCFGProcessLine(char *buff)
	{
	char buffcopy[256];
	int argc;
	char *argv[20];
	strcpy(buffcopy, buff);  /* unparsed version */
	argc = parse(buff, " \t,", argv);
	if (argc < 1) return 0;  /* ignore blank lines */
	if (argc < 2) return 0;  /* all statements currently have at least one parameter */
	switch (lookup(argv[0], statement_table, ncmds))
		{
		st_version: break;
		case st_loadpath: vrl_FileSetLoadpath(argv[1]); break;
		case st_include:
			{
			FILE *new = fopen(vrl_FileFixupFilename(argv[1]), "r");
			if (new)
				{
				vrl_ReadCFG(new);
				fclose(new);
				}
			}
			break;
		case st_device: driver_setup(argc, argv); break;
		case st_compass: show_compass = !stricmp(argv[1], "on"); break;
		case st_position: show_position = !stricmp(argv[1], "on"); break;
		case st_framerate: show_framerate = !stricmp(argv[1], "on"); break;
		case st_displaydriver:
			display_drivername = strdup(argv[1]);
			if (!stricmp(display_drivername, "RVD"))
				display_driver = vrl_DisplayDriverRVD;
//			else if (!stricmp(display_drivername, "WGT"))
//				display_driver = vrl_DisplayDriverWGT;
			else
				display_driver = vrl_DisplayDriverRVD;
			if (argc > 1) display_mode = strtoul(argv[2], NULL, 0);
			break;
		case st_devconfig:
			{
			DriverDetails *p;
			ChannelParms *channel_info;
			int channel;
			if (argc < 3) break;
			for (p = driverlist; p; p = p->next)
				if (!stricmp(p->name, argv[1]))
					break;
			if (p == NULL) break;  /* driver not found */
			if (isdigit(*argv[2]))
				channel = atoi(argv[2]);
			else
				channel = toupper(*argv[2]) - 'X' + ((toupper(argv[2][1]) == 'R') ? 3 : 0);
			for (channel_info = p->parms; channel_info; channel_info = channel_info->next)
				if (channel_info->channel == channel)
					break;
			if (channel_info == NULL)
				{
				channel_info = malloc(sizeof(ChannelParms));
				channel_info->channel = channel;
				channel_info->scale = float2scalar(1000);
				channel_info->deadzone = 0;
				channel_info->next = p->parms;
				p->parms = channel_info;
				}
			if (argc > 3)
				{
				if (toupper(*argv[3]) == 'A')
					channel_info->scale = float2angle(atof(&argv[3][1]));
				else
					channel_info->scale = float2scalar(atof(argv[3]));
				}
			if (argc > 4)
				channel_info->deadzone = float2scalar(atof(argv[4]));
			}
			break;
		default:
			vrl_ReadWLDfeature(argc, argv, buffcopy);
			break;
		}
	return 0;
	}

int vrl_ReadCFG(FILE *in)
	{
	char buff[256];
	while (getline(buff, sizeof(buff), in))
		vrl_ReadCFGProcessLine(buff);
	return 0;
	}

vrl_Device *vrl_ConfigureDevice(DriverDetails *drvr)
	{
	int serial = 0;
	vrl_SerialPort *port = NULL;
	vrl_DeviceDriverFunction *fn;
	ChannelParms *channel_info;
	if (drvr == NULL) return NULL;
	if (!stricmp(drvr->drivername, "keypad")) { fn = vrl_KeypadDevice; serial = 0; }
	else if (!stricmp(drvr->drivername, "mouse")) { fn = vrl_MouseDevice; serial = 0; }
	else if (!stricmp(drvr->drivername, "joystick")) { fn = vrl_JoystickDevice; serial = 0; }
	else if (!stricmp(drvr->drivername, "GDC")) { fn = vrl_GlobalDevice; serial = 1; }
	else if (!stricmp(drvr->drivername, "Cyberman")) { fn = vrl_CybermanDevice; serial = 1; }
	else if (!stricmp(drvr->drivername, "Spaceball")) { fn = vrl_SpaceballDevice; serial = 1; }
	else if (!stricmp(drvr->drivername, "RedBaron")) { fn = vrl_RedbaronDevice; serial = 1; }
	else if (!stricmp(drvr->drivername, "CTM")) { fn = vrl_CTMDevice; serial = 1; }
	/* add other devices in here, as well as making entries in avrildrv.h */
	else return NULL;
	if (serial)
		port = vrl_SerialOpen(drvr->address, drvr->irq, drvr->buffsize);
	drvr->device = vrl_DeviceOpen(fn, port);
	if (drvr->device == NULL)
		return NULL;
	vrl_DeviceSetMode(drvr->device, drvr->mode);
	for (channel_info = drvr->parms; channel_info; channel_info = channel_info->next)
		{
		vrl_DeviceSetScale(drvr->device, channel_info->channel, channel_info->scale);
		vrl_DeviceSetDeadzone(drvr->device, channel_info->channel, channel_info->deadzone);
		}
	return drvr->device;
	}

void vrl_ConfigureAllDevices()
	{
	DriverDetails *p;
	for (p = driverlist; p; p = p->next)
		vrl_ConfigureDevice(p);
	}

vrl_Device *vrl_ConfigFindDevice(char *name)
	{
	DriverDetails *p;
	for (p = driverlist; p; p = p->next)
		if (p->name)
			if (!stricmp(name, p->name))
				return p->device;
	return NULL;
	}

void vrl_ConfigSetCompassDisplay(vrl_Boolean flag)
	{
	show_compass = flag;
	}

vrl_Boolean vrl_ConfigGetCompassDisplay(void)
	{
	return show_compass;
	}

void vrl_ConfigToggleCompassDisplay(void)
	{
	show_compass = !show_compass;
	}

void vrl_ConfigSetPositionDisplay(vrl_Boolean flag)
	{
	show_position = flag;
	}

vrl_Boolean vrl_ConfigGetPositionDisplay(void)
	{
	return show_position;
	}

void vrl_ConfigTogglePositionDisplay(void)
	{
	show_position = !show_position;
	}

void vrl_ConfigSetFramerateDisplay(vrl_Boolean flag)
	{
	show_framerate = flag;
	}

vrl_Boolean vrl_ConfigGetFramerateDisplay(void)
	{
	return show_framerate;
	}

void vrl_ConfigToggleFramerateDisplay(void)
	{
	show_framerate = !show_framerate;
	}

char *vrl_ConfigGetDisplayDriverName(void)
	{
	return display_drivername;
	}

vrl_DisplayDriverFunction *vrl_ConfigGetDisplayDriver(void)
	{
	return display_driver;
	}

unsigned int vrl_ConfigGetDisplayMode(void)
	{
	return display_mode;
	}

void vrl_ConfigStartup(char *filename)
	{
	FILE *cfgfile = fopen(vrl_FileFixupFilename(filename ? filename : "avril.cfg"), "r");
	if (cfgfile)
		{
		vrl_ReadCFG(cfgfile);
		fclose(cfgfile);
		}
	vrl_DisplaySetDriver(vrl_ConfigGetDisplayDriver(), vrl_ConfigGetDisplayMode());
	vrl_SystemStartup();
	vrl_ConfigureAllDevices();
	}

