/* The "vrl_System" functions for AVRIL applications */

/* Written by Bernie Roehl, January 1994 */

/* Copyright 1994 by Bernie Roehl */

/*
   You may use this code for your own non-commercial projects without
   paying any fees or royalties.  "Non-commercial", in this context,
   means that the software you write is given away for free to anyone
   who wants it.
   
   Commercial use, including shareware, requires a licensing
   fee and a specific written agreement with the author.

   All programs created using this software (both commercial and
   non-commercial) must acknowledge the use of the AVRIL library,
   both in the documentation and in a banner screen at the start or
   end of the program.

   For more information, contact Bernie Roehl (broehl@uwaterloo.ca).

*/

#include <stdlib.h>  /* getenv() */
#include <string.h>  /* strstr() */
#include <signal.h>

#include "avril.h"

static vrl_Boolean system_running = 0;   /* non-zero while system is initialized and running */

void vrl_SystemStartRunning(void)
	{
	system_running = 1;
	}

void vrl_SystemStopRunning(void)
	{
	system_running = 0;
	}

vrl_Boolean vrl_SystemIsRunning(void)
	{
	return system_running;
	}

static vrl_Boolean need_to_redraw = 0;   /* non-zero if the screen needs updating */

void vrl_SystemRequestRefresh(void)
	{
	need_to_redraw = 1;
	}

vrl_Boolean vrl_SystemQueryRefresh(void)
	{
	return need_to_redraw;
	}

vrl_Boolean vrl_SystemStartup(void)
	{
	vrl_MathInit();
	if (vrl_DisplayInit(0))
		{
		printf("Could not enter graphics mode!\n");
		return -1;
		}
	atexit(vrl_DisplayQuit);
	vrl_MouseInit();
	atexit(vrl_MouseQuit);
	if (vrl_TimerInit())
		return -2;
	atexit(vrl_TimerQuit);
	if (vrl_RenderInit(800, 800, 500, 5, 65000))
		return -3;
	atexit(vrl_RenderQuit);
	atexit(vrl_DeviceCloseAll);
	atexit(vrl_SerialCloseAll);
	/* make sure that exit() [and therefore the atexit() functions] get
	   called if there are any fatal errors */
	signal(SIGABRT, exit);
	signal(SIGFPE, exit);
	signal(SIGILL, exit);
	signal(SIGINT, exit);
	signal(SIGSEGV, exit);
	signal(SIGTERM, exit);
	vrl_SystemStartRunning();
	vrl_SystemRequestRefresh();
	vrl_SystemRender(NULL);
	return 0;
	}

static void check_mouse(void)
	{
	unsigned int mouse_buttons;
	if (vrl_MouseGetUsage())  /* being used as 6D pointing device */
		return;
	if (!vrl_MouseRead(NULL, NULL, NULL))  /* mouse hasn't changed */
		return;
	vrl_MouseRead(NULL, NULL, &mouse_buttons);
	if (mouse_buttons)  /* button down */
		{
		int mouse_x, mouse_y;
		int win_x, win_y;
		unsigned int down_buttons = mouse_buttons;
		vrl_DisplayGetWindow(&win_x, &win_y, NULL, NULL);
		while (mouse_buttons)  /* wait for button release */
			vrl_MouseRead(&mouse_x, &mouse_y, &mouse_buttons);
		if (down_buttons & 0x07)
			vrl_ApplicationMouseUp(mouse_x - win_x, mouse_y - win_y, down_buttons);
		}
	}

void vrl_SystemRun(void)
	{
	vrl_ApplicationInit();
	while (vrl_SystemIsRunning())
		{
		vrl_Object *list;
		if (vrl_KeyboardCheck())
			vrl_ApplicationKey(vrl_KeyboardRead());
		check_mouse();
		vrl_TaskRun();
		vrl_DevicePollAll();
		list = vrl_WorldUpdate();
		if (vrl_SystemQueryRefresh())
			vrl_SystemRender(list);
		}
	}

static vrl_Time last_render_ticks;

vrl_Time vrl_SystemGetRenderTime(void)
	{
	return last_render_ticks;
	}

vrl_Time vrl_SystemGetFrameRate(void)
	{
	if (last_render_ticks == 0) last_render_ticks = 1;  /* so we don't divide by zero! */
	return vrl_TimerGetTickRate() / last_render_ticks;
	}

#include "vrlstats.h"  // DEBUG ONLY!!!

vrl_RenderStatus *vrl_SystemRender(vrl_Object *list)
	{
	static vrl_Object *lastlist = NULL;
	int pagenum;
	vrl_RenderStatus *stat;
	vrl_Time render_start;
	if (list == NULL)
		list = lastlist;
	else
		lastlist = list;
	pagenum = vrl_DisplayGetDrawPage();
	if (++pagenum >= vrl_DisplayGetNpages())
		pagenum = 0;
	vrl_DisplaySetDrawPage(pagenum);
	render_start = vrl_TimerRead();
	vrl_RenderBegin(vrl_WorldGetCamera(), vrl_WorldGetLights());
	vrl_RenderSetAmbient(vrl_WorldGetAmbient());
	if (vrl_WorldGetScreenClear())
		{
		if (vrl_WorldGetHorizon() && !vrl_DisplayGetDrawMode())
			vrl_RenderHorizon();
		else
			vrl_DisplayClear(vrl_WorldGetSkyColor());
		}
	vrl_ApplicationDrawUnder();
	stat = vrl_RenderObjlist(list);
	last_render_ticks = vrl_TimerRead() - render_start;
	vrl_ApplicationDrawOver(stat);
	vrl_MouseCursorHide();
	vrl_DisplayUpdate();
	vrl_DisplaySetViewPage(pagenum);
	vrl_MouseCursorShow();
	need_to_redraw = 0;
	return stat;
	}

void vrl_SystemCommandLine(int argc, char *argv[])
	{
	int i;
	vrl_Camera *cam;
	for (i = 1; i < argc; ++i)  /* i = 1 to skip argv[0] */
		{
		FILE *in = fopen(argv[i], "r");
		if (in == NULL) continue;
		if (strstr(argv[i], ".wld"))
			vrl_ReadWLD(in);
		else if (strstr(argv[i], ".fig"))
			vrl_ReadFIG(in, NULL, NULL);
		else if (strstr(argv[i], ".plg"))
			vrl_ReadObjectPLG(in);
		/* ignore anything else */
		fclose(in);
		}
	if (!vrl_WorldGetCamera())   /* need to have a camera */
		vrl_CameraCreate();
	vrl_WorldUpdate();
	}

