/* This is a sample program that demonstrates how to make use of the mouse
 * driver in SuperVGA modes.  The program also accepts an argument to
 * indicate the desired 'mode'.  Valid modes may be found by looking in the
 * graph.h header file.
 * USE:  WCL386 (or WCL) -ox etch.c
 * Note: -ox MUST be used or the stack pragmas will not work properly
 */

#include <conio.h>
#include <dos.h>
#include <graph.h>


void main( int argc, char *argv[] )
//=================================

{
    int                 mode;
    struct videoconfig  vc;

    if( argc == 2 ) {
	mode = atoi( argv[ 1 ] );
    } else {
	mode = _MAXRESMODE;
    }
    if( _setvideomode( mode ) == 0 ) {
	printf( "No graphics adapter present\n" );
	exit( 1 );
    }
    if( InitMouse() == 0 ) {
	printf( "No mouse driver present\n" );
	exit( 1 );
    }
    _getvideoconfig( &vc );
    SetMouseRange( 0, 0, vc.numxpixels - 1, vc.numypixels - 1 );
    _grtext( 5, 50, "Press <ESC> to clear screen, <END> to quit");
    _grtext( 5, 80, "You may also enter a mode (ie. 258) as a");
    _grtext( 5, 110, "parameter when invoking etch.exe");
    Etch();
    FiniMouse();
    _setvideomode( _DEFAULTMODE );
}


void Etch( void )
//===============

// Follow the mouse and draw while the mouse button is pressed.
// If 'Esc' is pressed, clear the screen. If 'End' is pressed, exit.

{
    int                 pen_down, button, ch;
    struct xycoord      curr_pos, prev_pos;

    CursorOn();
    pen_down = 0;

    for( ;; ) {
	GetPosition( &curr_pos, &button );
	if( button != 0 ) {     // button pressed
	    if( pen_down != 1 ) {
		pen_down = 1;
		_moveto( curr_pos.xcoord, curr_pos.ycoord );
		prev_pos = curr_pos;
	    } else {
		if( prev_pos.xcoord != curr_pos.xcoord || prev_pos.ycoord != curr_pos.ycoord ) {
		    CursorOff();
		    _lineto( curr_pos.xcoord, curr_pos.ycoord );
		    CursorOn();
		    prev_pos = curr_pos;
		}
	    }
	} else {
	    pen_down = 0;
	}
	if( kbhit() ) {
	    ch = getch();
	    if( ch == 0 ) {
		ch = 256 + getch();
	    }
	    if( ch == 27 ) {              /* ESC key */
		CursorOff();
		_clearscreen( _GCLEARSCREEN );
		CursorOn();
	    } else if( ch == 335 ) {      /* END key */
		return;
	    }
	}
    }
}


// Mouse Library


#define MOUSE_INT       0x33

int                     MouseCursorOn = 0;
int                     MouseX;
int                     MouseY;
int                     WeDrawCursor = 0;

#if !defined( __386__ )
  #pragma aux           GetStack = "mov dx,ss" \
				   "mov ax,sp" value [ dx ax ];
  #pragma aux           SetStack = "mov ss,dx" \
				   "mov sp,ax" parm [ dx ax ];
  #pragma aux           mouse_handler parm [ cx ] [ dx ]
#else
  #pragma aux           GetStack = "mov dx,ss" \
				   "mov eax,esp" value [ dx eax ];
  #pragma aux           SetStack = "mov ss,dx" \
				   "mov esp,eax" parm [ dx eax ];
  #pragma aux           mouse_handler parm [ ecx ] [ edx ]
#endif

#define STACKSIZE       1024

extern void far         *GetStack();
extern void             SetStack( char far * );


static void _loadds far mouse_handler( int x, int y )
//===================================================

{
    static char         newstack[ STACKSIZE ];
    static void far     *oldstack;

    if( MouseCursorOn ) {
	oldstack = GetStack();
	SetStack( newstack + STACKSIZE );
	DrawCursor();
	MouseX = x;
	MouseY = y;
	DrawCursor();
	SetStack( oldstack );
    } else {
	MouseX = x;
	MouseY = y;
    }
}


static void DrawCursor()
//======================

{
    int                 old_act;
    int                 old_style;
    struct xycoord      old_pos;

    old_act = _setplotaction( _GXOR );
    old_style = _getlinestyle();
    _setlinestyle( 0xffff );
    old_pos = _moveto( MouseX, MouseY );
    _lineto( MouseX + 15, MouseY + 15 );
    _moveto( MouseX, MouseY + 8 );
    _lineto( MouseX, MouseY );
    _lineto( MouseX + 8, MouseY );
    _moveto( old_pos.xcoord, old_pos.ycoord );
    _setlinestyle( old_style );
    _setplotaction( old_act );
}


int InitMouse( void )
//===================

{
    int                 result;
    union REGPACK       regs;
    struct xycoord      pos;
    struct videoconfig  vc;

    memset( &regs, 0, sizeof( union REGPACK ) );
    regs.w.ax = 0;
    intr( MOUSE_INT, &regs );
    result = regs.w.ax;
    if( result != 0 ) {
	_getvideoconfig( &vc );
	if( vc.mode >= _URES256COLOR ) {
	    // SuperVGA mode, we have to draw the mouse cursor ourselves
	    WeDrawCursor = 1;
	    regs.w.ax = 0x000C;
	    regs.w.cx = 0x0001;
#if defined( __386__ )
	    regs.x.edx = FP_OFF( &mouse_handler );
#else
	    regs.w.dx = FP_OFF( &mouse_handler );
#endif
	    regs.w.es = FP_SEG( &mouse_handler );
	    intr( MOUSE_INT, &regs );
	    // place mouse in centre of screen
	    MouseX = vc.numxpixels / 2;
	    MouseY = vc.numypixels / 2;
	    pos.xcoord = MouseX;
	    pos.ycoord = MouseY;
	    SetPosition( &pos );
	    MouseCursorOn = 1;
	    DrawCursor();
	}
    }
    return( result );
}


void CursorOn( void )
//===================

{
    union REGPACK       regs;

    if( WeDrawCursor ) {
	if( !MouseCursorOn ) {
	    DrawCursor();
	}
	MouseCursorOn = 1;
    } else {
	memset( &regs, 0, sizeof( union REGPACK ) );
	regs.w.ax = 1;
	intr( MOUSE_INT, &regs );
    }
}


void CursorOff( void )
//====================

{
    union REGPACK       regs;

    if( WeDrawCursor ) {
	if( MouseCursorOn ) {
	    DrawCursor();
	}
	MouseCursorOn = 0;
    } else {
	memset( &regs, 0, sizeof( union REGPACK ) );
	regs.w.ax = 2;
	intr( MOUSE_INT, &regs );
    }
}


void GetPosition( struct xycoord *pos, int *left )
//================================================

{
    union REGPACK       regs;

    memset( &regs, 0, sizeof( union REGPACK ) );
    regs.w.ax = 3;
    intr( MOUSE_INT, &regs );
    pos->xcoord = regs.w.cx;
    pos->ycoord = regs.w.dx;
    *left = regs.w.bx & 1;
}


void SetPosition( struct xycoord *pos )
//=====================================

{
    union REGPACK       regs;

    memset( &regs, 0, sizeof( union REGPACK ) );
    regs.w.ax = 4;
    regs.w.cx = pos->xcoord;
    regs.w.dx = pos->ycoord;
    intr( MOUSE_INT, &regs );
}


void SetMouseRange( int x1, int y1, int x2, int y2 )
//==================================================

{
    union REGPACK       regs;

    memset( &regs, 0, sizeof( union REGPACK ) );
    regs.w.ax = 7;
    regs.w.cx = x1;
    regs.w.dx = x2;
    intr( MOUSE_INT, &regs );
    regs.w.ax = 8;
    regs.w.cx = y1;
    regs.w.dx = y2;
    intr( MOUSE_INT, &regs );
}


void FiniMouse()
//==============

{
    union REGPACK       regs;

    if( WeDrawCursor ) {    // init again, to clear handler
	memset( &regs, 0, sizeof( union REGPACK ) );
	regs.w.ax = 0;
	intr( MOUSE_INT, &regs );
    }
}
