/*****************************************************************************
*   "Gif-Lib" - Yet another gif library.				     *
*									     *
* Written by:  Gershon Elber			IBM PC Ver 0.1,	Jul. 1989    *
******************************************************************************
* Program to dump GIF file into PostScript type printers		     *
* Options:								     *
* -x : force image to be horizontal.					     *
* -y : force image to be vertical.					     *
* -s x y : force image to be of given size.				     *
* -p x y : force image to be positioned at given position in page.	     *
* -i : invert the image.						     *
* -n n : number of copies.						     *
* -h : on line help.							     *
******************************************************************************
* History:								     *
* 22 Dec 89 - Version 1.0 by Gershon Elber.				     *
*****************************************************************************/

#ifdef __MSDOS__
#include <graphics.h>
#include <stdlib.h>
#include <alloc.h>
#include <io.h>
#include <dos.h>
#include <bios.h>
#endif /* __MSDOS__ */

#include <stdio.h>
#include <ctype.h>
#include <string.h>
#include <fcntl.h>
#include "gif_lib.h"
#include "getarg.h"

#define PROGRAM_NAME	"Gif2PS"
#define VERSION		" Version 1.0, "

#define PAGE_WIDTH      7.5		    /* All dimensions are in inches. */
#define PAGE_HEIGHT     9.0
#define FULL_PAGE_WIDTH 8.5
#define FULL_PAGE_HEIGHT 11.0

#define UNKNOWN_ORIENT		0
#define HORIZONTAL_ORIENT	1 
#define VERTICAL_ORIENT		2

#ifdef __MSDOS__
extern unsigned int
    _stklen = 16384;			      /* Increase default stack size */
#endif /* __MSDOS__ */

#ifdef SYSV
static char *VersionStr =
        "Gif library module,\t\tGershon Elber\n\
	(C) Copyright 1989 Gershon Elber, Non commercial use only.\n";
static char
    *CtrlStr = "Gif2PS x%- y%- s%-SizeX|SizeY!F!F p%-PosX|PosY!F!F i%- n%-#Copies!d h%- GifFile!*s";
#else
static char
    *VersionStr =
	PROGRAM_NAME
	"	IBMPC "
	VERSION
	"	Gershon Elber,	"
	__DATE__ ",   " __TIME__ "\n"
	"(C) Copyright 1989 Gershon Elber, Non commercial use only.\n";
static char
    *CtrlStr =
	PROGRAM_NAME
	" x%- y%- s%-SizeX|SizeY!F!F p%-PosX|PosY!F!F i%- n%-#Copies!d h%- GifFile!*s";
#endif /* SYSV */

static char
    *ProgramName;
/* Make some variables global, so we could access them faster: */
static int
    ImageNum = 0,
    BackGround = 0,
    ForceXFlag = FALSE,
    ForceYFlag = FALSE,
    SizeFlag = FALSE,
    PosFlag = FALSE,
    InvertFlag = FALSE,
    NumCopiesFlag = FALSE,
    HelpFlag = FALSE,
    NumOfCopies = 1,
    InterlacedOffset[] = { 0, 4, 2, 1 }, /* The way Interlaced image should. */
    InterlacedJumps[] = { 8, 8, 4, 2 },    /* be read - offsets and jumps... */
    PSOrientation;
static double PSSizeX, PSSizeY, PSPosX, PSPosY;
static GifColorType
    *ColorMap;

static void DumpScreen2PS(RowType *ScreenBuffer,
					int ScreenWidth, int ScreenHeight);
static void PutString(unsigned char *Line, int Len);

/******************************************************************************
* Interpret the command line and scan the given GIF file.		      *
******************************************************************************/
void main(int argc, char **argv)
{
    int	i, j, Error, NumFiles, Size, Row, Col, Width, Height, ExtCode, Count;
    GifRecordType RecordType;
    ByteType *Extension;
    char **FileName = NULL;
    RowType *ScreenBuffer;
    GifFileType *GifFile;

    if (strlen(ProgramName = argv[0]) == 0)		    /* DOS 3.x only! */
	ProgramName = PROGRAM_NAME;	  /* Do something reasonable for 2.x */

    if ((Error = GAGetArgs(argc, argv, CtrlStr, &ForceXFlag, &ForceYFlag,
		&SizeFlag, &PSSizeX, &PSSizeY,
		&PosFlag, &PSPosX, &PSPosY,
		&InvertFlag, &NumCopiesFlag, &NumOfCopies, &HelpFlag,
		&NumFiles, &FileName)) != FALSE ||
		(NumFiles > 1 && !HelpFlag)) {
	if (Error) GAPrintErrMsg(Error);
	else
	if (NumFiles > 1)
	    MESSAGE("Error in command line parsing - one GIF file please\n");
	GAPrintHowTo(CtrlStr);
	exit(1);
    }

    if (HelpFlag) {
	fprintf(stderr, VersionStr);
	GAPrintHowTo(CtrlStr);
	exit(0);
    }

    if (ForceXFlag) PSOrientation = HORIZONTAL_ORIENT;
    else
    if (ForceYFlag) PSOrientation = VERTICAL_ORIENT;
    else PSOrientation = UNKNOWN_ORIENT;
    
    if (NumFiles == 1) {
	if ((GifFile = DGifOpenFileName(*FileName)) == NULL) {
	    PrintGifError();
	    exit(-1);
	}
    }
    else {
	/* Use the stdin instead: */

#ifdef __MSDOS__
	setmode(0, O_BINARY);
#endif /* __MSDOS__ */
	if ((GifFile = DGifOpenFileHandle(0)) == NULL) {
	    PrintGifError();
	    exit(-1);
	}
    }

    /* Allocate the screen as vector of column of rows. We cannt allocate    */
    /* the all screen at once, as this broken minded CPU can allocate up to  */
    /* 64k at a time and our image can be bigger than that:		     */
    /* Note this screen is device independent - its the screen as defined by */
    /* the GIF file parameters itself.					     */
    if ((ScreenBuffer = (RowType *)
	malloc(GifFile -> SHeight * sizeof(RowType *))) == NULL)
	    EXIT("Failed to allocate memory required, aborted");

    Size = GifFile -> SWidth * sizeof(PixelType);/* Size in bytes of one row */
    if ((ScreenBuffer[0] = (RowType) malloc(Size)) == NULL)     /* First row */
	EXIT("Failed to allocate memory required, aborted");

    for (i=0; i<GifFile -> SWidth; i++)	  /* And set its color to BackGround */
	ScreenBuffer[0][i] = GifFile -> SBackGroundColor;
    for (i=1; i<GifFile -> SHeight; i++) {
	/* Allocate the other rows, and set their color to background too: */
	if ((ScreenBuffer[i] = (RowType) malloc(Size)) == NULL)
	    EXIT("Failed to allocate memory required, aborted\n");

	memcpy(ScreenBuffer[i], ScreenBuffer[0], Size);
    }

    /* Scan the content of the GIF file and load the image(s) in: */
    do {
	if (DGifGetRecordType(GifFile, &RecordType) == ERROR) {
	    PrintGifError();
	    exit(-1);
	}
	switch (RecordType) {
	    case IMAGE_DESC_RECORD_TYPE:
		if (DGifGetImageDesc(GifFile) == ERROR) {
		    PrintGifError();
		    exit(-1);
		}
		Row = GifFile -> ITop;  /* Image Position relative to Screen */
		Col = GifFile -> ILeft;
		Width = GifFile -> IWidth;
		Height = GifFile -> IHeight;
		fprintf(stderr, "\n%s: Image %d at (%d, %d) [%dx%d]:     ",
		    ProgramName, ++ImageNum, Col, Row, Width, Height);
		if (GifFile -> ILeft + GifFile -> IWidth > GifFile -> SWidth ||
		   GifFile -> ITop + GifFile -> IHeight > GifFile -> SHeight) {
		    fprintf(stderr, "Image %d is not confined to screen dimension, aborted\n");
		    exit(-2);
		}
		if (GifFile -> IInterlace) {
		    /* Need to perform 4 passes on the images: */
		    for (Count=i=0; i<4; i++)
			for (j=Row + InterlacedOffset[i]; j<Row + Height;
						 j += InterlacedJumps[i]) {
			    fprintf(stderr, "\b\b\b\b%-4d", Count++);
			    if (DGifGetLine(GifFile, &ScreenBuffer[j][Col],
				Width) == ERROR) {
				PrintGifError();
				exit(-1);
			    }
			}
		}
		else {
		    for (i=0; i<Height; i++) {
			fprintf(stderr, "\b\b\b\b%-4d", i);
			if (DGifGetLine(GifFile, &ScreenBuffer[Row++][Col],
				Width) == ERROR) {
			    PrintGifError();
			    exit(-1);
			}
		    }
		}
		break;
	    case EXTENSION_RECORD_TYPE:
		/* Skip any extension blocks in file: */
		if (DGifGetExtension(GifFile, &ExtCode, &Extension) == ERROR) {
		    PrintGifError();
		    exit(-1);
		}
		while (Extension != NULL) {
		    if (DGifGetExtensionNext(GifFile, &Extension) == ERROR) {
			PrintGifError();
			exit(-1);
		    }
		}
		break;
	    case TERMINATE_RECORD_TYPE:
		break;
	    default:		     /* Should be traps by DGifGetRecordType */
		break;
	}
    }
    while (RecordType != TERMINATE_RECORD_TYPE);

    /* Lets display it - set the global variables required and do it: */
    BackGround = GifFile -> SBackGroundColor;
    ColorMap = (GifFile -> IColorMap ? GifFile -> IColorMap :
				       GifFile -> SColorMap);
    DumpScreen2PS(ScreenBuffer, GifFile -> SWidth, GifFile -> SHeight);

    if (DGifCloseFile(GifFile) == ERROR) {
	PrintGifError();
	exit(-1);
    }
}

/******************************************************************************
* The real dumping routine.						      *
******************************************************************************/
static void DumpScreen2PS(RowType *ScreenBuffer,
					int ScreenWidth, int ScreenHeight)
{
    int i, j;
    double Aspect;
    ByteType *OutLine, Data;
    PixelType *Line;
    GifColorType *ColorMapEntry;

    /* If user did not enforce orientation, pick the best one. */
    if (PSOrientation == UNKNOWN_ORIENT)
	if (ScreenWidth > ScreenHeight)
	     PSOrientation = VERTICAL_ORIENT;
        else PSOrientation = HORIZONTAL_ORIENT;

    Aspect = ((double) ScreenHeight) / ((double) ScreenWidth);

    if (!SizeFlag)
	switch (PSOrientation) {
            case HORIZONTAL_ORIENT:
	        if (Aspect > PAGE_HEIGHT / PAGE_WIDTH) {
		    PSSizeX = PAGE_HEIGHT / Aspect; 
		    PSSizeY = PAGE_HEIGHT;
		}
		else {
		    PSSizeX = PAGE_WIDTH;
		    PSSizeY = PAGE_WIDTH * Aspect;
		}
		break;
	    case VERTICAL_ORIENT:
	        if (1 / Aspect > PAGE_HEIGHT / PAGE_WIDTH) {
		    PSSizeX = PAGE_HEIGHT * Aspect; 
		    PSSizeY = PAGE_HEIGHT;
		}
		else {
		    PSSizeX = PAGE_WIDTH;
		    PSSizeY = PAGE_WIDTH / Aspect;
		}
		break;
	}
    else {
	if (PAGE_WIDTH < PSSizeX) {
	    MESSAGE("X Size specified is too big, page size selected.\n");
	    PSSizeX = PAGE_WIDTH;
	}
	if (PAGE_HEIGHT < PSSizeY) {
	    MESSAGE("Y Size specified is too big, page size selected.\n");
	    PSSizeX = PAGE_HEIGHT;
	}
    }

    if (!PosFlag) {
	PSPosX = (FULL_PAGE_WIDTH - PSSizeX) / 2;
	PSPosY = (FULL_PAGE_HEIGHT - PSSizeY) / 2;
    }
    else {
	if (PSPosX + PSSizeX > PAGE_WIDTH || PSPosY + PSSizeY > PAGE_HEIGHT)
	    EXIT("Requested position will put image out of page, aborted\n");
    }

    /* Time to dump out the PostScript header: */
    printf("%%!\n");
    printf("%%%%Creator: %s\n", PROGRAM_NAME);
    printf("/#copies %d def\n", NumOfCopies);
    printf("gsave\n");
    printf("72 72 scale\t\t\t\t%% Lets talk inches.\n");
    printf("/oneline %d string def\t\t\t%% Allocate one scan line.\n",
	   ScreenWidth);
    printf("/drawimage {\n");
    printf("\t%d %d 8 [%d 0 0 %d 0 %d]\n", ScreenWidth, ScreenHeight,
	   ScreenWidth, -ScreenHeight, ScreenHeight);
    printf("\t{ currentfile oneline readhexstring pop } image\n");
    printf("} def\n");
    switch (PSOrientation) {
        case HORIZONTAL_ORIENT:
	    printf("%lf %lf translate\n", PSPosX, PSPosY);
	    printf("%lf %lf scale\n", PSSizeX, PSSizeY);
	    break;
	case VERTICAL_ORIENT:
	    printf("%lf %lf translate\n", PSPosX + PSSizeX, PSPosY);
	    printf("90 rotate\n");
	    printf("%lf %lf scale\n", PSSizeY, PSSizeX);
	    break;
    }
    printf("drawimage\n");

    if ((OutLine = (ByteType *) malloc(sizeof(ByteType) * ScreenWidth))
								== NULL)
	EXIT("Failed to allocate memory required, aborted");

    for (i=0; i<ScreenHeight; i++)
    {
	fprintf(stderr, "\b\b\b\b%-4d", ScreenHeight - i);
	
	Line = ScreenBuffer[i];
	for (j=0; j<ScreenWidth; j++) {
	    ColorMapEntry = &ColorMap[Line[j]];
	    Data = (30 * ((unsigned int) ColorMapEntry->Red) +
		    59 * ((unsigned int) ColorMapEntry->Green) +
		    11 * ((unsigned int) ColorMapEntry->Blue)) / 100;
	    OutLine[j] = InvertFlag ? 255 - Data : Data;
	}
	
	PutString(OutLine, ScreenWidth);
    }
    free(OutLine);

    printf("\nshowpage\n");
    printf("grestore\n");
}

/******************************************************************************
* Dumps the string of given length as 2 hexdigits per byte 39 bytes per line. *
******************************************************************************/
static void PutString(unsigned char *Line, int Len)
{
    int i;
    static Counter = 0;
    static char *Hex = "0123456789ABCDEF";

    for (i=0; i<Len; i++) {
	if (++Counter % 40 == 0) {
	    putchar('\n');
	    Counter = 1;
	}
	putchar(Hex[Line[i] >> 4]);
	putchar(Hex[Line[i] & 0x0f]);
    }
}
