//////////////////////////////////////////////////////////////////////////////
//    FILE: FONTS.CPP
//
//    A simple bitmap font class. Requires defined graphics mode 0x13
//////////////////////////////////////////////////////////////////////////////

#include <mstd.h>
#include <mobject.h>
#include <merror.h>
#include <mrect.h>
#include <mfileop.h>
#include <mmem.h>
#include <marray.h>
#include <mlist.h>
#include <mstaque.h>
#include <mgraph.h>

/*********************************************/
      MFont :: MFont (void)
/*********************************************
 DESCRIPTION: Creates a font with zero dimentions.
*/
{
    length = 0;
    width = 0;
    strcpy (Filename, "default.fnt\x0");
}

/*********************************************/
 void MFont :: LoadDirect
/*********************************************
 DESCRIPTION: Reads the font file into the memory. Font
	       file is formatted as follows: "Bitmap font
	       file" string, then the dimentions, then the
	       bitmaps. Each file contains NOL bitmaps.
*/
(int fp)
{
char str [ID_STR_LEN];

    read (fp, str, ID_STR_LEN * sizeof(char));    // Check if this is a font file
    if (strncmp (str, ID_STRING, ID_STR_LEN) == 0)
    {
       read (fp, &length, sizeof(BYTE));
       read (fp, &width, sizeof(BYTE));
       Atom :: Resize (NOL, (WORD)(length * width / 8) + 1);
       read (fp, mem, size);
    }
    else
       PrintTextError ("Not a font file!");
}

/*********************************************/
 void MFont :: SaveDirect
/*********************************************
 DESCRIPTION: Saves the font file on disk. Font
	       file is formatted as follows: "Bitmap font
	       file" string, then the dimentions, then the
	       bitmaps. Each file contains NOL bitmaps.
*/
(int fp)
{
BYTE str[ID_STR_LEN];

    sprintf (str, ID_STRING);
    write (fp, str, ID_STR_LEN * sizeof(BYTE));
    write (fp, &length, sizeof(BYTE));
    write (fp, &width, sizeof(BYTE));
    write (fp, mem, size);
}

/********************************************/
 void MFont :: ActivatePoint
/********************************************
 DESCRIPTION: Sets a point at coordinates (xpos, ypos)
	       in letter to TRUE (Font editor).
*/
(WORD letter, WORD xpos, WORD ypos)
{
BYTE *curlet = NULL;
WORD offset;

    curlet = (BYTE*)( Atom :: operator[] (letter));
    offset = ypos * length + xpos;
    while (offset > 7)
    {
       ++ curlet;
       offset -= 8;
    }
    curlet[0] |= bits [offset % 8];
}

/********************************************/
 void MFont :: DeactivatePoint
/********************************************
 DESCRIPTION: Sets a point at coordinates (xpos, ypos)
	       in letter to FALSE (Font editor).
*/
(WORD letter, WORD xpos, WORD ypos)
{
BYTE *curlet = NULL;
WORD offset;

    curlet = (BYTE*)( Atom :: operator[] (letter));
    offset = ypos * length + xpos;
    while (offset > 7)
    {
       ++ curlet;
       offset -= 8;
    }
    curlet[0] &= (~bits [offset % 8]);
}

/********************************************/
 void MFont :: TogglePoint
/********************************************
 DESCRIPTION: Toggles value of a point at
	      (xpos,ypos) in letter;
*/
(WORD letter, WORD xpos, WORD ypos)
{
BYTE *curlet = NULL;
WORD offset;

    curlet = (BYTE*)( Atom :: operator[] (letter));
    offset = ypos * length + xpos;
    while (offset > 7)
    {
       ++ curlet;
       offset -= 8;
    }
    curlet[0] ^= bits [offset % 8];
}


/********************************************/
 BOOL MFont :: ReadPoint
/********************************************
 DESCRIPTION: Returns the status of a point at
	       coordinates (xpos, ypos) in letter.
*/
(WORD letter, WORD xpos, WORD ypos)
{
BYTE *curlet = NULL;
WORD offset;

    curlet = (BYTE*)( Atom :: operator[] (letter));
    offset = ypos * length + xpos;
    while (offset > 7)
    {
       ++ curlet;
       offset -= 8;
    }
    return ((curlet[0] & bits [offset % 8]) != 0);
}

/********************************************/
 void MFont :: Resize
/********************************************
 DESCRIPTION: Resizes the font to new dimentions
	       and clears all the bitmaps.
*/
(WORD len, WORD wid)
{
    length = len; width = wid;
    Atom :: Resize (NOL, (WORD)(length*width/8) + 1);
    Atom :: Clear();
}

/*********************************************/
 int  MFont :: PrintString
/*********************************************/
(int x, int y, char *string)
{
int n,m;
    m = x;
    for (n = 0;n < strlen (string); ++ n)
       m += PrintCharacter (m, y, (int) string[n]);
    return (m - x);
}

/*********************************************/
 int  MFont :: PrintCharacter
/*********************************************/
(int x,int y,int ascii)
{
BYTE *curlet;        // Pointer to the letter bitmap.
int xpos,ypos,pos = 0;  // Cycle variables.
int mpos = 0;            // Maximum character length

    curlet = (BYTE*)( Atom :: operator[] (ascii));

    for (ypos = 0;ypos < width; ++ ypos)
    {
       for (xpos = 0;xpos < length; ++ xpos)
       {
	  if ((curlet[0] & bits[pos]) > 0)
	  {
	     putpixel (x + xpos, y + ypos);
	     if (mpos < xpos)
		mpos = xpos;
	  }

	  if (pos < 7)     // Skip to a new byte in memory if the
	     ++ pos;       //  end was reached in the previous one.
	  else
	  {
	     pos = 0;
	     ++ curlet;
	  }
       }
    }
    return (mpos + 2);
}

/*********************************************/
      MFont :: ~MFont (void)
/*********************************************/
{
    // All the deallocation is done by ATOM destructor.
}
