/////////////////////////////////////////////////////////////////////////////////
//     FILE: IMAGES.CPP
//
//   IMAGES is a class derived from cArray and features screen image          //
// operations like scanning, drawing and disk interactions.                   //
////////////////////////////////////////////////////////////////////////////////

#include <mstd.h>
#include <mobject.h>
#include <mrect.h>
#include <mfileop.h>
#include <mmem.h>
#include <marray.h>
#include <mimages.h>
#include <mlist.h>
#include <mstaque.h>
#include <mgraph.h>
#include <mgerr.h>

/*********************************************/
      MImage :: MImage
/*********************************************
 DESCRIPTION: Creates an image with given dimentions
	       from byte array nmem. All parameters
	       have defaults, defined in the header.
*/
(WORD len, WORD wid, BYTE *nmem) : cArray ((SIZE)len*wid, nmem)
{
    Length = len;
    Width = wid;
    CompSize = 0;
    PutType = NORMAL_PUT;
}

/*********************************************/
      MImage :: MImage
/*********************************************
 DESCRIPTION: A copy constructor.
*/
(const IMAGE& image) : cArray (image.size, (BYTE*)image.mem)
{
    if (&image != NULL)
    {
       Length = image.Length;
       Width = image.Width;
       CompSize = image.CompSize;
       PutType = image.PutType;
    }
}

/********************************************/
 void MImage :: Compress ()
/********************************************
 DESCRIPTION: compresses the image by deleting
	      the repetitions. Used in saving
	      the image.
*/
{
WORD count,n;
WORD opos = 0,npos = 0;
BYTE *bitmap;

    bitmap = (BYTE*)mem;
    do
    {
       count = 1;
       while ((bitmap[opos] == bitmap[opos+1]) && (opos + 1 < size) && (count < MAXWORD))
       {
	  ++ opos;
	  ++ count;
       }
       if (count < 5)
       {
          for (n = 0; n < count; ++ n)
	  {
             if (bitmap[opos] == (BYTE)REPEAT_SIGN)
                -- bitmap[opos];
	     bitmap[npos] = bitmap[opos];
	     ++ npos;
          }
       }
       else
       {
	  bitmap[npos] = (BYTE)(REPEAT_SIGN);
	  ++ npos;
	  bitmap[npos] = HIBYTE(count);
	  ++ npos;
          bitmap[npos] = LOBYTE(count);
          ++ npos;
          bitmap[npos] = bitmap[opos];
          ++ npos;
       }
       ++ opos;
    }
    while (opos < size);

    CompSize = npos;
}

/********************************************/
 void MImage :: Decompress ()
/********************************************
 DESCRIPTION: decompresses the image by restoring
	      the repetitions. Used in restoring
	      saved image.
*/
{
WORD count;
WORD opos = 0,npos = 0,n;
cArray buffer (size * sizeof(BYTE));
BYTE * bitmap = (BYTE*) mem;

    do
    {
       if (bitmap[opos] == REPEAT_SIGN)
       {
	  ++ opos;
	  count = MAKEWORD (bitmap [opos+1], bitmap [opos]);
	  opos += 2;   // Address bytes

	  for (n = 0; n < count; ++ n)
	  {
	     buffer[npos] = bitmap[opos];
	     ++ npos;
	  }
	  ++ opos;
       }
       else
       {
	  buffer[npos] = bitmap[opos];
	  ++ npos;
	  ++ opos;
       }
    }
    while (opos < CompSize);

    if (npos != size)
       PrintError ("Decompressed image is invalid!", FALSE);

    buffer.Move (PutOn, 0, bitmap, npos * sizeof(BYTE));
    CompSize = 0;
}

/*********************************************/
 inline void MImage :: MoveMasked
/*********************************************/
(WAY Way, WORD Offset, PTR Where, WORD HowMany, SIZE OfHowMuch, WORD Skip, WORD Ignore)
{
    if (Way == PutOn)
       MoveMemMasked ((BYTE*)mem + Offset, Where, HowMany, min (OfHowMuch, size), Skip, Ignore);
    else
       MoveMemMasked (Where, (BYTE*)mem + Offset, HowMany, min (OfHowMuch, size), Skip, Ignore);
}

/*********************************************/
 void MImage :: Blend
/*********************************************
 Blends two images pixel by pixel with boolean
 operations like AND, OR and NOT.
*/
(ADR with, ADR into)
{
WORD n;
BYTE *bitmap;
    bitmap = (BYTE*) mem;
    switch (PutType)
    {
	case SEETH_PUT:     // First image is obscure, except for T_COLOR
		for (n = 0;n < size; ++ n)
		{
		   if (bitmap[n] == T_COLOR)
		      into[n] = with[n];
		   else
		      into[n] = bitmap[n];
		}
		break;
	case AND_PUT:       // Pixels are ANDed
		for (n = 0;n < size; ++ n)
		   into[n] = with[n] & bitmap[n];
		break;
	case XOR_PUT:       // Pixels are XORed
		for (n = 0;n < size; ++ n)
		   into[n] = with[n] ^ bitmap[n];
		break;
	case OR_PUT:        // Pixels are ORed
		for (n = 0;n < size; ++ n)
		   into[n] = with[n] | bitmap[n];
		break;
	default:
		MoveMem (with, into, size);
		break;
    }
}

/*********************************************/
 void MImage :: Get
/*********************************************/
(int offx,int offy)
{
ADR src;
VP clipped;
WORD clip_offset;
    clipped.Set (offx, offy, offx + Length, offy + Width);
    clipscreen (&clipped);

    if (clipped.length > 0 && clipped.width > 0)
    {
       clip_offset = (clipped.top - offy) * Length + (clipped.left - offx);

       src = gr.dpy.address + clipped.top * gr.dpy.length + clipped.left;
       Move (GetFrom, clip_offset, src, clipped.width, clipped.length, Length - clipped.length, gr.dpy.length - clipped.length);
    }
}

/*********************************************/
 void MImage :: Put
/*********************************************/
(int offx, int offy, BOOL mask)
{
ADR dest;
VP clipped;
WORD clip_offset;
    clipped.Set (offx, offy, offx + Length, offy + Width);
    clipscreen (&clipped);

    if (clipped.length > 0 && clipped.width > 0)
    {
       clip_offset = (clipped.top - offy) * Length + (clipped.left - offx);

       dest = gr.dpy.address + clipped.top * gr.dpy.length + clipped.left;
       if (mask)
	  MoveMasked (PutOn, clip_offset, dest, clipped.width, clipped.length, gr.dpy.length - clipped.length, Length - clipped.length);
       else
	  Move (PutOn, clip_offset, dest, clipped.width, clipped.length, gr.dpy.length - clipped.length, Length - clipped.length);
    }
}

/********************************************/
 void MImage :: LoadDirect
/********************************************
 DESCRIPTION: Loads the image from file, and
	      decompresses it.
*/
(int fp)
{
    read (fp, &size, sizeof(WORD));
    read (fp, &CompSize, sizeof(WORD));
    read (fp, &Length, sizeof(WORD));
    read (fp, &Width, sizeof(WORD));
    Resize (size, sizeof(BYTE));
    read (fp, mem, CompSize);
    Decompress ();
}

/********************************************/
 inline BOOL MImage :: IsCompressed (void) const
/********************************************/
{
    return (CompSize == 0);
}

/********************************************/
 void MImage :: Redefine
/********************************************
 DESCRIPTION: Redefines image dimentions and, if
	      given, contents.
*/
(WORD nlen, WORD nwid, BYTE *nmem)
{
    Width = nwid;
    Length = nlen;
    size = Length*Width;
    CompSize = 0;

    Resize (size, sizeof(BYTE));
    if (nmem != NULL)
       Move (GetFrom, 0, nmem, size);
}

/********************************************/
 void MImage :: Redefine
/********************************************
 DESCRIPTION: Creates a duplicate of new_image.
*/
(const IMAGE& NewImage)
{
    if (NewImage.size > 0)
    {
       Width = NewImage.Width;
       Length = NewImage.Length;
       size = NewImage.size;
       CompSize = NewImage.CompSize;

       Resize (size, sizeof(BYTE));
       Move (GetFrom, 0, NewImage.mem, size);
    }
}

/********************************************/
 void MImage :: SaveDirect
/********************************************
 DESCRIPTION: Saves the image to file
	      in compressed form.
*/
(int fp)
{
    Compress();
    write (fp, &size, sizeof(WORD));
    write (fp, &CompSize, sizeof(WORD));
    write (fp, &Length, sizeof(WORD));
    write (fp, &Width, sizeof(WORD));
    write (fp, mem, CompSize);
    Decompress();
}

/*********************************************/
 void MImage :: SetPut
/*********************************************/
(int NewPut)
{
    if (NewPut >= NORMAL_PUT && NewPut <= XOR_PUT);
       PutType = NewPut;
}

/*********************************************/
 inline BYTE * MImage :: GetBitmap (void) const
/*********************************************/
{
    return ((BYTE*)mem);
}

/*********************************************/
 inline MImage :: operator BYTE*() const
/*********************************************/
{
    return ((BYTE*)mem);
}

/*********************************************/
      MImage :: ~MImage (void)
/*********************************************/
{
}

//////////////////////////////////////////////////////////////////////////////
