////////////////////////////////////////////////////////////////////////////////
// File: MOB.CPP
//
//    Moving object is an image with background, that can be moved on screen.
// Typical mob is a mouse.
////////////////////////////////////////////////////////////////////////////////

#include <mstd.h>
#include <mobject.h>
#include <mrect.h>
#include <mfileop.h>
#include <mmem.h>
#include <marray.h>
#include <mlist.h>
#include <mstaque.h>
#include <mgraph.h>
#include <mimages.h>
#include <mmob.h>

/********************************************/
      MMob :: MMob (void)
/********************************************
DESCRIPTION: First constructor for the moving
	     object class. Constructs empty
	     object with zero coordinates.
*/
{
    image = NULL;
    backgr = NULL;
    projection = NULL;
    SetPut (NORMAL_PUT);
    nx = getminx(); ny = getminy();
    ox = nx; oy = ny;
}

/********************************************/
      MMob :: MMob
/********************************************
DESCRIPTION: Second constructor for the moving
	     object class. Constructs object
	     with given coordinates and image.
*/
(int xpos,int ypos, IMAGE * picture)
{
    if (picture != NULL)
    {
       image = new IMAGE (*picture);
       backgr = new IMAGE (*picture);
       projection = new IMAGE (*picture);
       backgr->Get (xpos, ypos); 
       SetPut (NORMAL_PUT);
    }
    else
    {
       image = NULL;
       backgr = NULL;
       projection = NULL;
    }
    SetPosition (xpos, ypos);
}

/********************************************/
 void MMob :: SetImage
/********************************************/
(const IMAGE& new_image)
{
    if (backgr != NULL)
    {
       backgr->Put(ox,oy);
       backgr->Redefine (new_image);
    }
    else
       backgr = new IMAGE (new_image);
    backgr->Get(nx,ny);

    if (image != NULL)
       image->Redefine (new_image);
    else
       image = new IMAGE (new_image);

    if (projection != NULL)
       projection->Redefine (new_image);
    else
       projection = new IMAGE (new_image);
}

/********************************************/
 void MMob :: SetImage
/********************************************/
(WORD nlen, WORD nwid, const cArray& arr)
{
    if (backgr != NULL)
    {
       backgr->Put(ox,oy);
       backgr->Redefine (nlen,nwid,arr);
    }
    else
       backgr = new IMAGE (nlen,nwid, (BYTE*)arr);
    backgr->Get(nx,ny);

    if (image != NULL)
       image->Redefine (nlen,nwid,arr);
    else
       image = new IMAGE (nlen,nwid, (BYTE*)arr);

    if (projection != NULL)
       projection->Redefine (nlen,nwid,arr);
    else
       projection = new IMAGE (nlen,nwid, (BYTE*)arr);
}

/********************************************/
 void MMob :: LoadDirect
/********************************************/
(int fp)
{
    image->LoadDirect (fp);

    if (backgr != NULL)
    {
       backgr->Put(ox,oy);
       backgr->Redefine (image->Length, image->Width, image->GetBitmap());
    }
    else
       backgr = new IMAGE (image->Length, image->Width, image->GetBitmap());
    backgr->Get(nx,ny);

    if (projection != NULL)
       projection->Redefine (image->Length, image->Width, image->GetBitmap());
    else
       projection = new IMAGE (image->Length, image->Width, image->GetBitmap());
}

/********************************************/
 inline void MMob :: SaveDirect
/********************************************/
(int fp)
{
    image->SaveDirect (fp);
}

/********************************************/
 void MMob :: SetPosition
/********************************************
DESCRIPTION: Sets the position of the object.
*/
(int xpos,int ypos)
{
    if (xpos < 0)
       xpos = 0;
    if (ypos < 0)
       ypos = 0;
    if (xpos > MAXX - image->Length)
       xpos = MAXX - image->Length;
    if (ypos > MAXY - image->Width)
       ypos = MAXY - image->Width;
    nx = xpos; ny = ypos;
    ox = xpos; oy = ypos;
}

/********************************************/
 inline void MMob :: SetPut
/********************************************/
(int new_put)
{
    PutType = new_put;
    image->PutType = new_put;
    backgr->PutType = NORMAL_PUT;
    projection->PutType = NORMAL_PUT;
}

/********************************************/
 inline void MMob :: RescanBackgr (void)
/********************************************/
{
    if (backgr != NULL)
       backgr->Get (nx,ny);
}

/********************************************/
 void MMob :: Show (void)
/********************************************/
{
    if (PutType == NORMAL_PUT)
       image->Put (nx,ny);
    else if (PutType == SEETH_PUT)
       image->Put (nx,ny,TRUE);
    else
    {
       image->Blend (backgr->GetBitmap(), projection->GetBitmap());
       projection->Put (nx,ny);
    }
}

/********************************************/
 void MMob :: MoveTo
/********************************************
DESCRIPTION: Moves object to the new position.
*/
(int x,int y)
{
VP tArea;
int dx,dy,length,width,onx,ony;
IMAGE *buffer = NULL;              // Image, which are covers both starting
				   //         and the final image positions.

    if (image != NULL)
    {
       onx = nx; ony = ny;
       SetPosition (x, y);
       ox = onx; oy = ony;

       tArea.left = min (nx,ox);
       tArea.top = min (ny,oy);
       tArea.right = max (nx,ox) + image->Length;
       tArea.bottom = max (ny,oy) + image->Width;
       length = tArea.right - tArea.left;
       width = tArea.bottom - tArea.top;
       dx = abs (nx - ox);
       dy = abs (ny - oy);

       if (dx != 0 || dy != 0)  // If object moved
       {
	  if (dx <= image->Length && dy <= image->Width)  // If move overlaps
	  {                                                  // ͻ
	     buffer = new IMAGE (length,width);              //  Ŀ
							     // ͼ 
	     buffer->Get (tArea.left, tArea.top);	     //   

	     push_DC();
				   // Set buffer as output
	     setdpy (buffer->GetBitmap(), buffer->Length, buffer->Width);

				   // Output background and image onto buffer
	     backgr->Put (ox - tArea.left, oy - tArea.top);
	     backgr->Get (nx - tArea.left, ny - tArea.top);
	     if (PutType == NORMAL_PUT)
		image->Put (nx - tArea.left, ny - tArea.top);
	     else if (PutType == SEETH_PUT)
		image->Put (nx - tArea.left, ny - tArea.top, TRUE);
	     else
	     {
		image->Blend (backgr->GetBitmap(), projection->GetBitmap());
		projection->Put (nx - tArea.left, ny - tArea.top);
	     }

	     pop_DC();                             // Restore original DC.
	     buffer->Put (tArea.left, tArea.top);  // Put buffer on screen.
	     delete buffer;
	  }
	  else        // If object does not overlap, buffer is not needed.
	  {
	     backgr->Put (ox,oy);
	     backgr->Get (nx,ny);
	     Show();
	  }
       }
       else     // If object has not moved, simply redraw it.
	  Show();
    }
    else       // If the object is empty
       SetPosition (x,y);
}

/********************************************/
      MMob :: ~MMob (void)
/********************************************
DESCRIPTION: Destructor for the moving object class.
*/
{
    delete image;
    delete backgr;
    delete projection;
}
