////////////////////////////////////////////////////////////////////////////////
//   FILE: ARRAY.CPP
//
//   Arrays are derived from ATOM class (MEM.H). Include access operators ([])
//, type conversion ((BYTE*) & (WORD*)) and comparisons for STRING class
////////////////////////////////////////////////////////////////////////////////

#include <mstd.h>
#include <mobject.h>
#include <mrect.h>
#include <mfileop.h>
#include <mmem.h>
#include <marray.h>

BYTE bits [BITSPERBYTE] = {0x01,0x02,0x04,0x08,0x10,0x20,0x40,0x80};

///////////////////////////// bArray class ///////////////////////////////////
// An array of bits. Access to each bit is []. Memory is allocated for the  //
// required bits and rounded up to the nearest byte.                        //
//////////////////////////////////////////////////////////////////////////////

/********************************************/
      bArray :: bArray (void)
/********************************************/
{
}

/********************************************/
      bArray :: bArray
/********************************************
 The array is represented either by a single
  strand of bits, or by a two dimentional matrix,
   whichever is more convinient.
*/
(WORD len, WORD wid) : Atom ((WORD)(len*wid/BITSPERBYTE) + 1, sizeof(BYTE))
{
    length = len;
    width = wid;
}

/********************************************/
 void bArray :: Resize
/********************************************/
(SIZE n_Size)
{
    Atom::Resize ((int)(n_Size/BITSPERBYTE) + 1);
    Clear();
}

/********************************************/
      bArray :: bArray
/********************************************
 A copy constructor.
*/
(const bArray& arr) : Atom (arr.nElem, arr.elSize)
{
    Move (GetFrom, 0, arr.mem, size);
}

/********************************************/
 BOOL bArray :: operator[]
/********************************************
 DESCRIPTION: Access to a bit in single strand
	       array form. Returns TRUE if the
	       bit is 1, FALSE if the bit is 0.
*/
(WORD index)
{
BYTE *b;

    b = (BYTE*)mem + (WORD)(index / BITSPERBYTE);
    return (( *b & bits [index % BITSPERBYTE]) > 0);
}

/********************************************/
 inline BOOL bArray :: Read
/********************************************
 DESCRIPTION: Access to a bit in 2D matrix
	       array form. Returns TRUE if the
	       bit is 1, FALSE if the bit is 0.
*/
(WORD x, WORD y)
{
    return (operator[] (y * length + x));
}

/********************************************/
 void bArray :: Activate
/********************************************
 DESCRIPTION: Access to a bit in 2D matrix array
	       form. Sets the bit (x,y) to TRUE.
*/
(WORD x, WORD y)
{
WORD index;
BYTE *b;
    index = y * length + x;
    b = (BYTE*) Atom :: operator[]((WORD)(index/BITSPERBYTE));
    *b |= bits[index % BITSPERBYTE];
}

/********************************************/
 void bArray :: Deactivate
/********************************************
 DESCRIPTION: Access to a bit in 2D matrix array
	       form. Sets the bit (x,y) to FALSE.
*/
(WORD x, WORD y)
{
WORD index;
BYTE *b;
    index = y * length + x;
    b = (BYTE*) Atom :: operator[]((WORD)(index/BITSPERBYTE));
    *b &= !(bits[index % BITSPERBYTE]);
}

/********************************************/
 void bArray :: Activate
/********************************************
 DESCRIPTION: Access to a bit in 2D matrix array
	       form. Sets the bit (x,y) to TRUE.
*/
(WORD index)
{
BYTE *b;
    b = (BYTE*) Atom :: operator[]((WORD)(index/BITSPERBYTE));
    *b |= bits[index % BITSPERBYTE];
}

/********************************************/
 void bArray :: Deactivate
/********************************************
 DESCRIPTION: Access to a bit in 2D matrix array
	       form. Sets the bit (x,y) to FALSE.
*/
(WORD index)
{
BYTE *b;
    b = (BYTE*) Atom :: operator[]((WORD)(index/BITSPERBYTE));
    *b &= !(bits[index % BITSPERBYTE]);
}

#pragma argsused
/********************************************/
 void bArray :: Activate
/********************************************/
(const RECT& area)
{
    // Not implemented at this time.
}

#pragma argsused
/********************************************/
 void bArray :: Deactivate
/********************************************/
(const RECT& area)
{
    // Not implemented at this time.
}

/********************************************/
      bArray :: ~bArray (void)
/********************************************/
{
    // The destructor of ATOM does everything.
}

///////////////////////////// cArray class ///////////////////////////////////
// Simple byte array. Most functions are self-explanatory.                  //
//////////////////////////////////////////////////////////////////////////////

/********************************************/
      cArray :: cArray
/********************************************/
(void) : Atom (1, sizeof(BYTE))
{
    // All the allocation is done by ATOM constructor.
}

/********************************************/
      cArray :: cArray
/********************************************/
(SIZE len) : Atom (len, sizeof(BYTE))
{
    // All the allocation is done by ATOM constructor
}

/********************************************/
      cArray :: cArray
/********************************************/
(SIZE len, BYTE *arr) : Atom (len, sizeof(BYTE))
{
    // All the allocation is done by ATOM constructor.
    if (arr != NULL)
       Move (GetFrom, 0, arr, size);
}

/********************************************/
      cArray :: cArray
/********************************************/
(const cArray& arr) : Atom (arr.nElem, arr.elSize)
{
    Move (GetFrom, 0, arr, size);
}

/********************************************/
 BYTE& cArray :: operator[]
/********************************************/
(WORD index)
{
BYTE *b = NULL;                              // Atom::mem is of type void,
					     //  so conversion is required.
    b = (BYTE*) Atom :: operator[] (index);  // Get the pointer to mem[index]
    return (*b);
}

/********************************************/
      cArray :: operator BYTE*() const
/********************************************/
{
    return ((BYTE*)mem);
}

/********************************************/
 cArray          operator+
/********************************************
 DESCRIPTION: Appends a2 to a1. Returns the
	       resulting array by value. The
	       original arrays are not changed.
*/
(const cArray& a1, const cArray& a2)
{
BYTE *t;

    t = (BYTE*) AllocMem (a1.size + a2.size);
    MoveMem (a1.mem, t, a1.size);
    MoveMem (a2.mem, t + a1.size, a2.size);

    return (cArray (a1.nElem + a2.nElem, t));
}

/********************************************/
 cArray& cArray :: operator=
/********************************************
 DESCRIPTION: Sets cArray entries to the
	       given string.
*/
(BYTE * string)
{
    Resize (strlen (string) + 1);
    Clear();
    Move (GetFrom, 0, string, size);
    return (*this);
}

/********************************************/
      cArray :: ~cArray (void)
/********************************************/
{
    // All the deallocation is done by ATOM destructor.
}

///////////////////////////// wArray class ///////////////////////////////////
// Single strand array of type WORD.                                        //
//////////////////////////////////////////////////////////////////////////////

/********************************************/
      wArray :: wArray
/********************************************
 DESCRIPTION: Constructs a wArray with size 1.
*/
(void) : Atom (1, sizeof(WORD))
{
}

/********************************************/
      wArray :: wArray
/********************************************
 DESCRIPTION: Constructs a wArray with size len.
*/
(SIZE len) : Atom (len, sizeof(WORD))
{
}

/********************************************/
      wArray :: wArray
/********************************************
 DESCRIPTION: Constructs a wArray from a set of
	       integers pointed by arr with size len.
*/
(WORD len, WORD *arr) : Atom (len, sizeof(WORD))
{
    Move (GetFrom, 0, arr, size);
}

/********************************************/
      wArray :: wArray
/********************************************
 DESCRIPTION: Constructs a wArray that
	       is a duplicate of arr.
*/
(const wArray& arr) : Atom (arr)
{
}

/********************************************/
 WORD& wArray :: operator[]
/********************************************/
(WORD index)
{
WORD *w = NULL;
    w = (WORD*) Atom :: operator[](index);
    return (*w);
}

/********************************************/
 inline wArray :: operator WORD*() const
/********************************************
 DESCRIPTION: Returns pointer of type
	       WORD to the array.
*/
{
    return ((WORD*)mem);
}

/********************************************/
 wArray          operator+
/********************************************
 DESCRIPTION: Appends a2 to a1. Returns new
	       wArray by value, leaving the
	       original arrays intact.
*/
(const wArray& a1, const wArray& a2)
{
WORD *t;
    t = (WORD*) AllocMem (a1.size + a2.size);
    MoveMem (a1.mem, t, a1.size);
    MoveMem (a2.mem, t + a1.size, a2.size);
    return (wArray(a1.nElem + a2.nElem, t));
}

/********************************************/
      wArray :: ~wArray (void)
/********************************************/
{
    // All the deallocation is done by ATOM destructor.
}

