/*
   Strip Chart Library Test Example.
   This program demonstrates the capabilities of the Library.

   Copyright (c) 1993 by Christopher Lim.  All rights reserved.
*/

/*
   Function func() simulates digitizing a sine wave.
   SPS defined below is the simulated sampling rate.
   The program can be made to run faster by using a lower sampling rate.
*/
#define SPS 300.0

#include "sc.h"
#include <stdio.h>
#include <math.h>
#include <conio.h>
#include <stdlib.h>
#include <malloc.h>
#ifdef BORLAND
#include <alloc.h> /* for coreleft() */
#endif

/*------------------------------------------------------------*/

int FreeMem(void)
{
   /* Prints out free memory.
      This function is compiler dependent.
    */
#ifdef BORLAND
    printf("Free memory: %ld\n",(unsigned long)coreleft());
#endif
#ifdef MICROSOFT
   printf("Free memory: %ld\n",(unsigned long)_memavl());
#endif
    return 0;
}

/*------------------------------------------------------------*/

double func(double t)
{
    /* Test function response for time t */
    double x;
    double amplitude = 0.5;
    double period = 0.5;
    x = amplitude * sin(2 * 3.1416 * t/period);
    return(x);
}

/*------------------------------------------------------------*/

char *Label(SC_WINDOW *w, double seconds, unsigned long labelCount)
{
   static char buf[80];
   static int minutes = 0;
   static int hours = 0;

   while (seconds >= 60.0)
   {
      seconds -= 60.0;
      minutes++;
   }
   if (minutes >= 60)
   {
      minutes %= 60;
      hours += minutes/60;
   }
   sprintf(buf,"%02d:%02d:%05.2f (%ld)",hours,minutes,seconds,
      labelCount);
   return buf;
}

/*------------------------------------------------------------*/

void GetPrinterType(unsigned char *printerType, unsigned char *printerRes,
   char **s)
{
   int c;
   unsigned char i,j;

   printf("Select printer type and resolution code: \n");
   printf("[1] 9 PIN LOW RES\n");
   printf("[2] 9 PIN HIGH RES\n");
   printf("[3] 24 PIN LOW RES\n");
   printf("[4] 24 PIN HIGH RES\n");
   printf("Select code [1,2,3 or 4]: ");
   c = getch();
   printf("%c\n",c);
   switch (c)
   {
        case '1': i = SC_9PIN; j = SC_LOW_RES;
             *s = "9 PIN LOW RES";
             break;
        case '2': i = SC_9PIN; j = SC_HIGH_RES;
             *s = "9 PIN HIGH RES";
             break;
        case '3': i = SC_24PIN; j = SC_LOW_RES;
             *s = "24 PIN LOW RES";
             break;
        case '4': i = SC_24PIN; j = SC_HIGH_RES;
             *s = "24 PIN HIGH RES";
             break;
        default:
             printf("Aborted\n");
             exit(1);
   }
   printf("%s\n",*s);
   *printerType = i;
   *printerRes = j;
   return;
}

/*------------------------------------------------------------*/

void ErrorTrap(char *msg)
{
    printf("%s\n",msg);
    exit(1);
}

/*------------------------------------------------------------*/

void main()
{
   int i,j;
   int error;
   double scan;
   double a;
   SC_WINDOW *w1,*w2,*w3;
   SC_WINDOW *text, *text1, *text2, *text3, *text4, *text5;
   SC_WINDOW *text6, *text7, *text8;
   SC_WINDOW *time;
   SC_WINDOW *event1, *event2, *event3;
   double sps = SPS; /* simulated sampling rate */
   double t;
   unsigned char printerType, printerRes;
   static char string[128];
   char *s;

   if (!SC_PrinterReady(LPT1))
   {
      printf("Printer at LPT1 is not ready!\n");
      exit(1);
   }

   GetPrinterType(&printerType,&printerRes,&s);

   FreeMem();

   /* Error trap */
   SC_SetErrorTrap(ErrorTrap);

   /* Initialize: 8.5 inch paper, plot speed = 1 inch per second */
   SC_Init(LPT1, printerType,printerRes,INCHES, 8.5, 1.0,&error);

   /* Horizontal Grids: 4 divisions */
   SC_SetHorizontalGrids(4,&error);

   /* Time grids every 1 second */
   SC_SetTimeGrids(1.0,&error);

   /* Format of Y labels */
   SC_SetYlabels(10.0/72.0,"%+0.1f",3.0, &error);

   /* Text window to display title */
   text = SC_AddTextWindow(0.5,14.0/72.0,&error);

   /* Add time axis at the top of the plot */
   time = SC_AddTimeWindow(SC_yoffset+0.125,20.0/72.0, LABELS_ABOVE_AXIS,
      2.0,5,Label,&error);

   /* Add trace windows */
   text1 = SC_AddTextWindow(SC_yoffset+0.125,10.0/72.0,&error);
   w1 = SC_AddTraceWindow(SC_yoffset+0.125, 1.0,
     -1.0,1.0,SCF_BORDER|SCF_GRID|SCF_YLABEL,&error);

   text2 = SC_AddTextWindow(SC_yoffset+0.125,10.0/72.0,&error);
   w2 = SC_AddTraceWindow(SC_yoffset+0.125, 1.0,
     -1.0,1.0,SCF_BORDER|SCF_GRID|SCF_YLABEL,&error);

   text3 = SC_AddTextWindow(SC_yoffset+0.125,10.0/72.0,&error);
   w3 = SC_AddTraceWindow(SC_yoffset+0.125, 1.0,
     -1.0,1.0,SCF_CLIP|SCF_BORDER|SCF_GRID|SCF_YLABEL,&error);

   /* Add event windows */
   text4 = SC_AddTextWindow(SC_yoffset+0.125,10.0/72.0,&error);
   event1 = SC_AddEventWindow(SC_yoffset+0.125,0.125,
      SCM_PULSE,0,&error);
   event2 = SC_AddEventWindow(SC_yoffset+0.125,0.125,
      SCM_TICK,0,&error);
   event3 = SC_AddEventWindow(SC_yoffset+0.125,0.125,
      SCM_BAR,HALF_FILL,&error);

   /* Add time axis at the bottom of the plot */
   time = SC_AddTimeWindow(SC_yoffset+0.125,20.0/72.0, LABELS_BELOW_AXIS,
      2.0,5,Label,&error);

   /* Add text window to display character set */
   text5 = SC_AddTextWindow(SC_yoffset+0.25,12.0/72.0,&error);
   text6 = SC_AddTextWindow(SC_yoffset+0.125,12.0/72.0,&error);
   text7 = SC_AddTextWindow(SC_yoffset+0.125,10.0/72.0,&error);
   text8 = SC_AddTextWindow(SC_yoffset+0.125,8.0/72.0,&error);

   /* initialize text windows */
   sprintf(string,"STRIP CHART LIBRARY EXAMPLES [%s]",s);
   SC_Text(text,string,&error);
   SC_Text(text1,"Trace Window with gain ranging:",&error);
   SC_Text(text2,"Trace Window with no clipping:",&error);
   SC_Text(text3,"Trace Window with clipping:",&error);
   SC_Text(text4,"Event Windows:",&error);

   /* Draw character set */
   for (i=33,j=0; i < 128; i++,j++)
      string[j] = i;
   string[j] = 0; /* terminating NULL */
   SC_Text(text5,string,&error);
   SC_Text(text6,"Text Window: Scalable fonts", &error);
   SC_Text(text7,"Text Window: Scalable fonts", &error);
   SC_Text(text8,"Text Window: Scalable fonts",&error);

   /* draw traces */
   a = 1.0; /* initial amplitude */
   for (t=0.0; t < 9.5; t += 1.0/sps)
   {
      if (kbhit())
      {
         printf("User aborted.\n");
         break;
      }

      if (t >= 3.0 && t < 6.0 && a == 1.0)
      { /* Quadruple the amplitude at 2.0 <= t < 4.0 */
         a = 4.0;
         SC_SetTraceRange(w1,-2.0,2.0,&error);
         SC_Text(text1,"4 x Amplitude",&error);
      }

      if (t >= 6.0 && a == 4.0)
      { /* 1/4 the amplitude at t >= 4.0 */
         a = 1.0;
         SC_SetTraceRange(w1,-1.0,1.0,&error);
         SC_Text(text1,"1/4 x Amplitude",&error);
      }

      scan = a * func(t);

      SC_Trace(w1,scan,&error);
      SC_Trace(w2,scan,&error);
      SC_Trace(w3,scan,&error);

      /* Set event high when scan is positive */
      SC_Event(event1,scan > 0.0 ? 1:-1, &error);
      SC_Event(event2,scan > 0.0 ? 1:-1, &error);
      SC_Event(event3,scan > 0.0 ? 1:-1, &error);

      SC_Draw();
      SC_Advance(1.0/sps); /* next */
   }

   SC_Close();
   FreeMem();
}


