/*
 * Display a color-mapped image on a PC VGA. Images are produced by ART and
 * MEDIAN.
 */

#include <stdio.h>
#include <stdlib.h>
#include <dos.h>
#include <io.h>
#include <bios.h>
#include <string.h>
#include <memory.h>
#include <signal.h>
#include "vort.h"
#include "pcdisp.h"

#ifdef TC
#ifndef _STKLEN
unsigned _stklen = 4096;
#else
unsigned _stklen = _STKLEN;
#endif
#endif


#define	MAX_ADAPTERS	50
#define	MAX_IMAGE_WIDTH	2048
#define	PALETTE_SIZE	256
#define SET_TEXT()	{union REGS regs; regs.x.ax = 3; int86(0x10, &regs, &regs);}

/* image positioning flag values */
#define	POSN_UL		0
#define	POSN_UR		1
#define	POSN_CT		2
#define	POSN_LL		3
#define	POSN_LR		4

/* adapter id values */
#define	ADAPTER_NONE		0
#define	ADAPTER_ATI			1
#define	ADAPTER_EVEREX		2
#define	ADAPTER_TRIDENT		3
#define	ADAPTER_VIDEO7		4
#define	ADAPTER_PARADISE	5
#define	ADAPTER_CHIPS		6
#define	ADAPTER_TSENG		7
#define	ADAPTER_TSENG4		8
#define	ADAPTER_AHEADA		9
#define	ADAPTER_AHEADB		10
#define	ADAPTER_OTI			11
#define	ADAPTER_VGA			99


/* palette entry structure */
#pragma pack(1)
typedef struct {

	UCHAR           r, g, b;

}               PALETTE;

#pragma pack()

/* description of an image, where it lives, where to put it and how */
typedef struct {

	char           *fn;	/* image filename */

	int             type;	/* image type */

	int             x, y;	/* video co-ords to place image */

	int             xlen, ylen;	/* dimensions of image */

	char            posn_flag;	/* image positioning flag */

	PALETTE        *pal;	/* palette; NULL = use current palette */

	int             pal_size;	/* size of colour palette */

	int             darkest_pal;	/* palette index of darkest colour */

	union {

		char far       *ptr;	/* position in regular memory */

		ULONG           EMS_ofs;	/* position in EMS memory */

	}               adr;

	UINT           EMS_h;	/* handle of EMS memory; -1 if in regular
				 * memory */

}               IMAGE_DESCR;



/* image info */
IMAGE_DESCR    *images = NULL;

int             max_images = 0;



/* adapter info */
int             max_adapters = 0;

ADAPTER        *adapters;

ADAPTER         v_adapter;


/* these next 3 MUST!!! be in the near data segment */
/* and DS == SS, since we rely on it! */
UINT            adapter_x = 0;

UINT            adapter_y = 0;

PFI             adapter_bank = NULL;



/* for reading images */
UCHAR          *r, *g, *b, *s;


/* EMS stuff */
int             EMS_available = FALSE;

char far       *EMS_addr = NULL;

UINT            EMS_handle = 0xFFFF;

UINT            EMS_max_pages = 0;

UCHAR far      *EMS_base();

UINT            init_EMS();

void            lmemmove(UCHAR far *, UCHAR far *, int);


/* CTRL-C trap processor */
void             ctrl_C(void);



int
main(argc, argv)
	int             argc;

	char           *argv[];

{

	int             ac;

	char          **av;

	int             mode = 0, i, c, key, animate = FALSE, overscan = FALSE,
	                posn_flag = POSN_CT, recognise_palette = FALSE,
	                disable_EMS = FALSE, auto_detect = TRUE;

	int             current_image = 0;

	ULONG           EMS_offset = 0L;


	ac = --argc;

	av = ++argv;

	/* load adapter info (for usage()) */
	load_adapters(&adapters, &max_adapters, FALSE);


	/* count image filenames */
	while (ac-- > 0) {
		if (**av == '-' || **av == '/') {
			/* skip parameters */
			switch (tolower((*av)[1])) {
			case 'd':
				auto_detect = FALSE;
				break;
			case 'a':
			case 'm':
			case 'p':
				if (((*av)[2]) == 0) {
					++av;
					--ac;
				}
				break;

			default:
				break;

			}
		} else {
			max_images++;
		}

		++av;
	}

	if (auto_detect) {

		/* unload the adapter info */
		unload_adapters(adapters);

		max_adapters = 0;


		/*
		 * then load only the info relevent to the card we know we
		 * have
		 */
		if (!load_adapters(&adapters, &max_adapters, TRUE)) {

			/* failed auto detecting - load all info again! */
			unload_adapters(adapters);

			max_adapters = 0;

			load_adapters(&adapters, &max_adapters, FALSE);

		}
	}
	if (max_images < 1)
		usage();

	if (!(images = calloc(max_images, sizeof(IMAGE_DESCR)))) {

		fail("no memory for image descriptions!");

	}
	ac = argc;

	av = argv;


	/* parse cmd line arguments, store image filenames */
	while (ac-- > 0) {

		if (**av == '-' || **av == '/') {

			switch (tolower((*av)[1])) {


			case '?':

				usage();

				break;


			case 'a':

				animate = TRUE;

				if (((*av)[2]) == 0) {

					/*
					 * check for 'p' parameter to this
					 * 'a' parameter
					 */
					if (ac - 1 >= 0) {

						if (stricmp(*(av + 1), "p") == 0) {

							av++;

							ac--;

							break;

						}
					}
				} else
					sscanf(&((*av)[2]), "%c", &c);

				if (toupper(c) == 'P')
					recognise_palette = TRUE;

				break;


			case 'd':
				auto_detect = FALSE;
				break;


			case 'm':

				if (((*av)[2]) == 0) {

					if (--ac >= 0)
						sscanf(*++av, "%d", &mode);

					else
						usage();

				} else
					sscanf(&((*av)[2]), "%d", &mode);

				if (mode < 0 || mode >= max_adapters)
					usage();

				break;


			case 'p':

				if (((*av)[2]) == 0) {

					if (--ac >= 0)
						sscanf(*++av, "%d", &posn_flag);

					else
						usage();

				} else
					sscanf(&((*av)[2]), "%d", &posn_flag);

				if (mode < 0 || mode >= max_adapters)
					usage();

				break;


			case 'o':

				overscan = TRUE;

				break;


			case 'e':

				disable_EMS = TRUE;

				break;


			default:

				usage();

				break;

			}
		} else {

			/*
			 * t'was an image, store the filename and current
			 * image position flag
			 */
			images[current_image].fn = *av;

			images[current_image].posn_flag = posn_flag;

			current_image++;

		}
		++av;

	}

	/* allocate buffers for maximum possible (likely) image width */
	s = (UCHAR *) malloc(MAX_IMGWIDTH);

	r = (UCHAR *) malloc(MAX_IMGWIDTH);

	g = (UCHAR *) malloc(MAX_IMGWIDTH);

	b = (UCHAR *) malloc(MAX_IMGWIDTH);


	/* set the adapter */
	v_adapter = adapters[mode];

	adapter_x = v_adapter.width;

	adapter_y = v_adapter.height;

	adapter_bank = v_adapter.select_bank;

	free(adapters);


	signal(SIGINT, ctrl_C);	/* trap user aborts */


	if (!disable_EMS) {

		/*
		 * test for EMS; set flag TRUE if available; allocate all of
		 * EMS
		 */
		EMS_handle = init_EMS();

	}
	/* display the images */
	if (!animate) {

		/* display just one at a time please */
		v_adapter.init();

		single_display(max_images, overscan, EMS_handle);

	} else {

		/* load each image into memory */
		for (current_image = 0; current_image < max_images; current_image++) {

			if (EMS_available) {

				/*
				 * try to load image to EMS; if we fail try
				 * regular memory
				 */
				if (!load_image2EMS(current_image, (current_image == 0) || recognise_palette, EMS_handle, &EMS_offset)) {

					if (!load_image(current_image, (current_image == 0) || recognise_palette))
						break;

				}
			} else if (!load_image(current_image, (current_image == 0) || recognise_palette))
				break;

			position_image(current_image);

			if (recognise_palette)
				check_duplicate_cmap(current_image);

		}

		if (current_image < 1) {

			fail("couldn't load the first image!");

		}
		/* set number of images actually loaded */
		max_images = current_image;


		v_adapter.init();

		v_adapter.palette(images[0].pal, images[0].pal_size);

		if (overscan)
			v_adapter.overscan(images[0].darkest_pal);


		/* display one after the other real fast */
		animated_display(max_images);


		/* unload all the images */
		for (current_image = 0; current_image < max_images; current_image++) {

			unload_image(current_image);

		}
	}

	uninit_EMS();


	/* reset video mode */
	SET_TEXT();


	exit(0);

}



int
single_display(n, overscan, EMS_h)
	int             n;

	int             overscan;

	UINT            EMS_h;

{

	int             current_image;

	int             key = 0;

	int             request_abort = FALSE;

	int             direction = 1;

	int             image_loaded;

	ULONG           EMS_ofs;

	IMAGE_DESCR    *img;


	current_image = 0;

	do {


		if (EMS_available) {

			EMS_ofs = 0L;

			image_loaded = load_image2EMS(current_image, TRUE, EMS_h, &EMS_ofs);

		} else
			image_loaded = load_image(current_image, TRUE);


		if (image_loaded) {

			img = &(images[current_image]);

			position_image(current_image);


			/* display image from memory */
			if (img->pal) {

				v_adapter.palette(img->pal, img->pal_size);

				if (overscan)
					v_adapter.overscan(img->darkest_pal);

			}
			display_image(current_image);

			unload_image(current_image);

		} else {

			/*
			 * can't load image into memory, display image
			 * directly from file
			 */
			display_image_file(current_image, overscan);

		}

		/* wait for operator */
		while (!get_key(&key));

		switch (toupper(key)) {

		case '+':
			direction = 1;
			break;

		case '-':
			direction = -1;
			break;

		case 0x1b:

		case 'Q':
			request_abort = TRUE;

		default:
			break;

		}

		/* clear video */
		v_adapter.fill(0, 0, adapter_x, adapter_y, img->darkest_pal);


		/* next image */
		current_image = (current_image + direction) % n;

		if (current_image < 0)
			current_image = n - 1;


	} while (!request_abort);

}



int
animated_display(n)
	int             n;

{

#define		SNAIL_FACTOR	1000L
#define		MAX_DELAY		100
	int             i = 0, request_abort = FALSE, hold = FALSE, single_step = FALSE,
	                delay = 0;

	ULONG           snail;

	int             key;


	do {

		/* check for user intervention */
		if (get_key(&key)) {

			switch (toupper(key)) {

			case 0x1b:

			case 'Q':
				request_abort = TRUE;
				break;

			case 'H':
				hold = !hold;
				break;

			case 'S':
				single_step = !single_step;
				break;

			case 'D':
				if (delay == 0)
					delay = MAX_DELAY;

				else
					delay = 0;

				break;

			case '+':
				if (single_step)
					i = (i + 1) % n;

				else if (++delay > MAX_DELAY)
					delay = MAX_DELAY;

				break;

			case '-':
				if (single_step) {

					i--;

					if (i < 0)
						i = n - 1;

				} else if (--delay < 0)
					delay = 0;

				break;

			default:
				break;

			}
		}
		if (!hold) {

			if (delay) {

				/* slow her up a bit */
				for (snail = delay * SNAIL_FACTOR; snail; snail--);

			}
			/* set image palette */
			if (images[i].pal) {

				v_adapter.palette(images[i].pal, images[i].pal_size);

				/*
				 * if (overscan)
				 * v_adapter.overscan(images[i].darkest_pal);
				 */
			}
			display_image(i);

			if (!single_step)
				i = (i + 1) % n;

		}
	} while (!request_abort);

}



int
display_image_file(current_image, overscan)
	int             current_image;

	int             overscan;

{

	image          *im;

	int             w, h, x, y;

	UCHAR          *sptr;

	IMAGE_DESCR    *img;


	img = &(images[current_image]);

	if ((im = openimage(img->fn, "r")) == (image *) NULL) {

		fprintf(stderr, "pcdisp: can't open file %s!\n", img->fn);

		return 1;

	}
	img->xlen = w = imagewidth(im);

	img->ylen = h = imageheight(im);


	/* remap palette */
	make_palette(im, img);

	v_adapter.palette(img->pal, img->pal_size);

	if (overscan)
		v_adapter.overscan(img->darkest_pal);


	/* display image line by line */
	imagepos(im, 0, 0);

	switch (imagetype(im)) {


	case PIX_CMAP:

	case PIX_RLECMAP:

		for (y = 0; y < h; y++) {

			readmappedline(im, s);

			v_adapter.putline((char far *) s, (UINT) w, (UINT) img->x, (UINT) img->y + y);

		}
		break;


	default:

		for (y = 0; y < h; y++) {

			readrgbline(im, r, g, b);

			sptr = s;

			for (x = 0; x < w; x++) {

				*sptr++ = (UCHAR) ((90 * r[x] + 140 * g[x] + 25 * b[x]) >> 8);

			}
			v_adapter.putline((char far *) s, (UINT) w, (UINT) img->x, (UINT) img->y + y);

		}
		break;

	}
	closeimage(im);

	return 0;

}




int
display_image(current_image)
	int             current_image;

{

	IMAGE_DESCR    *img;


	img = &(images[current_image]);


	if (img->EMS_h == 0xFFFF) {

		v_adapter.block(img->adr.ptr, img->xlen, img->ylen, img->x, img->y);

	} else {

		v_adapter.EMSdecode(img->EMS_h, img->adr.EMS_ofs, img->x, img->y, img->xlen, img->ylen);

	}
}



int
load_image(current_image, set_palette)
	int             current_image;

	int             set_palette;

{

	image          *im;

	int             x, y, w, h;

	UCHAR          *fptr;

	IMAGE_DESCR    *img;


	img = &(images[current_image]);


	if ((im = openimage(img->fn, "r")) == (image *) NULL) {

		fprintf(stderr, "pcdisp: can't open file %s!\n", img->fn);

		return FALSE;

	}
	img->xlen = w = imagewidth(im);

	img->ylen = h = imageheight(im);


	/* validate size of buffer required */
	if (((long) w * (long) h) > 65500L) {

		/*
		 * can't load image into memory without heaps of stuffing
		 * around,
		 */
		closeimage(im);

		return FALSE;

	}
	if (!(img->adr.ptr = fptr = malloc(w * h))) {

		fprintf(stderr, "pcdisp: can't alloc %u bytes for image!\n", (long) (w * h));

		closeimage(im);

		return FALSE;

	}
	img->EMS_h = 0xFFFF;


	/* extract palette if required */
	if (set_palette) {

		make_palette(im, img);

	} else
		img->pal = NULL;


	/* load image */
	imagepos(im, 0, 0);

	switch (imagetype(im)) {


	case PIX_CMAP:

	case PIX_RLECMAP:

		for (y = 0; y < h; y++) {

			readmappedline(im, fptr);

			fptr += w;

		}
		break;


	default:

		for (y = 0; y < h; y++) {

			readrgbline(im, r, g, b);

			for (x = 0; x < w; x++) {

				*fptr = (UCHAR) ((90 * r[x] + 140 * g[x] + 25 * b[x]) >> 8);

				fptr++;

			}
		}
		break;

	}

	closeimage(im);

	return TRUE;

}



/****
 * Loads the menu image file defined by m into EMS
 * provided the adapter we are displaying image on can
 * handle it.
 *
 * We should be more careful of the EMS available and only
 * use what is actually there.
  **/

int
load_image2EMS(current_image, set_palette, EMS_h, EMS_ofs)
	int             current_image;

	int             set_palette;

	UINT            EMS_h;

	ULONG          *EMS_ofs;

{

	image          *im = NULL;

	int             fd;

	int             i;

	ULONG           len, img_len, abs_ofs;

	UINT            page;

	char            buffer[4096];

	UINT            ofs, n;

	IMAGE_DESCR    *img;


	/* check that the adapter knows how to use EMS */
	if (!v_adapter.EMSdecode)
		return FALSE;


	img = &(images[current_image]);


	/* extract VORT header info, like colour map, dimensions */
	if ((im = openimage(img->fn, "r")) == (image *) NULL) {

		fprintf(stderr, "pcdisp: can't open file %s!\n", img->fn);

		goto fail;

	}
	/* only load colour mapped images into EMS */
	if (imagetype(im) != PIX_CMAP && imagetype(im) != PIX_RLECMAP)
		goto fail;


	img->xlen = imagewidth(im);

	img->ylen = imageheight(im);


	/* extract palette if required */
	if (set_palette) {

		make_palette(im, img);

	} else
		img->pal = NULL;


	/* determine size of image part of VORT file */
	fd = imagefile(im);

	len = filelength(fd);

	imagepos(im, 0, 0);

	len -= tell(fd);


	img->EMS_h = EMS_h;

	img->adr.EMS_ofs = *EMS_ofs;

	page = *EMS_ofs / 16384;

	ofs = *EMS_ofs % 16384;

	EMS_map(EMS_h, page);


	/* store image length into EMS (possibly spanning a page boundary) */
	n = min(16384 - ofs, sizeof(len));

	lmemmove(EMS_addr + ofs, (char far *) &len, n);

	ofs += n;

	if (ofs >= 16384) {

		if (++page >= EMS_max_pages)
			goto fail;

		EMS_map(EMS_h, page);

		n = sizeof(len) - n;

		lmemmove(EMS_addr, (char far *) &len, n);

		ofs = n;

	}
	/* slap image directly to EMS without any decoding */
	while (len) {

		n = (sizeof(buffer) > len) ? len : sizeof(buffer);

		if (read(fd, buffer, n) == n) {

			len -= n;

			if (n + ofs >= 16384) {

				/* size will overflow page */
				lmemmove(EMS_addr + ofs, (char far *) buffer, 16384 - ofs);

				n -= 16384 - ofs;

				if (++page >= EMS_max_pages)
					goto fail;

				EMS_map(EMS_h, page);

				lmemmove(EMS_addr, (char far *) buffer + (16384 - ofs), n);

				ofs = n;

			} else {

				lmemmove(EMS_addr + ofs, (char far *) buffer, n);

				ofs += n;

			}
		} else
			goto fail;

	}

	closeimage(im);


	*EMS_ofs = page * 16384L + ofs;


	return TRUE;


fail:
	if (im)
		closeimage(im);

	img->adr.EMS_ofs = 0L;

	img->EMS_h = 0xFFFF;

	return FALSE;

}



int
unload_image(current_image)
	int             current_image;

{

	IMAGE_DESCR    *img;


	img = &(images[current_image]);

	if (img->pal)
		free(img->pal);


	if (img->EMS_h != 0xFFFF) {

		img->adr.EMS_ofs = 0L;

	} else if (img->adr.ptr)
		free(img->adr.ptr);

}



int
make_palette(im, img)
	image          *im;

	IMAGE_DESCR    *img;

{

	int             i, r1, g1, b1, cmap_size;

	long            darkest = 999999999, intensity;


	/* generate palette */
	if (imagetype(im) == PIX_CMAP || imagetype(im) == PIX_RLECMAP) {


		img->pal_size = cmap_size = cmapsize(im);

		if (!(img->pal = malloc(cmap_size * sizeof(PALETTE)))) {

			fprintf(stderr, "pcdisp: can't allocate image palette %d bytes long!\n", cmap_size * sizeof(PALETTE));

			return;

		}
		/* set up colour palette for colour map files */
		for (i = 0; i < cmap_size; i++) {


			/* 6-bits only per value please */
			r1 = img->pal[i].r = im->red[i] >> 2;

			g1 = img->pal[i].g = im->green[i] >> 2;

			b1 = img->pal[i].b = im->blue[i] >> 2;


			/*
			 * check if this colour is darker than previous
			 * darkest
			 */
			if ((intensity = r1 * r1 + g1 * g1 + b1 * b1) < darkest) {

				darkest = intensity;

				img->darkest_pal = i;

			}
		}

		if (img->pal_size < PALETTE_SIZE && darkest != 0) {

			/*
			 * palette not yet full & darkest not black - add a
			 * black entry to colour map
			 */
			img->pal[cmap_size - 1].r = 0;

			img->pal[cmap_size - 1].g = 0;

			img->pal[cmap_size - 1].b = 0;

			img->darkest_pal = cmap_size - 1;

			img->pal_size++;

		}
	} else {

		img->pal_size = cmap_size = PALETTE_SIZE;

		if (!(img->pal = malloc(cmap_size * sizeof(PALETTE)))) {

			fprintf(stderr, "pcdisp: can't allocate image palette %d bytes long!\n", cmap_size * sizeof(PALETTE));

			return;

		}
		/* create a B&W colour map */
		for (i = 0; i < cmap_size; i++) {

			/* gray scale: 6-bits only per value */
			img->pal[i].r = i >> 2;

			img->pal[i].g = i >> 2;

			img->pal[i].b = i >> 2;

		}
		img->darkest_pal = 0;

	}
}



/****
 * Set (x, y) co-ords of image on screen
 * depending on posn_flag in image description.
  **/

int
position_image(current_image)
	int             current_image;

{

	IMAGE_DESCR    *img;


	img = &(images[current_image]);

	switch (img->posn_flag) {

	default:

	case POSN_UL:
		img->x = 0;
		img->y = 0;
		break;

	case POSN_UR:
		img->x = adapter_x - img->xlen;
		img->y = 0;
		break;

	case POSN_LL:
		img->x = 0;
		img->y = adapter_y - img->ylen;
		break;

	case POSN_LR:
		img->x = adapter_x - img->xlen;
		img->y = adapter_y - img->ylen;
		break;

	case POSN_CT:
		img->x = (adapter_x - img->xlen) / 2;
		img->y = (adapter_y - img->ylen) / 2;
		break;

	}
	if (img->x < 0)
		img->x = 0;

	if (img->y < 0)
		img->y = 0;

}



/****
 * If previous image's palette is the same as
 * this image's, then throw this image's palette
 * away since there is no need to store it.
  **/

int
check_duplicate_cmap(current_image)
	int             current_image;

{

	int             i;

	IMAGE_DESCR    *img, *prev_img;


	if (current_image == 0)
		return;


	img = &(images[current_image]);


	/* find a previous image with a colour map */
	for (i = current_image - 1; i > 0 && images[i].pal == NULL; i--);

	prev_img = &(images[i]);

	if (prev_img->pal_size != img->pal_size)
		return;


	/* check colour maps are the same */
	if (memcmp(img->pal, prev_img->pal, sizeof(PALETTE) * img->pal_size) != 0)
		return;


	/* image colour maps are the same, throw current one away */
	free(img->pal);

	img->pal = NULL;

	img->pal_size = 0;

}



int
usage()
{

	int             i;


	fprintf(stderr, "usage: pcdisp fname [fname...] [-m mode] [-p n] [-a [p]] [-d] [-e] [-o] [-p n] [-?]\n\n");

	fprintf(stderr, "where:\n");

	fprintf(stderr, "       fname    - filenames of image to display (may be wildcard)\n");

	fprintf(stderr, "       -m mode  - video mode to use\n");

	for (i = 0; i < max_adapters; i++)
		fprintf(stderr, "                   = %2d %s\n", i, adapters[i].descr);

	fprintf(stderr, "       -a [p]   - animate the images specified\n");

	fprintf(stderr, "                  specify 'p' to enable recognition of different palettes\n");

	fprintf(stderr, "       -d       - disable auto adapter detect\n");

	fprintf(stderr, "       -e       - do not use EMS if available\n");

	fprintf(stderr, "       -o       - set overscan colour to darkest colour in colour map\n");

	fprintf(stderr, "       -p n     - image position flag; values are:\n");

	fprintf(stderr, "                  0 - Upper left; 1 - Upper right\n");

	fprintf(stderr, "                  2 - Centre (default);\n");

	fprintf(stderr, "                  3 - Lower left; 4 - Lower right\n");

	fprintf(stderr, "       -?       - this message\n");

	exit(1);

}


int
fail(msg)
	char           *msg;

{

	fprintf(stderr, "pcdisp: %s", msg);

	exit(1);

}


int
get_key(key)
	int            *key;

{

/*
	*key = _bios_keybrd(_KEYBRD_READY);

	if (*key)
		*key = _bios_keybrd(_KEYBRD_READ) & 0xff;
*/

	if (kbhit())
		*key = getch();

	return *key;

}



/*
 * Check if EMS is available, that there is enough and set the base page
 * address. Allocate ALL available EMS memory for us to use.
 */

UINT
init_EMS()
{

	UINT            EMS_handle = 0xFFFF;


	EMS_available = FALSE;


	if (EMS_rdy()) {

		EMS_max_pages = EMS_count();


		/* allocate ALL of EMS memory available */
		EMS_handle = EMS_alloc(EMS_max_pages);

		if (EMS_handle != 0xFFFF) {

			EMS_addr = EMS_base();

			EMS_available = TRUE;

		}
	}
	return EMS_handle;

}



int
uninit_EMS()
{

	EMS_release(EMS_handle);

}



/****
 * Handle user abort requests.
  **/

void
ctrl_C()
{

	signal(SIGINT, SIG_IGN);/* ignore any abort requests in this routine */

	uninit_EMS();

	exit(1);

}



/*****
 * Setup adapter structures.
  **/

int
load_adapters(a, n, auto_detect)
	ADAPTER       **a;

	int            *n;

	int             auto_detect;

{

	ADAPTER        *adapters;

	int             card = ADAPTER_NONE;


	if (!(adapters = malloc(MAX_ADAPTERS * sizeof(ADAPTER)))) {

		fail("no memory for adapter descriptions!");

	}

	fprintf(stderr, "in load_adatpters...\n");
	/* determine what type of card we have */
	if (auto_detect) {

		if ((card = adapter_detect()) == ADAPTER_NONE)
			return 0;

	}
	(*n) += setup_vga320x200(&adapters[(*n)]);

	(*n) += setup_vga320x400(&adapters[(*n)]);

	if (!auto_detect || card == ADAPTER_TSENG4)
		(*n) += setup_tseng4000(&adapters[(*n)]);

	if (!auto_detect || card == ADAPTER_TSENG)
		(*n) += setup_tseng(&adapters[(*n)]);

	if (!auto_detect || card == ADAPTER_VIDEO7)
		(*n) += setup_video7(&adapters[(*n)]);

	if (!auto_detect || card == ADAPTER_PARADISE)
		(*n) += setup_paradise(&adapters[(*n)]);

	if (!auto_detect || card == ADAPTER_CHIPS)
		(*n) += setup_chips(&adapters[(*n)]);

	if (!auto_detect || card == ADAPTER_TRIDENT)
		(*n) += setup_trident(&adapters[(*n)]);

	if (!auto_detect || card == ADAPTER_ATI)
		(*n) += setup_ati(&adapters[(*n)]);

	if (!auto_detect || card == ADAPTER_EVEREX)
		(*n) += setup_everex(&adapters[(*n)]);

	if (!auto_detect || card == ADAPTER_AHEADA)
		(*n) += setup_aheada(&adapters[(*n)]);

	if (!auto_detect || card == ADAPTER_AHEADB)
		(*n) += setup_aheadb(&adapters[(*n)]);

	if (!auto_detect || card == ADAPTER_OTI)
		(*n) += setup_oaktech(&adapters[(*n)]);


	*a = adapters;

	fprintf(stderr, "returing from load_adapters..\n");
	return 1;

}


int
unload_adapters(a)
	ADAPTER        *a;

{

	free(a);

}

