/***************************************************************************/
/*																		   */
/*								  ZFONT.C								   */
/*																		   */
/***************************************************************************/
/*		(c) 1995 Zephyr Software, Stephen L. Balkum and Daniel A. Sill	   */
/***************************************************************************/
/*                                                                         */
/*                     NON-EXCLUSIVE LICENSE FOR USE                       */
/* This SOFTWARE is provided under a license and may only be used in       */
/* accordance with that license.  This SOFTWARE or any copies may not be   */
/* distributed by any means, electronic or otherwise, to any third         */
/* individual or party.  The title and ownership of this SOFTWARE remains  */
/* with Zephyr Software, Stephen L. Balkum and Daniel A. Sill.  This file  */
/* may change without notice and there is no commitment of support by      */
/* Zephyr Software, Stephen L. Balkum and Daniel A. Sill.                  */
/* The content of this file is confidential.                               */
/*                                                                         */
/*                     NO WARRANTIES AND NO LIABILITY                      */
/* Stephen L. Balkum and Daniel A. Sill provide no warranties, either      */
/* expressed or implied, of merchantability, or fitness, for a particular  */
/* use or purpose of this SOFTWARE and documentation.  In no event shall   */
/* Stephen L. Balkum or Daniel A. Sill be held liable for any damages      */
/* resulting from the use or misuse of the SOFTWARE and documentation.     */
/*                                                                         */
/*                    U.S. GOVERNMENT RESTRICTED RIGHTS                    */
/* Use, duplication, or disclosure of the SOFTWARE and documentation by    */
/* the U.S. Government is subject to the restictions as set forth in       */
/* subparagraph (c)(1)(ii) of the Rights in Technical Data and Computer    */
/* Software clause at DFARS 252.227-7013.  Contractor/manufacturer is      */
/* Stephen L. Balkum and Daniel A. Sill, P.O. Box 7704, Austin, Texas      */
/* 78713-7704.                                                             */
/*                                                                         */
/* The copyright information found in this code must remain intact and may */
/* not be commented out or otherwise removed from the compiling process.   */
/*                                                                         */
/* By using this SOFTWARE or documentation, you agree to the above terms   */
/* and conditions.                                                         */
/***************************************************************************/

/*--------------------------------------------------------------------------\
| This file contains the function definitions for a scalable font system.
|
| Revision History:
|
| Feb ??, 1995: Completed and passed initial coding tests.
| Apr 03, 1995: Changed to use XMS for glyph storage.  Initial tests pass.
| Apr 04, 1995: Incorporated custom file access for memory model and
|               compiler independence.  Final tests pass.
\--------------------------------------------------------------------------*/


#include "svgacc.h"
#include "zfont.h"


/*---------------------------------------------------------------------------
| _zFontCopyright:  This string identifies this code and its authors.  In
|   accordance with the license agreement, this data may not be commented out
|   or otherwise removed from the compiling process.
*/
static char far _zFontCopyright[] = "zFont: (c) 1995 Zephyr Software, Stephen L. Balkum and Daniel A. Sill";


/*---------------------------------------------------------------------------
| _zFontCurrent:  pointer to the currently selected font.  For protection
| 	this variable is declared static and can only be seen by functions local
| 	to this module.  It is initialized to zero and can, therefore, be used as
|   a flag for the successful initialization of the zFont routines.
*/
static zFont far* _zFontCurrent = 0;


/*---------------------------------------------------------------------------
| _zTempSpace1, _zTempSpace2, _zTempSpace3:  Pointers for three buffers of
|   ample size for the the point stack and calls to the 2D transformation
|	functions.
*/
byte far *_zTempSpace1 = 0;
byte far *_zTempSpace2 = 0;
byte far *_zTempSpace3 = 0;


/*---------------------------------------------------------------------------
| lcbufseg, lcbufoff:  pointer to the internal buffer of svgaxx.  This buffer
|   will be used for the point stack and the transformations.  Care should be
|   taken to provide space for the call to fillpoly.
*/
extern unsigned short far lcbufseg;
extern unsigned short far lcbufoff;


/*---------------------------------------------------------------------------
| These functions provide support for this module and are declared static for
| protection.  They are defined at the end of this module.
*/
static D2Point far* far _zQuadSpline(byte quality, D2Point far* pointstack, D2Point far* p1, D2Point far* p2, D2Point far* p3);
static D2Point far* far _zPushPoint(D2Point far* pointstack, D2Point far* new_point);
static unsigned short far _zFileOpen(const char far* filename);
static unsigned short far _zFileClose(unsigned short filehandle);
static long far _zFileRead(unsigned short filehandle, void far* buffer, unsigned short count);
static long far _zFileLength(unsigned short filehandle);


/*---------------------------------------------------------------------------
| zFontInitialize()
|
| This function establishes the temporary working buffers for this module.
| The internal buffer of the svgaxx library is used to provide three 12K
| work areas.  For this reason the whichvga function must be called first as
| it establishes the svgaxx internal buffers.  Calling this function out of
| order will probably crash the system.
|
| Inputs: none
|
| Return: none
*/
void far zFontInitialize(void)
{
    _zTempSpace1 = (byte far *)(((unsigned long)lcbufseg << 16) + (unsigned long)lcbufoff);
    _zTempSpace2 = _zTempSpace1 + 12288;
    _zTempSpace3 = _zTempSpace2 + 12288;
}


/*---------------------------------------------------------------------------
| zFontLoad()
|
| This function loads a font from disk, defines the values in the zFont
| structure and stores the data in extended memory.  This requires that the
| whichxms function be called first successfully.
|
| Inputs: fontname = pointer to filename for font data
|         destfont = pointer to zFont structure for this font
|
| Return: 1 for success, 0 for XMS error (see xmserror()), -1 for file error
|         (see errno)
*/
int far zFontLoad(const char far *fontname, zFont far* destfont)
{
    unsigned int filehandle;
    long flength;
    unsigned int kbytes;
    unsigned int i, j;
    long k, t;

    filehandle = _zFileOpen(fontname);
    if(!filehandle)
    {
        // file error
        return(-1);
    }

    flength = _zFileLength(filehandle);
    if(flength < sizeof(destfont->_GlyphOffset))
    {
        // file error
        return(-1);
    }

    /* determine required size of xms buffer */
    kbytes = (unsigned int)(flength>>10);
    if(flength & 0x3FF)
        kbytes++;

    destfont->_XMSHandle = xmsallocate(kbytes);
    if(!destfont->_XMSHandle)
    {
        // xms error
        _zFileClose(filehandle);
        return(0);
    }

	_zFileRead(filehandle, destfont->_GlyphOffset, sizeof(destfont->_GlyphOffset));

    /* determine length of each individual glyph */
    for(i=0;i<256;i++)
    {
        /* look for the next closest glyph offset */
        k = 0x7FFFFFFF;
        for(j=0;j<256;j++)
        {
            if(j != i) // don't check against itself
            {
                if(destfont->_GlyphOffset[j] > destfont->_GlyphOffset[i])
                {
                    t = destfont->_GlyphOffset[j] - destfont->_GlyphOffset[i];
                    if(t < k)
                        k = t;
                }
            }
        }
        // now check against the end of the file
        t = flength - destfont->_GlyphOffset[i];
        if(t < k)
            k = t;

        destfont->_GlyphLength[i] = k;
    }

    /* subtract 0x400 from each offset */
    for(i=0;i<256;i++)
        destfont->_GlyphOffset[i] -= 0x400;

    /* read in each glyph */
    k = 0;
    flength -= sizeof(destfont->_GlyphOffset);
    while(k < flength)
    {
        t = _zFileRead(filehandle, _zTempSpace1, 16384);
        if(t < 0)
        {
            // file error
            _zFileClose(filehandle);
            xmsfree(destfont->_XMSHandle);
            return(-1);
        }
        xmsput(_zTempSpace1, destfont->_XMSHandle, k, t);
        k += t;
	}
	_zFileClose(filehandle);

    /* success */
    return(1);
}


/*---------------------------------------------------------------------------
| zFontUnload()
|
| This function merely frees the extended memory used by the font.  It serves
| as a reminder that this is required before exiting a program.
|
| Inputs: destfont = pointer to zFont structure for font to unload
|
| Return: none
*/
void far zFontUnload(zFont far* font)
{
    xmsfree(font->_XMSHandle);
}


/*---------------------------------------------------------------------------
| zFontSet()
|
| This function assigns the hidden variable _zFontCurrent.
|
| Inputs: destfont = pointer to font to make active
|
| Return: none
*/
void far zFontSet(zFont far* newfont)
{
    _zFontCurrent = newfont;
}


/*---------------------------------------------------------------------------
| zFontDraw()
|
| This is the heart of the zFont module.  This function draws the given
| string on the screen in the font pointed to by _zFontCurrent.  The quality
| of the curves, location and orientation, and color are all options.
|
| Inputs: fcolr   = fill color for the string
|         angle   = orientation angle of string in degrees
|         height  = the desired pixel height of the character bounding box
|         quality = determines number of points calculated in curves (see
|                   _zQuadSpline)
|         strng   = the text to be drawn
|         x, y    = the location to draw the bottom, left corner of the first
|                   character
|
| Return: none
*/
void far zFontDraw(int fcolr, int angle, int height, byte quality, const char far *strng, int x, int y)
{
	/* temporaries and counters */
	unsigned short i, k;
	unsigned int j, h;

    int glyphw, glyphh;         /* dimensions of each glyph */
	int xoffset;                /* offset from beginning of text (disregards angle) */
	unsigned int length;        /* num chars in string */
	unsigned int numlines;      /* num lines to draw char */
	unsigned short numcontours; /* num contours in char */
	unsigned short numpoints;   /* num points in contour */
	short far* thisglyph;       /* local pointer to current glyph */
	short far* p;               /* local pointer to current contour */
	
	D2Point far* pointstack;    /* current location in stack of line points, reset for each contour */

	D2Point start_point;        /* need to remember where we started */
	byte start_state;
	
	D2Point p1, p2, p3;         /* working points */
	
	#define VALUE_MASK          (0x3FFF)
	#define VALUE_SIGN_BIT      (0x2000)
	#define VALUE_INNER_CONTOUR (0x4000)
	#define VALUE_OFF_CURVE     (0x8000)
	#define VALUE_FLAG_MASK     (0xC000)

	//This macro takes our 14 bit signed number and returns a 16 bit signed number
	#define VALUE(v) \
	((short)( ((v) & VALUE_SIGN_BIT) ? ((v) | VALUE_FLAG_MASK) : ((v) & VALUE_MASK) ))

	byte stateflags = 0;        /* current curve status */
	#define CURR_STATE_LINE 0x01
    #define CURR_STATE_CURVE 0x00
	
    xoffset = 0;

    /* get length of string */
    length = 0;
    k = 0;
    while(strng[k])
    {
        k++;
        length++;
    }

    /* if no font is set, then force an early exit. */
    if(!_zFontCurrent)
        length = 0;

	for(k=0;k<length;k++) {

        /* glyph is copied into tempspace3 since pointstack is in tempspace1
         * and transformed points sit in tempspace2
         */
        thisglyph = (short far*)_zTempSpace3;
        xmsget(_zFontCurrent->_XMSHandle,
        	_zFontCurrent->_GlyphOffset[*(byte far *)(strng+k)],
            thisglyph,
            _zFontCurrent->_GlyphLength[*(byte far *)(strng+k)]);

		/* first short is width, second is height, third is number of contours */
        glyphw = thisglyph[0];
        glyphh = thisglyph[1];
		numcontours = thisglyph[2];
		p = &thisglyph[3];

        for(j=0;j<numcontours;j++) {
            pointstack = (D2Point far*)_zTempSpace1; /* reset pointstack */
			numpoints = *p; /* first short is number of points in contour */
			p++;
			
			/* read and push beginning state and first point */
			if(!(*p & VALUE_OFF_CURVE))
				start_state = stateflags = CURR_STATE_LINE;
			else
				start_state = stateflags = CURR_STATE_CURVE;
			start_point.x = VALUE(*p);
			p++;
			start_point.y = VALUE(*p);
			p++;
			p1 = start_point;
			p2 = start_point;
			// place first point on stack
            pointstack = _zPushPoint(pointstack, &start_point);

			/****************************************/
			/* always maintain previous point in p2 */
            /****************************************/
			for(i=1;i<numpoints;i++) {

				if(stateflags == CURR_STATE_LINE) {
					if(!(*p & VALUE_OFF_CURVE)) { /* next point is on curve */
						p3.x = VALUE(*p);
						p++;
						p3.y = VALUE(*p);
						p++;
						// line thru p2, p3
                        pointstack = _zPushPoint(pointstack, &p3);
						p1 = p2;
						p2 = p3;
					} else {
						p3.x = VALUE(*p);
						p++;
						p3.y = VALUE(*p);
						p++;
						p1 = p3;
						p1.x += p2.x;
						p1.x >>= 1;
						p1.y += p2.y;
						p1.y >>= 1;
						// line thru p2, p1
                        pointstack = _zPushPoint(pointstack, &p1);
						p1 = p2;
						p2 = p3;
						stateflags = CURR_STATE_CURVE;
					}
				} else {
					if(!(*p & VALUE_OFF_CURVE)) { /* next point is on curve */
						p1.x += p2.x;
						p1.x >>= 1;
						p1.y += p2.y;
						p1.y >>= 1;
						// line thru p1, p2
                        pointstack = _zPushPoint(pointstack, &p2);
						p3.x = VALUE(*p);
						p++;
						p3.y = VALUE(*p);
						p++;
						// line thru p2, p3
                        pointstack = _zPushPoint(pointstack, &p3);
						p1 = p2;
						p2 = p3;
						stateflags = CURR_STATE_LINE;
					} else {
						p3.x = VALUE(*p);
						p++;
						p3.y = VALUE(*p);
						p++;
                        // _zQuadSpline thru p1, p2, p3
                        pointstack = _zQuadSpline(quality, pointstack, &p1, &p2, &p3);
						p1 = p2;
						p2 = p3;
					}
				}
			}

			/* make one more pass thru using the start_point */
			if(stateflags & CURR_STATE_LINE) {
				if(!(start_state & CURR_STATE_LINE)) { /* start_point is off curve */
					start_point.x += p2.x;
					start_point.x >>= 1;
					start_point.y += p2.y;
					start_point.y >>= 1;
				}
				// line thru p2, start_point
                pointstack = _zPushPoint(pointstack, &start_point);
			} else {
				if(start_state & CURR_STATE_LINE) { /* next point is on curve */
					p1.x += p2.x;
					p1.x >>= 1;
					p1.y += p2.y;
					p1.y >>= 1;
					// line thru p1, p2
                    pointstack = _zPushPoint(pointstack, &p2);
					// line thru p2, start_point
                    pointstack = _zPushPoint(pointstack, &start_point);
				} else {
                    // _zQuadSpline thru p1, p2, start_point
                    pointstack = _zQuadSpline(quality, pointstack, &p1, &p2, &start_point);
				}
			}

            /* calculate number of points */
            numlines = ((unsigned long)pointstack & 0x0000FFFF);
            numlines -= ((unsigned long)_zTempSpace1 & 0x0000FFFF);
            numlines >>= 2; /* /= sizeof(D2Point); */

			/* translate to specified offset relative to string */
            d2translate(numlines, xoffset, 0, (D2Point far*)_zTempSpace1, (D2Point far*)_zTempSpace2);

			/* if necessary, rotate the points relative to its offset in string */
			if(angle) {
                d2rotate(numlines, 0, 0, angle, (D2Point far*)_zTempSpace2, (D2Point far*)_zTempSpace2);
			}

			/* scale the points */
			h = height;
			h <<= 8; /* 256 = 1 for d2scale factor */
            h /= (glyphh + 1);
			d2scale(numlines, h, h, (D2Point far*)_zTempSpace2, (D2Point far*)_zTempSpace2);

			/* translate to specified position */
			d2translate(numlines, x, y, (D2Point far*)_zTempSpace2, (D2Point far*)_zTempSpace2);

            fillpoly(fcolr, numlines, (D2Point far*)_zTempSpace2);
        }
		
		/* offset for next character */
		xoffset += (glyphw + 1);
	}
}


/*---------------------------------------------------------------------------
| _zQuadSpline()
|
| Calculates the points for a quadratic spline and places them on the stack.
| The number of points calculated is 2^quality.
|
| Inputs: quality    = power of 2 number of points to calculate
|         pointstack = a stack for the calculated points
|         p1         = point 1
|         p2         = point 2
|         p3         = point 3
|
| Return: the new top of the point stack
*/
static D2Point far* far _zQuadSpline(byte quality, D2Point far* pointstack, D2Point far* p1, D2Point far* p2, D2Point far* p3)
{
	#define QUADSPLINESHIFT 4
	#define QUADSPLINEFIXED (1 << QUADSPLINESHIFT)
	#define NUMQUADSPLINESTEPS 2 //16
	#define QUADSPLINESTEP (QUADSPLINEFIXED/NUMQUADSPLINESTEPS)
	#define QUADSPLINESTEP2 (QUADSPLINESTEP * QUADSPLINESTEP)

	byte i, numquadsplinesteps; // counter variable and number of segments as determined
								// by specified quality
	byte quadsplinestep, quadsplinestep2;   // shifts to create QUADSPLINEFIXED/numquadsplinesteps
											// and the same squared

	D2Point current_pos, new_pos;
	long ax, ay, bx, by;
	long tx, ty, t;
	long txd1, txd2;
	long tyd1, tyd2;
	
	quality &= 0x03; // only a quality of 0-3 is allowed
	
	numquadsplinesteps = 1;
	numquadsplinesteps <<= quality;
	
	quadsplinestep = 4;
	quadsplinestep -= quality;
	
	quadsplinestep2 = quadsplinestep;
	quadsplinestep2 <<= 1;
	
	ax = p1->x;
	ax += p3->x;
	ax >>= 1;
	ax -= p2->x;
	ax <<= QUADSPLINESHIFT;
	
	bx = p2->x;
	bx -= p1->x;
	bx <<= (QUADSPLINESHIFT * 2);
	
	tx = p1->x;
	tx += p2->x;
	tx >>= 1;
	tx <<= (QUADSPLINESHIFT * 3);
	
	ay = p1->y;
	ay += p3->y;
	ay >>= 1;
	ay -= p2->y;
	ay <<= QUADSPLINESHIFT;
	
	by = p2->y;
	by -= p1->y;
	by <<= (QUADSPLINESHIFT * 2);
	
	ty = p1->y;
	ty += p2->y;
	ty >>= 1;
	ty <<= (QUADSPLINESHIFT * 3);
	
	txd1 = (ax << quadsplinestep2) + (bx << quadsplinestep);
	txd2 = (ax << 1) << quadsplinestep2;
	
	tyd1 = (ay << quadsplinestep2) + (by << quadsplinestep);
	tyd2 = (ay << 1) << quadsplinestep2;
	
    for (i = 0; i < numquadsplinesteps; i++ ) {
		tx += txd1;
		t = tx;
		t >>= (QUADSPLINESHIFT * 3);
		new_pos.x = (short)t;
		txd1 += txd2;
		
		ty += tyd1;
		t = ty;
		t >>= (QUADSPLINESHIFT * 3);
		new_pos.y = (short)t;
		tyd1 += tyd2;

        pointstack = _zPushPoint(pointstack, &new_pos);
    }
	return(pointstack);
}


/*---------------------------------------------------------------------------
| _zPushPoint()
|
| Pushes a new 2D point on the point stack and returns the new top of the
| stack.
|
| Inputs: pointstack = a initial top of the stack
|         new_point  = point to be pushed
|
| Return: the new top of the point stack
*/
static D2Point far* far _zPushPoint(D2Point far* pointstack, D2Point far* new_point)
{
	*pointstack = *new_point;
	pointstack++;
	return(pointstack);
}


/*---------------------------------------------------------------------------
| _zFileOpen()
|
| Opens a file for binary reading.  This function is implemented to provide
| memory model and compiler independence.
|
| Inputs: filename = path and name of file to open
|
| Return: the handle of the opened file or zero for failure
*/
static unsigned short far _zFileOpen(const char far* filename)
{
    unsigned short fnseg, fnoff;

    fnseg = ((unsigned long)filename >> 16);
    fnoff = ((unsigned long)filename & 0x0000FFFF);

    asm
    {
        push 	ds
        mov     dx, fnoff
        mov     ds, fnseg
        mov     ax, 0x3D00
        int     0x21
        jnc     short isgood
        mov     ax, 0
isgood:
        pop     ds
    }
}


/*---------------------------------------------------------------------------
| _zFileClose()
|
| Closes a file opened by _zFileOpen.  This function is implemented to
| provide memory model and comiler independence.
|
| Inputs: filehandle = handle of file to close
|
| Return: 1 for success, 0 for failure
*/
static unsigned short far _zFileClose(unsigned short filehandle)
{
    asm
    {
        mov     bx, filehandle
        mov     ax, 0x3E00
        int     0x21
        mov     ax, 1
        jnc     short isgood
        mov     ax, 0
isgood:
    }
}


/*---------------------------------------------------------------------------
| _zFileRead()
|
| Reads a specified number of bytes from a file opened by _zFileOpen.  This
| function is implemented to provide memory model and comiler independence.
|
| Inputs: filehandle = handle of file to close
|         buffer     = pointer to memory buffer to receive bytes
|         count      = number of bytes to read
|
| Return: number of bytes read, zero for end-of-file, -1 for error
*/
static long far _zFileRead(unsigned short filehandle, void far* buffer, unsigned short count)
{
    unsigned short bufseg, bufoff;

    bufseg = ((unsigned long)buffer >> 16);
    bufoff = ((unsigned long)buffer & 0x0000FFFF);

    asm
    {
        push    ds
        mov     bx, filehandle
        mov     cx, count
        mov     dx, bufoff
        mov     ds, bufseg
        mov     ax, 0x3F00
        int     0x21
        pop     ds
        jnc     short isgood
        mov     ax, -1
isgood:
        cwd
    }
}


/*---------------------------------------------------------------------------
| _zFileLength()
|
| Determines the size in bytes of a file opened by _zFileOpen.  This function
| is implemented to provide memory model and comiler independence.
|
| Inputs: filehandle = handle of file to close
|
| Return: length of file in bytes, -1 if failure
*/
static long far _zFileLength(unsigned short filehandle)
{
    unsigned short posmsw, poslsw, lenmsw, lenlsw;
    asm
    {
        /* get current file position */
        mov     ax, 0x4201
        mov     bx, filehandle
        mov     cx, 0
        mov     dx, cx
        int     0x21
        jc     	short isbad
        mov     posmsw, dx
        mov     poslsw, ax

        /* get file length */
        mov     ax, 0x4202
        mov     bx, filehandle
        mov     cx, 0
        mov     dx, cx
        int     0x21
        mov     lenmsw, dx
        mov     lenlsw, ax

        /* restore file position */
        mov     ax, 0x4200
        mov     bx, filehandle
        mov     dx, posmsw
        mov     cx, poslsw
        int     0x21

        /* prepare return value */
        mov     dx, lenmsw
        mov     ax, lenlsw
        jmp     short done
isbad:
        mov     ax, -1
        cwd
done:
    }
}
