/*
 * GTCMOUSE.C : Display a mouse cursor with graphics resolution in Text
 *            Mode on VGA video adaptors. Character pattern rewritting,
 *	      is used for this purpose. It works only with the 80x25 color
 *            mode and with large/huge compliler models. TurboC v2.0 was used.
 *
 * Written : 24 July 1992,
 * Author : A.Moshovos, Grad. Stud., Computer Science Dept., Univ. Of Crete,
 *         Heraklion, Hellas (Greece)
 * e-mail : moshovos@csi.forth.gr
 *
 * IMPORTANT :
 
 * This code is provided "as is" without express or implied waranty.
 * Permission to freely use, copy and distribute this code without any fee
 * is granted as long as those statements and the copyright notice
 * (gtcmouse_cid string) are included in both source and executable code.
 *  Additionaly in the case of commercial a written notification is required,
 * via surface mail, to :
 *       A.Moshovos, Paysania 21-23, Thessaloniki, 54352-GR, Greece.
 *
 * HOW TO USE GTCMOUSE :
 *
 *  I.initialization & exit functions :
 *	1st : gtcmouse_init(); -> initializes internal structures and hooks int 8
 *  on exit : gtcmouse_close();-> restores int 8 handler.
 *
 * II.interface functions :
 * 	gtcmouse_show(x,y); -> displays the cursor at (x,y) ([0,640],[0,400]).
 *	gtcmouse_hide();    -> hide mouse cursor
 *	gtcmouse_reveal();  -> enable a previous hiden cursor
 *	gtcmouse_getxy(&x,&y); -> returns the cursor text screen coordinates ([0,79],[0,24]).
 *
  NOTES:
  1. gtcmouse internally uses four characters, by default the 220-223 chars.
  The first of the four is defined by the GMOUSE_CHAR const. Change that value
  if it is necessary to use those characters.

  2. The VGA memory access routines have been tested only on
  few adaptors. if problems occur one could use the bios function 0x1100 of
  int 0x10 in place of the vga_char() function. But video synchronization
  problems occur on many VGA cards if this method is used (screen flushing
  or waving).

  3. Not all the characters propagate the value of bit 0 to the 9th bit when using
  fonts of 8x16 size on VGA adaptors. Certainly the ascci characters do not,
  while the characters used for cursor displaying do. For that, some
  undesirable effects occur when the cursor passes over characters that have
  some of 0 bits set. This problems can be bypassed only by redesigning
  the font.

  4. It is assumed that the VGA adaptor is currently set to the 8x25 color
  mode with 8x16 character size. The vga_sync function will hang if used
  with monochrome emulation. ( this is due to the address of the status
  register which is 0x3da on color mode and 0x3ba on monochrome mode)


*/

#include <stdio.h>
#include <dos.h>

#define GMOUSE_FONT_SZ 4080			/* 256 characters x 16 bytes/char */
#define GMOUSE_CHAR 220				/* first of the 4 characters used to display the cursor */

void interrupt gtcmouse_int();			/* new int 8 handler */
void interrupt (*gtcmouse_int8)();		/* points to old int 8 handler */
char gtcmouse_cid[] = { "GTCmouse v1.0 , Jul 24 1992, author: A.Moshovos, Thes/niki Greece" };
unsigned char gtcmouse_font[GMOUSE_FONT_SZ];	/* font patterns */
unsigned char gtcmouse_cfill[16] = {            /* mouse cursor pattern, one character 8x16 */
 0x00,0x40,0x40,0x60,0x60,0x70,0x70,0x78,
 0x78,0x7e,0x7e,0x48,0x08,0x04,0x04,0x00
				};
unsigned char gtcmouse_cmask[16] = {		/* mouse mask pattern */
 0x1f,0x1f,0x1f,0x0f,0x0f,0x07,0x07,0x03,
 0x03,0x01,0x01,0x01,0xe1,0xf1,0xf1,0xf1
				};
unsigned gtcmouse_cfl[16],gtcmouse_cml[16];	/* shifted mouse patterns */
unsigned char gtcmouse_scr[4];			/* chars under mouse cursor */
int	gtcmouse_px,gtcmouse_py;		/* previous mouse coordinates (text)*/
int	gtcmouse_x,gtcmouse_y;			/* current mouse coordinates (text)*/
int	gtcmouse_on;				/* 1 if the cursor is displayed */
int	gtcmouse_xr,gtcmouse_yr;    		/* previous coordinates (graphics) */

gtcmouse_vpoke(x,y,v)				/* vga 80x25 color mode assumed */
{						/* video ram starts at b800:0000 */
 if (x>79) return;
 pokeb(0xb800,(x<<1)+(y*160),v);
}

gtcmouse_vpeek(x,y)
{
 if (x>79) return;
 return(peekb(0xb800,(x<<1)+(y*160)));
}

gtcmouse_init()
/* Copy the vga 8x16 font patterns into a local buffer.
 * Communicate with the mouse driver and modify the sensitivity and the vertical/horizontal limits.
 * if a mouse driver is not found 0 is returned
 * Hook the timer (0x08) interrupt.
 */
{
 struct REGPACK r;

 r.r_ax=0x0000; intr(0x33,&r); if (r.r_ax==0) return(0); /* driver installation check */
 vga_char(255,0,gtcmouse_font,0);
 r.r_ax=0x000f; r.r_cx=r.r_dx=1; intr(0x33,&r);		/* sensitivity */
 r.r_ax=0x0007; r.r_cx=0; r.r_dx=80*8*8; intr(0x33,&r);	/* vert/horz limits */
 r.r_ax=0x0008; r.r_cx=0; r.r_dx=25*16*16; intr(0x33,&r);
 gtcmouse_x=gtcmouse_y=-1; gtcmouse_px=gtcmouse_py=0;		/* initialize cursor on screen */
 gtcmouse_save(0,0);
 gtcmouse_on=1;
 gtcmouse_show(1,1);
 gtcmouse_int8=getvect(0x8); setvect(0x8,gtcmouse_int);	/* hook int 8 */
 return(1);
}

gtcmouse_restore(x,y)
/* restore the chars under the cursor */
{
 gtcmouse_vpoke(x,y,gtcmouse_scr[0]);
 gtcmouse_vpoke(x+1,y,gtcmouse_scr[1]);
 gtcmouse_vpoke(x,y+1,gtcmouse_scr[2]);
 gtcmouse_vpoke(x+1,y+1,gtcmouse_scr[3]);
}

gtcmouse_save(x,y)
/* save the chars under the cursor */
{
 gtcmouse_scr[0]=gtcmouse_vpeek(x,y);
 gtcmouse_scr[1]=gtcmouse_vpeek(x+1,y);
 gtcmouse_scr[2]=gtcmouse_vpeek(x,y+1);
 gtcmouse_scr[3]=gtcmouse_vpeek(x+1,y+1);
}

gtcmouse_cp(f,t)
/* copy the pattern of character f to character t */
{
 int i;

 f<<=4;
 t<<=4;
 for (i=0; i<16; i++) gtcmouse_font[t++]=gtcmouse_font[f++];
}

gtcmouse_show(x,y)
/* Display the mouse cursor :
 * 1st : return if the coordinates are not changed.
 * restore the characters under the previous cursor position
 * save the characters under the new cursor position
 * copy their patterns to the characters used for the cursor
 * and insert the cursor pattern.
 * Finally display them and update the cursor position counters.
 */

{
 int i,j,o;
 int xo,yo;

 if (gtcmouse_on==0) return;
 gtcmouse_x=x>>3; gtcmouse_y=y>>4;
 if (gtcmouse_xr==x && gtcmouse_yr==y) return(1);
 xo=7-x&0x07; yo=y&0x0f;
 gtcmouse_restore(gtcmouse_px,gtcmouse_py);
 gtcmouse_save(gtcmouse_x,gtcmouse_y);
 gtcmouse_cp(gtcmouse_scr[0],GMOUSE_CHAR);
 gtcmouse_cp(gtcmouse_scr[1],GMOUSE_CHAR+1);
 gtcmouse_cp(gtcmouse_scr[2],GMOUSE_CHAR+2);
 gtcmouse_cp(gtcmouse_scr[3],GMOUSE_CHAR+3);
 j=GMOUSE_CHAR<<4;
 o=~(0xffff<<xo);
 for (i=0; i<16; i++) {
	gtcmouse_cfl[i]=((unsigned)gtcmouse_cfill[i])<<xo;
	gtcmouse_cml[i]=o|((((unsigned)gtcmouse_cmask[i])|0xff00)<<xo);
 }
 for (i=0; i<16; i++)  {
  o=(i+yo<16) ? j+yo+i : j+16+yo+i;
  gtcmouse_font[o+16]&=gtcmouse_cml[i]&0xff;
  gtcmouse_font[o]&=(gtcmouse_cml[i])>>8;
  gtcmouse_font[o+16]|=gtcmouse_cfl[i]&0xff;
  gtcmouse_font[o]|=(gtcmouse_cfl[i])>>8;
 }
 vga_char(4,GMOUSE_CHAR,gtcmouse_font+j,1);
 gtcmouse_vpoke(gtcmouse_x,gtcmouse_y,GMOUSE_CHAR);
 gtcmouse_vpoke(gtcmouse_x+1,gtcmouse_y,GMOUSE_CHAR+1);
 gtcmouse_vpoke(gtcmouse_x,gtcmouse_y+1,GMOUSE_CHAR+2);
 gtcmouse_vpoke(gtcmouse_x+1,gtcmouse_y+1,GMOUSE_CHAR+3);

 gtcmouse_px=gtcmouse_x; gtcmouse_py=gtcmouse_y;
 gtcmouse_xr=x; gtcmouse_yr=y;
}

gtcmouse_hide()
/* hide the cursor */
{
 if (gtcmouse_on) gtcmouse_restore(gtcmouse_px,gtcmouse_py);
 gtcmouse_on=0;
}

gtcmouse_reveal()
/* enable the cursor */
{
 if (gtcmouse_on==0) {
	gtcmouse_save(-1,-1);
	gtcmouse_px=gtcmouse_py=-1;
	gtcmouse_on=1;
	gtcmouse_show(gtcmouse_x,gtcmouse_y);
 }
 gtcmouse_on=1;
}

gtcmouse_getxy(x,y) int *x,*y;
/* returns the cursor coordinates (text 80x25) */
{
 *x=gtcmouse_x;
 *y=gtcmouse_y;
}

gtcmouse_mouse_getxy(x,y) int *x,*y;
/* calls the mouse driver to get the mouse coordinates */
{
 struct REGPACK r;

 r.r_ax=0x0003;
 intr(0x33,&r);
 *x=r.r_cx;
 *y=r.r_dx;
}

void interrupt gtcmouse_int()
/* new timer interrupt :
 * get the new mouse coordinates and display the cursor
 */
{
 int x,y;
 (*gtcmouse_int8)();
 gtcmouse_mouse_getxy(&x,&y);
 gtcmouse_show(x>>3,y>>4);
}

gtcmouse_close()
/* simply returns the timer interrupt to its previous owner */
{
 setvect(0x8,gtcmouse_int8);
}


/*-------------------------------------------------------------------------*/
/* VGA direct memory access, those routines should work fine with any vga
 * card.
 */
unsigned R41,R44,R42,RE5,RE6,RE7,RE4;
unsigned p2[] = { 1,2,4,8 };
bnk_prolog(bank)
{
 outportb(0x3c4,2); R42=inportb(0x3c5); outportb(0x3c5,p2[bank]);
 outportb(0x3c4,4); R44=inportb(0x3c5); outportb(0x3c5,7);
 outportb(0x3ce,5); RE5=inportb(0x3cf); outportb(0x3cf,0);
 outportb(0x3ce,6); RE6=inportb(0x3cf); outportb(0x3cf,4);
 outportb(0x3ce,4); RE4=inportb(0x3cf); outportb(0x3cf,bank);
 outportb(0x3ce,7); RE7=inportb(0x3cf); outportb(0x3cf,0xff);
}
bnk_epilog()
{
 outportb(0x3c4,2); outportb(0x3c5,R42);
 outportb(0x3c4,4); outportb(0x3c5,R44);
 outportb(0x3ce,5); outportb(0x3cf,RE5);
 outportb(0x3ce,6); outportb(0x3cf,RE6);
 outportb(0x3ce,4); outportb(0x3cf,RE4);
 outportb(0x3ce,7); outportb(0x3cf,RE7);
}
vga_char(chars,first,buff,set) unsigned char *buff;
{
 int i,il;
 unsigned char *mp,*mpe;
 unsigned pat_s,pat_o;

 pat_s=FP_SEG(buff); pat_o=FP_OFF(buff);
 mp=MK_FP(0xa000,first*32);
 mpe=mp+chars*32;
 disable();

 while (inportb(0x3da)&0x08);
 while (inportb(0x3da)&0x08==0);

 bnk_prolog(2);
 for (i=0; mp<mpe;) {
  if (set) *mp++=*buff++;
  else *buff++=*mp++;
  i++;
  if (i%16==0) mp+=16;
 }
 bnk_epilog();
 enable();
}


