/* PLPRINT.C : Printer output routines for PICLAB.  The only command
** handler here is for the PRINT command.
*/

#include <stdlib.h>
#include <string.h>
#include <memory.h>
#include <bios.h>

#include "piclab.h"
#include "pjtbls"

static int outfile;
static U8 *inlines[3], *outbuf;
static int pxoff, pyoff, ptype, pwidth = 0, transparent = 0;

static int printblock(int, char *);
static int startprint(void);
static int endprint(void);
static int pjcolor(U8 *);
static int pjmono(void);

static int printblock(int len, char *block)
{
	int r;

	for (r = len; len>0; --len,++block) {
		if (_bios_printer(_PRINTER_WRITE, 0, (U16)*block) & 0x29) r = -1;
	}
	return r;
}

/* Open printer output file, write initialization string.
*/
static int startprint(void)
{
    char initstring[80];
    int l;

	if (!*printfile) {
		if (_bios_printer(_PRINTER_INIT, 0, 0) & 0x29) return 3;
	} else {
    	if ((outfile = p_open(printfile, outbuf, BUFSIZE, 512, WRITE)) < 0) return 3;
	}
    switch (ptype) {
        case 0: /* PAINTJET */
            pl_sprintf(initstring,
            "\033&k%dW\033*t180R\033*r%ds3U\033*b%dM\033&a%dy%dX\033*r1A",
            transparent ? 3 : 0, pwidth<<1, 0, pyoff, pxoff);
            break;
        case 1: /* LASERJET */
            pl_sprintf(initstring,
            "\033E\033&l0e0O\033&s1C\0339\033*t%dR\033*r1A",
            laserdpi);
            break;
        default:    return 1;
    }
    l = strlen(initstring);
	if (*printfile) {
	    memcpy(linebuf(outfile), initstring, l);
    	if (p_putline(outfile, l) != l) return 3;
	} else {
		if (printblock(l, initstring) != l) return 3;
	}
	return 0;
}

/* Write cleanup string, close output file.
*/
static int endprint()
{
    int l;
    char temp[80];

    switch (ptype) {
        case 0: /* PAINTJET */
        case 1: /* LASERJET */
            pl_sprintf(temp, "\033*r0B\n");
            break;
        default:    return 1;
    }
    l = strlen(temp);
	if (*printfile) {
    	memcpy(linebuf(outfile), temp, l);
    	if (p_putline(outfile, l) != l) return 3;
	    memcpy(linebuf(outfile), crlf, 2);
    	if (p_putline(outfile, 2) != 2) return 3;
    	p_close(outfile);
	} else {
		if (printblock(l, temp) != l) return 3;
		if (printblock(2, crlf) != 2) return 3;
	}
    return 0;
}

/* Pjcolor() takes an array of indices from the ditherline() routine and
** shuffles the bits around into the bizarre order the PaintJet needs.  Those
** of you without 16 fingers must be content that it works.
*/
static int pjcolor(U8 *indices)
{
    static char pcode2[20], pcode3[20];
    int w, p, x, cl, l;
    register U8 b1, b2;
    U8 *ip;
    U32 mem;
    U16 m0, m1, m2, m3, *pp, *patline;
    U8 plane[3][2][180];
    char *bp;

    mem = mark();
    if ((patline = (U16 *)talloc(pwidth * sizeof(U16))) == NULL) return 2;
    ip = indices;
    pp = patline;
	if (new->flags & 1) {
		ip += pwidth;
		for (x=0; x<pwidth; ++x) *pp++ = pjpatterns[*--ip];
	} else {
		for (x=0; x<pwidth; ++x) *pp++ = pjpatterns[*ip++];
	}

    w = (pwidth+3) >> 2;
    pl_sprintf(pcode2, "\033*b%dV", w);
    pl_sprintf(pcode3, "\033*b%dW", w);
    cl = strlen(pcode3);

    for (p=0; p<3; ++p) {
        m3 = 1 << p; m2 = m3 << 4; m1 = m2 << 4; m0 = m1 << 4;
        pp = patline;

        for (x=0; x<w; ++x) {
            b1 = b2 = 0;
            if (*pp   & m0) b1 |= 0x80;
            if (*pp   & m1) b1 |= 0x40;
            if (*pp   & m2) b2 |= 0x80;
            if (*pp++ & m3) b2 |= 0x40;

            if (*pp   & m0) b1 |= 0x20;
            if (*pp   & m1) b1 |= 0x10;
            if (*pp   & m2) b2 |= 0x20;
            if (*pp++ & m3) b2 |= 0x10;

            if (*pp   & m0) b1 |= 0x08;
            if (*pp   & m1) b1 |= 0x04;
            if (*pp   & m2) b2 |= 0x08;
            if (*pp++ & m3) b2 |= 0x04;

            if (*pp   & m0) b1 |= 0x02;
            if (*pp   & m1) b1 |= 0x01;
            if (*pp   & m2) b2 |= 0x02;
            if (*pp++ & m3) b2 |= 0x01;

            plane[p][0][x] = b1;
            plane[p][1][x] = b2;
        }
    }
    for (p=0; p<3; ++p) {
        bp = ((p==2) ? pcode3 : pcode2);
        l = strlen(bp);

		if (*printfile) {
	        memcpy(linebuf(outfile), bp, l);
    	    if (p_putline(outfile, l) != l) return 3;
        	memcpy(linebuf(outfile), plane[p][0], w);
	        if (p_putline(outfile, w) != w) return 3;
		} else {
			if (printblock(l, bp) != l) return 3;
			if (printblock(w, plane[p][0]) != w) return 3;
		}
    }
    for (p=0; p<3; ++p) {
        bp = ((p==2) ? pcode3 : pcode2);
        l = strlen(bp);

		if (*printfile) {
	        memcpy(linebuf(outfile), bp, l);
    	    if (p_putline(outfile, l) != l) return 3;
        	memcpy(linebuf(outfile), plane[p][1], w);
	        if (p_putline(outfile, w) != w) return 3;
		} else {
			if (printblock(l, bp) != l) return 3;
			if (printblock(w, plane[p][1]) != w) return 3;
		}
    }
    release(mem);
    return 0;
}

static int pjmono()
{
    pl_printf(	"Image must be in full color format for printing on the\r\n"
				"HP PaintJet.  Issue COLOR and/or UNMAP commands as\r\n"
				"needed and print again.\r\n");
    return 0;
}

/* Print NEW buffer to HP PaintJet.
*/
int pjprint(void)
{
    int i, r;
    U8 *rgb[3], *pjline;
    U16 lines;
    struct _plane *ip[3];

    if (new->planes != 3) return pjmono();
    if ((pwidth = new->width - xorigin) > 720) pwidth = 720;
    if ((r = startprint()) != 0) return r;

    startdither = 1;
    rv = pjrv; gv = pjgv; bv = pjbv; table2 = pjtable2;

    if ((pjline = (U8 *)talloc(pwidth)) == NULL) return 2;
    for (i=0; i<3; ++i) if ((ip[i] = openplane(i, new, READ)) == NULL) return 3;

    for (lines=0; lines < new->height; ++lines) {
        for (i=0; i<3; ++i) if (getline(ip[i]) == 0) return 4;
        for (i=0; i<3; ++i) rgb[i] = (U8 *)(ip[i]->linebuf + xorigin);
        ditherline(pwidth, rgb, pjline);
        if ((r = pjcolor(pjline)) != 0) return r;
        pl_trace(lines);
    }

    for (i=0; i<3; ++i) closeplane(ip[i]);
    if ((r = endprint()) != 0) return r;
    return 0;
}

/* Print NEW buffer to HP LaserJet.
*/
int ljprint(void)
{
    int r, l;
    U8 *gray, *bitmap[2];
    U16 lines, bw;
    struct _plane *ip;
    char code[20];

    if (new->planes != 1 || (new->flags & 2)) {
        pl_printf(	"Image must be grayscale to print on LaserJet.  Issue "
					"GRAY command,\r\nthen PRINT again.\r\n");
        return 0;
    }
    if (laserdpi == 300) bleed = 70; else bleed = 20;
    if ((pwidth = new->width - xorigin) > 1200) pwidth = 1200;
    if ((r = startprint()) != 0) return r;

    bw = (2 * pwidth + 7) >> 3;
    startdither = 1;
    bitmap[0] = (U8 *)talloc(bw);
    if ((bitmap[1] = (U8 *)talloc(bw)) == NULL) return 2;
    if ((ip = openplane(0, new, READ)) == NULL) return 3;

    for (lines=0; lines < new->height; ++lines) {
        if (getline(ip) == 0) return 4;
        gray = (U8 *)(ip->linebuf + xorigin);
        bitmapline(pwidth, gray, bitmap);
        pl_sprintf(code, "\033*b%dW", bw);
        l = strlen(code);

		if (*printfile) {
	        memcpy(linebuf(outfile), code, l);
    	    if (p_putline(outfile, l) != l) return 3;
        	memcpy(linebuf(outfile), bitmap[0], bw);
	        if (p_putline(outfile, bw) != bw) return 3;
    	    memcpy(linebuf(outfile), code, l);
        	if (p_putline(outfile, l) != l) return 3;
	        memcpy(linebuf(outfile), bitmap[1], bw);
    	    if (p_putline(outfile, bw) != bw) return 3;
		} else {
			if (printblock(l, code) != l) return 3;
			if (printblock(bw, bitmap[0]) != bw) return 3;
			if (printblock(l, code) != l) return 3;
			if (printblock(bw, bitmap[1]) != bw) return 3;
		}
        pl_trace(lines);
    }
    closeplane(ip);
    if ((r = endprint()) != 0) return r;
    return 0;
}

/* Command handler for PRINT.  Arguments, if any, are stored and printed
** after image.
*/
int printpic(int ac, argument *av)
{
    int r;

    if (new->planes == 0) return 7;
	if (ac > 1) pxoff = (int)av[1].fval;
	if (ac > 2) pyoff - (int)av[2].fval;
	if (ac > 3) pl_warn(1);

	for (ptype = 0; printers[ptype].name; ++ptype) {
		if (strcmp(printer, printers[ptype].name) == 0) {
	        pl_printf("Printing...\r\n");
    	    if ((outbuf = (U8 *)talloc(BUFSIZE+512)) == NULL) return 2;
			r = (*printers[ptype].printfunc)();
			break;
		}
	}
	if (printers[ptype].name == NULL) {
        pl_printf("Unrecognized printer type.\r\n");
        return 0;
	}
    if (r == 0) pl_printf(done);
    return r;
}
