/* PLTGA.C : Targa file handling routines for PICLAB.  The Targa file type
** is actually a collection of at least 50 different ways to store picture
** data that are all loosely related.  The lookup tables tgatbl1[] and
** tgatbl2[] are used to smooth out some of the differences in these formats.
**
** The third byte of the header contains a file type code, and the 17th byte
** contains the size in bits of each pixel entry.  These are the two indices
** into tgatbl1[].	The number at that position is a file type code used for
** the remainder of the code.  Tgatbl2[] is indexed by this code and contains
** a byte of flags laid out as follows:
**
**		Bits 7-3	Not used
**		Bit 2		True if format is monochrome
**		Bit 1		True if format is run-length encoded
**		Bit 0		True if format is color-mapped
**
** This module contains handlers for the commands TLOAD, TSAVE.
*/

#include <stdlib.h>
#include <memory.h>

#include "piclab.h"

static U8 tgatbl1[][4] = {
	{0},
	{	1,	2,	0,	0	},
	{	0,	3,	4,	5	},
	{	6,	0,	0,	0	},
	{0},{0},{0},{0},{0},
	{	7,	8,	0,	0	},
	{	0,	9, 10, 11	},
	{  12,	0,	0,	0	} };

static U8 tgatbl2[13] = { 0,
	0x01, 0x01, 0x00, 0x00, 0x00, 0x04,
	0x03, 0x03, 0x02, 0x02, 0x02, 0x06	};

static U8 c5to8bits[32] = {
    0,   8,  16,  24,  32,  41,  49,  57,
   65,  74,  82,  90,  98, 106, 115, 123,
  131, 139, 148, 156, 164, 172, 180, 189,
  197, 205, 213, 222, 230, 238, 246, 255 };

static int psize, bcount, btype;

/* Get one pixel from compressed TGA stream.
*/
static U32 cgetpixel(int tf)
{
	static U32 repeat;

	if (--bcount <= 0) {
		bcount = p_getc(tf);
		if (bcount & 0x80) {
			bcount = (bcount & 0x7F) + 1;
			btype = 1;
			p_getline(tf, psize);
			memcpy(&repeat, linebuf(tf), psize);
		} else {
			++bcount;
			btype = 0;
		}
	}
	if (btype == 0) {
		p_getline(tf, psize);
		memcpy(&repeat, linebuf(tf), psize);
	}
	return repeat;
}

/* Handler for TLOAD command.  Only argument is filename to load.  All of
** the flags are set by info in the file header.
*/
int loadtga(int ac, argument *av)
{
	char *path;
	U8 FAR *tbuf, *fbuf, *lbuf[3];
	int i, np, idsize, maptype, tfile, ftype, flipped,
		r, maplen, mapentsize, tflags;
	U16 line, h, w, x, cv;
	struct _plane *p[3];
	U32 pixelbuf;

	if (ac < 2) {
		pl_printf("Syntax is [T]LOAD <filename>\n");
		return 0;
	}
	if (ac > 2) pl_warn(1);

	histvalid = 0;
	if ((tbuf = (U8 *)talloc(BUFSIZE + 6144)) == NULL) return 2;
	path = makepath(picdir, av[1].cval, "TGA");
	pl_printf(reading, path);
	if ((tfile = p_open(path, tbuf, BUFSIZE, 6144, READ)) < 0) return 10;
	if (p_getline(tfile, 18) != 18) return 4;
	fbuf = linebuf(tfile);

	GET16(fbuf[12],w);
	GET16(fbuf[14],h);
	flipped = ((fbuf[17] & 0x20) == 0);
	idsize = fbuf[0];
	maptype = fbuf[1];
	GET16(fbuf[5],maplen);
	mapentsize = fbuf[7] >> 3;

	if (fbuf[16] == 15) ++fbuf[16];
	psize = fbuf[16] >> 3;

	if ((ftype = tgatbl1[fbuf[2]][psize-1]) == 0) return 6;
	tflags = tgatbl2[ftype];
	if (ftype > 6) bcount = btype = 0;
	if ((tflags & 0x04) || (tflags & 0x01)) np = 1; else np = 3;

	if ((r = begintransform()) != 0) return r;
	if (flipped) new->flags |= 1; else new->flags &= ~1;
	new->planes = (U8)np;
	if (tflags & 1) new->flags |= 2; else new->flags &= ~2;
	new->width = w;
	new->height = h;
	for (i=0; i<np; ++i) if ((p[i] = openplane(i, new, WRITE)) == NULL) return 3;

	if (idsize) {
		i = p_getline(tfile, idsize); /* skip ID field */
		if (i != idsize) return 4;
	}
	if (maptype && maplen > 0) {
		i = p_getline(tfile, (maplen * mapentsize));
		if (i != (maplen * mapentsize)) return 4;
		if ((tflags & 0x01) != 0) {
			fbuf = linebuf(tfile);
			for (i=0; i<maplen; ++i) {
				switch (mapentsize) {
					case 2:
						GET16(*fbuf,cv);
						fbuf += 2;
						gcmap[0][i] = c5to8bits[(cv & 0x7C00) >> 10];
						gcmap[1][i] = c5to8bits[(cv & 0x03E0) >> 5];
						gcmap[2][i] = c5to8bits[cv & 0x001F];
						break;
					case 3:
					case 4:
						gcmap[2][i] = *fbuf++;
						gcmap[1][i] = *fbuf++;
						gcmap[0][i] = *fbuf++;
						if (mapentsize == 4) ++fbuf;
						break;
					default: return 6;
				}
			}
		}
	}

	for (line = 0; line < h; ++line) {
		for (i=0; i<np; ++i) lbuf[i] = p[i]->linebuf;
		switch (ftype) {
			case 1:
			case 6:
				if (p_getline(tfile, w) != w) return 4;
				memcpy(lbuf[0], linebuf(tfile), w);
				break;
			case 2:
				if (p_getline(tfile, 2*w) != 2*w) return 4;
				fbuf = linebuf(tfile);
				for (x=0; x<w; ++x) {
					lbuf[0][x] = *fbuf++;
					++fbuf;
				}
				break;
			case 3:
				if (p_getline(tfile, 2*w) != 2*w) return 4;
				fbuf = linebuf(tfile);
				for (x=0; x<w; ++x) {
					GET16(*fbuf,cv);
					fbuf += 2;
					lbuf[0][x] = c5to8bits[(cv & 0x7C00) >> 10];
					lbuf[1][x] = c5to8bits[(cv & 0x03E0) >> 5];
					lbuf[2][x] = c5to8bits[cv & 0x001F];
				}
				break;
			case 4:
				if (p_getline(tfile, 3*w) != 3*w) return 4;
				fbuf = linebuf(tfile);
				for (x=0; x<w; ++x) {
					lbuf[2][x] = *fbuf++;
					lbuf[1][x] = *fbuf++;
					lbuf[0][x] = *fbuf++;
				}
				break;
			case 5:
				if (p_getline(tfile, 4*w) != 4*w) return 4;
				fbuf = linebuf(tfile);
				for (x=0; x<w; ++x) {
					lbuf[2][x] = *fbuf++;
					lbuf[1][x] = *fbuf++;
					lbuf[0][x] = *fbuf++;
					++fbuf; 				/* Skip overlay byte */
				}
				break;
			case 7:
			case 8:
			case 12:
				for (x=0; x<w; ++x) *lbuf[0]++ = (U8)cgetpixel(tfile);
				break;
			case 9:
				for (x=0; x<w; ++x) {
					pixelbuf = cgetpixel(tfile);
					lbuf[0][x] = (U8)((pixelbuf & 0x7C00) >> 7);
					lbuf[1][x] = (U8)((pixelbuf & 0x03E0) >> 2);
					lbuf[2][x] = (U8)((pixelbuf & 0x001F) << 3);
				}
				break;
			case 10:
			case 11:
				for (x=0; x<w; ++x) {
					pixelbuf = cgetpixel(tfile);
					lbuf[2][x] = (U8)pixelbuf;
					lbuf[1][x] = *((U8 *)&pixelbuf + 1);
					lbuf[0][x] = *((U8 *)&pixelbuf + 2);
				}
				break;
			default:	return 6;
		}
		for (i=0; i<np; ++i) if (putline(p[i]) == 0) return 3;
		pl_trace(line);
	}
	p_close(tfile);
	for (i=0; i<np; ++i) closeplane(p[i]);
	pl_printf(done);
	return 0;
}

/* Handler for TSAVE command.  First argument is filename to save; other
** arguments if present select the size of values output to file (16/24/32
** bits) or compression mode.
*/
int savetga(int ac, argument *av)
{
	struct _plane *p[3];
	int i, tfile, np, ts, compress;
	U16 line, x, w, word;
	U8 *fbuf, FAR *tbuf, *lbuf[3];
	char *path;

	if (transpend) return 8;
	if (new->planes == 0) return 7;
	ts = 24;
	compress = 0;
	if (ac > 4) pl_warn(1);
	if (ac < 2) {
		pl_printf("Syntax is [T]SAVE <filename> [<bits>] [COMPRESSED]\n");
		return 0;
	} else {
		path = makepath(picdir, (*++av).cval, "TGA");
		while (--ac > 1) {
			if (*(*++av).cval == 'C') compress = 1;
			else ts = (int)av->fval;
		}
	}
	if (compress) {
		pl_printf("Sorry, compressed Targa file saving is not implemented in version %s.\n", version);
		pl_printf("File will be stored uncompressed.\n");
	}

	pl_printf(writing, path);
	if ((tbuf = talloc(BUFSIZE + 6144)) == NULL) return 2;
	if ((tfile = p_open(path, tbuf, BUFSIZE, 6144, WRITE)) < 0) return 3;

	np = new->planes;
	for (i=0; i<np; ++i) if ((p[i] = openplane(i, new, READ)) == NULL) return 3;
	fbuf = linebuf(tfile);

	for (i=0; i<18; ++i) fbuf[i] = 0;
	if (np > 1) {
		fbuf[16] = (U8)ts;
		fbuf[2] = 2;
	} else if (new->flags & 2) {
		fbuf[16] = 8;
		fbuf[1] = fbuf[2] = 1;
		PUT16(fbuf[5], palette);
		fbuf[7] = 24;
	} else {
		fbuf[16] = 8;
		fbuf[2] = 3;
	}
	PUT16(fbuf[12],new->width);
	PUT16(fbuf[14],new->height);
	if (ts == 16) fbuf[17] |= 0x01;
	else if (ts == 32) fbuf[17] |= 0x08;
	if ((new->flags & 0x01) == 0) fbuf[17] |= 0x20;
	if (p_putline(tfile, 18) != 18) return 3;

	if (new->flags & 2) {
		fbuf = linebuf(tfile);
		for (i=0; i<palette; ++i) {
			*fbuf++ = gcmap[2][i];
			*fbuf++ = gcmap[1][i];
			*fbuf++ = gcmap[0][i];
		}
		if (p_putline(tfile, 3*palette) != 3*palette) return 3;
	}

	w = new->width;
	for (line = 0; line < new->height; ++line) {
		fbuf = linebuf(tfile);
		for (i=0; i<np; ++i) {
			if (getline(p[i]) == 0) return 4;
			lbuf[i] = p[i]->linebuf;
		}
		if (np == 3) {
			if (ts == 16) {
				for (x=0; x<w; ++x) {
					word = ((lbuf[2][x] >> 3) & 0x001F) +
						   (((U16)lbuf[1][x] << 2) & 0x03E0) +
						   (((U16)lbuf[0][x] << 7) & 0x7C00);
					*fbuf++ = (U8)(word & 0xFF);
					*fbuf++ = (U8)(word >> 8);
				}
				if (p_putline(tfile, 2*w) != 2*w) return 3;
			} else {
				for (x=0; x<w; ++x) {
					*fbuf++ = lbuf[2][x];
					*fbuf++ = lbuf[1][x];
					*fbuf++ = lbuf[0][x];
					if (ts == 32) *fbuf++ = 0;
				}
				if (ts == 24) {
					if (p_putline(tfile, 3*w) != 3*w) return 3;
				} else if (p_putline(tfile, 4*w) != 4*w) return 3;
			}
		} else {
			memcpy(fbuf, lbuf[0], w);
			if (p_putline(tfile, w) != w) return 3;
		}
		pl_trace(line);
	}
	for (i=0; i<np; ++i) closeplane(p[i]);
	pl_printf(done);
	p_close(tfile);

	return 0;
}
