// Routines to initialize math tables for the integer math library
// All code by Dave Stampe, last updated 23/12/93

/*
 This code is part of the VR-386 project, created by Dave Stampe.
 VR-386 is a desendent of REND386, created by Dave Stampe and
 Bernie Roehl.  Almost all the code has been rewritten by Dave
 Stampre for VR-386.

 Copyright (c) 1994 by Dave Stampe:
 May be freely used to write software for release into the public domain
 or for educational use; all commercial endeavours MUST contact Dave Stampe
 (dstampe@psych.toronto.edu) for permission to incorporate any part of
 this software or source code into their products!  Usually there is no
 charge for under 50-100 items for low-cost or shareware products, and terms
 are reasonable.  Any royalties are used for development, so equipment is
 often acceptable payment.

 ATTRIBUTION:  If you use any part of this source code or the libraries
 in your projects, you must give attribution to VR-386 and Dave Stampe,
 and any other authors in your documentation, source code, and at startup
 of your program.  Let's keep the freeware ball rolling!

 DEVELOPMENT: VR-386 is a effort to develop the process started by
 REND386, improving programmer access by rewriting the code and supplying
 a standard API.  If you write improvements, add new functions rather
 than rewriting current functions.  This will make it possible to
 include you improved code in the next API release.  YOU can help advance
 VR-386.  Comments on the API are welcome.

 CONTACT: dstampe@psych.toronto.edu
*/

#include <stdlib.h>
#include <alloc.h>
#include <math.h>
#include <stdio.h>

#include "intmath.h"


/* this stuff moved out of integer core so that there are
   no uncompilable TC references in integer core library */

#define XFSC 536870912   /* 2**29 for shifting float to <3.29> fixed format */

static float xfsc = XFSC;

// long sintable[258];    /* what these contain: for fast trig */
// long atantable[258];


#define SINESIZE 258
#define ATANSIZE 258

long *sintable  = NULL;
long *atantable = NULL;

static int fill_sine()
{
  int i;

  if(sintable) return 0;
  sintable = malloc(SINESIZE * sizeof(long));
  if(sintable==NULL) return 1;

  if(atantable) return 0;
  atantable = malloc(ATANSIZE * sizeof(long));
  if(atantable==NULL) return 1;

  for (i = 0; i < 256; i++)
	sintable[i] = (XFSC * sin(3.14159/2/256 * i));
  sintable[256] = XFSC;
  sintable[257] = XFSC;

  for (i = 0; i < 256; i++)
	atantable[i] = 180.0/3.14159*65536.0 * atan(i/256.0);
  atantable[256] = atantable[257] = 45*65536L;

  return 0;
}


/* tables for sphere object clipping:   		 */
/* these are needed for REND386's renderer pipeline only */
/* tou can leave this out if using math elsewhere 	 */

/* zoom range: FOV = 2*atan(1/zoom)          */
/* or about 150 to 7 degrees 		     */

// long sclip_C[800]; /* C = 1/sqrt(zoom^2 + 1)
// long sclip_M[800]; /* M = zoom * C

#define CLIPSIZE 800

long *sclip_C = NULL;
long *sclip_M = NULL;

static fill_sclip()
{
  int i;
  float n;

  if(sclip_C) return 0;
  sclip_C = malloc(CLIPSIZE * sizeof(long));
  if(sclip_C==NULL) return 1;

  if(sclip_M) return 0;
  sclip_M = malloc(CLIPSIZE * sizeof(long));
  if(sclip_M==NULL) return 1;

  for (i = 0; i < 800; i++)
    {
      n = 1.0/sqrt((i/16.0)*(i/16.0) + 1);
      sclip_C[i] = XFSC * n;
      sclip_M[i] = XFSC * ((i/16.0) * n) ;
    }
  return 0;
}

//  int sqrtable[1024];		// table for fast 16-bit sqrt lookup
				// expects index to be normalized
#define SQRTSIZE 1024

int *sqrtable = NULL;

static fill_sqrt()
{
  int i;

  if(sqrtable) return 0;
  sqrtable = malloc(SQRTSIZE * sizeof(int));
  if(sqrtable==NULL) return 1;

  for (i = 0; i < 1024; i++)
	sqrtable[i] = 1024*sqrt(i);
  return 0;
}


static void free_math()
{
  if (sqrtable)  free(sqrtable);

  if (sclip_C)   free(sclip_C);  // these two for REND386 only
  if (sclip_M)   free(sclip_M);  //

  if (sintable)  free(sintable);
  if (atantable) free(atantable);
}


int init_math()		// returns 0 if OK, else out of memory
{
 atexit(free_math);
 if(fill_sqrt()) return 1;
 if(fill_sine()) return 1;
 if(fill_sclip()) return 1;   // comment out if not using for REND386
 return 0;
}
