//* Reads a Configuration file
// Written by Dave Stampe, 9/1/94
// Relies heavily on the code in wparse.c


/*
 This code is part of the VR-386 project, created by Dave Stampe.
 VR-386 is a desendent of REND386, created by Dave Stampe and
 Bernie Roehl.  Almost all the code has been rewritten by Dave
 Stampre for VR-386.

 Copyright (c) 1994 by Dave Stampe:
 May be freely used to write software for release into the public domain
 or for educational use; all commercial endeavours MUST contact Dave Stampe
 (dstampe@psych.toronto.edu) for permission to incorporate any part of
 this software or source code into their products!  Usually there is no
 charge for under 50-100 items for low-cost or shareware products, and terms
 are reasonable.  Any royalties are used for development, so equipment is
 often acceptable payment.

 ATTRIBUTION:  If you use any part of this source code or the libraries
 in your projects, you must give attribution to VR-386 and Dave Stampe,
 and any other authors in your documentation, source code, and at startup
 of your program.  Let's keep the freeware ball rolling!

 DEVELOPMENT: VR-386 is a effort to develop the process started by
 REND386, improving programmer access by rewriting the code and supplying
 a standard API.  If you write improvements, add new functions rather
 than rewriting current functions.  This will make it possible to
 include you improved code in the next API release.  YOU can help advance
 VR-386.  Comments on the API are welcome.

 CONTACT: dstampe@psych.toronto.edu
*/


#include <stdio.h>
#include <string.h>
#include <stdlib.h>
#include <alloc.h>
#include <dos.h>       /* delay() */

#include "vr_api.h"
#include "intmath.h"
#include "pcdevice.h"
#include "wparse.h"

extern int do_screen_clear;
extern STEREO default_stereo;
extern unsigned char palette[];
extern int npalette;

extern char *fix_fname(char *fname);

extern char loadpath[];

#define match(a,b)  !strnicmp(a, b, strlen(b))

static char ps[] = " \t\n,#";
static char st[] = "%s";

/************* READ A CFG FILE *********/


// LIST OF COMMANDS

static char *cfgcommands[] = {
	"loadpath", "palette", "skycolor", "groundcolor", "screencolor",
	"worldscale", "key", "control", "viewframe", "hither", "yon",
	"eyespacing", "screendist", "include", "screenwidth",
	"convergence", "stepsize", "options", "videodev",
	"mousedev", "headdev", "glovedev", "ptrdev",
	"switchdev", "glovecursor", "ptrcursor", "segaport", "switchport",
	"pgloveport", "pglovetime", "stereoset", "stereotype",
	"stereoleft", "stereoright",
	NULL
};

enum com_codes {
	c_loadpath = 0, c_palette, c_skycolor, c_groundcolor, c_screencolor,
	c_worldscale, c_key, c_control, c_viewframe, c_hither, c_yon,
	c_eyespacing, c_screendist, c_include, c_screenwidth,
	c_convergence, c_stepsize, c_options, c_videodev,
	c_mousedev, c_headdev, c_glovedev, c_ptrdev,
	c_switchdev, c_glovecursor, c_ptrcursor, c_segaport, c_switchport,
	c_pgloveport, c_pglovetime, c_stereoset, c_stereotype,
	c_stereoleft, c_stereoright
};


void read_config_file(FILE *in)
{
  char obuff[256];
  char inbuff[256], *buff, fname[100];
  char *args;
  char *pname;
  int i,cmd;

  while (fgets(inbuff, sizeof(inbuff), in))
    {
      strcpy(obuff,inbuff);
      buff = strtok(inbuff,"#");
      for (buff = inbuff; isspace(*buff); ++buff);
      if(buff[0]==0) continue;
      for (args = buff+1; isalpha(*args); ++args);
      args++;
      buff = strtok(buff," \t\n");

      for (cmd = 0; cfgcommands[cmd]; cmd++)
	if (!stricmp(cfgcommands[cmd], buff)) break;
	  if (cfgcommands[cmd] == NULL) continue;

      switch(cmd)
	{
	   case c_loadpath:
	       sscanf(args, st, loadpath);
	       break;

	   case c_videodev:
	     {
	       extern char vdname[];
	       extern int vdmode;

	       sscanf(args, "%s %x", vdname, &vdmode);
	     } 	break;

	   case c_mousedev:
	     {
	       extern char mdname[];

	       sscanf(args, st, mdname);
	     } break;

	   case c_headdev:
	     {
	       extern char hdname[];
	       extern POSE hd_offset;
	       float x,y,z,rx,ry,rz;

	       sscanf(args, "%s %f %f %f %f %f %f", hdname, &x, &y, &z, &rx, &ry, &rz);
	       hd_offset.x = x;
	       hd_offset.y = y;
	       hd_offset.z = z;
	       hd_offset.rx = float2angle(rx);
	       hd_offset.ry = float2angle(ry);
	       hd_offset.rz = float2angle(rz);
	     }
		break;

	   case c_glovedev:
	     {
	       extern char gpdname[];
	       extern int have_glove;
	       extern POSE ptr_scale;
	       float x,y,z,rx,ry,rz;

	       have_glove = 1;
	       sscanf(args, "%s %f %f %f %f %f %f", gpdname, &x, &y, &z, &rx, &ry, &rz);
	       ptr_scale.x = float2scale(x);
	       ptr_scale.y = float2scale(y);
	       ptr_scale.z = float2scale(z);
	       ptr_scale.rx = float2scale(rx);
	       ptr_scale.ry = float2scale(ry);
	       ptr_scale.rz = float2scale(rz);
	     }	break;

	   case c_ptrdev:
	     {
	       extern char gpdname[];
	       extern int have_ptr;
	       extern POSE ptr_scale;
	       float x,y,z,rx,ry,rz;

	       have_ptr = 1;
	       sscanf(args, "%s %f %f %f %f %f %f", gpdname, &x, &y, &z, &rx, &ry, &rz);
	       ptr_scale.x = float2scale(x);
	       ptr_scale.y = float2scale(y);
	       ptr_scale.z = float2scale(z);
	       ptr_scale.rx = float2scale(rx);
	       ptr_scale.ry = float2scale(ry);
	       ptr_scale.rz = float2scale(rz);
	     } 	break;

	   case c_switchdev:
	     {
	       extern char swdname[];

	       sscanf(args, st, swdname);
	     }	break;

	   case c_glovecursor:
	     {
	       extern char gpcursor[];
	       sscanf(args, st, gpcursor);
	     }break;

	   case c_ptrcursor:
	     {
	       extern char gpcursor[];
	       sscanf(args, st, gpcursor);
	     } break;

	   /* CONFIG: DONE IMMEDIATELY */

	   case c_switchport:  /* synonym with no trademark problems */
	   case c_segaport:
	     {
	       sscanf(args, "%x %x %x %x %x", &sega_address,
			&sega_mask, &sega_left, &sega_right, &sega_doff, &sega_port_image);
	     }	break;

	   case c_pgloveport:
	     {
	       sscanf(args, "%x %x %x %x %x %x %x %x", &glove_in_port, &glove_out_port,
			&glove_write_mask, &glove_none_mask, &glove_latch_mask, &glove_clock_mask,
			&glove_clock_latch, &glove_data_mask, &port_image );
	     }	break;

	   case c_pglovetime:
	     {
	       sscanf(args, "%d %d", &glove_bit_delay, &glove_byte_delay );
	     }	break;

	   case c_stereoset:
	     {
	       float ws = 1.0;
	       if(!stereo_type) break;
	       sscanf(args, "%ld %ld %ld %ld %ld %f", &(default_stereo.phys_screen_dist),
			&(default_stereo.phys_screen_width), &(default_stereo.pixel_width),
			&(default_stereo.phys_eye_spacing), &(default_stereo.phys_convergence), &ws);
	       default_stereo.world_scaling = 65536.0 * ws;
	     } 	break;

	   case c_stereotype:
	     {
	       char sty[80];

	       sscanf(args, st, sty);
	       if (match(sty, "OFF")) stereo_type = MONOSCOPIC;
	       else if (match(sty, "SWITCH")) stereo_type = SWITCHED;
	       else if (match(sty, "SEPARATE")) stereo_type = SEPARATE;
	       else if (match(sty, "SPLIT")) stereo_type = SPLITLR;
	     }	break;

	   case c_stereoleft:
	     {
	       float ry;
	       if(!stereo_type) break;
#define STWL &default_stereo.window[LEFT_EYE]
	       sscanf(args, "%d %d %f %ld %ld %ld %ld %ld", STWL.xoff, STWL.orientation,
			&ry, STWL.l, STWL.t, STWL.r, STWL.b );
	       *STWL.ry = float2angle(ry);
	     } break;

	   case c_stereoright:
	     {
	       float ry;
	       if(!stereo_type) break;
#define STWR &default_stereo.window[RIGHT_EYE]
	       sscanf(args, "%d %d %f %ld %ld %ld %ld %ld", STWR.xoff, STWR.orientation,
			&ry, STWR.l, STWR.t, STWR.r, STWR.b );
	       *STWR.ry = float2angle(ry);
	     }break;

	   case c_include:
	     {
	       char fname[100];
	       FILE *in;
	       char *f;
	       strcpy(fname,strtok(NULL,ps));
	       if ((in = fopen(f = fix_fname(fname), "r")) != NULL)
		 {
		   read_config_file(in);
		   fclose(in);
		 }
	       else
		 errprintf("Could not open '%s'as include file", f);
	     }break;

	   case c_hither:
	     {
	      COORD h = atol(strtok(NULL,ps));
	      if(h<1) h=1;
	      set_camera_hither(default_camera,h);
	      break;
	     }
	   case c_yon:
	     {
	      COORD y = atol(strtok(NULL,ps));
	      if(y>530000000L) y=530000000L;
	      set_camera_yon(default_camera,y);
	      break;
	     }
	   case c_eyespacing:
	       default_stereo.phys_eye_spacing = atol(strtok(NULL,ps));
	       break;
	   case c_screendist:
	       default_stereo.phys_screen_dist = atol(strtok(NULL,ps));
	       break;
	   case c_screenwidth:
	       default_stereo.phys_screen_width = atol(strtok(NULL,ps));
	       default_stereo.pixel_width = atoi(strtok(NULL,ps));
	       break;
	   case c_convergence:
	       default_stereo.phys_convergence = atol(strtok(NULL,ps));
	       break;
	 }
     }
  return;
}

