// RENDERER CORE INCLUDES AND DEFINES
// 26/12/93 by Dave Stampe
// All algorithms and code (c) 1993, 1994 by Dave Stampe

/*
 This code is part of the VR-386 project, created by Dave Stampe.
 VR-386 is a desendent of REND386, created by Dave Stampe and
 Bernie Roehl.  Almost all the code has been rewritten by Dave
 Stampre for VR-386.

 Copyright (c) 1994 by Dave Stampe:
 May be freely used to write software for release into the public domain
 or for educational use; all commercial endeavours MUST contact Dave Stampe
 (dstampe@psych.toronto.edu) for permission to incorporate any part of
 this software or source code into their products!  Usually there is no
 charge for under 50-100 items for low-cost or shareware products, and terms
 are reasonable.  Any royalties are used for development, so equipment is
 often acceptable payment.

 ATTRIBUTION:  If you use any part of this source code or the libraries
 in your projects, you must give attribution to VR-386 and Dave Stampe,
 and any other authors in your documentation, source code, and at startup
 of your program.  Let's keep the freeware ball rolling!

 DEVELOPMENT: VR-386 is a effort to develop the process started by
 REND386, improving programmer access by rewriting the code and supplying
 a standard API.  If you write improvements, add new functions rather
 than rewriting current functions.  This will make it possible to
 include you improved code in the next API release.  YOU can help advance
 VR-386.  Comments on the API are welcome.

 CONTACT: dstampe@psych.toronto.edu
*/



#include "vr_ctypes.h"

// This file is the best documentation for the renderer
// and how to use it.  To use the renderer, you will need
// graphics routines as defined in F3DKITD.H, a stub for
// accessptr(), the INTMATH library, and some user callback
// routines described below. These are: (read more below):

// accessptr(void *) : allows MAP-EMM system to make memory accessible

// void user_render_poly(WORD vertex_count, WORD *pcoords,
//			 SURFACE poly_color, COORD max_depth);

// unsigned user_poly_color(POLY *p, unsigned pcolor,
//			unsigned nppoints, long maxz);

// void user_setup_blitter();
// void user_reset_blitter();

// if in doubt, look through RENDCODE.C



/************* VIEWPORT CONTROL ************/


	/* fetches the viewpoint position, even if set by matrix */
extern void real_viewpoint(VIEW *v, long *x, long *y, long *z);

	/* compute eye point/ angle movement factors only     */
	/* uses the pan/tilt/roll/x/y/z data in viewport      */
	/* use if view position has changed, but nothing else */
	/* still need to call render_set_view afterwards      */
extern void fast_view_factors(VIEW *v);

	/* 100% compatible with original REND386 viewport call */
	/* don't use for new code  (bit of a waste of time)    */
extern void compute_view_factors(VIEW *v);

	// sets the view point/angle from a matrix.  This
	// is the only way to go for real VR work!
	// still need to call render_set_view afterwards
extern void matrix_view_factors(VIEW *v,MATRIX m);

	// extract a matrix from the view.  Should not be needed
	// if you're just using matrices to move viewpoint, but
	// might be useful if the viewport is on the end of a lot
	// of transforms (i.e stereoscopic)
extern void view_to_matrix(VIEW *v,MATRIX m);

/************ MEMORY AND RENDERER SETUP ************/

#define MAXVERTICES 20	// polys should have no more than 20 vertices

extern void *vtxram;          /* memory allocation area start */

	// releases renderer resources.  Already done at exit
extern void reset_render();

	// initializes renderer, also initializes integer math library
	// allocates <mem> K of memory: up to 63K for poly and vertex
	// working copies, which limits numbers visible on screen.
	// <polys> determines how many polys or objects can be seen:
	// use about 3000
	// between renders, this memory can be re-used for trmporaries
	// by use of vtxram.  Be careful!
extern void *setup_render(unsigned mem, int polys);


/******** DRAWING CLIPPED POLYS ***********/

	// Some routines (i.e the horizon drawing) need a way to
	// draw XY clipped polys through the renderer.  This routine
	// accepts arrays of X and Y screen coordinates (+/-32000 range)
	// then clips and draws them immediately.  No depth sorting, and
	// the color is passed directly to user_render_poly
extern void render_ext_poly(int npoints, int vx[20], int vy[20], unsigned color);

/********** SCREEN-POINT MONITOR ***********/

// these routines control/read a monitor which looks at the polys
// going out during rendering to see where in the world a screen point is.
// because the renderer can be called many times to draw a world divided
// by splits, the monitor records across render calls.

	// turns the monitor on, and clears it
	// sets the screen point to be tested
extern void set_screen_monitor(int x, int y);

	// turns off the monitor.
	// data may still be read out later.
extern void clear_screen_monitor();

	// returns the object that was visible at the point
	// returns NULL if no object was found
extern OBJECT *screen_monitor_object();

	// returns pointer to poly that was visible at the point
	// returns NULL if no poly was found.
extern POLY *screen_monitor_poly();

	// returns vertex number of poly that was closest (within 50 pixels)
	// of the test point.  This is NOT the vertex number in the object!
	// if no vertex was close enough, -1 is returned
extern int screen_monitor_vertex();


/************* DRAWING POLYS *************/

	// YOU must supply this routine, to draw the polys given
	// the vertices and color.  Simple, really.  The poly_color was
	// supplied by user_poly_color(), another of your routines.
	// So these two can do a lot of work together.
extern void user_render_poly(WORD vertex_count, WORD *pcoords,
			     SURFACE poly_color, COORD max_depth);

/*********** COLOR AND LIGHTING SUPPORT ***********/

	// YOU must supply this routine!
	// the renderer will call this routine to get the color
	// the poly is to be drawn in.  You have the poly (to be passed
	// to compute_poly_cosine() if desired), the number of points
	// (1 if dot, 2 if line, >2 if poly) and the maximum depth of
	// the poly (for depth lighting).
	// You are given the poly's color field in <pcolor>, which
	// you must translate into a color that your poly drawing
	// routine user_render_poly() can handle.  See COLORMAP.C
	// for an example.  These callbacks make it easier to upgrade
	// the renderer to do new things in the future.
extern unsigned user_poly_color(POLY *p, unsigned pcolor,
				unsigned nppoints, long maxz);

	// given poly, computes its cosine*127 from its first
	// vertex to the given (light) point.  Used for lighting,
	// may be used by the color callback routine above.
	// if vect is set, the <xyz> point is a directional vector
	// of the light (spotlight), else it is the position of
	// the light source itself
extern int compute_poly_cosine(POLY *p, long x, long y, long z, int vect);


/******* OBJECT REPRESENTATION UPDATE CALLBACK ********/

	// sets up routine to be called when an object's
	// representation is to be displayed that needs updating
	// usually used to move representations to match object
	// called when the object's and representation's update_count field
	// do not match.  This could happen if the object is moved, which
	// increments its update_count.  Then when the object is drawn,
	// the representation must be moved to match.  You can increment
	// the representation's count too, or leave it to be called every
	// time the object is visible.
extern void *set_renderer_update_handler(void (*handler)(OBJECT *o) );



/*************** OBJECT LIST RENDERING *************/
// at last! something high-level!

	// these two are YOUR routines, called before and after a burst
	// of rendering in case you need to set up/reset video hardware.
	// VGA drivers should call setup_hdwe(0) and reset_hdwe() for
	// these respectively.
extern void user_setup_blitter();
extern void user_reset_blitter();

	// render an objectlist.  The renderer will call
	// user_setup_blitter(), then process and sort the
	// objects.  Finally, the polys will be blasted out to
	// user_render_poly() to be drawn.
	// You must call render_set_view before this, but you
	// can draw bunches of object lists with repeated calls
	// (say, for a world divided by splits)
extern void subrender(OBJLIST *objlist);

	// this is a little more expensive, but calls the
	// wiewport setup itself.  Good if all you'll draw
	// is one objectlist.  The code for it is shown below:
extern void render(OBJLIST *objlist, VIEW *view);

//  void render(OBJLIST *objlist, VIEW *view)
//    {
//      render_set_view(view);
//      subrender(objlist);
//    }

/************* VIEWPORT SETUP AND INITIALIZE *********/

// viewport must have been processed by
// initialize_screen_factors() in the past,
// and then viewpoint can be changed with
// fast_view_factors() or matrix_view_factors

	/* copy viewport data to fast access area */
	/* to prepare renderer to draw a view     */
	/* do after viewpoint or look angle has changed */
extern void render_set_view(VIEW *v);

	/* compute screen and viewport factors.  These stay constant   */
	/* over eye point changes.  Call this to initialize a viewport */
	/* and after the view parameters such as zoom or stereo change */
extern void initialize_screen_factors(VIEW *v);



