/*
   The following code is one of my attempts at improving PoV's supersampling
   function. The supersample function starts by dividing the pixels to be
   supersampled into 4 sub-regions and then traces a ray in each region. If
   the colour difference between any two sub-pixels is greater than the
   anti-alias threshold the offending sub-pixels are further broken up into
   4 more regions and so on. Where possible, the test rays are reused when
   sub-dividing the pixel. MAX_SSLEVEL determines the maximum number of times
   that a pixel will be sub-divided. The maximum number of rays that will be
   traced per supersampled pixel is 4^MAX_SSLEVEL. JITTER determines whether
   the rays will be randomly jittered.

						       Steve Anger
*/


void Sub_Sample (COLOUR *result, DBL x, DBL y, DBL wx, DBL wy,
		 COLOUR *lastcol, DBL lastx, DBL lasty, int sslevel);

#define MAX_SSLEVEL 2     /* 2 or 3 usually work well */
#define JITTER 1          /* 1 = jittered, 0 = not jittered */

void Supersample (result, x, y, Width, Height)
    COLOUR *result;
    int x, y, Width, Height;
{
    COLOUR dummy;

    Make_Colour (&dummy, 0.0, 0.0, 0.0);
    Number_Of_Pixels_Supersampled++;
    Sub_Sample (result, (DBL)x, (DBL)y, 1.0, 1.0, &dummy, (DBL)x, (DBL)y, 0);
}


void Sub_Sample (result, x, y, wx, wy, lastcol, lastx, lasty, sslevel)
    COLOUR  *result, *lastcol;
    DBL     x, y, wx, wy, lastx, lasty;
    int     sslevel;
{
    COLOUR col[4];
    DBL    sx[4], sy[4], jx[4], jy[4], jit_x, jit_y;
    int    refine[4], quad, i;

    /* Calculate the centers of the four new sub-pixels */
    sx[0] = x - wx/4.0;   sy[0] = y + wy/4.0;
    sx[1] = x + wx/4.0;   sy[1] = y + wy/4.0;
    sx[2] = x - wx/4.0;   sy[2] = y - wy/4.0;
    sx[3] = x + wx/4.0;   sy[3] = y - wy/4.0;

    /* Determine which sub-pixel the test ray hit */
    if (lastx == x && lasty == y)
	quad == -1;
    else {
	quad =  (lastx < x) ? 0 : 1;
	quad += (lasty < y) ? 2 : 0;
    }

    for (i = 0; i < 4; i++) {
	if (i == quad) {
            /* Re-use the test ray for this sub-pixel */
	    Make_Colour (&col[i], lastcol->Red, lastcol->Green, lastcol->Blue);

            jx[i] = lastx;
            jy[i] = lasty;
	}
	else {
#if JITTER
	    jit_x = (rand3d((int)(100.0*sx[i]), (int)(100.0*sy[i]))
                     & 0x7FFF) / 32768.0 * (wx/4.0) - (wx/8.0);
	    jit_y = (rand3d((int)(100.0*sx[i]) + 10, (int)(100.0*sy[i]) + 10)
                     & 0x7FFF) / 32768.0 * (wy/4.0) - (wy/8.0);
#else
	    jit_x = 0.0;
	    jit_y = 0.0;
#endif

	    jx[i] = sx[i] + jit_x;
	    jy[i] = sy[i] + jit_y;

	    Create_Ray (VP_Ray, Frame.Screen_Width, Frame.Screen_Height,
			jx[i], jy[i]);

	    Trace_Level = 0;
	    Trace (VP_Ray, &col[i]);
	    Clip_Colour (&col[i], &col[i]);
	}

	refine[i] = FALSE;
    }

    /* Find the pixels that need to be refined */
    if (sslevel < MAX_SSLEVEL-1) {
	if (Colour_Distance (&col[0], &col[1]) >= Frame.Antialias_Threshold) {
	    refine[0] = TRUE;
	    refine[1] = TRUE;
	}

	if (Colour_Distance (&col[2], &col[3]) >= Frame.Antialias_Threshold) {
	    refine[2] = TRUE;
	    refine[3] = TRUE;
	}

	if (Colour_Distance (&col[0], &col[2]) >= Frame.Antialias_Threshold) {
	    refine[0] = TRUE;
	    refine[2] = TRUE;
	}

	if (Colour_Distance (&col[1], &col[3]) >= Frame.Antialias_Threshold) {
	    refine[1] = TRUE;
	    refine[3] = TRUE;
	}
    }

    Make_Colour (result, 0.0, 0.0, 0.0);

    for (i = 0; i < 4; i++) {
	if (refine[i]) {
	    Sub_Sample (&col[i], sx[i], sy[i], wx/2.0, wy/2.0,
                        &col[i], jx[i], jy[i], sslevel+1);
        }

	Add_Colour (result, result, &col[i]);
    }

    Scale_Colour (result, result, 0.25);
}

